#!/usr/bin/env python

from mapping import *

######################################################################################
#   file   : transpose.py  (for the transpose application)
#   date   : may 2014
#   author : Alain Greiner
#######################################################################################
#  This file describes the mapping of the multi-threaded "transpose" 
#  application on a multi-clusters, multi-processors architecture.
#  This include both the mapping of virtual segments on the clusters,
#  and the mapping of tasks on processors.
#  This mapping uses 5 platform parameters, (obtained from the "mapping" argument)
#  - x_size    : number of clusters in a row
#  - y_size    : number of clusters in a column
#  - x_width   : number of bits coding x coordinate
#  - y_width   : number of bits coding y coordinate
#  - procs_max : number of processors per cluster
####################################################################################

#########################
def transpose( mapping ):

    x_size    = mapping.x_size
    y_size    = mapping.y_size
    procs_max = mapping.procs_max
    x_width   = mapping.x_width
    y_width   = mapping.y_width

    # define vsegs base & size
    code_base  = 0x10000000
    code_size  = 0x00010000     # 64 Kbytes (replicated in each cluster)
    
    data_base  = 0x20000000
    data_size  = 0x00010000     # 64 Kbytes (non replicated)

    ptab_base  = 0x30000000
    ptab_size  = 0x00040000     # 256 Kbytes (replicated in each cluster)

    stack_base = 0x40000000 
    stack_size = 0x00100000     # 1 Mbytes (to be divided between all tasks)

    heap_base  = 0x50000000 
    heap_size  = 0x00010000     # 64 Kbytes (to be shared by all tasks)

    # create vspace
    vspace = mapping.addVspace( name = 'transpose', startname = 'trsp_data' )
    
    # data vseg : shared (only in cluster[0,0])
    mapping.addVseg( vspace, 'trsp_data', data_base , data_size, 
                     'C_WU', vtype = 'ELF', x = 0, y = 0, pseg = 'RAM', 
                     binpath = 'build/transpose/transpose.elf',
                     local = False )

    # code vsegs : local (one copy in each cluster)
    for x in xrange (x_size):
        for y in xrange (y_size):
            mapping.addVseg( vspace, 'trsp_code_%d_%d' %(x,y), code_base , code_size,
                             'CXWU', vtype = 'ELF', x = x, y = y, pseg = 'RAM', 
                             binpath = 'build/transpose/transpose.elf',
                             local = True )

    # ptab vsegs : local (one specific ptab per cluster)
    for x in xrange (x_size):
        for y in xrange (y_size):
            mapping.addVseg( vspace, 'trsp_ptab_%d_%d' %(x,y), ptab_base , ptab_size,
                            'C_WU', vtype = 'PTAB', x = x, y = y, pseg = 'RAM', 
                            align = 13,
                            local = True )

    # stacks vsegs: local (one stack per processor, procs_max stacks per cluster)            
    for x in xrange (x_size):
        for y in xrange (y_size):
            for p in xrange( procs_max ):
                proc_id = (((x * y_size) + y) * procs_max) + p
                size    = stack_size / (x_size * y_size * procs_max)
                base    = stack_base + (proc_id * size)
                mapping.addVseg( vspace, 'trsp_stack_%d_%d_%d' % (x,y,p), base, size,
                                 'C_WU', vtype = 'BUFFER', x = x , y = y , pseg = 'RAM',
                                 local = True )

    # heap vsegs: shared (all heap segments can be accessed by all tasks)
    for x in xrange (x_size):
        for y in xrange (y_size):
            cluster_id = (x * y_size) + y
            size  = heap_size / (x_size * y_size)
            base  = heap_base + (cluster_id * size)
            mapping.addVseg( vspace, 'trsp_heap_%d_%d' % (x,y), base, size, 
                             'C_WU', vtype = 'BUFFER', x = x, y = y, pseg = 'RAM',
                             local = False )

    # distributed tasks / one task per processor
    for x in xrange (x_size):
        for y in xrange (y_size):
            for p in xrange( procs_max ):
                trdid = (((x * y_size) + y) * procs_max) + p
                mapping.addTask( vspace, 'trsp_%d_%d_%d' % (x,y,p), trdid, x, y, p,
                                 'trsp_stack_%d_%d_%d' % (x,y,p),
                                 'trsp_heap_%d_%d' % (x,y), 0 )

    # extend mapping name
    mapping.name += '_transpose'

    return vspace  # useful for test
            
################################ test ######################################################

if __name__ == '__main__':

    vspace = transpose( Mapping( 'test', 2, 2, 4 ) )
    print vspace.xml()


# Local Variables:
# tab-width: 4;
# c-basic-offset: 4;
# c-file-offsets:((innamespace . 0)(inline-open . 0));
# indent-tabs-mode: nil;
# End:
#
# vim: filetype=python:expandtab:shiftwidth=4:tabstop=4:softtabstop=4

