///////////////////////////////////////////////////////////////////////////////////
// File      : rdk_driver.c
// Date      : 13/02/2014
// Author    : alain greiner
// Maintainer: cesar fuguet
// Copyright (c) UPMC-LIP6
///////////////////////////////////////////////////////////////////////////////////
// The rdk_driver.c and rdk_driver.h files are part ot the GIET-VM kernel.
//
// This driver supports a virtual disk implemented as a memory segment, 
// in the address space. 
//
// The _rdk_read() and _rdk_write() blocking functions use a software memcpy,
// whatever the selected mode. They return only when the transfer is completed.
// As the number of concurrent accesses is not bounded, these functions
// don't use the _ioc_lock variable.
//
// The SEG_RDK_BASE virtual address must be defined in the hard_config.h
// file when the USE_RAMDISK flag is set.
///////////////////////////////////////////////////////////////////////////////////

#include <giet_config.h>
#include <hard_config.h>
#include <rdk_driver.h>
#include <utils.h>
#include <tty_driver.h>
#include <ctx_handler.h>

#if !defined(SEG_RDK_BASE) 
# error: You must define SEG_RDK_BASE in the hard_config.h file
#endif

///////////////////////////////////////////////////////////////////////////////
//       _rdk_init()
// This function does nothing, and return 0 for success.
///////////////////////////////////////////////////////////////////////////////
unsigned int _rdk_init( unsigned int channel )
{
    return 0;
}

///////////////////////////////////////////////////////////////////////////////
//     _rdk_read()
// Transfer data from the RAMDISK to a memory buffer. 
// - mode     : BOOT / KERNEL / USER
// - lba      : first block index on the block device
// - buffer   : virtual base address of the memory buffer
// - count    : number of blocks to be transfered.
// Returns 0 if success, > 0 if error.
///////////////////////////////////////////////////////////////////////////////
unsigned int _rdk_read( unsigned int lba, 
                        unsigned int buffer, 
                        unsigned int count) 
{

#if GIET_DEBUG_IOC_DRIVER
_printf("\n[IOC DEBUG] Enter _rdk_read() at cycle %d"
        "\n - vaddr   = %x"
        "\n - sectors = %d"
        "\n - lba     = %x",
        _get_proctime(), buffer, count, lba );
#endif

#if USE_IOC_RDK
    char* src = (char*)SEG_RDK_BASE + (512*lba);
    char* dst = (char*)buffer;
    memcpy( dst, src, count*512 );
    return 0;
#else
    _printf("[GIET ERROR] _rdk_read() should not be used if USE_IOC_RDK not set\n");
    return 1;
#endif
}

///////////////////////////////////////////////////////////////////////////////
//     _rdk_write()
// Transfer data from a memory buffer to the block device. 
// - mode     : BOOT / KERNEL / USER
// - lba      : first block index on the block device
// - buffer   : virtual base address of the memory buffer 
// - count    : number of blocks to be transfered.
// Returns 0 if success, > 0 if error.
///////////////////////////////////////////////////////////////////////////////
unsigned int _rdk_write( unsigned int lba, 
                         unsigned int buffer, 
                         unsigned int count ) 
{

#if GIET_DEBUG_IOC_DRIVER
_printf("\n[IOC DEBUG] Enter _rdk_write() at cycle %d"
        "\n - vaddr   = %x"
        "\n - sectors = %d"
        "\n - lba     = %x",
        _get_proctime(), buffer, count, lba );
#endif

#if USE_IOC_RDK
    char* dst = (char*)SEG_RDK_BASE + (512*lba);
    char* src = (char*)buffer;
    memcpy( dst, src, count*512 );
    return 0;
#else
    _printf("[GIET ERROR] _rdk_write() should not be used if USE_IOC_RDK not set\n");
    return 1;
#endif
}

///////////////////////////////////////////////////////////////////////////////
//     _rdk_get_block_size()
// This function returns the block_size.
///////////////////////////////////////////////////////////////////////////////
unsigned int _rdk_get_block_size() 
{
    return 512;
}


// Local Variables:
// tab-width: 4
// c-basic-offset: 4
// c-file-offsets:((innamespace . 0)(inline-open . 0))
// indent-tabs-mode: nil
// End:
// vim: filetype=c:expandtab:shiftwidth=4:tabstop=4:softtabstop=4

