//////////////////////////////////////////////////////////////////////////////////
// File     : stdio.h         
// Date     : 01/04/2010
// Author   : alain greiner & Joel Porquet
// Copyright (c) UPMC-LIP6
///////////////////////////////////////////////////////////////////////////////////
// The stdio.c and stdio.h files are part of the GIET_VM nano-kernel.
// This library contains all user-level functions that contain a system call
// to access protected or shared ressources.
///////////////////////////////////////////////////////////////////////////////////

#ifndef _STDIO_H
#define _STDIO_H

// These define must be synchronised with 
// the _syscall_vector defined in file sys_handler.c

#define SYSCALL_PROCID            0x00
#define SYSCALL_PROCTIME          0x01
#define SYSCALL_TTY_WRITE         0x02
#define SYSCALL_TTY_READ          0x03
#define SYSCALL_TIMER_START       0x04
#define SYSCALL_TIMER_STOP        0x05
#define SYSCALL_TTY_GET_LOCK      0x06
#define SYSCALL_TTY_RELEASE_LOCK  0x07
#define SYSCALL_HEAP_INFO         0x08
#define SYSCALL_LOCAL_TASK_ID     0x09
#define SYSCALL_GLOBAL_TASK_ID    0x0A
#define SYSCALL_FB_CMA_INIT       0x0B
#define SYSCALL_FB_CMA_WRITE      0x0C
#define SYSCALL_FB_CMA_STOP       0x0D
#define SYSCALL_EXIT              0x0E
#define SYSCALL_PROC_NUMBER       0x0F

#define SYSCALL_FB_SYNC_WRITE     0x10
#define SYSCALL_FB_SYNC_READ      0x11
#define SYSCALL_THREAD_ID         0x12
#define SYSCALL_FREE_13           0x13
#define SYSCALL_FREE_14           0x14
#define SYSCALL_FREE_15           0x15
#define SYSCALL_FREE_16           0x16
#define SYSCALL_FREE_17           0x17
#define SYSCALL_FREE_18           0x18
#define SYSCALL_CTX_SWITCH        0x19
#define SYSCALL_VOBJ_GET_VBASE    0x1A
#define SYSCALL_GET_XY            0x1B
#define SYSCALL_NIC_CMA_START     0x1C
#define SYSCALL_NIC_CMA_STOP      0x1D
#define SYSCALL_NIC_SYNC_READ     0x1E
#define SYSCALL_NIC_SYNC_WRITE    0x1F

#define SYSCALL_FAT_OPEN          0x20
#define SYSCALL_FAT_READ          0x21
#define SYSCALL_FAT_WRITE         0x22
#define SYSCALL_FAT_LSEEK         0x23
#define SYSCALL_FAT_FSTAT         0x24
#define SYSCALL_FAT_CLOSE         0x25

//////////////////////////////////////////////////////////////////////////////////
// NULL pointer definition
//////////////////////////////////////////////////////////////////////////////////

#define NULL (void *)0

//////////////////////////////////////////////////////////////////////////////////
// This generic C function is used to implement all system calls.
// It writes the system call arguments in the proper registers,
// and tells GCC what has been modified by system call execution.
//////////////////////////////////////////////////////////////////////////////////
static inline int sys_call( int call_no,
                            int arg_0, 
                            int arg_1, 
                            int arg_2, 
                            int arg_3 ) 
{
    register int reg_no_and_output asm("v0") = call_no;
    register int reg_a0 asm("a0") = arg_0;
    register int reg_a1 asm("a1") = arg_1;
    register int reg_a2 asm("a2") = arg_2;
    register int reg_a3 asm("a3") = arg_3;

    asm volatile(
            "syscall"
            : "+r" (reg_no_and_output), /* input/output argument */
              "+r" (reg_a0),             
              "+r" (reg_a1),
              "+r" (reg_a2),
              "+r" (reg_a3),
              "+r" (reg_no_and_output)
            : /* input arguments */
            : "memory",
            /* These persistant registers will be saved on the stack by the
             * compiler only if they contain relevant data. */
            "at",
            "v1",
            "ra",
            "t0",
            "t1",
            "t2",
            "t3",
            "t4",
            "t5",
            "t6",
            "t7",
            "t8",
            "t9"
               );
    return (volatile int)reg_no_and_output;
}

//////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////
//               MIPS32 related system calls 
//////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////

//////////////////////////////////////////////////////////////////////////
// This function returns the processor (x,y,lpid) identifier: 
// (x,y) are the cluster coordinates / lpid is the local processor index.
//////////////////////////////////////////////////////////////////////////
extern void giet_proc_xyp( unsigned int* cluster_x,
                           unsigned int* cluster_y,
                           unsigned int* lpid );

//////////////////////////////////////////////////////////////////////////
// This function returns the local processor time.
//////////////////////////////////////////////////////////////////////////
extern int giet_proctime();

//////////////////////////////////////////////////////////////////////////
// This function returns a pseudo-random value derived from the processor 
// cycle count. This value is comprised between 0 & 65535.
/////////////////////////////////////////////////////////////////////////
extern int giet_rand();

//////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////
//             TTY device related system calls 
//////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////

//////////////////////////////////////////////////////////////////////////
// This function is a modified version of the mutek_printf().
// It uses a private terminal allocated to the calling task in the boot.
// ("use_tty" argument in xml mapping), and does not take the TTY lock.
// It calls several times the _tty_write system function.
// Only a limited number of formats are supported:
//   - %d : signed decimal
//   - %u : unsigned decimal
//   - %x : 32 bits hexadecimal
//   - %l : 64 bits hexadecimal
//   - %c : char
//   - %s : string
// In case or error returned by syscall, it makes a giet_exit().
//////////////////////////////////////////////////////////////////////////
extern void giet_tty_printf( char* format, ... );

//////////////////////////////////////////////////////////////////////////
// This function is a modified version of the mutek_printf().
// It uses the kernel TTY0 as a shared terminal, and it takes the
// TTY lock to get exclusive access during the format display.
// It calls several times the _tty_write system function.
// Only a limited number of formats are supported:
//   - %d : signed decimal
//   - %u : unsigned decimal
//   - %x : 32 bits hexadecimal
//   - %l : 64 bits hexadecimal
//   - %c : char
//   - %s : string
// In case or error returned by syscall, it makes a giet_exit().
//////////////////////////////////////////////////////////////////////////
extern void giet_shr_printf( char* format, ... );

//////////////////////////////////////////////////////////////////////////
// This blocking function fetches a single character from the private
// terminal allocated to the calling task in the boot.
// It uses the TTY_RX_IRQ interrupt, and the associated kernel buffer.
// In case or error returned by syscall, it makes a giet_exit().
//////////////////////////////////////////////////////////////////////////
extern void giet_tty_getc( char* byte );

//////////////////////////////////////////////////////////////////////////
// This blocking function fetches a string from the private terminal 
// allocated to the calling task to a fixed length buffer.
// The terminal index must be defined in the task context in the boot.
// It uses the TTY_RX_IRQ interrupt, and the associated kernel buffer.
// - Up to (bufsize - 1) characters (including the non printable characters)
//   are copied into buffer, and the string is completed by a NUL character.
// - The <LF> character is interpreted, and the function close the string
//   with a NUL character if <LF> is read.
// - The <DEL> character is interpreted, and the corresponding character(s) 
//   are removed from the target buffer.
// - It does not provide an echo.
// In case or error returned by syscall, it makes a giet_exit().
/////////////////////////////////////////////////////////////////////////
extern void giet_tty_gets( char* buf, unsigned int bufsize );

/////////////////////////////////////////////////////////////////////////
// This blocking function fetches a string of decimal characters (most
// significant digit first) to build a 32-bit unsigned integer from
// the private TTY terminal allocated to the calling task. 
// The terminal index must be defined in the task context in the boot.
// It uses the TTY_RX_IRQ interrupt, and the associated kernel buffer.
// - The non-blocking system function _tty_read is called several times,
//   and the decimal characters are written in a 32 characters buffer 
//   until a <LF> character is read.
// - It ignores non-decimal characters, and displays an echo 
//   system function) for each decimal character.
// - The <DEL> character is interpreted, and previous characters can be cancelled. 
// - When the <LF> character is received, the string is converted to an
//   unsigned int value. If the number of decimal digit is too large for the 32
//   bits range, the zero value is returned.
// In case or error returned by syscall, it makes a giet_exit().
//////////////////////////////////////////////////////////////////////////
extern void giet_tty_getw( unsigned int* val );

//////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////
//                TIMER device related system calls 
//////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////

//////////////////////////////////////////////////////////////////////////
// This function activates the private user timer allocated 
// to the calling task in the boot phase.
// In case or error returned by syscall, it makes a giet_exit().
//////////////////////////////////////////////////////////////////////////
extern void giet_timer_start();

//////////////////////////////////////////////////////////////////////////
// This function stops the private user timer allocated 
// to the calling task.
// In case or error returned by syscall, it makes a giet_exit().
//////////////////////////////////////////////////////////////////////////
extern void giet_timer_stop();
 
//////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////
//                Frame buffer device related system calls 
//////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////

//////////////////////////////////////////////////////////////////////////
// This blocking function use a memory copy strategy to transfer data 
// from the frame buffer device in kernel space to an user buffer.
//     offset : offset (in bytes) in the frame buffer
//     buffer : base address of the user buffer
//     length : number of bytes to be transfered
// In case or error returned by syscall, it makes a giet_exit().
//////////////////////////////////////////////////////////////////////////
extern void giet_fb_sync_read( unsigned int offset, 
                               void*        buffer, 
                               unsigned int length );

//////////////////////////////////////////////////////////////////////////
// This blocking function use a memory copy strategy to transfer data 
// from a user buffer to the frame buffer device in kernel space.
//     offset : offset (in bytes) in the frame buffer
//     buffer : base address of the memory buffer
//     length : number of bytes to be transfered
// In case or error returned by syscall, it makes a giet_exit().
//////////////////////////////////////////////////////////////////////////
extern void giet_fb_sync_write( unsigned int offset, 
                                void*        buffer, 
                                unsigned int length );

//////////////////////////////////////////////////////////////////////////
// This function initializes the two chbuf SRC an DST used by the CMA 
// controller and activates the CMA channel allocated to the calling task.
// - buf0   : first user buffer virtual address
// - buf1   : second user buffer virtual address
// - length : buffer size (bytes)
// In case or error returned by syscall, it makes a giet_exit().
//////////////////////////////////////////////////////////////////////////
extern void giet_fb_cma_init( void*        buf0, 
                              void*        buf1,
                              unsigned int length );

//////////////////////////////////////////////////////////////////////////
// This function initializes the two chbuf SRC an DST used by the CMA 
// controller and activates the CMA channel allocated to the calling task.
// - buf0   : first user buffer virtual address
// - buf0   : second user buffer virtual address
// - length : buffer size (bytes)
// In case or error returned by syscall, it makes a giet_exit().
//////////////////////////////////////////////////////////////////////////
extern void giet_fb_cma_write( unsigned int buf_id );

//////////////////////////////////////////////////////////////////////////
// This function desactivates the CMA channel allocated to the task.
// In case or error returned by syscall, it makes a giet_exit().
//////////////////////////////////////////////////////////////////////////
extern void giet_fb_cma_stop();

//////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////
//                  NIC related system calls 
//////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////

//////////////////////////////////////////////////////////////////////////
// This function initializes the memory chbuf used by the CMA controller, 
// activates the NIC channel allocated to the calling task, 
// and activates the two CMA channels.
// - tx     : RX channel if 0 / TX channel if non 0
// - buf0   : first user buffer virtual address
// - buf1   : second user buffer virtual address
// - length : buffer size (bytes)
// In case or error returned by syscall, it makes a giet_exit().
//////////////////////////////////////////////////////////////////////////
extern void giet_nic_cma_start();

//////////////////////////////////////////////////////////////////////////
// This function desactivates the NIC channel and the two CMA channels
// allocated to the calling task.
// In case or error returned by syscall, it makes a giet_exit().
//////////////////////////////////////////////////////////////////////////
extern void giet_nic_cma_stop();

//////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////
//               FAT related system calls 
//////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////

//////////////////////////////////////////////////////////////////////////
// Open a file identified by a pathname, and contained in the system FAT.
// The read/write flags are not supported yet: no effect.
// Return -1 in case or error.
//////////////////////////////////////////////////////////////////////////
extern int giet_fat_open(  const char*  pathname,
                           unsigned int flags );

///////////////////////////////////////////////////////////////////////////////////
// Read "count" sectors from a file identified by "fd", skipping "offset"
// sectors in file, and writing into the user "buffer". 
// The user buffer base address shoulb be 64 bytes aligned.
// In case or error returned by syscall, it makes a giet_exit().
///////////////////////////////////////////////////////////////////////////////////
extern void giet_fat_read(  unsigned int fd,
                            void*        buffer,
                            unsigned int count,
                            unsigned int offset );

///////////////////////////////////////////////////////////////////////////////////
// Write "count" sectors into a file identified by "fd", skipping "offset"
// sectors in file, and reading from the user "buffer". 
// The user buffer base address shoulb be 64 bytes aligned.
// In case or error returned by syscall, it makes a giet_exit().
///////////////////////////////////////////////////////////////////////////////////
extern void giet_fat_write( unsigned int fd,
                            void*        buffer,
                            unsigned int count,
                            unsigned int offset );

///////////////////////////////////////////////////////////////////////////////////
// Change the lseek file pointer value for a file identified by "fd".
// In case or error returned by syscall, it makes a giet_exit().
///////////////////////////////////////////////////////////////////////////////////
extern void giet_fat_lseek( unsigned int fd,
                            unsigned int offset,
                            unsigned int whence );

///////////////////////////////////////////////////////////////////////////////////
// Returns general informations of a file identified by "fd". 
// (Only the file_size in sectors for this moment)
///////////////////////////////////////////////////////////////////////////////////
extern void giet_fat_fstat( unsigned int fd );

//////////////////////////////////////////////////////////////////////////
// Close a file identified by "fd".
//////////////////////////////////////////////////////////////////////////
extern void giet_fat_close( unsigned int fd );

//////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////
//                    Task context system calls
//////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////

//////////////////////////////////////////////////////////////////////////
// This functions returns the local task id.
// If processor has n tasks the local task index is ranging from 0 to n-1
//////////////////////////////////////////////////////////////////////////
extern int giet_proc_task_id();

//////////////////////////////////////////////////////////////////////////
// This functions returns the global task id, (unique in the system).
//////////////////////////////////////////////////////////////////////////
extern int giet_global_task_id(); 

//////////////////////////////////////////////////////////////////////////
// This functions returns the thread index of the task in its vspace.
//////////////////////////////////////////////////////////////////////////
extern int giet_thread_id(); 

//////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////
//                    Miscelaneous system calls
//////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////

//////////////////////////////////////////////////////////////////////////
// This function stops execution of the calling task with a TTY message, 
// the user task is descheduled and becomes not runable. 
// It does not consume processor cycles anymore.
//////////////////////////////////////////////////////////////////////////
extern void giet_exit( char* string );

//////////////////////////////////////////////////////////////////////////
// This function uses the giet_exit() system call
// and kill the calling task if the condition is false.
//////////////////////////////////////////////////////////////////////////
extern void giet_assert( unsigned int condition, 
                         char*        string );

//////////////////////////////////////////////////////////////////////////
// The user task calling this function is descheduled and
// the processor is allocated to another task.
//////////////////////////////////////////////////////////////////////////
extern void giet_context_switch();

//////////////////////////////////////////////////////////////////////////
// This function writes in argument "vobj_vaddr" the virtual base address 
// of a vobj (defined in the mapping_info data structure), identified by 
// the two arguments "vspace_name" and "vobj_name".
// In case or error returned by syscall, it makes a giet_exit().
// ( vobj not defined or wrong vspace )
//////////////////////////////////////////////////////////////////////////
extern void giet_vobj_get_vbase( char*         vspace_name, 
                                 char*         vobj_name, 
                                 unsigned int* vobj_vaddr);

//////////////////////////////////////////////////////////////////////////
// This function returns in the "buffer" argument the number of processors 
// in the cluster specified by the "cluster_xy" argument.
// In case or error returned by syscall, it makes a giet_exit().
//////////////////////////////////////////////////////////////////////////
extern void giet_procnumber( unsigned int cluster_xy,
                             unsigned int buffer );

//////////////////////////////////////////////////////////////////////////
// This function supports access to the task's heap or to a remote heap:
// - If (x < X_SIZE) and (y < Y_SIZE), this function returns the base 
//   address and length of the heap associated to any task running 
//   on cluster(x,y) => remote heap
// - Else, this function returns the base address and length of the 
//   heap associated to the calling task => local heap
//////////////////////////////////////////////////////////////////////////
extern void giet_heap_info( unsigned int* vaddr, 
                            unsigned int* length,
                            unsigned int  x,
                            unsigned int  y );

//////////////////////////////////////////////////////////////////////////
// This function takes as input a virtual address (ptr argument), 
// and returns through the (px,py) arguments the coordinates of 
// the cluster containing the physical address associated to ptr. 
// In case of error (unmapped virtual address), it makes a giet_exit().
//////////////////////////////////////////////////////////////////////////
extern void giet_get_xy( void*          ptr, 
                         unsigned int*  px,
                         unsigned int*  py );

#endif

// Local Variables:
// tab-width: 4
// c-basic-offset: 4
// c-file-offsets:((innamespace . 0)(inline-open . 0))
// indent-tabs-mode: nil
// End:
// vim: filetype=c:expandtab:shiftwidth=4:tabstop=4:softtabstop=4

