//////////////////////////////////////////////////////////////////////////////////////
// File     : tim_driver.h
// Date     : 01/11/2013
// Author   : alain greiner 
// Copyright (c) UPMC-LIP6
///////////////////////////////////////////////////////////////////////////////////
// The tim_driver.c and tim_driver.h files are part ot the GIET-VM nano-kernel.
// This driver supports the SoCLib vci_multi_timer component.
//
// It can exist several multi_timers in the architecture (at most one per cluster),
// and each one can contain several timers (called channels).
//
// There is two types of timers: 
// - "system" timers : one per processor, used for context switch.
//   local_id in [0, NB_PROCS_MAX-1],
// - "user" timers : requested by the task in the mapping_info data structure.
//   For each user timer, the timer_id is stored in the context of the task.
// The global index is cluster_xy * (NB_PROCS_MAX + NB_TIM_CHANNELS) + local_id
//
// The NB_PROCS_MAX and NB_TIM_CHANNELS values must be defined in hard_config.h file.
//
// The virtual base address of the segment associated to a channel is:
//     SEG_TIM_BASE + cluster_xy * PERI_CLUSTER_INCREMENT + TIMER_SPAN * timer_id
//
// The SEG_TIM_BASE and PERI_CLUSTER_INCREMENT must be defined in hard_config.h.
/////////////////////////////////////////////////////////////////////////////////////

#ifndef _GIET_TIM_DRIVER_H_
#define _GIET_TIM_DRIVER_H_

///////////////////////////////////////////////////////////////////////////////////
//                    registers offsets
///////////////////////////////////////////////////////////////////////////////////

enum TIMER_registers 
{
    TIMER_VALUE     = 0,
    TIMER_MODE      = 1,
    TIMER_PERIOD    = 2,
    TIMER_RESETIRQ  = 3,
    /**/
    TIMER_SPAN      = 4,
};

///////////////////////////////////////////////////////////////////////////////////
//                   access functions 
///////////////////////////////////////////////////////////////////////////////////

///////////////////////////////////////////////////////////////////////////////////
// This function activates a timer in the vci_timer external peripheral.
// - channel : Timer channel global index
// - period  : interrupt period (cycles)
///////////////////////////////////////////////////////////////////////////////////
extern int _timer_start( unsigned int channel, 
                         unsigned int period );

///////////////////////////////////////////////////////////////////////////////////
// This function desactivates a timer in the vci_timer external component.
///////////////////////////////////////////////////////////////////////////////////
extern int _timer_stop( unsigned int channel );

///////////////////////////////////////////////////////////////////////////////////
// This function resets the timer counter. To do so, it read the period,
// and re-write it in the timer register, what causes the count to restart.
///////////////////////////////////////////////////////////////////////////////////
extern int _timer_reset_cpt( unsigned int channel );

///////////////////////////////////////////////////////////////////////////////////
// This Interrupt Service Routine handles the IRQs generated by the "user" timers.
// It can be a HWI or a PTI.
// The channel argument is the user timer global index.
// The ISR acknowledges the IRQ, registers the event in the proper entry
// of the _user_timer_event[] array, and a log message is displayed on TTY0.
///////////////////////////////////////////////////////////////////////////////////
extern void _timer_isr( unsigned int irq_type,
                        unsigned int irq_id,
                        unsigned int channel );


#endif

// Local Variables:
// tab-width: 4
// c-basic-offset: 4
// c-file-offsets:((innamespace . 0)(inline-open . 0))
// indent-tabs-mode: nil
// End:
// vim: filetype=c:expandtab:shiftwidth=4:tabstop=4:softtabstop=4

