///////////////////////////////////////////////////////////////////////////////////
// File     : sys_handler.c
// Date     : 01/04/2012
// Author   : alain greiner and joel porquet
// Copyright (c) UPMC-LIP6
///////////////////////////////////////////////////////////////////////////////////

#include <sys_handler.h>
#include <tty_driver.h>
#include <tim_driver.h>
#include <ioc_driver.h>
#include <nic_driver.h>
#include <mmc_driver.h>
#include <cma_driver.h>
#include <ctx_handler.h>
#include <fat32.h>
#include <utils.h>
#include <kernel_utils.h>
#include <vmem.h>
#include <hard_config.h>
#include <giet_config.h>
#include <mapping_info.h>

#if !defined(SEG_BOOT_MAPPING_BASE) 
# error: You must define SEG_BOOT_MAPPING_BASE in the hard_config.h file
#endif

////////////////////////////////////////////////////////////////////////////
//     Channel allocators for peripherals
//     (TTY[0] is reserved for kernel)
////////////////////////////////////////////////////////////////////////////

unsigned int _tty_channel_allocator = 1;
unsigned int _tim_channel_allocator = 0;
unsigned int _cma_channel_allocator = 0;
unsigned int _nic_channel_allocator = 0;

////////////////////////////////////////////////////////////////////////////
//    Initialize the syscall vector with syscall handlers
// Note: This array must be synchronised with the define in file stdio.h
////////////////////////////////////////////////////////////////////////////
const void * _syscall_vector[64] = 
{
    &_sys_proc_xyp,         /* 0x00 */
    &_get_proctime,         /* 0x01 */
    &_sys_tty_write,        /* 0x02 */
    &_sys_tty_read,         /* 0x03 */
    &_sys_tty_alloc,        /* 0x04 */
    &_sys_tty_get_lock,     /* 0x05 */
    &_sys_tty_release_lock, /* 0x06 */
    &_sys_heap_info,        /* 0x07 */
    &_sys_local_task_id,    /* 0x08 */
    &_sys_global_task_id,   /* 0x09 */ 
    &_sys_fbf_cma_alloc,    /* 0x0A */
    &_sys_fbf_cma_start,    /* 0x0B */
    &_sys_fbf_cma_display,  /* 0x0C */
    &_sys_fbf_cma_stop,     /* 0x0D */
    &_sys_task_exit,        /* 0x0E */
    &_sys_procs_number,     /* 0x0F */

    &_sys_fbf_sync_write,   /* 0x10 */
    &_sys_fbf_sync_read,    /* 0x11 */
    &_sys_thread_id,        /* 0x12 */
    &_sys_ukn,              /* 0x13 */
    &_sys_tim_alloc,        /* 0x14 */
    &_sys_tim_start,        /* 0x15 */ 
    &_sys_tim_stop,         /* 0x16 */
    &_sys_ukn,              /* 0x17 */
    &_sys_ukn,              /* 0x18 */   
    &_context_switch,       /* 0x19 */
    &_sys_vobj_get_vbase,   /* 0x1A */
    &_sys_vobj_get_length,  /* 0x1B */
    &_sys_xy_from_ptr,      /* 0x1C */
    &_sys_nic_alloc,        /* 0x1D */
    &_sys_nic_sync_send,    /* 0x1E */
    &_sys_nic_sync_receive, /* 0x1F */

    &_fat_user_open,        /* 0x20 */
    &_fat_user_read,        /* 0x21 */
    &_fat_user_write,       /* 0x22 */
    &_fat_user_lseek,       /* 0x23 */
    &_fat_fstat,            /* 0x24 */
    &_fat_close,            /* 0x25 */
    &_sys_ukn,              /* 0x26 */
    &_sys_ukn,              /* 0x27 */
    &_sys_ukn,              /* 0x28 */
    &_sys_ukn,              /* 0x29 */
    &_sys_ukn,              /* 0x2A */
    &_sys_ukn,              /* 0x2B */
    &_sys_ukn,              /* 0x2C */
    &_sys_ukn,              /* 0x2D */
    &_sys_ukn,              /* 0x2E */
    &_sys_ukn,              /* 0x2F */

    &_sys_ukn,              /* 0x30 */
    &_sys_ukn,              /* 0x31 */
    &_sys_ukn,              /* 0x32 */
    &_sys_ukn,              /* 0x33 */
    &_sys_ukn,              /* 0x34 */
    &_sys_ukn,              /* 0x35 */ 
    &_sys_ukn,              /* 0x36 */
    &_sys_ukn,              /* 0x37 */
    &_sys_ukn,              /* 0x38 */   
    &_sys_ukn,              /* 0x39 */
    &_sys_ukn,              /* 0x3A */
    &_sys_ukn,              /* 0x3B */
    &_sys_ukn,              /* 0x3C */
    &_sys_ukn,              /* 0x3D */
    &_sys_ukn,              /* 0x3E */
    &_sys_ukn,              /* 0x3F */
};

//////////////////////////////////////////////////////////////////////////////
//             TTY related syscall handlers 
//////////////////////////////////////////////////////////////////////////////

////////////////////
int _sys_tty_alloc()
{
    // get a new TTY terminal index
    unsigned int channel = _tty_channel_allocator;
    unsigned int thread  = _get_context_slot( CTX_TRDID_ID );
    unsigned int vspace  = _get_context_slot( CTX_VSID_ID );

    if ( channel >= NB_TTY_CHANNELS )
    {
        _printf("\n[GIET_ERROR] in _sys_tty_alloc() : not enough TTY channels\n");
        return -1;
    }
    else
    {
        _printf("\n[GIET WARNING] TTY channel %d allocated "
                " to thread %d in vspace %d\n", channel, thread, vspace );
    }

    // register timer index in task context
    _set_context_slot( CTX_TTY_ID, _tty_channel_allocator );

    // update timer allocator
    _tty_channel_allocator++;

    return 0;
}

/////////////////////////////////////////////////
int _sys_tty_write( const char*  buffer,    
                    unsigned int length,    // number of characters
                    unsigned int channel)   // channel index 
{
    unsigned int  nwritten;

    // compute and check tty channel
    if( channel == 0xFFFFFFFF )  channel = _get_context_slot(CTX_TTY_ID);
    if( channel >= NB_TTY_CHANNELS ) return -1;

    // write string to TTY channel
    for (nwritten = 0; nwritten < length; nwritten++) 
    {
        // check tty's status 
        if ( _tty_get_register( channel, TTY_STATUS ) & 0x2 )  break;

        // write one byte
        if (buffer[nwritten] == '\n') 
        {
            _tty_set_register( channel, TTY_WRITE, (unsigned int)'\r' );
        }
        _tty_set_register( channel, TTY_WRITE, (unsigned int)buffer[nwritten] );
    }
    
    return nwritten;
}

////////////////////////////////////////////////
int _sys_tty_read( char*        buffer, 
                   unsigned int length,    // unused
                   unsigned int channel)   // channel index
{
    // compute and check tty channel
    if( channel == 0xFFFFFFFF )  channel = _get_context_slot(CTX_TTY_ID);
    if( channel >= NB_TTY_CHANNELS ) return -1;

    // read one character from TTY channel
    if (_tty_rx_full[channel] == 0) 
    {
        return 0;
    }
    else 
    {
        *buffer = _tty_rx_buf[channel];
        _tty_rx_full[channel] = 0;
        return 1;
    }
}

///////////////////////////////////////////
int _sys_tty_get_lock( unsigned int   channel,
                       unsigned int * save_sr_ptr )
{
    // compute and check tty channel
    if( channel == 0xFFFFFFFF )  channel = _get_context_slot(CTX_TTY_ID);
    if( channel >= NB_TTY_CHANNELS ) return -1;

    _it_disable( save_sr_ptr );
    _get_lock( &_tty_lock[channel] );
    return 0;
}

///////////////////////////////////////////////
int _sys_tty_release_lock( unsigned int   channel,
                           unsigned int * save_sr_ptr )
{
    // compute and check tty channel
    if( channel == 0xFFFFFFFF )  channel = _get_context_slot(CTX_TTY_ID);
    if( channel >= NB_TTY_CHANNELS ) return -1;

    _release_lock( &_tty_lock[channel] );
    _it_restore( save_sr_ptr );
    return 0;
}

//////////////////////////////////////////////////////////////////////////////
//             TIM related syscall handlers 
//////////////////////////////////////////////////////////////////////////////

////////////////////
int _sys_tim_alloc()
{
    // get a new timer index 
    unsigned int channel = _tim_channel_allocator;
    unsigned int thread  = _get_context_slot( CTX_TRDID_ID );
    unsigned int vspace  = _get_context_slot( CTX_VSID_ID );

    if ( channel >= NB_TIM_CHANNELS )
    {
        _printf("\n[GIET_ERROR] in _sys_tim_alloc() : not enough TIM channels\n");
        return -1;
    }
    else
    {
        _printf("\n[GIET WARNING] TIM channel %d allocated "
                " to thread %d in vspace %d\n", channel, thread, vspace );
    }

    // register timer index in task context
    _set_context_slot( CTX_TIM_ID, channel );

    // update timer allocator
    _tim_channel_allocator++;

    return 0;
}

/////////////////////////////////////////
int _sys_tim_start( unsigned int period )
{
    // get timer index
    unsigned int channel = _get_context_slot( CTX_TIM_ID );
    if ( channel >= NB_TIM_CHANNELS )
    {
        _printf("\n[GIET_ERROR] in _sys_tim_start() : not enough TIM channels\n");
        return -1;
    }

    // start timer
    _timer_start( channel, period );

    return 0;
}

///////////////////
int _sys_tim_stop()
{
    // get timer index
    unsigned int channel = _get_context_slot( CTX_TIM_ID );
    if ( channel >= NB_TIM_CHANNELS )
    {
        _printf("\n[GIET_ERROR] in _sys_tim_stop() : illegal timer index\n");
        return -1;
    }

    // stop timer
    _timer_stop( channel );

    return 0;
}

//////////////////////////////////////////////////////////////////////////////
//             NIC related syscall handlers 
//////////////////////////////////////////////////////////////////////////////

////////////////////
int _sys_nic_alloc()
{
    // get a new NIC channel index 
    unsigned int channel = _nic_channel_allocator;
    unsigned int thread  = _get_context_slot( CTX_TRDID_ID );
    unsigned int vspace  = _get_context_slot( CTX_VSID_ID );

    if ( channel >= NB_NIC_CHANNELS )
    {
        _printf("\n[GIET_ERROR] in _sys_nic_alloc() : not enough NIC channels\n");
        return -1;
    }
    else
    {
        _printf("\n[GIET WARNING] NIC channel %d allocated "
                " to thread %d in vspace %d\n", channel, thread, vspace );
    }

    // register channel index in task context
    _set_context_slot( CTX_NIC_ID, channel );

    // update NIC channel allocator
    _nic_channel_allocator++;

    return 0;
}

////////////////////////////////////
int _sys_nic_sync_send( void* vbuf )
{
    unsigned int ppn;
    unsigned int flags;
    unsigned int vaddr = (unsigned int)vbuf;

    // get NIC channel index
    unsigned int channel = _get_context_slot( CTX_NIC_ID );
    if ( channel >= NB_NIC_CHANNELS )
    {
        _printf("\n[GIET_ERROR] in _sys_nic_sync_send() : illegal NIC channel index\n");
        return -1;
    }

    // get page table pointer
    unsigned int user_ptab = _get_context_slot( CTX_PTAB_ID );

    // Compute user buffer physical address and check access rights
    unsigned int ko = _v2p_translate( (page_table_t*)user_ptab,
                                      vaddr,
                                      &ppn,
                                      &flags );
    if ( ko )
    {
        _printf("\n[GIET ERROR] in _sys_nic_sync_send() : user buffer unmapped\n");
        return -1;
    }
    if ( (flags & PTE_U) == 0 )
    {
        _printf("\n[GIET ERROR] in _sys_nic_sync_send() : illegal buffer address\n");
        return -1;
    }
    unsigned long long pbuf = ((unsigned long long)ppn << 12) | (vaddr & 0x00000FFF);

    _nic_sync_send( channel, pbuf );

    return 0;
}

///////////////////////////////////////
int _sys_nic_sync_receive( void* vbuf )
{
    unsigned int ppn;
    unsigned int flags;
    unsigned int vaddr = (unsigned int)vbuf;

    // get NIC channel index
    unsigned int channel = _get_context_slot( CTX_NIC_ID );
    if ( channel >= NB_NIC_CHANNELS )
    {
        _printf("\n[GIET_ERROR] in _sys_nic_sync_send() : illegal NIC channel index\n");
        return -1;
    }

    // get page table pointer
    unsigned int user_ptab = _get_context_slot( CTX_PTAB_ID );

    // Compute user buffer physical address and check access rights
    unsigned int ko = _v2p_translate( (page_table_t*)user_ptab,
                                      vaddr,
                                      &ppn,
                                      &flags );
    if ( ko )
    {
        _printf("\n[GIET ERROR] in _sys_nic_sync_send() : user buffer unmapped\n");
        return -1;
    }
    if ( (flags & PTE_U) == 0 )
    {
        _printf("\n[GIET ERROR] in _sys_nic_sync_send() : illegal buffer address\n");
        return -1;
    }
    unsigned long long pbuf = ((unsigned long long)ppn << 12) | (vaddr & 0x00000FFF);

    _nic_sync_receive( channel, pbuf );

    return 0;
}

/////////////////////////////////////////////////////////////////////////////////////////
//    FBF related syscall handlers
/////////////////////////////////////////////////////////////////////////////////////////

// Array of fbf_chbuf descriptors, indexed by the CMA channel index.
__attribute__((section (".unckdata")))
volatile fbf_chbuf_t _fbf_chbuf[NB_CMA_CHANNELS] __attribute__((aligned(32)));

// Physical addresses of these fbf_chbuf descriptors (required for L2 cache sync)
__attribute__((section (".unckdata")))
unsigned long long _fbf_chbuf_paddr[NB_CMA_CHANNELS];

/////////////////////////////////////////////
int _sys_fbf_sync_write( unsigned int offset,
                         void*        buffer,
                         unsigned int length )
{
    char* fbf_address = (char *)SEG_FBF_BASE + offset;
    memcpy( fbf_address, buffer, length);

    return 0;
}

/////////////////////////////////////////////
int _sys_fbf_sync_read(  unsigned int offset,
                         void*        buffer,
                         unsigned int length )
{
    char* fbf_address = (char *)SEG_FBF_BASE + offset;
    memcpy( buffer, fbf_address, length);

    return 0;
}

////////////////////////
int _sys_fbf_cma_alloc()
{
#if NB_CMA_CHANNELS > 0

   // get a new CMA channel index 
    unsigned int channel = _cma_channel_allocator;
    unsigned int thread  = _get_context_slot( CTX_TRDID_ID );
    unsigned int vspace  = _get_context_slot( CTX_VSID_ID );

    if ( channel >= NB_CMA_CHANNELS )
    {
        _printf("\n[GIET ERROR] in _sys_fbf_cma_alloc() : not enough CMA channels\n");
        return -1;
    }
    else
    {
        _printf("\n[GIET WARNING] FBF_CMA channel %d allocated "
                " to thread %d in vspace %d\n", channel, thread, vspace );
    }

    // register channel index in task context
    _set_context_slot( CTX_FBCMA_ID, channel );

    // update CMA channel allocator
    _cma_channel_allocator++;

    return 0;

#else

    _printf("\n[GIET ERROR] in _fb_cma_start() : NB_CMA_CHANNELS = 0\n");
    return -1;

#endif
} // end sys_fbf_cma_alloc()

////////////////////////////////////////////
int _sys_fbf_cma_start( void*        vbase0, 
                        void*        vbase1,  
                        unsigned int length ) 
{
#if NB_CMA_CHANNELS > 0

    unsigned int       ptab;            // page table virtual address
    unsigned int       ko;              // unsuccessfull V2P translation
    unsigned int       vaddr;           // virtual address
    unsigned int       flags;           // protection flags
    unsigned int       ppn;             // physical page number
    unsigned long long chbuf_paddr;     // physical address of source chbuf descriptor

    // get channel index
    unsigned int channel = _get_context_slot( CTX_FBCMA_ID );

    if ( channel >= NB_CMA_CHANNELS )
    {
        _printf("\n[GIET ERROR] in _fbf_cma_start() : CMA channel index too large\n");
        return -1;
    }

#if GIET_DEBUG_FBF_CMA
_printf("\n[FBF_CMA DEBUG] enters _sys_fbf_cma_start()\n"
        " - channel      = %d\n"
        " - buf0   vbase = %x\n"
        " - buf1   vbase = %x\n"
        " - buffer size  = %x\n",
        channel,
        (unsigned int)vbase0,
        (unsigned int)vbase1,
        length );
#endif

    // checking user buffers virtual addresses and length alignment
    if ( ((unsigned int)vbase0 & 0x3) || ((unsigned int)vbase1 & 0x3) || (length & 0x3) ) 
    {
        _printf("\n[GIET ERROR] in _fbf_cma_start() : user buffer not word aligned\n");
        return -1;
    }

    // get page table virtual address
    ptab = _get_context_slot(CTX_PTAB_ID);

    // compute frame buffer physical address and initialize _fbf_chbuf[channel]
    vaddr = ((unsigned int)SEG_FBF_BASE);
    ko    = _v2p_translate( (page_table_t*) ptab, 
                            (vaddr >> 12),
                            &ppn, 
                            &flags );
    if (ko) 
    {
        _printf("\n[GIET ERROR] in _fb_cma_start() : frame buffer unmapped\n");
        return -1;
    }

    _fbf_chbuf[channel].fbf    = ((paddr_t)ppn << 12) | (vaddr & 0x00000FFF);

    // Compute user buffer 0 physical addresses and intialize _fbf_chbuf[channel]
    vaddr = (unsigned int)vbase0; 
    ko = _v2p_translate( (page_table_t*) ptab, 
                         (vaddr >> 12),
                         &ppn, 
                         &flags );
    if (ko) 
    {
        _printf("\n[GIET ERROR] in _fbf_cma_start() : user buffer 0 unmapped\n");
        return -1;
    } 
    if ((flags & PTE_U) == 0) 
    {
        _printf("\n[GIET ERROR] in _fbf_cma_start() : user buffer 0 not in user space\n");
        return -1;
    }

    _fbf_chbuf[channel].buf0 = ((paddr_t)ppn << 12) | (vaddr & 0x00000FFF);

    // Compute user buffer 1 physical addresses and intialize _fbf_chbuf[channel]
    vaddr = (unsigned int)vbase1; 
    ko = _v2p_translate( (page_table_t*) ptab, 
                         (vaddr >> 12),
                         &ppn, 
                         &flags );
    if (ko) 
    {
        _printf("\n[GIET ERROR] in _fbf_cma_start() : user buffer 1 unmapped\n");
        return -1;
    } 
    if ((flags & PTE_U) == 0) 
    {
        _printf("\n[GIET ERROR] in _fbf_cma_start() : user buffer 1 not in user space\n");
        return -1;
    }

    _fbf_chbuf[channel].buf1 = ((paddr_t)ppn << 12) | (vaddr & 0x00000FFF);

    // initializes buffer length
    _fbf_chbuf[channel].length = length;

    // Compute and register physical adress of the chbuf descriptor
    vaddr = (unsigned int)(&_fbf_chbuf[channel]);
    ko = _v2p_translate( (page_table_t*) ptab, 
                         (vaddr >> 12),
                         &ppn, 
                         &flags );
    if (ko) 
    {
        _printf("\n[GIET ERROR] in _fbf_cma_start() : chbuf descriptor unmapped\n");
        return -1;
    }
 
    chbuf_paddr = (((paddr_t)ppn) << 12) | (vaddr & 0x00000FFF);

    _fbf_chbuf_paddr[channel] = chbuf_paddr;


    if ( USE_IOB )
    {
        // SYNC request for channel descriptor
        _mmc_sync( chbuf_paddr, 32 );
    }

#if GIET_DEBUG_FBF_CMA
_printf(" - fbf    pbase = %l\n"
        " - buf0   pbase = %l\n"
        " - buf1   pbase = %l\n"
        " - chbuf  pbase = %l\n",
        _fbf_chbuf[channel].fbf,
        _fbf_chbuf[channel].buf0,
        _fbf_chbuf[channel].buf1,
        chbuf_paddr );
#endif

    // call CMA driver to start transfer
    _cma_start_channel( channel, 
                        chbuf_paddr,
                        2,
                        chbuf_paddr + 16,
                        1,
                        length );
    return 0;

#else

    _printf("\n[GIET ERROR] in _sys_fbf_cma_start() : NB_CMA_CHANNELS = 0\n");
    return -1;

#endif
} // end _sys_fbf_cma_start()

/////////////////////////////////////////////////////
int _sys_fbf_cma_display( unsigned int buffer_index )
{
#if NB_CMA_CHANNELS > 0

    volatile paddr_t buf_paddr;
    unsigned int     full = 1;

    // get channel index
    unsigned int channel = _get_context_slot( CTX_FBCMA_ID );

    if ( channel >= NB_CMA_CHANNELS )
    {
        _printf("\n[GIET ERROR] in _sys_fbf_cma_display() : CMA channel index too large\n");
        return -1;
    }

#if GIET_DEBUG_FBF_CMA
_printf("\n[FBF_CMA DEBUG] enters _sys_fb_cma_display()\n"
        " - channel      = %d\n"
        " - buffer       = %d\n",
        channel, buffer_index );
#endif

    // waiting user buffer empty
    while ( full )
    {  
        if ( USE_IOB )
        {
            // INVAL L1 cache for the chbuf descriptor,
            _dcache_buf_invalidate( (unsigned int)&_fbf_chbuf[channel], 32 );

            // INVAL L2 cache for the chbuf descriptor,
            _mmc_inval( _fbf_chbuf_paddr[channel], 32 );
        }

        // read user buffer descriptor
        if ( buffer_index == 0 ) buf_paddr = _fbf_chbuf[channel].buf0;
        else                     buf_paddr = _fbf_chbuf[channel].buf1;

        full = ( (unsigned int)(buf_paddr>>63) );
    }

    if ( USE_IOB )
    {
        // SYNC request for the user buffer, because 
        // it will be read from XRAM by the CMA component
        _mmc_sync( buf_paddr, _fbf_chbuf[channel].length );
    }

    // set user buffer status 
    if ( buffer_index == 0 ) _fbf_chbuf[channel].buf0 = buf_paddr | 0x8000000000000000ULL;
    else                     _fbf_chbuf[channel].buf1 = buf_paddr | 0x8000000000000000ULL;

    // reset fbf buffer status
    _fbf_chbuf[channel].fbf  = _fbf_chbuf[channel].fbf & 0x7FFFFFFFFFFFFFFFULL;

    if ( USE_IOB )
    {
        // SYNC request for the channel descriptor, because
        // it will be read from XRAM by the CMA component
        _mmc_sync( _fbf_chbuf_paddr[channel], 32 );
    }

#if GIET_DEBUG_FBF_CMA
_printf(" - fbf    pbase = %l\n"
        " - buf0   pbase = %l\n"
        " - buf1   pbase = %l\n",
        _fbf_chbuf[channel].fbf,
        _fbf_chbuf[channel].buf0,
        _fbf_chbuf[channel].buf1 );
#endif


    return 0;

#else

    _printf("\n[GIET ERROR] in _sys_fbf_cma_display() : no CMA channel allocated\n");
    return -1;

#endif
} // end _sys_fbf_cma_display()

///////////////////////
int _sys_fbf_cma_stop()
{
#if NB_CMA_CHANNELS > 0

    // get channel index
    unsigned int channel = _get_context_slot( CTX_FBCMA_ID );

    if ( channel >= NB_CMA_CHANNELS )
    {
        _printf("\n[GIET ERROR] in _sys_fbf_cma_stop() : CMA channel index too large\n");
        return -1;
    }

    // Desactivate CMA channel
    _cma_stop_channel( channel );

    return 0;

#else

    _printf("\n[GIET ERROR] in _sys_fbf_cma_stop() : no CMA channel allocated\n");
    return -1;

#endif
} // end _sys_fbf_cma_stop()


//////////////////////////////////////////////////////////////////////////////
//           Miscelaneous syscall handlers 
//////////////////////////////////////////////////////////////////////////////

///////////////
int _sys_ukn() 
{
    _printf("\n[GIET ERROR] Undefined System Call / EPC = %x\n", _get_epc() );
    return -1;
}

////////////////////////////////////
int _sys_proc_xyp( unsigned int* x,
                   unsigned int* y,
                   unsigned int* p )
{
    unsigned int gpid = _get_procid();  // global processor index from CPO register

    *x = (gpid >> (Y_WIDTH + P_WIDTH)) & ((1<<X_WIDTH)-1);
    *y = (gpid >> P_WIDTH) & ((1<<Y_WIDTH)-1);
    *p = gpid & ((1<<P_WIDTH)-1);

    return 0;
}

//////////////////////////////////
int _sys_task_exit( char* string ) 
{
    unsigned int date       = _get_proctime();

    unsigned int gpid       = _get_procid();
    unsigned int cluster_xy = gpid >> P_WIDTH;
    unsigned int y          = cluster_xy & ((1<<Y_WIDTH)-1);
    unsigned int x          = cluster_xy >> Y_WIDTH;
    unsigned int lpid       = gpid & ((1<<P_WIDTH)-1);

    unsigned int task_id    = _get_context_slot(CTX_LTID_ID);

    // print exit message
    _printf("\n[GIET] Exit task %d on processor[%d,%d,%d] at cycle %d"
            "\n       Cause : %s\n\n",
            task_id, x, y, lpid, date, string );

    // goes to sleeping state
    _set_context_slot(CTX_RUN_ID, 0);

    // deschedule
    _context_switch();

    return 0;
} 

//////////////////////
int _context_switch() 
{
    unsigned int save_sr;

    _it_disable( &save_sr );
    _ctx_switch();
    _it_restore( &save_sr );

    return 0;
}

////////////////////////
int _sys_local_task_id()
{
    return _get_context_slot(CTX_LTID_ID);
}

/////////////////////////
int _sys_global_task_id()
{
    return _get_context_slot(CTX_GTID_ID);
}

////////////////////
int _sys_thread_id()
{
    return _get_context_slot(CTX_TRDID_ID);
}

//////////////////////////////////////
int _sys_procs_number( unsigned int  x, 
                       unsigned int  y, 
                       unsigned int* number )
{
    mapping_header_t * header  = (mapping_header_t *)SEG_BOOT_MAPPING_BASE;
    mapping_cluster_t * cluster = _get_cluster_base(header);

    if ( (x < X_SIZE) && (y < Y_SIZE) )
    {
        *number = cluster[(x*Y_SIZE)+y].procs;
        return 0;
    }
    else 
    {
        _printf("\n[GIET ERROR] in _sys_procs_number() : illegal (x,y) coordinates\n" );
        return -1;
    }
}

///////////////////////////////////////////////////////
int _sys_vobj_get_vbase( char*             vspace_name, 
                         char*             vobj_name, 
                         unsigned int*     vbase ) 
{
    mapping_header_t * header = (mapping_header_t *)SEG_BOOT_MAPPING_BASE;
    mapping_vspace_t * vspace = _get_vspace_base(header);
    mapping_vobj_t * vobj     = _get_vobj_base(header);

    unsigned int vspace_id;
    unsigned int vobj_id;

    // scan vspaces 
    for (vspace_id = 0; vspace_id < header->vspaces; vspace_id++) 
    {
        if (_strncmp( vspace[vspace_id].name, vspace_name, 31) == 0) 
        {
            // scan vobjs
            for (vobj_id = vspace[vspace_id].vobj_offset; 
                 vobj_id < (vspace[vspace_id].vobj_offset + vspace[vspace_id].vobjs); 
                 vobj_id++) 
            {
                if (_strncmp(vobj[vobj_id].name, vobj_name, 31) == 0) 
                {
                    *vbase = vobj[vobj_id].vbase;
                    return 0;
                }
            } 
        }
    } 
    return -1;    // not found 
}

/////////////////////////////////////////////////////////
int _sys_vobj_get_length( char*         vspace_name, 
                          char*         vobj_name,
                          unsigned int* length ) 
{
    mapping_header_t * header = (mapping_header_t *)SEG_BOOT_MAPPING_BASE;
    mapping_vspace_t * vspace = _get_vspace_base(header);
    mapping_vobj_t * vobj     = _get_vobj_base(header);

    unsigned int vspace_id;
    unsigned int vobj_id;

    // scan vspaces 
    for (vspace_id = 0; vspace_id < header->vspaces; vspace_id++) 
    {
        if (_strncmp( vspace[vspace_id].name, vspace_name, 31) == 0) 
        {
            // scan vobjs
            for (vobj_id = vspace[vspace_id].vobj_offset; 
                 vobj_id < (vspace[vspace_id].vobj_offset + vspace[vspace_id].vobjs); 
                 vobj_id++) 
            {
                if (_strncmp(vobj[vobj_id].name, vobj_name, 31) == 0) 
                {
                    *length = vobj[vobj_id].length;
                    return 0;
                }
            } 
        }
    } 
    return -1;    // not found 
}

////////////////////////////////////////
int _sys_xy_from_ptr( void*         ptr,
                      unsigned int* x,
                      unsigned int* y )
{
    unsigned int ret;
    unsigned int ppn;
    unsigned int flags;
    unsigned int vpn  = (((unsigned int)ptr)>>12);
    
    // get the page table pointer
    page_table_t* pt = (page_table_t*)_get_context_slot( CTX_PTAB_ID ); 

    // compute the physical address
    if ( (ret = _v2p_translate( pt, vpn, &ppn, &flags )) ) return -1;

    *x = (ppn>>24) & 0xF;
    *y = (ppn>>20) & 0xF;
    return 0;
}

/////////////////////////////////////////
int _sys_heap_info( unsigned int* vaddr, 
                    unsigned int* length,
                    unsigned int  x,
                    unsigned int  y ) 
{
    mapping_header_t * header  = (mapping_header_t *)SEG_BOOT_MAPPING_BASE;
    mapping_task_t *   tasks   = _get_task_base(header);
    mapping_vobj_t *   vobjs   = _get_vobj_base(header);
    mapping_vspace_t * vspaces = _get_vspace_base(header);

    unsigned int task_id;
    unsigned int vspace_id;
    unsigned int vobj_id = 0xFFFFFFFF;

    // searching the heap vobj_id
    if ( (x < X_SIZE) && (y < Y_SIZE) )  // searching a task in cluster(x,y)
    {
        // get vspace global index
        vspace_id = _get_context_slot(CTX_VSID_ID);

        // scan all tasks in vspace
        unsigned int min = vspaces[vspace_id].task_offset ;
        unsigned int max = min + vspaces[vspace_id].tasks ;
        for ( task_id = min ; task_id < max ; task_id++ )
        {
            if ( tasks[task_id].clusterid == (x * Y_SIZE + y) )
            {
                vobj_id = tasks[task_id].heap_vobj_id;
                if ( vobj_id != 0xFFFFFFFF ) break;
            }
        }
    }
    else                                // searching in the calling task 
    {
        task_id = _get_context_slot(CTX_GTID_ID);
        vobj_id = tasks[task_id].heap_vobj_id;
    }

    // analysing the vobj_id
    if ( vobj_id != 0xFFFFFFFF ) 
    {
        *vaddr  = vobjs[vobj_id].vbase;
        *length = vobjs[vobj_id].length;
        return 0;
    }
    else 
    {
        *vaddr  = 0;
        *length = 0;
        return -1;
    }
}  // end _sys_heap_info()


// Local Variables:
// tab-width: 4
// c-basic-offset: 4
// c-file-offsets:((innamespace . 0)(inline-open . 0))
// indent-tabs-mode: nil
// End:
// vim: filetype=c:expandtab:shiftwidth=4:tabstop=4:softtabstop=4

