/////////////////////////////////////////////////////////////////////////////////////////////
// File   : main.c   (for classif application)
// Date   : november 2014
// author : Alain Greiner
/////////////////////////////////////////////////////////////////////////////////////////////
// This multi-threaded application takes a Gigabit Ethernet packets stream, and
// makes a packet analysis and classification, based on the source MAC address.
// It uses the NIC peripheral, and the chained buffer build by the CMA component
// to consume packets on the Gigabit Ethernet port. 
//
// It is described as a TCG (Task and Communication Graph) containing
// - one "load" task per cluster.
// - from one to three "analyse" tasks per cluster.
// In each cluster, the "load" task communicates with the local "analyse" tasks through 
// a local MWMR fifo containing NB_PROCS_MAX containers (one item = one container).
// The MWMR fifo descriptor and the data buffer containing the containers are defined
// as global variables distributed in (up to) 16 clusters. 
//
// WARNING: the platform cannot contain more than 16 clusters: (X_SIZE < 4) && (Y_SIZE < 4)
//
// 1) The "load" task transfer one container from the kernel chbuf associated to the
//    NIC_RX channel, to a private buffer. Then it copies this bufferer to the local MWMR fifo.
//    Each "load" task loads CONTAINERS_MAX containers before exit, and the
//    task in cluster[0,0] displays the results stored in global counters filled
//    by the "analyse" tasks.
//
// 2) The "analyse" task transfer one container from the local MWMR fifo to a private
//    local buffer. It analyse each packet contained in the container, compute the
//    packet type, depending on the 4 MSB bits of the source MAC address,
//    and increment the corresponding counters.
//
// It uses the he following hardware parameters, defined in the hard_config.h file:
// - X_SIZE       : number of clusters in a row
// - Y_SIZE       : number of clusters in a column
// - NB_PROCS_MAX : number of processors per cluster
/////////////////////////////////////////////////////////////////////////////////////////////

#include "stdio.h"
#include "barrier.h"
#include "malloc.h"
#include "user_lock.h"
#include "mwmr_channel.h"
#include "hard_config.h"

#define LOAD_VERBOSE     0
#define ANALYSE_VERBOSE  0
#define CONTAINERS_MAX   10

///////////  distributed data /////////////////////////////////////////////////////////////
// - fifo_x_y is the local MWMR fifo descriptor
// - data_x_y is the local MWMR fifo data buffer
// - sync_x_y is the local variable signaling MWMR fifo initialisation
///////////  distributed data /////////////////////////////////////////////////////////////

#if ( (X_SIZE > 0) && (Y_SIZE > 0) )
__attribute__((section (".data_0_0")))  mwmr_channel_t fifo_0_0;
__attribute__((section (".data_0_0")))  unsigned int   data_0_0[NB_PROCS_MAX<<10];
__attribute__((section (".data_0_0")))  unsigned int   sync_0_0 = 0;
#endif
#if ( (X_SIZE > 0) && (Y_SIZE > 1) )
__attribute__((section (".data_0_1")))  mwmr_channel_t fifo_0_1;
__attribute__((section (".data_0_1")))  unsigned int   data_0_1[NB_PROCS_MAX<<10];
__attribute__((section (".data_0_1")))  unsigned int   sync_0_1 = 0;
#endif
#if ( (X_SIZE > 0) && (Y_SIZE > 2) )
__attribute__((section (".data_0_2")))  mwmr_channel_t fifo_0_2;
__attribute__((section (".data_0_2")))  unsigned int   data_0_2[NB_PROCS_MAX<<10];
__attribute__((section (".data_0_2")))  unsigned int   sync_0_2 = 0;
#endif
#if ( (X_SIZE > 0) && (Y_SIZE > 3) )
__attribute__((section (".data_0_3")))  mwmr_channel_t fifo_0_3;
__attribute__((section (".data_0_3")))  unsigned int   data_0_3[NB_PROCS_MAX<<10];
__attribute__((section (".data_0_3")))  unsigned int   sync_0_3 = 0;
#endif
#if ( (X_SIZE > 1) && (Y_SIZE > 0) )
__attribute__((section (".data_1_0")))  mwmr_channel_t fifo_1_0;
__attribute__((section (".data_1_0")))  unsigned int   data_1_0[NB_PROCS_MAX<<10];
__attribute__((section (".data_1_0")))  unsigned int   sync_1_0 = 0;
#endif
#if ( (X_SIZE > 1) && (Y_SIZE > 1) )
__attribute__((section (".data_1_1")))  mwmr_channel_t fifo_1_1;
__attribute__((section (".data_1_1")))  unsigned int   data_1_1[NB_PROCS_MAX<<10];
__attribute__((section (".data_1_1")))  unsigned int   sync_1_1 = 0;
#endif
#if ( (X_SIZE > 1) && (Y_SIZE > 2) )
__attribute__((section (".data_1_2")))  mwmr_channel_t fifo_1_2;
__attribute__((section (".data_1_2")))  unsigned int   data_1_2[NB_PROCS_MAX<<10];
__attribute__((section (".data_1_2")))  unsigned int   sync_1_2 = 0;
#endif
#if ( (X_SIZE > 1) && (Y_SIZE > 3) )
__attribute__((section (".data_1_3")))  mwmr_channel_t fifo_1_3;
__attribute__((section (".data_1_3")))  unsigned int   data_1_3[NB_PROCS_MAX<<10];
__attribute__((section (".data_1_3")))  unsigned int   sync_1_3 = 0;
#endif
#if ( (X_SIZE > 2) && (Y_SIZE > 0) )
__attribute__((section (".data_2_0")))  mwmr_channel_t fifo_2_0;
__attribute__((section (".data_2_0")))  unsigned int   data_2_0[NB_PROCS_MAX<<10];
__attribute__((section (".data_2_0")))  unsigned int   sync_2_0 = 0;
#endif
#if ( (X_SIZE > 2) && (Y_SIZE > 1) )
__attribute__((section (".data_2_1")))  mwmr_channel_t fifo_2_1;
__attribute__((section (".data_2_1")))  unsigned int   data_2_1[NB_PROCS_MAX<<10];
__attribute__((section (".data_2_1")))  unsigned int   sync_2_1 = 0;
#endif
#if ( (X_SIZE > 2) && (Y_SIZE > 2) )
__attribute__((section (".data_2_2")))  mwmr_channel_t fifo_2_2;
__attribute__((section (".data_2_2")))  unsigned int   data_2_2[NB_PROCS_MAX<<10];
__attribute__((section (".data_2_2")))  unsigned int   sync_2_2 = 0;
#endif
#if ( (X_SIZE > 2) && (Y_SIZE > 3) )
__attribute__((section (".data_2_3")))  mwmr_channel_t fifo_2_3;
__attribute__((section (".data_2_3")))  unsigned int   data_2_3[NB_PROCS_MAX<<10];
__attribute__((section (".data_2_3")))  unsigned int   sync_2_3 = 0;
#endif
#if ( (X_SIZE > 3) && (Y_SIZE > 0) )
__attribute__((section (".data_3_0")))  mwmr_channel_t fifo_3_0;
__attribute__((section (".data_3_0")))  unsigned int   data_3_0[NB_PROCS_MAX<<10];
__attribute__((section (".data_3_0")))  unsigned int   sync_3_0 = 0;
#endif
#if ( (X_SIZE > 3) && (Y_SIZE > 1) )
__attribute__((section (".data_3_1")))  mwmr_channel_t fifo_3_1;
__attribute__((section (".data_3_1")))  unsigned int   data_3_1[NB_PROCS_MAX<<10];
__attribute__((section (".data_3_1")))  unsigned int   sync_3_1 = 0;
#endif
#if ( (X_SIZE > 3) && (Y_SIZE > 2) )
__attribute__((section (".data_3_2")))  mwmr_channel_t fifo_3_2;
__attribute__((section (".data_3_2")))  unsigned int   data_3_2[NB_PROCS_MAX<<10];
__attribute__((section (".data_3_2")))  unsigned int   sync_3_2 = 0;
#endif
#if ( (X_SIZE > 3) && (Y_SIZE > 3) )
__attribute__((section (".data_3_3")))  mwmr_channel_t fifo_3_3;
__attribute__((section (".data_3_3")))  unsigned int   data_3_3[NB_PROCS_MAX<<10];
__attribute__((section (".data_3_3")))  unsigned int   sync_3_3 = 0;
#endif

/////////// shared variables in cluster[0,0] //////////////////////////

__attribute__((section (".data_0_0")))  unsigned int count[16];

__attribute__((section (".data_0_0")))  giet_barrier_t barrier;

__attribute__((section (".data_0_0")))  unsigned int global_init_ok = 0;

__attribute__((section (".data_0_0")))  unsigned int nic_channel;


/////////////////////////////////////////
__attribute__ ((constructor)) void load()
/////////////////////////////////////////
{
    // get processor identifiers
    unsigned int    x;
    unsigned int    y;
    unsigned int    l;
    giet_proc_xyp( &x, &y, &l );

    if (X_SIZE > 4 )  giet_exit("The X_SIZE parameter cannot be larger than 4\n");
    if (Y_SIZE > 4 )  giet_exit("The Y_SIZE parameter cannot be larger than 4\n");

    // local buffer to store one container
    unsigned int  temp[1024];

    // get pointer on local MWMR fifo descriptor and data buffer
    unsigned int    offset = ((x * 4) + y) * 0x10000;
    mwmr_channel_t* fifo = (mwmr_channel_t*)(((unsigned int)&fifo_0_0) + offset);
    unsigned int*   data = (unsigned int*)  (((unsigned int)data_0_0)  + offset);
    unsigned int*   sync = (unsigned int*)  (((unsigned int)&sync_0_0) + offset);

    if ( (x==X_SIZE-1) && (y==Y_SIZE-1) )
    giet_shr_printf("\n*** Task load starts on P[%d,%d,%d] at cycle %d\n"
                    "      &fifo = %x / &data = %x / &sync = %x\n",
                    x, y, l, giet_proctime(), 
                    (unsigned int)fifo, (unsigned int)data, (unsigned int)sync ); 

    // Task load on cluster[0,0] makes global initialisation:
    // - NIC & CMA channels allocation & initialisation.
    // - barrier for all load tasks initialisation.
    // Other load task wait completion.
    if ( (x==0) && (y==0) )
    {
        // get NIC_RX channel 
        nic_channel = giet_nic_rx_alloc();

        // start CMA transfer
        giet_nic_rx_start();

        // barrier init
        barrier_init( &barrier, X_SIZE * Y_SIZE );

        global_init_ok = 1;
    }
    else
    {
        while ( global_init_ok == 0 ) asm volatile ("nop");
    }    

    // Each load task initialises local MWMR fifo (width = 4kbytes / depth = NB_PROCS_MAX)
    mwmr_init( fifo , data , 1024 , NB_PROCS_MAX );

    // signal MWMR fifo initialisation completion
    *sync = 1;

    // main loop (on containers)
    unsigned int container = 0;
    while ( container < CONTAINERS_MAX ) 
    { 
        // get one container from kernel chbuf
        giet_nic_rx_move( nic_channel, temp );

        // get packets number
        unsigned int npackets = temp[0] & 0x0000FFFF;
        unsigned int nwords   = temp[0] >> 16;

        if ( (x==X_SIZE-1) && (y==Y_SIZE-1) )
        giet_shr_printf("\nTask load on P[%d,%d,%d] get container %d at cycle %d"
                        " : %d packets / %d words\n",
                        x, y, l, container, giet_proctime(), npackets, nwords );

        // put container to MWMR channel
        mwmr_write( fifo, temp, 1 );

        container++;
    }

    // all load tasks synchronise before result display
    barrier_wait( &barrier );

    // Task load in cluster[0,0] displays counters and stops NIC / CMA transfer
    if ( (x==0) && (y==0) )
    {
        giet_shr_printf("\n@@@@ Clasification Results @@@\n"
                        " - TYPE 0 : %d packets\n"
                        " - TYPE 1 : %d packets\n"
                        " - TYPE 2 : %d packets\n"
                        " - TYPE 3 : %d packets\n"
                        " - TYPE 4 : %d packets\n"
                        " - TYPE 5 : %d packets\n"
                        " - TYPE 6 : %d packets\n"
                        " - TYPE 7 : %d packets\n"
                        " - TYPE 8 : %d packets\n"
                        " - TYPE 9 : %d packets\n"
                        " - TYPE A : %d packets\n"
                        " - TYPE B : %d packets\n"
                        " - TYPE C : %d packets\n"
                        " - TYPE D : %d packets\n"
                        " - TYPE E : %d packets\n"
                        " - TYPE F : %d packets\n",
            count[0x0], count[0x1], count[0x2], count[0x3],
            count[0x4], count[0x5], count[0x6], count[0x7],
            count[0x8], count[0x9], count[0xA], count[0xB],
            count[0xC], count[0xD], count[0xE], count[0xF] );

        giet_nic_rx_stop();

        giet_nic_rx_stats();

    }

    // all load tasks synchronise before exit
    barrier_wait( &barrier );

    giet_exit("Task completed");
 
} // end load()

////////////////////////////////////////////
__attribute__ ((constructor)) void analyse()
////////////////////////////////////////////
{
    // get processor identifiers
    unsigned int    x;
    unsigned int    y;
    unsigned int    l;
    giet_proc_xyp( &x, &y, &l );

    // local buffer to store one container
    unsigned int  temp[1024];

    // get pointer on MWMR channel descriptor 
    unsigned int    offset = ((x * 4) + y) * 0x10000;
    mwmr_channel_t* fifo = (mwmr_channel_t*)(((unsigned int)&fifo_0_0) + offset);
    unsigned int*   sync = (unsigned int*)  (((unsigned int)&sync_0_0) + offset);

    if ( (x==X_SIZE-1) && (y==Y_SIZE-1) )
    giet_shr_printf("\n*** Task analyse starts on P[%d,%d,%d] at cycle %d\n"
                    "       &fifo = %x / &sync = %x\n",
                    x, y, l, giet_proctime(), 
                    (unsigned int)fifo, (unsigned int)sync );
    
    // wait MWMR channel initialisation (done by task load)
    while ( *sync == 0 ) asm volatile ("nop");

    // infinite loop (on containers)
    unsigned int nwords;     // number of words in container
    unsigned int npackets;   // number of packets in container
    unsigned int length;     // number of bytes in current packet
    unsigned int word;       // current packet first word in container
    unsigned int type;       // current packet type
    unsigned int p;          // current packet index
    while ( 1 )
    { 
        // get one container from MWMR fifo
        mwmr_read( fifo, temp, 1 );

        // get packets number
        npackets = temp[0] & 0x0000FFFF;
        nwords   = temp[0] >> 16;

        if ( (x==X_SIZE-1) && (y==Y_SIZE-1) )
        giet_shr_printf("\nTask analyse on P[%d,%d,%d] get container at cycle %d"
                        " : %d packets / %d words\n",
						x, y, l, giet_proctime(), npackets, nwords );

        // initialize word index in container
        word = 34;

        // loop on packets
        for( p = 0 ; p < npackets ; p++ )
        {
            // get packet length from container header
            if ( (p & 0x1) == 0 )  length = temp[1+(p>>1)] >> 16;
            else                   length = temp[1+(p>>1)] & 0x0000FFFF;

            // get packet type (source mac address 4 MSB bits)
            type = (temp[word+1] & 0x0000F000) >> 12;

            // increment counter
            atomic_increment( &count[type], 1 );

            // update word index 
            if ( length & 0x3 ) word += (length>>2)+1;
            else                word += (length>>2);
        }
    }
} // end analyse()

