///////////////////////////////////////////////////////////////////////////////////
// File     : sys_handler.h
// Date     : 01/04/2012
// Author   : alain greiner and joel porquet
// Copyright (c) UPMC-LIP6
///////////////////////////////////////////////////////////////////////////////////
// The sys_handler.c and sys_handler.h files are part of the GIET-VM nano-kernel.
// It define the syscall_vector[] (at the end of this file), as well as the 
// associated syscall handlers.
///////////////////////////////////////////////////////////////////////////////////

#ifndef _SYS_HANDLER_H
#define _SYS_HANDLER_H

#include "giet_config.h"
#include "locks.h"

#if !defined ( GIET_NIC_NBUFS )
# error: You must define GIET_NIC_NBUFS in the giet_config.h file
#endif

#if !defined ( GIET_NIC_NFAKE )
# error: You must define GIET_NIC_NFAKE in the giet_config.h file
#endif

#if !defined ( GIET_NIC_BUFSIZE )
# error: You must define GIET_NIC_BUFSIZE in the giet_config.h file
#endif

#if !defined ( GIET_NIC_TIMEOUT )
# error: You must define GIET_NIC_TIMEOUT in the giet_config.h file
#endif

#if !defined ( GIET_NIC_MAC4 )
# error: You must define GIET_NIC_MAC4 in the giet_config.h file
#endif

#if !defined ( GIET_NIC_MAC2 )
# error: You must define GIET_NIC_MAC2 in the giet_config.h file
#endif

#if ( (GIET_NIC_NBUFS + GIET_NIC_NFAKE) % 8 )
#error: GIET_NIC_NBUFS + GIET_NIC_NFAKE must be multiple of 8 for alignment
#endif

///////////////////////////////////////////////////////////////////////////////////
//     Syscall Vector Table (indexed by syscall index)
///////////////////////////////////////////////////////////////////////////////////

extern const void * _syscall_vector[64];

///////////////////////////////////////////////////////////////////////////////////
// This structure is used by the CMA component to move a stream 
// of images from two buffers in user space to the frame buffer in kernel space.
// it must be 64 bytes aligned.
// It contains two chbuf arrays:
// - The SRC chbuf contains two buffers (buf0 & buf1), that can be in user space.
// - The DST cbuf contains one single buffer (fbf), that is the frame buffer.
// - The length field define the buffer size (bytes)
///////////////////////////////////////////////////////////////////////////////////

typedef struct fbf_chbuf_s
{
    unsigned long long  buf0;        // physical address + status for user buffer 0 
    unsigned long long  buf1;        // physical address + status for user buffer 1
    unsigned long long  fbf;         // physical address + status for user buffer 0 
    unsigned int        length;      // buffer length (bytes)
    unsigned int        padding[9];  // 64 bytes alignment
} fbf_chbuf_t;   

//////////////////////////////////////////////////////////////////////////////////
// This structure is used by the CMA component to move a stream
// of packet containers between the NIC component an a chbuf containing
// a variable number of buffers in kernel space.
// The same structure is used for both TX or RX transfers.
// It must be 64 bytes aligned.
// The single buffer size and the number of buffers must be defined by the 
// GIET_NIC_BUFSIZE and GIET_NIC_NBUFS parameters in giet_config.h.
// - The buffer array implements the chbuf, and is concurently accessed 
//   by the CMA component and by the kernel code.
// - The lock must be taken by the kernel code, because several user tasks
//   can concurently try to consume buffers in the chbuf.
// - The index is only used by the kernel, and define the currently pointed 
//   buffer for read (RX transfer) or write (TX transfer).
//////////////////////////////////////////////////////////////////////////////////

typedef struct nic_chbuf_s
{
    unsigned long long  buffer[GIET_NIC_NBUFS];  // Kernel CHBUF
    unsigned long long  unused[GIET_NIC_NFAKE];  // padding for 64 bytes alignment
    unsigned int        index;                   // current buffer index
    unsigned int        padding[15];             // padding for 64 bytes alignment
} nic_chbuf_t;


//////////////////////////////////////////////////////////////////////////////////
//    TTY related syscall handlers
//////////////////////////////////////////////////////////////////////////////////

int _sys_tty_alloc();

int _sys_tty_write( const char*  buffer,
                    unsigned int length,
                    unsigned int channel );

int _sys_tty_read(  char*        buffer,
                    unsigned int length,
                    unsigned int channel );

int _sys_tty_get_lock( unsigned int   channel,
                       unsigned int * save_sr_ptr );

int _sys_tty_release_lock( unsigned int   channel,
                           unsigned int * save_sr_ptr );

//////////////////////////////////////////////////////////////////////////////
//    TIM related syscall handlers
//////////////////////////////////////////////////////////////////////////////

int _sys_tim_alloc();

int _sys_tim_start( unsigned int period );

int _sys_tim_stop();

//////////////////////////////////////////////////////////////////////////////
//    NIC related syscall handlers
//////////////////////////////////////////////////////////////////////////////

int _sys_nic_alloc( unsigned int is_rx );

int _sys_nic_start( unsigned int is_rx );

int _sys_nic_move( unsigned int is_rx,
                   unsigned int nic_channel,
                   void*        buffer );

int _sys_nic_stop( unsigned int is_rx );

int _sys_nic_clear( unsigned int is_rx );

int _sys_nic_stats( unsigned int is_rx );

//////////////////////////////////////////////////////////////////////////////
//    FBF related syscall handlers
//////////////////////////////////////////////////////////////////////////////

int _sys_fbf_sync_write( unsigned int offset,
                         void*        buffer,
                         unsigned int length );

int _sys_fbf_sync_read(  unsigned int offset,
                         void*        buffer,
                         unsigned int length );

int _sys_fbf_cma_alloc();

int _sys_fbf_cma_start( void*        vbase0, 
                        void*        vbase1,  
                        unsigned int length );

int _sys_fbf_cma_display( unsigned int buffer_index );

int _sys_fbf_cma_stop();

//////////////////////////////////////////////////////////////////////////////
//    Miscelaneous syscall handlers
//////////////////////////////////////////////////////////////////////////////

int _sys_ukn();

int _sys_proc_xyp( unsigned int* x,
                   unsigned int* y,
                   unsigned int* p );

int _sys_task_exit( char* string );

int _context_switch();

int _sys_local_task_id();

int _sys_global_task_id();

int _sys_thread_id();

int _sys_procs_number( unsigned int  x,
                       unsigned int  y, 
                       unsigned int* number );

int _sys_vobj_get_vbase( char*         vspace_name,
                         char*         vobj_name,
                         unsigned int* vbase );

int _sys_vobj_get_length( char*         vspace_name, 
                          char*         vobj_name,
                          unsigned int* length );

int _sys_xy_from_ptr( void*          ptr,
                      unsigned int*  x,
                      unsigned int*  y );

int _sys_heap_info( unsigned int* vaddr, 
                    unsigned int* length,
                    unsigned int  x,
                    unsigned int  y ); 

#endif

// Local Variables:
// tab-width: 4
// c-basic-offset: 4
// c-file-offsets:((innamespace . 0)(inline-open . 0))
// indent-tabs-mode: nil
// End:
// vim: filetype=c:expandtab:shiftwidth=4:tabstop=4:softtabstop=4

