///////////////////////////////////////////////////////////////////////////////////
// File     : locks.h
// Date     : 01/12/2014
// Author   : alain greiner
// Copyright (c) UPMC-LIP6
///////////////////////////////////////////////////////////////////////////////////
// The locks.c and locks.h files are part of the GIET-VM nano-kernel.
// They define both atomic increment operations and three types of locks.
///////////////////////////////////////////////////////////////////////////////////

#ifndef GIET_LOCKS_H
#define GIET_LOCKS_H

#include "hard_config.h"

///////////////////////////////////////////////////////////////////////////////////
//      Simple lock structure and access functions
///////////////////////////////////////////////////////////////////////////////////

typedef struct simple_lock_s
{
    unsigned int value;          // lock taken if non zero
    unsigned int padding[15];    // for 64 bytes alignment
} simple_lock_t;

extern void _simple_lock_acquire( simple_lock_t* lock );

extern void _simple_lock_release( simple_lock_t* lock );

///////////////////////////////////////////////////////////////////////////////////
//      Queuing lock structure and access functions
///////////////////////////////////////////////////////////////////////////////////

typedef struct spin_lock_s 
{
    unsigned int current;        // current slot index
    unsigned int free;           // next free tiket index
    unsigned int padding[14];    // for 64 bytes alignment
} spin_lock_t;

extern unsigned int _atomic_increment( unsigned int* ptr,
                                       unsigned int  increment );

extern void _spin_lock_init( spin_lock_t* lock );

extern void _spin_lock_acquire( spin_lock_t* lock );

extern void _spin_lock_release( spin_lock_t* lock );

//////////////////////////////////////////////////////////////////////////////////
//      SBT lock structures and access functions
//////////////////////////////////////////////////////////////////////////////////

typedef struct lock_node_s 
{
    unsigned int            taken;           // lock taken if non zero
    unsigned int            level;           // hierarchical level (0 is bottom)
    struct lock_node_s*     parent;          // pointer on parent node (NULL for root)
    struct lock_node_s*     child0;          // pointer on children node
    struct lock_node_s*     child1;          // pointer on children node
    unsigned int            x;               // cluster x coordinate        
    unsigned int            y;               // cluster y coordinate           
    unsigned int            padding[9];      // for 64 bytes alignment         
} lock_node_t;

typedef struct sbt_lock_s 
{
    unsigned int    ntasks;                   // total number of expected tasks
    lock_node_t*    node[X_SIZE][Y_SIZE][9];  // array of pointers on SBT nodes 
} sbt_lock_t;

extern void _sbt_lock_init( sbt_lock_t*   lock );

extern void _sbt_lock_acquire( sbt_lock_t*  lock );

extern void _sbt_lock_release( sbt_lock_t*  lock );

#endif

// Local Variables:
// tab-width: 4
// c-basic-offset: 4
// c-file-offsets:((innamespace . 0)(inline-open . 0))
// indent-tabs-mode: nil
// End:
// vim: filetype=c:expandtab:shiftwidth=4:tabstop=4:softtabstop=4

