//////////////////////////////////////////////////////////////////////////////////
// File     : barrier.h         
// Date     : 01/04/2012
// Author   : alain greiner
// Copyright (c) UPMC-LIP6
///////////////////////////////////////////////////////////////////////////////////
// The barrier.c and barrier.h files are part of the GIET-VM nano-kernel.
// This user-level library provides a synchronisation service between several
// tasks sharing the same address space in a parallel multi-tasks application.
//
// There is actually two types of barriers:
// 1) The "giet_barrier_t" is a simple sense-reversing barrier.
//    It can be safely used several times (in a loop for example),
//    but it does not scale, and should not be used when the number
//    of tasks is larger than few tens.
//
// 2) The giet_sbt_barrier_t" can be used in multi-clusters architectures,
//    and is implemented as a physically distributed Sliced-Binary-Tree (SBT).
//    WARNING: The following placement constraints must be respected:
//    - The number of involved clusters must be a power of 2.
//    - The number of involved tasks in a cluster is the same in all clusters.
//    - The involved clusters form a mesh[N][N] or a mesh[N][N/2]
//    - The lower left involved cluster is cluster(0,0)  
//
// Neither the barrier_init(), nor the barrier_wait() function require a syscall.
// For both types of barriers, the barrier initialisation should be done by
// one single task.
///////////////////////////////////////////////////////////////////////////////////

#ifndef _BARRIER_H_
#define _BARRIER_H_

#include "hard_config.h"

///////////////////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////////////
//  simple barrier structure and access functions
///////////////////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////////////

typedef struct giet_barrier_s 
{
    char         name[32];   // barrier name
    unsigned int sense;      // barrier state (toggle)
    unsigned int ntasks;     // total number of expected tasks
    unsigned int count;      // number of not arrived tasks
} giet_barrier_t;

///////////////////////////////////////////////////
extern void barrier_init( giet_barrier_t* barrier,
                          unsigned int    ntasks );   

////////////////////////////////////////////////////
extern void barrier_wait( giet_barrier_t* barrier );

///////////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////////
// SBT barrier structures and access functions
//////////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////////

typedef struct sbt_node_s 
{
    unsigned int       arity;           // number of children (must be 2 or 4)
    unsigned int       count;           // number of not arrived children
    unsigned int       sense;           // barrier state (toggle)
    unsigned int       level;           // hierarchical level (0 is bottom)
    struct sbt_node_s* parent;          // pointer on parent node (NULL for root)
    struct sbt_node_s* child0;          // pointer on children node
    struct sbt_node_s* child1;          // pointer on children node
    unsigned int       padding;         // for 32 bytes alignment
} sbt_node_t;

typedef struct giet_sbt_barrier_s 
{
    char            name[32];                 // barrier name
    unsigned int    ntasks;                   // total number of expected tasks
    sbt_node_t*     node[X_SIZE][Y_SIZE][9];  // array of pointers on SBT nodes 
} giet_sbt_barrier_t;

///////////////////////////////////////////////////////////
extern void sbt_barrier_init( giet_sbt_barrier_t*  barrier,
                              unsigned int         nclusters,
                              unsigned int         ntasks );   

/////////////////////////////////////////////////////////////
extern void sbt_barrier_wait( giet_sbt_barrier_t*  barrier );

/////////////////////////////////////////////
void sbt_build( giet_sbt_barrier_t*  barrier,
                unsigned int         x,
                unsigned int         y,
                unsigned int         level, 
                sbt_node_t*          parent,
                unsigned int         ntasks );

///////////////////////////////////////
void sbt_decrement( sbt_node_t* node );

///////////////////////////////////
void sbt_release( sbt_node_t* node,
                  unsigned int expected );



#endif

// Local Variables:
// tab-width: 4
// c-basic-offset: 4
// c-file-offsets:((innamespace . 0)(inline-open . 0))
// indent-tabs-mode: nil
// End:
// vim: filetype=c:expandtab:shiftwidth=4:tabstop=4:softtabstop=4

