///////////////////////////////////////////////////////////////////////////////////
// File     : sys_handler.c
// Date     : 01/04/2012
// Author   : alain greiner and joel porquet
// Copyright (c) UPMC-LIP6
///////////////////////////////////////////////////////////////////////////////////

#include <sys_handler.h>
#include <tty_driver.h>
#include <tim_driver.h>
#include <ioc_driver.h>
#include <nic_driver.h>
#include <mmc_driver.h>
#include <cma_driver.h>
#include <ctx_handler.h>
#include <fat32.h>
#include <utils.h>
#include <kernel_malloc.h>
#include <tty0.h>
#include <vmem.h>
#include <hard_config.h>
#include <giet_config.h>
#include <mapping_info.h>

#if !defined(SEG_BOOT_MAPPING_BASE) 
# error: You must define SEG_BOOT_MAPPING_BASE in the hard_config.h file
#endif

#if !defined(NB_TTY_CHANNELS) 
# error: You must define NB_TTY_CHANNELS in the hard_config.h file
#endif

#if (NB_TTY_CHANNELS < 1)
# error: NB_TTY_CHANNELS cannot be smaller than 1!
#endif

#if !defined(NB_TIM_CHANNELS) 
# error: You must define NB_TIM_CHANNELS in the hard_config.h file
#endif

#if !defined(NB_NIC_CHANNELS) 
# error: You must define NB_NIC_CHANNELS in the hard_config.h file
#endif

#if !defined(NB_CMA_CHANNELS) 
# error: You must define NB_CMA_CHANNELS in the hard_config.h file
#endif

#if !defined(GIET_NO_HARD_CC)
# error: You must define GIET_NO_HARD_CC in the giet_config.h file
#endif

#if !defined ( GIET_NIC_MAC4 )
# error: You must define GIET_NIC_MAC4 in the giet_config.h file
#endif

#if !defined ( GIET_NIC_MAC2 )
# error: You must define GIET_NIC_MAC2 in the giet_config.h file
#endif


////////////////////////////////////////////////////////////////////////////
//     Channel allocators for peripherals
//     (TTY[0] is reserved for kernel)
////////////////////////////////////////////////////////////////////////////

__attribute__((section(".kdata")))
unsigned int _tty_channel_allocator    = 1;

__attribute__((section(".kdata")))
unsigned int _tim_channel_allocator    = 0;

__attribute__((section(".kdata")))
unsigned int _cma_channel_allocator    = 0;

__attribute__((section(".kdata")))
unsigned int _nic_rx_channel_allocator = 0;

__attribute__((section(".kdata")))
unsigned int _nic_tx_channel_allocator = 0;

////////////////////////////////////////////////////////////////////////////
// These global variables is defined in tty0.c and tty_driver.c files.
////////////////////////////////////////////////////////////////////////////

extern sqt_lock_t _tty0_sqt_lock;

extern unsigned int _tty_rx_full[NB_TTY_CHANNELS];

extern unsigned int _tty_rx_buf[NB_TTY_CHANNELS];

////////////////////////////////////////////////////////////////////////////
//     NIC_RX and NIC_TX chbuf arrays 
////////////////////////////////////////////////////////////////////////////

__attribute__((section(".kdata")))
nic_chbuf_t  _nic_rx_chbuf[NB_NIC_CHANNELS] __attribute__((aligned(64)));

__attribute__((section(".kdata")))
nic_chbuf_t  _nic_tx_chbuf[NB_NIC_CHANNELS] __attribute__((aligned(64)));

////////////////////////////////////////////////////////////////////////////
// FBF related chbuf descriptors array, indexed by the CMA channel index.
// Physical addresses of these chbuf descriptors required for L2 cache sync.
////////////////////////////////////////////////////////////////////////////

__attribute__((section(".kdata")))
fbf_chbuf_t _fbf_chbuf[NB_CMA_CHANNELS] __attribute__((aligned(64)));

__attribute__((section(".kdata")))
unsigned long long _fbf_chbuf_paddr[NB_CMA_CHANNELS];

////////////////////////////////////////////////////////////////////////////
//    Initialize the syscall vector with syscall handlers
// Note: This array must be synchronised with the define in file stdio.h
////////////////////////////////////////////////////////////////////////////

__attribute__((section(".kdata")))
const void * _syscall_vector[64] = 
{
    &_sys_proc_xyp,         /* 0x00 */
    &_get_proctime,         /* 0x01 */
    &_sys_tty_write,        /* 0x02 */
    &_sys_tty_read,         /* 0x03 */
    &_sys_tty_alloc,        /* 0x04 */
    &_sys_tty_get_lock,     /* 0x05 */
    &_sys_tty_release_lock, /* 0x06 */
    &_sys_heap_info,        /* 0x07 */
    &_sys_local_task_id,    /* 0x08 */
    &_sys_global_task_id,   /* 0x09 */ 
    &_sys_fbf_cma_alloc,    /* 0x0A */
    &_sys_fbf_cma_start,    /* 0x0B */
    &_sys_fbf_cma_display,  /* 0x0C */
    &_sys_fbf_cma_stop,     /* 0x0D */
    &_sys_task_exit,        /* 0x0E */
    &_sys_procs_number,     /* 0x0F */

    &_sys_fbf_sync_write,   /* 0x10 */
    &_sys_fbf_sync_read,    /* 0x11 */
    &_sys_thread_id,        /* 0x12 */
    &_sys_ukn,              /* 0x13 */
    &_sys_tim_alloc,        /* 0x14 */
    &_sys_tim_start,        /* 0x15 */ 
    &_sys_tim_stop,         /* 0x16 */
    &_sys_ukn,              /* 0x17 */
    &_sys_ukn,              /* 0x18 */   
    &_context_switch,       /* 0x19 */
    &_sys_vobj_get_vbase,   /* 0x1A */
    &_sys_vobj_get_length,  /* 0x1B */
    &_sys_xy_from_ptr,      /* 0x1C */
    &_sys_ukn,              /* 0x1D */
    &_sys_ukn,              /* 0x1E */
    &_sys_ukn,              /* 0x1F */

    &_fat_user_open,        /* 0x20 */
    &_fat_user_read,        /* 0x21 */
    &_fat_user_write,       /* 0x22 */
    &_fat_user_lseek,       /* 0x23 */
    &_fat_fstat,            /* 0x24 */
    &_fat_close,            /* 0x25 */
    &_sys_ukn,              /* 0x26 */
    &_sys_ukn,              /* 0x27 */
    &_sys_ukn,              /* 0x28 */
    &_sys_ukn,              /* 0x29 */
    &_sys_ukn,              /* 0x2A */
    &_sys_ukn,              /* 0x2B */
    &_sys_ukn,              /* 0x2C */
    &_sys_ukn,              /* 0x2D */
    &_sys_ukn,              /* 0x2E */
    &_sys_ukn,              /* 0x2F */

    &_sys_nic_alloc,        /* 0x30 */
    &_sys_nic_start,        /* 0x31 */
    &_sys_nic_move,         /* 0x32 */
    &_sys_nic_stop,         /* 0x33 */
    &_sys_nic_stats,        /* 0x34 */
    &_sys_nic_clear,        /* 0x35 */ 
    &_sys_ukn,              /* 0x36 */
    &_sys_ukn,              /* 0x37 */
    &_sys_ukn,              /* 0x38 */   
    &_sys_ukn,              /* 0x39 */
    &_sys_ukn,              /* 0x3A */
    &_sys_ukn,              /* 0x3B */
    &_sys_ukn,              /* 0x3C */
    &_sys_ukn,              /* 0x3D */
    &_sys_ukn,              /* 0x3E */
    &_sys_ukn,              /* 0x3F */
};


//////////////////////////////////////////////////////////////////////////////
//             TTY related syscall handlers 
//////////////////////////////////////////////////////////////////////////////

////////////////////
int _sys_tty_alloc()
{
    // get a new TTY terminal index
    unsigned int channel = _atomic_increment( &_tty_channel_allocator, 1 );
    unsigned int thread  = _get_context_slot( CTX_TRDID_ID );
    unsigned int vspace  = _get_context_slot( CTX_VSID_ID );

    if ( channel >= NB_TTY_CHANNELS )
    {
        _printf("\n[GIET_ERROR] in _sys_tty_alloc() : not enough TTY channels\n");
        return -1;
    }
    else
    {
        _printf("\n[GIET WARNING] TTY channel %d allocated "
                " to thread %d in vspace %d\n", channel, thread, vspace );
        _set_context_slot( CTX_TTY_ID, channel );
        return 0;
    }
}

/////////////////////////////////////////////////
int _sys_tty_write( const char*  buffer,    
                    unsigned int length,    // number of characters
                    unsigned int channel)   // channel index 
{
    unsigned int  nwritten;

    // compute and check tty channel
    if( channel == 0xFFFFFFFF )  channel = _get_context_slot(CTX_TTY_ID);
    if( channel >= NB_TTY_CHANNELS ) return -1;

    // write string to TTY channel
    for (nwritten = 0; nwritten < length; nwritten++) 
    {
        // check tty's status 
        if ( _tty_get_register( channel, TTY_STATUS ) & 0x2 )  break;

        // write one byte
        if (buffer[nwritten] == '\n') 
        {
            _tty_set_register( channel, TTY_WRITE, (unsigned int)'\r' );
        }
        _tty_set_register( channel, TTY_WRITE, (unsigned int)buffer[nwritten] );
    }
    
    return nwritten;
}

////////////////////////////////////////////////
int _sys_tty_read( char*        buffer, 
                   unsigned int length,    // unused
                   unsigned int channel)   // channel index
{
    // compute and check tty channel
    if( channel == 0xFFFFFFFF )  channel = _get_context_slot(CTX_TTY_ID);
    if( channel >= NB_TTY_CHANNELS ) return -1;

    // read one character from TTY channel
    if (_tty_rx_full[channel] == 0) 
    {
        return 0;
    }
    else 
    {
        *buffer = _tty_rx_buf[channel];
        _tty_rx_full[channel] = 0;
        return 1;
    }
}

///////////////////////////////////////////
int _sys_tty_get_lock( unsigned int   channel,       // unused
                       unsigned int * save_sr_ptr )
{
    // check tty channel
    if( channel != 0 )  return 1;

    _it_disable( save_sr_ptr );
    _sqt_lock_acquire( &_tty0_sqt_lock );
    return 0;
}

///////////////////////////////////////////////
int _sys_tty_release_lock( unsigned int   channel,
                           unsigned int * save_sr_ptr )
{
    // check tty channel
    if( channel != 0 )  return 1;

    _sqt_lock_release( &_tty0_sqt_lock );
    _it_restore( save_sr_ptr );
    return 0;
}

//////////////////////////////////////////////////////////////////////////////
//             TIM related syscall handlers 
//////////////////////////////////////////////////////////////////////////////

////////////////////
int _sys_tim_alloc()
{
    // get a new timer index 
    unsigned int channel = _atomic_increment( &_tim_channel_allocator, 1 );
    unsigned int thread  = _get_context_slot( CTX_TRDID_ID );
    unsigned int vspace  = _get_context_slot( CTX_VSID_ID );

    if ( channel >= NB_TIM_CHANNELS )
    {
        _printf("\n[GIET_ERROR] in _sys_tim_alloc() : not enough TIM channels\n");
        return -1;
    }
    else
    {
        _printf("\n[GIET WARNING] TIM channel %d allocated "
                " to thread %d in vspace %d\n", channel, thread, vspace );
        _set_context_slot( CTX_TIM_ID, channel );
        return 0;
    }
}

/////////////////////////////////////////
int _sys_tim_start( unsigned int period )
{
    // get timer index
    unsigned int channel = _get_context_slot( CTX_TIM_ID );
    if ( channel >= NB_TIM_CHANNELS )
    {
        _printf("\n[GIET_ERROR] in _sys_tim_start() : not enough TIM channels\n");
        return -1;
    }

    // start timer
    _timer_start( channel, period );

    return 0;
}

///////////////////
int _sys_tim_stop()
{
    // get timer index
    unsigned int channel = _get_context_slot( CTX_TIM_ID );
    if ( channel >= NB_TIM_CHANNELS )
    {
        _printf("\n[GIET_ERROR] in _sys_tim_stop() : illegal timer index\n");
        return -1;
    }

    // stop timer
    _timer_stop( channel );

    return 0;
}

//////////////////////////////////////////////////////////////////////////////
//             NIC related syscall handlers 
//////////////////////////////////////////////////////////////////////////////

#define NIC_CONTAINER_SIZE 4096

////////////////////////////////////////
int _sys_nic_alloc( unsigned int is_rx,
                    unsigned int xmax,
                    unsigned int ymax )
{
    // check xmax / ymax parameters
    if ( xmax > X_SIZE )
    {
        _printf("\n[GIET_ERROR] in _sys_nic_alloc() xmax argument too large\n");
        return -1;
    }
    if ( ymax > Y_SIZE )
    {
        _printf("\n[GIET_ERROR] in _sys_nic_alloc() ymax argument too large\n");
        return -1;
    }

    // get a NIC_RX or NIC_TX channel index 
    unsigned int nic_channel;
    unsigned int cma_channel;

    if ( is_rx ) nic_channel = _atomic_increment( &_nic_rx_channel_allocator, 1 );
    else         nic_channel = _atomic_increment( &_nic_tx_channel_allocator, 1 );

    if ( (nic_channel >= NB_NIC_CHANNELS) )
    {
        _printf("\n[GIET_ERROR] in _sys_nic_alloc() not enough NIC channels\n");
        return -1;
    }

    // get a CMA channel index
    cma_channel = _atomic_increment( &_cma_channel_allocator, 1 );

    if ( cma_channel >= NB_CMA_CHANNELS )
    {
        _printf("\n[GIET_ERROR] in _sys_nic_alloc() not enough CMA channels\n");
        return -1;
    }

#if GIET_DEBUG_NIC
unsigned int thread  = _get_context_slot( CTX_TRDID_ID );
_printf("\n[GIET DEBUG NIC] Task %d enters sys_nic_alloc() at cycle %d\n"
        " nic_channel = %d / cma_channel = %d\n"
        thread , _get_proctime() , nic_channel , cma_channel );
#endif

    // register nic_index and cma_index in task context
    if ( is_rx )
    {
        _set_context_slot( CTX_NIC_RX_ID, nic_channel );
        _set_context_slot( CTX_CMA_RX_ID, cma_channel );
    }
    else
    {
        _set_context_slot( CTX_NIC_TX_ID, nic_channel );
        _set_context_slot( CTX_CMA_TX_ID, cma_channel );
    }

    // physical addresses to be registered in the CMA registers
    unsigned long long nic_chbuf_pbase;     // NIC chbuf physical address
    unsigned long long ker_chbuf_pbase;     // kernel chbuf physical address

    // These variables are used for the various V2P translation
    unsigned int       ptab  = _get_context_slot(CTX_PTAB_ID);
    unsigned int       ppn;
    unsigned int       flags;
    unsigned int       vaddr;

    // allocate one kernel container per cluster in the (xmax / ymax) mesh
    unsigned int        cx;              // cluster X coordinate
    unsigned int        cy;              // cluster Y coordinate
    unsigned int        index;           // container index in chbuf
    unsigned long long  cont_paddr;      // container physical address

    for ( cx = 0 ; cx < xmax ; cx++ )
    {
        for ( cy = 0 ; cy < ymax ; cy++ )
        {
            // compute index in chbuf
            index = (cx * ymax) + cy; 

            // allocate the kernel container
            vaddr = (unsigned int)_remote_malloc( NIC_CONTAINER_SIZE, cx, cy );

            if ( vaddr == 0 )  // not enough kernel heap memory in cluster[cx,cy]
            {
                _printf("\n[GIET_ERROR] in _sys_nic_alloc() not enough kenel heap"
                        " in cluster[%d,%d]\n", cx, cy );
                return -1;
            }

            // compute container physical address
            _v2p_translate( (page_table_t*)ptab,
                            vaddr>>12,
                            &ppn,
                            &flags );
            cont_paddr = (((unsigned long long)ppn) << 12) | (vaddr & 0x00000FFF);

            // initialize chbuf entry
            if ( is_rx ) _nic_rx_chbuf[nic_channel].buffer[index].desc = cont_paddr;
            else         _nic_tx_chbuf[nic_channel].buffer[index].desc = cont_paddr;

#if GIET_DEBUG_NIC
_printf("\n[GIET DEBUG NIC] Task %d in _sys_nic_start()"
        " allocates container in cluster[%d,%d] : vaddr = %x / paddr = %l\n",
        thread , cx , cy , vaddr , cont_paddr );
#endif
        }
    }

    // complete kernel chbuf initialisation
    if ( is_rx )
    {
        _nic_rx_chbuf[nic_channel].xmax = xmax;
        _nic_rx_chbuf[nic_channel].ymax = ymax;
    }
    else
    {
        _nic_tx_chbuf[nic_channel].xmax = xmax;
        _nic_tx_chbuf[nic_channel].ymax = ymax;
    }

    // compute the NIC chbuf descriptor physical address
    unsigned int offset;
    if ( is_rx ) offset = 0x4000;
    else         offset = 0x4080;
    nic_chbuf_pbase = (((unsigned long long)((X_IO << Y_WIDTH) + Y_IO))<<32) |
                      (SEG_NIC_BASE + (nic_channel<<15) + offset);

#if GIET_DEBUG_NIC
_printf("\n[GIET DEBUG NIC] Task %d in _sys_nic_start()"
        " get NIC chbuf : paddr = %l\n",
        thread , nic_chbuf_pbase );
#endif

    // compute the kernel chbuf descriptor physical address
    if ( is_rx ) vaddr = (unsigned int)( &_nic_rx_chbuf[nic_channel] );
    else         vaddr = (unsigned int)( &_nic_tx_chbuf[nic_channel] );
    _v2p_translate( (page_table_t*)ptab,
                     vaddr>>12,
                     &ppn,
                     &flags );
    ker_chbuf_pbase = (((unsigned long long)ppn) << 12) | (vaddr & 0x00000FFF);

#if GIET_DEBUG_NIC
_printf("\n[GIET DEBUG NIC] Task %d in _sys_nic_start()"
        " get kernel chbuf : vaddr = %x / paddr = %l\n",
        thread , vaddr , ker_chbuf_pbase );
#endif

    // sync the kernel chbuf in L2 after write in L2
    _mmc_sync( ker_chbuf_pbase, sizeof( nic_chbuf_t ) );

    // initializes CMA registers defining the source & destination chbufs 
    if ( is_rx )               // NIC to kernel
    {
        _cma_set_register( cma_channel, CHBUF_SRC_DESC , (unsigned int)(nic_chbuf_pbase) );
        _cma_set_register( cma_channel, CHBUF_SRC_EXT  , (unsigned int)(nic_chbuf_pbase>>32) );
        _cma_set_register( cma_channel, CHBUF_SRC_NBUFS, 2 );
        _cma_set_register( cma_channel, CHBUF_DST_DESC , (unsigned int)(ker_chbuf_pbase) );
        _cma_set_register( cma_channel, CHBUF_DST_EXT  , (unsigned int)(ker_chbuf_pbase>>32) );
        _cma_set_register( cma_channel, CHBUF_DST_NBUFS, xmax * ymax );
    }
    else                      // kernel to NIC
    {
        _cma_set_register( cma_channel, CHBUF_SRC_DESC , (unsigned int)(ker_chbuf_pbase) );
        _cma_set_register( cma_channel, CHBUF_SRC_EXT  , (unsigned int)(ker_chbuf_pbase>>32) );
        _cma_set_register( cma_channel, CHBUF_SRC_NBUFS, xmax * ymax );
        _cma_set_register( cma_channel, CHBUF_DST_DESC , (unsigned int)(nic_chbuf_pbase) );
        _cma_set_register( cma_channel, CHBUF_DST_EXT  , (unsigned int)(nic_chbuf_pbase>>32) );
        _cma_set_register( cma_channel, CHBUF_DST_NBUFS, 2 );
    }

#if GIET_DEBUG_NIC
_printf("\n[GIET DEBUG NIC] Task %d exit _sys_nic_alloc() at cycle %d\n",
        thread, _get_proctime() );
#endif

    return nic_channel;
} // end _sys_nic_alloc()


////////////////////////////////////////
int _sys_nic_start( unsigned int is_rx,
                    unsigned int channel )
{
    unsigned int nic_channel;
    unsigned int cma_channel;

    // get NIC channel index and CMA channel index from task context
    if ( is_rx )
    {
        nic_channel = _get_context_slot( CTX_NIC_RX_ID );
        cma_channel = _get_context_slot( CTX_CMA_RX_ID );
    }
    else
    {
        nic_channel = _get_context_slot( CTX_NIC_TX_ID );
        cma_channel = _get_context_slot( CTX_CMA_TX_ID );
    }

#if GIET_DEBUG_NIC
unsigned int thread  = _get_context_slot( CTX_TRDID_ID );
_printf("\n[GIET DEBUG NIC] Task %d in _sys_nic_start() at cycle %d"
        " get NIC channel = %d / CMA channel = %d\n",
        thread, _get_proctime(), nic_channel, cma_channel );
#endif

    // check NIC and CMA channels index
    if ( nic_channel != channel )
    {
        _printf("\n[GIET_ERROR] in _sys_nic_start(): illegal NIC channel\n");
        return -1;
    }
    if ( cma_channel >= NB_CMA_CHANNELS )
    {
        _printf("\n[GIET_ERROR] in _sys_nic_start(): illegal CMA channel\n");
        return -1;
    }

    // start CMA transfer
    _cma_set_register( cma_channel, CHBUF_BUF_SIZE , NIC_CONTAINER_SIZE );
    _cma_set_register( cma_channel, CHBUF_PERIOD   , 0 );                   // OUT_OF_ORDER mode
    _cma_set_register( cma_channel, CHBUF_RUN      , 1 );

    // activates NIC channel
    _nic_channel_start( nic_channel, is_rx, GIET_NIC_MAC4, GIET_NIC_MAC2 ); 

#if GIET_DEBUG_NIC
_printf("\n[GIET DEBUG NIC] Task %d exit _sys_nic_start() at cycle %d\n",
        thread , _get_proctime() );
#endif

    return 0;
}  // end sys_nic_start()


//////////////////////////////////////
int _sys_nic_move( unsigned int is_rx,
                   unsigned int channel,
                   void*        buffer )
{

#if GIET_DEBUG_NIC
unsigned int thread  = _get_context_slot( CTX_TRDID_ID );
_printf("\n[GIET DEBUG NIC] Task %d enters _sys_nic_move() at cycle %d\n",
        thread , _get_proctime() );
#endif

    // check NIC channel index
    if ( channel >= NB_NIC_CHANNELS )
    {
        _printf("\n[GIET_ERROR] in _sys_nic_move() : illegal NIC channel index\n");
        return -1;
    }

    // get kernel chbuf virtual address
    nic_chbuf_t* chbuf;
    if ( is_rx )  chbuf = &_nic_rx_chbuf[channel];
    else          chbuf = &_nic_tx_chbuf[channel];

    // get xmax / ymax parameters
    unsigned int xmax = chbuf->xmax;
    unsigned int ymax = chbuf->ymax;

    // get cluster coordinates for the processor running the calling task
    unsigned int  procid = _get_procid();
    unsigned int  cx     = procid >> (Y_WIDTH + P_WIDTH);
    unsigned int  cy     = (procid >> P_WIDTH) & ((1<<Y_WIDTH)-1);
    
    // check processor coordinates / (xmax,ymax)
    if ( cx >= xmax )
    {
        _printf("\n[GIET_ERROR] in _sys_nic_move() : processor X coordinate = %d"
                " / xmax = %d\n", cx , xmax );
        return -1;
    }
    if ( cy >= ymax )
    {
        _printf("\n[GIET_ERROR] in _sys_nic_move() : processor Y coordinate = %d"
                " / ymax = %d\n", cy , ymax );
        return -1;
    }
    
    unsigned long long user_buffer_paddr;    // user buffer physical address 
    unsigned long long kernel_buffer_paddr;  // kernel buffer physical address
    unsigned long long kernel_chbuf_paddr;   // kernel chbuf physical address
    unsigned long long buffer_desc;          // kernel buffer descriptor
    unsigned long long buffer_desc_paddr;    // kernel buffer descriptor physical address
    unsigned int       index;                // kernel buffer index in chbuf

    // The following variables are used for V2P translation
    unsigned int ptab = _get_context_slot( CTX_PTAB_ID );
    unsigned int ppn;
    unsigned int flags;
    unsigned int vaddr;

    // Compute user buffer physical address and check access rights
    vaddr = (unsigned int)buffer;
    _v2p_translate( (page_table_t*)ptab,
                     vaddr>>12,
                     &ppn,
                     &flags );

    if ( (flags & PTE_U) == 0 )
    {
        _printf("\n[GIET ERROR] in _sys_nic_tx_move() : illegal buffer address\n");
        return -1;
    }
    user_buffer_paddr = ((unsigned long long)ppn << 12) | (vaddr & 0x00000FFF);

#if GIET_DEBUG_NIC
_printf("\n[GIET DEBUG NIC] Task %d in _sys_nic_move() get user buffer : paddr = %l\n",
        thread, user_buffer_paddr );
#endif

    // compute kernel chbuf physical address (required for sync)
    vaddr = (unsigned int)chbuf;
    _v2p_translate( (page_table_t*)ptab,
                     vaddr>>12,
                     &ppn,
                     &flags );
    kernel_chbuf_paddr = ((unsigned long long)ppn << 12) | (vaddr & 0x00000FFF);

    // poll local kernel container status until success
    while ( 1 )
    {
        // compute buffer index and buffer descriptor paddr
        index = (ymax * cx) + cy;
        buffer_desc_paddr = kernel_chbuf_paddr + (index<<6);

        // inval buffer descriptor in L2 before read in L2
        _mmc_inval( buffer_desc_paddr , 8 );
        buffer_desc = chbuf->buffer[index].desc;

#if GIET_DEBUG_NIC
_printf("\n[GIET DEBUG NIC] Task %d in _sys_nic_move() read buffer descriptor %d\n"
        " at cycle = %d / paddr = %l / buffer descriptor = %l\n",
        thread, index, _get_proctime(), buffer_desc_paddr, buffer_desc );
#endif

        // test buffer status and break if found
        if ( ( is_rx != 0 ) && (buffer_desc >> 63) == 1 )  break;
        if ( ( is_rx == 0 ) && (buffer_desc >> 63) == 0 )  break;
    }

    // compute kernel buffer physical address
    kernel_buffer_paddr = buffer_desc & 0x0000FFFFFFFFFFFFULL;
    
    // move one container
    if ( is_rx )              // RX transfer
    {
        // inval kernel buffer in L2 before read in L2
        _mmc_inval( kernel_buffer_paddr, NIC_CONTAINER_SIZE );

        // transfer data from kernel buffer to user buffer
        _physical_memcpy( user_buffer_paddr, 
                          kernel_buffer_paddr, 
                          NIC_CONTAINER_SIZE );
#if GIET_DEBUG_NIC
_printf("\n[GIET DEBUG NIC] Task %d in _sys_nic_move() transfer "
        "kernel buffer %l to user buffer %l at cycle %d\n",
        thread , kernel_buffer_paddr , user_buffer_paddr , _get_proctime() );
#endif

    }
    else                      // TX transfer
    {
        // transfer data from user buffer to kernel buffer
        _physical_memcpy( kernel_buffer_paddr, 
                          user_buffer_paddr, 
                          NIC_CONTAINER_SIZE );

        // sync kernel buffer in L2 after write in L2
        _mmc_sync( kernel_buffer_paddr, NIC_CONTAINER_SIZE );

#if GIET_DEBUG_NIC
_printf("\n[GIET DEBUG NIC] Task %d in _sys_nic_move() transfer "
        "user buffer %l to kernel buffer %l at cycle %d\n",
        thread , user_buffer_paddr , kernel_buffer_paddr , _get_proctime() );
#endif

    }

    // update kernel chbuf status 
    if ( is_rx ) chbuf->buffer[index].desc = kernel_buffer_paddr & 0x0000FFFFFFFFFFFFULL;
    else         chbuf->buffer[index].desc = kernel_buffer_paddr | 0x8000000000000000ULL;

    // sync kernel chbuf in L2 after write in L2
    _mmc_sync( kernel_chbuf_paddr + (index<<6) , 8 );

#if GIET_DEBUG_NIC
_printf("\n[GIET DEBUG NIC] Task %d get buffer %d  and exit _sys_nic_move() at cycle %d\n",
        thread , index , _get_proctime() );
#endif

    return 0;
} // end _sys_nic_move()


////////////////////////////////////////
int _sys_nic_stop( unsigned int is_rx,
                   unsigned int channel )
{
    unsigned int nic_channel;
    unsigned int cma_channel;

    // get NIC channel index and CMA channel index
    if ( is_rx )
    {
        nic_channel = _get_context_slot( CTX_NIC_RX_ID );
        cma_channel = _get_context_slot( CTX_CMA_RX_ID );
    }
    else
    {
        nic_channel = _get_context_slot( CTX_NIC_TX_ID );
        cma_channel = _get_context_slot( CTX_CMA_TX_ID );
    }

    // check NIC and CMA channels index
    if ( nic_channel != channel )
    {
        _printf("\n[GIET_ERROR] in _sys_nic_stop(): illegal NIC channel\n");
        return -1;
    }
    if ( cma_channel >= NB_CMA_CHANNELS )
    {
        _printf("\n[GIET_ERROR] in _sys_nic_stop(): illegal CMA channel\n");
        return -1;
    }

    // desactivates the NIC channel
    _nic_channel_stop( nic_channel, is_rx );

    // desactivates the CMA channel
    _cma_channel_stop( cma_channel );

    return 0;
}  // end _sys_nic_stop()

////////////////////////////////////////
int _sys_nic_clear( unsigned int is_rx,
                    unsigned int channel )
{
    unsigned int nic_channel;

    // get NIC channel 
    if ( is_rx )  nic_channel = _get_context_slot( CTX_NIC_RX_ID );
    else          nic_channel = _get_context_slot( CTX_NIC_TX_ID );

    if ( nic_channel != channel )
    {
        _printf("\n[GIET_ERROR] in _sys_nic_clear(): illegal NIC channel\n");
        return -1;
    }

    if ( is_rx )
    {
        _nic_set_global_register( NIC_G_NPKT_RX_G2S_RECEIVED       , 0 );
        _nic_set_global_register( NIC_G_NPKT_RX_DES_TOO_SMALL      , 0 );
        _nic_set_global_register( NIC_G_NPKT_RX_DES_TOO_BIG        , 0 );
        _nic_set_global_register( NIC_G_NPKT_RX_DES_MFIFO_FULL     , 0 );
        _nic_set_global_register( NIC_G_NPKT_RX_DES_CRC_FAIL       , 0 );
        _nic_set_global_register( NIC_G_NPKT_RX_DISPATCH_RECEIVED  , 0 );
        _nic_set_global_register( NIC_G_NPKT_RX_DISPATCH_BROADCAST , 0 );
        _nic_set_global_register( NIC_G_NPKT_RX_DISPATCH_DST_FAIL  , 0 );
        _nic_set_global_register( NIC_G_NPKT_RX_DISPATCH_CH_FULL   , 0 );
    } 
    else
    {
        _nic_set_global_register( NIC_G_NPKT_TX_DISPATCH_RECEIVED  , 0 );
        _nic_set_global_register( NIC_G_NPKT_TX_DISPATCH_TRANSMIT  , 0 );
        _nic_set_global_register( NIC_G_NPKT_TX_DISPATCH_TOO_BIG   , 0 );
        _nic_set_global_register( NIC_G_NPKT_TX_DISPATCH_TOO_SMALL , 0 );
        _nic_set_global_register( NIC_G_NPKT_TX_DISPATCH_SRC_FAIL  , 0 );
        _nic_set_global_register( NIC_G_NPKT_TX_DISPATCH_BYPASS    , 0 );
        _nic_set_global_register( NIC_G_NPKT_TX_DISPATCH_BROADCAST , 0 );
    }
    return 0;
}  // en _sys_nic_clear()

////////////////////////////////////////
int _sys_nic_stats( unsigned int is_rx,
                    unsigned int channel )
{
    unsigned int nic_channel;

    // get NIC channel 
    if ( is_rx )  nic_channel = _get_context_slot( CTX_NIC_RX_ID );
    else          nic_channel = _get_context_slot( CTX_NIC_TX_ID );

    if ( nic_channel != channel )
    {
        _printf("\n[GIET_ERROR] in _sys_nic_stats(): illegal NIC channel\n");
        return -1;
    }

    if ( is_rx )
    {
        unsigned int received   = _nic_get_global_register( NIC_G_NPKT_RX_G2S_RECEIVED       );
        unsigned int too_small  = _nic_get_global_register( NIC_G_NPKT_RX_DES_TOO_SMALL      );
        unsigned int too_big    = _nic_get_global_register( NIC_G_NPKT_RX_DES_TOO_BIG        );
        unsigned int fifo_full  = _nic_get_global_register( NIC_G_NPKT_RX_DES_MFIFO_FULL     );
        unsigned int crc_fail   = _nic_get_global_register( NIC_G_NPKT_RX_DES_CRC_FAIL       );
        unsigned int broadcast  = _nic_get_global_register( NIC_G_NPKT_RX_DISPATCH_BROADCAST );
        unsigned int dst_fail   = _nic_get_global_register( NIC_G_NPKT_RX_DISPATCH_DST_FAIL  );
        unsigned int ch_full    = _nic_get_global_register( NIC_G_NPKT_RX_DISPATCH_CH_FULL   );

        _printf("\n### Network Controller RX Statistics ###\n"
                "- packets received : %d\n"
                "- too small        : %d\n"
                "- too big          : %d\n"
                "- fifo full        : %d\n" 
                "- crc fail         : %d\n" 
                "- dst mac fail     : %d\n" 
                "- channel full     : %d\n" 
                "- broadcast        : %d\n",
                received,
                too_small,
                too_big,
                fifo_full,
                crc_fail,
                dst_fail,
                ch_full,
                broadcast );
    } 
    else
    {
        unsigned int received   = _nic_get_global_register( NIC_G_NPKT_TX_DISPATCH_RECEIVED  );
        unsigned int too_big    = _nic_get_global_register( NIC_G_NPKT_TX_DISPATCH_TOO_BIG   );
        unsigned int too_small  = _nic_get_global_register( NIC_G_NPKT_TX_DISPATCH_TOO_SMALL );
        unsigned int src_fail   = _nic_get_global_register( NIC_G_NPKT_TX_DISPATCH_SRC_FAIL  );
        unsigned int bypass     = _nic_get_global_register( NIC_G_NPKT_TX_DISPATCH_BYPASS    );
        unsigned int broadcast  = _nic_get_global_register( NIC_G_NPKT_TX_DISPATCH_BROADCAST );

        _printf("\n### Network Controller TX Statistics ###\n"
                "- packets received : %d\n"
                "- too small        : %d\n"
                "- too big          : %d\n"
                "- src mac fail     : %d\n" 
                "- bypass           : %d\n" 
                "- broadcast        : %d\n",
                received,
                too_big,
                too_small,
                src_fail,
                bypass,
                broadcast );
    }
    return 0;
}  // end _sys_nic_stats()

/////////////////////////////////////////////////////////////////////////////////////////
//    FBF related syscall handlers
/////////////////////////////////////////////////////////////////////////////////////////

/////////////////////////////////////////////
int _sys_fbf_sync_write( unsigned int offset,
                         void*        buffer,
                         unsigned int length )
{
    char* fbf_address = (char *)SEG_FBF_BASE + offset;
    memcpy( fbf_address, buffer, length);

    return 0;
}

/////////////////////////////////////////////
int _sys_fbf_sync_read(  unsigned int offset,
                         void*        buffer,
                         unsigned int length )
{
    char* fbf_address = (char *)SEG_FBF_BASE + offset;
    memcpy( buffer, fbf_address, length);

    return 0;
}

////////////////////////
int _sys_fbf_cma_alloc()
{
   // get a new CMA channel index 
    unsigned int channel = _atomic_increment( &_cma_channel_allocator, 1 );
    unsigned int thread  = _get_context_slot( CTX_TRDID_ID );
    unsigned int vspace  = _get_context_slot( CTX_VSID_ID );

    if ( channel >= NB_CMA_CHANNELS )
    {
        _printf("\n[GIET ERROR] in _sys_fbf_cma_alloc() : not enough CMA channels\n");
        return -1;
    }
    else
    {
        _printf("\n[GIET WARNING] FBF_CMA channel %d allocated "
                " to thread %d in vspace %d\n", channel, thread, vspace );
        _set_context_slot( CTX_CMA_FB_ID, channel );
        return 0;
    }
} // end sys_fbf_cma_alloc()

////////////////////////////////////////////
int _sys_fbf_cma_start( void*        vbase0, 
                        void*        vbase1,  
                        unsigned int length ) 
{
#if NB_CMA_CHANNELS > 0

    unsigned int       ptab;            // page table virtual address
    unsigned int       vaddr;           // virtual address
    unsigned int       flags;           // protection flags
    unsigned int       ppn;             // physical page number
    unsigned long long chbuf_paddr;     // physical address of source chbuf descriptor

    // get channel index
    unsigned int channel = _get_context_slot( CTX_CMA_FB_ID );

    if ( channel >= NB_CMA_CHANNELS )
    {
        _printf("\n[GIET ERROR] in _fbf_cma_start() : CMA channel index too large\n");
        return -1;
    }

#if GIET_DEBUG_FBF_CMA
_printf("\n[FBF_CMA DEBUG] enters _sys_fbf_cma_start()\n"
        " - channel      = %d\n"
        " - buf0   vbase = %x\n"
        " - buf1   vbase = %x\n"
        " - buffer size  = %x\n",
        channel,
        (unsigned int)vbase0,
        (unsigned int)vbase1,
        length );
#endif

    // checking user buffers virtual addresses and length alignment
    if ( ((unsigned int)vbase0 & 0x3) || ((unsigned int)vbase1 & 0x3) || (length & 0x3) ) 
    {
        _printf("\n[GIET ERROR] in _fbf_cma_start() : user buffer not word aligned\n");
        return -1;
    }

    // get page table virtual address
    ptab = _get_context_slot(CTX_PTAB_ID);

    // compute frame buffer physical address and initialize _fbf_chbuf[channel]
    vaddr = ((unsigned int)SEG_FBF_BASE);
    _v2p_translate( (page_table_t*) ptab, 
                    (vaddr >> 12),
                    &ppn, 
                    &flags );

    _fbf_chbuf[channel].fbf.desc = ((paddr_t)ppn << 12) | (vaddr & 0x00000FFF);

    // Compute user buffer 0 physical addresses and intialize _fbf_chbuf[channel]
    vaddr = (unsigned int)vbase0; 
    _v2p_translate( (page_table_t*) ptab, 
                    (vaddr >> 12),
                    &ppn, 
                    &flags );
    if ((flags & PTE_U) == 0) 
    {
        _printf("\n[GIET ERROR] in _fbf_cma_start() : user buffer 0 not in user space\n");
        return -1;
    }

    _fbf_chbuf[channel].buf0.desc = ((paddr_t)ppn << 12) | (vaddr & 0x00000FFF);

    // Compute user buffer 1 physical addresses and intialize _fbf_chbuf[channel]
    vaddr = (unsigned int)vbase1; 
    _v2p_translate( (page_table_t*) ptab, 
                    (vaddr >> 12),
                    &ppn, 
                    &flags );
    if ((flags & PTE_U) == 0) 
    {
        _printf("\n[GIET ERROR] in _fbf_cma_start() : user buffer 1 not in user space\n");
        return -1;
    }

    _fbf_chbuf[channel].buf1.desc = ((paddr_t)ppn << 12) | (vaddr & 0x00000FFF);

    // initializes buffer length
    _fbf_chbuf[channel].length = length;

    // Compute and register physical adress of the chbuf descriptor
    vaddr = (unsigned int)(&_fbf_chbuf[channel]);
    _v2p_translate( (page_table_t*) ptab, 
                    (vaddr >> 12),
                    &ppn, 
                    &flags );
 
    chbuf_paddr = (((paddr_t)ppn) << 12) | (vaddr & 0x00000FFF);

    _fbf_chbuf_paddr[channel] = chbuf_paddr;


    if ( USE_IOB )
    {
        // SYNC request for channel descriptor
        _mmc_sync( chbuf_paddr, 32 );
    }

#if GIET_DEBUG_FBF_CMA
_printf(" - fbf    pbase = %l\n"
        " - buf0   pbase = %l\n"
        " - buf1   pbase = %l\n"
        " - chbuf  pbase = %l\n",
        _fbf_chbuf[channel].fbf.desc,
        _fbf_chbuf[channel].buf0.desc,
        _fbf_chbuf[channel].buf1.desc,
        chbuf_paddr );
#endif

    // call CMA driver to start transfer
    _cma_channel_start( channel, 
                        chbuf_paddr,
                        2,
                        chbuf_paddr + 128,
                        1,
                        length );
    return 0;

#else

    _printf("\n[GIET ERROR] in _sys_fbf_cma_start() : NB_CMA_CHANNELS = 0\n");
    return -1;

#endif
} // end _sys_fbf_cma_start()

/////////////////////////////////////////////////////
int _sys_fbf_cma_display( unsigned int buffer_index )
{
#if NB_CMA_CHANNELS > 0

    volatile unsigned int long long user_buffer_desc;
    volatile unsigned int           full = 1;

    // get channel index
    unsigned int channel = _get_context_slot( CTX_CMA_FB_ID );

    if ( channel >= NB_CMA_CHANNELS )
    {
        _printf("\n[GIET ERROR] in _sys_fbf_cma_display() : CMA channel index too large\n");
        return -1;
    }

#if GIET_DEBUG_FBF_CMA
_printf("\n[FBF_CMA DEBUG] enters _sys_fb_cma_display()\n"
        " - channel      = %d\n"
        " - buffer       = %d\n",
        channel, buffer_index );
#endif

    if ( buffer_index == 0 )    // user buffer 0
    {
        // waiting user buffer released by the CMA component)
        while ( full )
        {  
            // INVAL L1 and L2 cache copies of user buffer descriptor, because
            // it has been modified in RAM by the CMA component 
            _dcache_buf_invalidate( (unsigned int)&_fbf_chbuf[channel].buf0.desc , 8 );
            _mmc_inval( _fbf_chbuf_paddr[channel] , 8 );

            // get user buffer descriptor
            user_buffer_desc = _fbf_chbuf[channel].buf0.desc;

            // get user buffer descriptor status
            full = ( (unsigned int)(user_buffer_desc>>63) );
        }

        // SYNC request for the user buffer, because 
        // it will be read from XRAM by the CMA component
        _mmc_sync( user_buffer_desc, _fbf_chbuf[channel].length );

        // set user buffer status, and SYNC request, because this buffer
        // descriptor will be read from XRAM by the CMA component
        _fbf_chbuf[channel].buf0.desc = user_buffer_desc | 0x8000000000000000ULL;
        _mmc_sync( _fbf_chbuf_paddr[channel] , 8 );

        // reset fbf buffer status, and SYNC request, because this buffer
        // descriptor will be read from XRAM by the CMA component
        _fbf_chbuf[channel].fbf.desc  = _fbf_chbuf[channel].fbf.desc & 0x7FFFFFFFFFFFFFFFULL;
        _mmc_sync( _fbf_chbuf_paddr[channel] + 128 , 8 );
    }
    else                        // user buffer 1
    {
        // waiting user buffer released by the CMA component)
        while ( full )
        {  
            // INVAL L1 and L2 cache copies of user buffer descriptor, because
            // it has been modified in RAM by the CMA component 
            _dcache_buf_invalidate( (unsigned int)&_fbf_chbuf[channel].buf1.desc , 8 );
            _mmc_inval( _fbf_chbuf_paddr[channel] + 64 , 8 );

            // get user buffer descriptor
            user_buffer_desc = _fbf_chbuf[channel].buf1.desc;

            // get user buffer descriptor status
            full = ( (unsigned int)(user_buffer_desc>>63) );
        }

        // SYNC request for the user buffer, because 
        // it will be read from XRAM by the CMA component
        _mmc_sync( user_buffer_desc, _fbf_chbuf[channel].length );

        // set user buffer status, and SYNC request, because this buffer
        // descriptor will be read from XRAM by the CMA component
        _fbf_chbuf[channel].buf1.desc = user_buffer_desc | 0x8000000000000000ULL;
        _mmc_sync( _fbf_chbuf_paddr[channel] , 8 );

        // reset fbf buffer status, and SYNC request, because this buffer
        // descriptor will be read from XRAM by the CMA component
        _fbf_chbuf[channel].fbf.desc  = _fbf_chbuf[channel].fbf.desc & 0x7FFFFFFFFFFFFFFFULL;
        _mmc_sync( _fbf_chbuf_paddr[channel] + 128 , 8 );
    }

#if GIET_DEBUG_FBF_CMA
_printf(" - fbf    desc = %l\n"
        " - buf0   desc = %l\n"
        " - buf1   desc = %l\n",
        _fbf_chbuf[channel].fbf.desc,
        _fbf_chbuf[channel].buf0.desc,
        _fbf_chbuf[channel].buf1.desc );
#endif

    return 0;

#else

    _printf("\n[GIET ERROR] in _sys_fbf_cma_display() : no CMA channel allocated\n");
    return -1;

#endif
} // end _sys_fbf_cma_display()


///////////////////////
int _sys_fbf_cma_stop()
{
#if NB_CMA_CHANNELS > 0

    // get channel index
    unsigned int channel = _get_context_slot( CTX_CMA_FB_ID );

    if ( channel >= NB_CMA_CHANNELS )
    {
        _printf("\n[GIET ERROR] in _sys_fbf_cma_stop() : CMA channel index too large\n");
        return -1;
    }

    // Desactivate CMA channel
    _cma_channel_stop( channel );

    return 0;

#else

    _printf("\n[GIET ERROR] in _sys_fbf_cma_stop() : no CMA channel allocated\n");
    return -1;

#endif
} // end _sys_fbf_cma_stop()


//////////////////////////////////////////////////////////////////////////////
//           Miscelaneous syscall handlers 
//////////////////////////////////////////////////////////////////////////////

///////////////
int _sys_ukn() 
{
    _printf("\n[GIET ERROR] Undefined System Call / EPC = %x\n", _get_epc() );
    return -1;
}

////////////////////////////////////
int _sys_proc_xyp( unsigned int* x,
                   unsigned int* y,
                   unsigned int* p )
{
    unsigned int gpid = _get_procid();  // global processor index from CPO register

    *x = (gpid >> (Y_WIDTH + P_WIDTH)) & ((1<<X_WIDTH)-1);
    *y = (gpid >> P_WIDTH) & ((1<<Y_WIDTH)-1);
    *p = gpid & ((1<<P_WIDTH)-1);

    return 0;
}

//////////////////////////////////
int _sys_task_exit( char* string ) 
{
    unsigned int date       = _get_proctime();

    unsigned int gpid       = _get_procid();
    unsigned int cluster_xy = gpid >> P_WIDTH;
    unsigned int y          = cluster_xy & ((1<<Y_WIDTH)-1);
    unsigned int x          = cluster_xy >> Y_WIDTH;
    unsigned int lpid       = gpid & ((1<<P_WIDTH)-1);

    unsigned int task_id    = _get_context_slot(CTX_LTID_ID);

    // print exit message
    _printf("\n[GIET] Exit task %d on processor[%d,%d,%d] at cycle %d"
            "\n       Cause : %s\n\n",
            task_id, x, y, lpid, date, string );

    // goes to sleeping state
    _set_context_slot(CTX_RUN_ID, 0);

    // deschedule
    _context_switch();

    return 0;
} 

//////////////////////
int _context_switch() 
{
    unsigned int save_sr;

    _it_disable( &save_sr );
    _ctx_switch();
    _it_restore( &save_sr );

    return 0;
}

////////////////////////
int _sys_local_task_id()
{
    return _get_context_slot(CTX_LTID_ID);
}

/////////////////////////
int _sys_global_task_id()
{
    return _get_context_slot(CTX_GTID_ID);
}

////////////////////
int _sys_thread_id()
{
    return _get_context_slot(CTX_TRDID_ID);
}

////////////////////////////////////////////
int _sys_procs_number( unsigned int* x_size,
                       unsigned int* y_size,
                       unsigned int* nprocs )
{
    mapping_header_t * header   = (mapping_header_t *)SEG_BOOT_MAPPING_BASE;
    mapping_cluster_t * cluster = _get_cluster_base(header);

    unsigned int x;
    unsigned int y;
    unsigned int okmin = 1;
    unsigned int okmax = 1;

    // compute max values
    unsigned int xmax  = header->x_size;
    unsigned int ymax  = header->y_size;
    unsigned int procs = cluster[0].procs;

    // check the (ymax-1) lower rows
    for ( y = 0 ; y < ymax-1 ; y++ )
    {
        for ( x = 0 ; x < xmax ; x++ )
        {
            if (cluster[x*ymax+y].procs != procs ) okmin = 0;
        }
    }

    // check the upper row
    for ( x = 0 ; x < xmax ; x++ )
    {
        if (cluster[x*ymax+ymax-1].procs != procs ) okmax = 0;
    }

    // return values
    if ( okmin && okmax )
    {
        *x_size = xmax;
        *y_size = ymax;
        *nprocs = procs;
    }
    else if ( okmin )
    {
        *x_size = xmax;
        *y_size = ymax-1;
        *nprocs = procs;
    }
    else
    {
        *x_size = 0;
        *y_size = 0;
        *nprocs = 0;
    }
    return 0;
}

///////////////////////////////////////////////////////
int _sys_vobj_get_vbase( char*             vspace_name, 
                         char*             vobj_name, 
                         unsigned int*     vbase ) 
{
    mapping_header_t * header = (mapping_header_t *)SEG_BOOT_MAPPING_BASE;
    mapping_vspace_t * vspace = _get_vspace_base(header);
    mapping_vobj_t * vobj     = _get_vobj_base(header);

    unsigned int vspace_id;
    unsigned int vobj_id;

    // scan vspaces 
    for (vspace_id = 0; vspace_id < header->vspaces; vspace_id++) 
    {
        if (_strncmp( vspace[vspace_id].name, vspace_name, 31) == 0) 
        {
            // scan vobjs
            for (vobj_id = vspace[vspace_id].vobj_offset; 
                 vobj_id < (vspace[vspace_id].vobj_offset + vspace[vspace_id].vobjs); 
                 vobj_id++) 
            {
                if (_strncmp(vobj[vobj_id].name, vobj_name, 31) == 0) 
                {
                    *vbase = vobj[vobj_id].vbase;
                    return 0;
                }
            } 
        }
    } 
    return -1;    // not found 
}

/////////////////////////////////////////////////////////
int _sys_vobj_get_length( char*         vspace_name, 
                          char*         vobj_name,
                          unsigned int* length ) 
{
    mapping_header_t * header = (mapping_header_t *)SEG_BOOT_MAPPING_BASE;
    mapping_vspace_t * vspace = _get_vspace_base(header);
    mapping_vobj_t * vobj     = _get_vobj_base(header);

    unsigned int vspace_id;
    unsigned int vobj_id;

    // scan vspaces 
    for (vspace_id = 0; vspace_id < header->vspaces; vspace_id++) 
    {
        if (_strncmp( vspace[vspace_id].name, vspace_name, 31) == 0) 
        {
            // scan vobjs
            for (vobj_id = vspace[vspace_id].vobj_offset; 
                 vobj_id < (vspace[vspace_id].vobj_offset + vspace[vspace_id].vobjs); 
                 vobj_id++) 
            {
                if (_strncmp(vobj[vobj_id].name, vobj_name, 31) == 0) 
                {
                    *length = vobj[vobj_id].length;
                    return 0;
                }
            } 
        }
    } 
    return -1;    // not found 
}

////////////////////////////////////////
int _sys_xy_from_ptr( void*         ptr,
                      unsigned int* x,
                      unsigned int* y )
{
    unsigned int ppn;
    unsigned int flags;
    unsigned int vpn  = (((unsigned int)ptr)>>12);
    
    // get the page table pointer
    page_table_t* pt = (page_table_t*)_get_context_slot( CTX_PTAB_ID ); 

    // compute the physical address
    _v2p_translate( pt, vpn, &ppn, &flags );

    *x = (ppn>>24) & 0xF;
    *y = (ppn>>20) & 0xF;
    return 0;
}

/////////////////////////////////////////
int _sys_heap_info( unsigned int* vaddr, 
                    unsigned int* length,
                    unsigned int  x,
                    unsigned int  y ) 
{
    mapping_header_t * header  = (mapping_header_t *)SEG_BOOT_MAPPING_BASE;
    mapping_task_t *   tasks   = _get_task_base(header);
    mapping_vobj_t *   vobjs   = _get_vobj_base(header);
    mapping_vspace_t * vspaces = _get_vspace_base(header);

    unsigned int task_id;
    unsigned int vspace_id;
    unsigned int vobj_id = 0xFFFFFFFF;

    // searching the heap vobj_id
    if ( (x < X_SIZE) && (y < Y_SIZE) )  // searching a task in cluster(x,y)
    {
        // get vspace global index
        vspace_id = _get_context_slot(CTX_VSID_ID);

        // scan all tasks in vspace
        unsigned int min = vspaces[vspace_id].task_offset ;
        unsigned int max = min + vspaces[vspace_id].tasks ;
        for ( task_id = min ; task_id < max ; task_id++ )
        {
            if ( tasks[task_id].clusterid == (x * Y_SIZE + y) )
            {
                vobj_id = tasks[task_id].heap_vobj_id;
                if ( vobj_id != 0xFFFFFFFF ) break;
            }
        }
    }
    else                                // searching in the calling task 
    {
        task_id = _get_context_slot(CTX_GTID_ID);
        vobj_id = tasks[task_id].heap_vobj_id;
    }

    // analysing the vobj_id
    if ( vobj_id != 0xFFFFFFFF ) 
    {
        *vaddr  = vobjs[vobj_id].vbase;
        *length = vobjs[vobj_id].length;
        return 0;
    }
    else 
    {
        *vaddr  = 0;
        *length = 0;
        return -1;
    }
}  // end _sys_heap_info()


// Local Variables:
// tab-width: 4
// c-basic-offset: 4
// c-file-offsets:((innamespace . 0)(inline-open . 0))
// indent-tabs-mode: nil
// End:
// vim: filetype=c:expandtab:shiftwidth=4:tabstop=4:softtabstop=4

