///////////////////////////////////////////////////////////////////////////////////
// File     : irq_handler.h
// Date     : 01/04/2012
// Author   : alain greiner 
// Copyright (c) UPMC-LIP6
///////////////////////////////////////////////////////////////////////////////////
// The irq_handler.c and irq_handler.h files are part of the GIET-VM nano-kernel.
// They contain the code of the _irq_demux() function that access the XICU or
// ICU component (Interupt Controler Unit), and the various ISRs (Interrupt
// Service Routine) associated to the various ISR types.
///////////////////////////////////////////////////////////////////////////////////

#ifndef _IRQ_HANDLER_H
#define _IRQ_HANDLER_H

////////////////////////////////////////////////////////////////////////////////
// This enum must consistent with the values defined in 
// xml_driver.c / irq_handler.c / mapping.py 
///////////////////////////////////////////////////////////////////////////////

enum isr_type_t
{
    ISR_DEFAULT = 0,
    ISR_TICK    = 1,
    ISR_TTY_RX  = 2,
    ISR_TTY_TX  = 3,
    ISR_BDV     = 4,
    ISR_TIMER   = 5,
    ISR_WAKUP   = 6,
    ISR_NIC_RX  = 7,
    ISR_NIC_TX  = 8,
    ISR_CMA     = 9,
    ISR_MMC     = 10,
    ISR_DMA     = 11,
    ISR_SPI     = 12,
    ISR_MWR     = 13,
};

///////////////////////////////////////////////////////////////////////////////////
//    irq_handler functions
///////////////////////////////////////////////////////////////////////////////////

///////////////////////////////////////////////////////////////////////////////////
// This function access the ICU or XICU component (Interrupt Controler Unit)
// to get the interrupt vector entry. There is one ICU or XICU component per
// cluster, and this component can support up to NB_PROCS_MAX output IRQs.
// It returns the highest priority active interrupt index (smaller
// indexes have the highest priority).
// Any value larger than 31 means "no active interrupt", and no ISR is executed.
//
// There is three interrupt vectors per processor (stored in the processor's
// scheduler) for the three HWI, PTI, and WTI interrupts types.
// Each interrupt vector entry contains three bits fields:
// - isr_id     bits[15:0]  : defines the type of ISR to be executed.
// - channel_id bits[30:16] : defines the channel for multi-channels peripherals.
// - valid      bit 31      : valid interrupt vector entry
// If the peripheral is replicated in clusters, the channel_id is 
// a global index : channel_id = cluster_id * NB_CHANNELS_MAX + loc_id   
///////////////////////////////////////////////////////////////////////////////////
extern void _irq_demux();

///////////////////////////////////////////////////////////////////////////////////
// This default ISR is called  when the interrupt handler is called, 
// and there is no active IRQ. It simply displays a warning message on TTY[0].
///////////////////////////////////////////////////////////////////////////////////
extern void _isr_default();

///////////////////////////////////////////////////////////////////////////////////
// This ISR can only be executed after a WTI (IPI) to force a context switch
// on a remote processor. The context switch is only executed if the current task
// is the IDLE_TASK, or if the value written in the mailbox is non zero.
///////////////////////////////////////////////////////////////////////////////////
extern void _isr_wakup( unsigned int irq_type,
                        unsigned int irq_id,
                        unsigned int channel );

/////////////////////////////////////////////////////////////////////////////////////
// This ISR is in charge of context switch, and handles the IRQs generated by
// the "system" timers. It can be PTI in case of XCU, or it can be HWI generated
// by an external timer in case of ICU.
// The ISR acknowledges the IRQ, and calls the _ctx_switch() function.
/////////////////////////////////////////////////////////////////////////////////////
extern void _isr_tick( unsigned int irq_type,
                       unsigned int irq_id,
                       unsigned int channel );

#endif

// Local Variables:
// tab-width: 4
// c-basic-offset: 4
// c-file-offsets:((innamespace . 0)(inline-open . 0))
// indent-tabs-mode: nil
// End:
// vim: filetype=c:expandtab:shiftwidth=4:tabstop=4:softtabstop=4

