///////////////////////////////////////////////////////////////////////////////////
// File     : irq_handler.c
// Date     : 01/04/2012
// Author   : alain greiner 
// Copyright (c) UPMC-LIP6
///////////////////////////////////////////////////////////////////////////////////

#include <giet_config.h>
#include <irq_handler.h>
#include <sys_handler.h>
#include <ctx_handler.h>
#include <tim_driver.h>
#include <xcu_driver.h>
#include <tty_driver.h>
#include <nic_driver.h>
#include <cma_driver.h>
#include <mmc_driver.h>
#include <bdv_driver.h>
#include <dma_driver.h>
#include <spi_driver.h>
#include <mapping_info.h>
#include <utils.h>
#include <tty0.h>

#if NB_TIM_CHANNELS
extern volatile unsigned char _user_timer_event[NB_TIM_CHANNELS] ;
#endif

/////////////////////////////////////////////////////////////////////////
// These ISR_TYPE names for display must be consistent with values in
// irq_handler.h / mapping.py / xml_driver.c
/////////////////////////////////////////////////////////////////////////
__attribute__((section(".kdata")))
char* _isr_type_name[] = { "DEFAULT",
                           "TICK"   ,
                           "TTY_RX" ,
                           "TTY_TX" ,
                           "BDV"    ,
                           "TIMER"  ,
                           "WAKUP"  ,
                           "NIC_RX" ,
                           "NIC_TX" ,
                           "CMA"    ,
                           "MMC"    ,
                           "DMA"    ,
                           "SPI"    ,
                           "MWR"    };
/////////////////
void _irq_demux() 
{
    unsigned int gpid           = _get_procid();
    unsigned int cluster_xy     = gpid >> P_WIDTH;
    unsigned int x              = cluster_xy >> Y_WIDTH;
    unsigned int y              = cluster_xy & ((1<<Y_WIDTH)-1);
    unsigned int lpid           = gpid & ((1<<P_WIDTH)-1);
    unsigned int irq_id;
    unsigned int irq_type;

    // get the highest priority active IRQ index 
    unsigned int icu_out_index = lpid * IRQ_PER_PROCESSOR;

    _xcu_get_index( cluster_xy, icu_out_index, &irq_id, &irq_type );

    if (irq_id < 32) 
    {
        static_scheduler_t* psched = (static_scheduler_t*)_get_sched();
        unsigned int        entry = 0;
        unsigned int        isr_type;
        unsigned int        channel;

        if      (irq_type == IRQ_TYPE_HWI) entry = psched->hwi_vector[irq_id];
        else if (irq_type == IRQ_TYPE_PTI) entry = psched->pti_vector[irq_id];
        else if (irq_type == IRQ_TYPE_WTI) entry = psched->wti_vector[irq_id];
        else
        {
            _printf("\n[GIET ERROR] illegal irq_type in irq_demux()\n");
            _exit();
        }

        isr_type   = (entry    ) & 0x0000FFFF;
        channel    = (entry>>16) & 0x00007FFF;

#if GIET_DEBUG_IRQS // we don't take the TTY lock to avoid deadlocks
char* irq_type_str[] = { "HWI", "WTI", "PTI" }; 
_puts("\n[IRQS DEBUG] Processor[");
_putd(x);
_puts(",");
_putd(y);
_puts(",");
_putd(lpid);
_puts("] enters _irq_demux() at cycle ");
_putd(_get_proctime() );
_puts("\n  ");
_puts(irq_type_str[irq_type] );
_puts(" : irq_id = ");
_putd(irq_id);
_puts(" / isr_type = ");
_putd(isr_type);
_puts(" / channel = ");
_putd(channel);
_puts("\n");
#endif

        // ISR call
        if      ( isr_type == ISR_TICK   ) _isr_tick   ( irq_type, irq_id, channel );
        else if ( isr_type == ISR_WAKUP  ) _isr_wakup  ( irq_type, irq_id, channel );
        else if ( isr_type == ISR_BDV    ) _bdv_isr    ( irq_type, irq_id, channel );
        else if ( isr_type == ISR_CMA    ) _cma_isr    ( irq_type, irq_id, channel );
        else if ( isr_type == ISR_TTY_RX ) _tty_rx_isr ( irq_type, irq_id, channel );
        else if ( isr_type == ISR_TTY_TX ) _tty_tx_isr ( irq_type, irq_id, channel );
        else if ( isr_type == ISR_NIC_RX ) _nic_rx_isr ( irq_type, irq_id, channel );
        else if ( isr_type == ISR_NIC_TX ) _nic_tx_isr ( irq_type, irq_id, channel );
        else if ( isr_type == ISR_TIMER  ) _timer_isr  ( irq_type, irq_id, channel );
        else if ( isr_type == ISR_MMC    ) _mmc_isr    ( irq_type, irq_id, channel );
        else if ( isr_type == ISR_DMA    ) _dma_isr    ( irq_type, irq_id, channel );
        else if ( isr_type == ISR_SPI    ) _spi_isr    ( irq_type, irq_id, channel );
        else if ( isr_type == ISR_MWR    ) _mwr_isr    ( irq_type, irq_id, channel );
        else
        {
            _printf("\n[GIET ERROR] in _irq_demux() :"
                    " illegal ISR type on processor[%d,%d,%d] at cycle %d\n"
                    " - irq_type = %d\n"
                    " - irq_id   = %d\n"
                    " - isr_type = %x\n",
                    x, y, lpid, _get_proctime(), irq_type, irq_id, isr_type );
        }
    }
    else   // no interrupt active
    {
        _isr_default();
    } 
}

///////////////////
void _isr_default()
{
    unsigned int gpid       = _get_procid();
    unsigned int cluster_xy = gpid >> P_WIDTH;
    unsigned int x          = cluster_xy >> Y_WIDTH;
    unsigned int y          = cluster_xy & ((1<<Y_WIDTH)-1);
    unsigned int lpid       = gpid & ((1<<P_WIDTH)-1);

    _printf("\n[GIET WARNING] IRQ handler called but no active IRQ "
            "on processor[%d,%d,%d] at cycle %d\n",
            x, y, lpid, _get_proctime() );
}


////////////////////////////////////////////////////////////
void _isr_wakup( unsigned int irq_type,   // HWI / WTI / PTI
                 unsigned int irq_id,     // index returned by ICU
                 unsigned int channel )   // unused
{
    unsigned int gpid       = _get_procid();
    unsigned int cluster_xy = gpid >> P_WIDTH;
    unsigned int x          = cluster_xy >> Y_WIDTH;
    unsigned int y          = cluster_xy & ((1<<Y_WIDTH)-1);
    unsigned int lpid       = gpid & ((1<<P_WIDTH)-1);

    unsigned int task       = _get_current_task_id();
    unsigned int value;

    if ( irq_type != IRQ_TYPE_WTI )
    {
        _puts("[GIET ERROR] _isr_wakup() not called by a WTI on processor[");
        _putd( x );
        _puts(",");
        _putd( y );
        _puts(",");
        _putd( lpid );
        _puts("] at cycle ");
        _putd( _get_proctime() );
        _puts("\n");
        _exit();
    }

    // get mailbox value and acknowledge WTI
    _xcu_get_wti_value( cluster_xy, irq_id, &value );

#if GIET_DEBUG_IRQS // we don't take the TTY lock to avoid deadlocks
_puts("\n[IRQS DEBUG] Processor[");
_putd( x );
_puts(",");
_putd( y );
_puts(",");
_putd( lpid );
_puts("] enters _isr_wakup() at cycle ");
_putd( _get_proctime() );
_puts("\n  WTI / mailbox data = ");
_putx( value );
_puts(" / current task index = ");
_putd( task );
_puts("\n  ");
#endif

    // context swich if required
    if ( (task == IDLE_TASK_INDEX) || (value != 0) ) _ctx_switch();
} // end _isr_wakup

///////////////////////////////////////////////////////////
void _isr_tick( unsigned int irq_type,   // HWI / WTI / PTI
                unsigned int irq_id,     // index returned by ICU
                unsigned int channel )   // channel index if HWI
{
    unsigned int gpid       = _get_procid();
    unsigned int cluster_xy = gpid >> P_WIDTH;
    unsigned int x          = cluster_xy >> Y_WIDTH;
    unsigned int y          = cluster_xy & ((1<<Y_WIDTH)-1);
    unsigned int lpid       = gpid & ((1<<P_WIDTH)-1);

    if ( irq_type != IRQ_TYPE_PTI )
    {
        _puts("[GIET ERROR] _isr_tick() not called by a PTI on processor[");
        _putd( x );
        _puts(",");
        _putd( y );
        _puts(",");
        _putd( lpid );
        _puts("] at cycle ");
        _putd( _get_proctime() );
        _puts("\n");
        _exit();
    }

    // acknowledge PTI
    _xcu_timer_reset_irq( cluster_xy, irq_id );

#if GIET_DEBUG_IRQS  // we don't take the TTY lock to avoid deadlock
_puts("\n[IRQS DEBUG] Processor[");
_putd( x );
_puts(",");
_putd( y );
_puts(",");
_putd( lpid );
_puts("] enters _isr_tick() at cycle ");
_putd( _get_proctime() );
_puts("\n  ");
#endif

    // context switch
    _ctx_switch();
}  // end _isr_tick


// Local Variables:
// tab-width: 4
// c-basic-offset: 4
// c-file-offsets:((innamespace . 0)(inline-open . 0))
// indent-tabs-mode: nil
// End:
// vim: filetype=c:expandtab:shiftwidth=4:tabstop=4:softtabstop=4

