///////////////////////////////////////////////////////////////////////////////////////
// File     : boot.c
// Date     : 01/11/2013
// Author   : alain greiner
// Copyright (c) UPMC-LIP6
///////////////////////////////////////////////////////////////////////////////////////
// The boot.c file contains the bootloader for the GIET-VM static OS.  
//
// This code has been written for the MIPS32 processor.
// The virtual adresses are on 32 bits and use the (unsigned int) type. The 
// physicals addresses can have up to 40 bits, and use the  (unsigned long long) type.
// It natively supports clusterised shared memory multi-processors architectures, 
// where each processor is identified by a composite index [x,y,p],
// and where there is one physical memory bank per cluster.
//
// The boot.elf file is stored on disk and is loaded into memory by proc[0,0,0],
// executing the generic preloader (stored in ROM). The boot-loader code itself
// is executed in parallel by all proc[x,y,0], and performs the following tasks:
// - load into memory various binary files, from a FAT32 file system.
// - build the various page tables (one page table per vspace). 
// - initialize the shedulers (one scheduler per processor).
// - initialize the external peripherals.
//
// 1) The binary files to be loaded are:
//    - the "map.bin" file contains the hardware architecture description and the
//      mapping directives. It must be stored in the the seg_boot_mapping segment 
//      (at address SEG_BOOT_MAPPING_BASE defined in hard_config.h file).
//    - the "kernel.elf" file contains the kernel binary code and data.
//    - the various "application.elf" files.
//
// 2) The "map.bin" file contains the C binary structure defining: 
//    - the hardware architecture: number of clusters, number or processors, 
//      size of the memory segments, and peripherals in each cluster.
//    - The structure of the various multi-threaded software applications:
//      number of tasks, communication channels.
//    - The mapping: placement of virtual segments (vseg) in the physical 
//      segments (pseg), placement of software tasks on the processors, 
//
// 3) The GIET-VM uses the paged virtual memory to provides two services:
//    - classical memory protection, when several independant applications compiled
//      in different virtual spaces are executing on the same hardware platform.
//    - data placement in NUMA architectures, to control the placement 
//      of the software objects (vsegs) on the physical memory banks (psegs).
//
//    The max number of vspaces (GIET_NB_VSPACE_MAX) is a configuration parameter,
//    and - for each application - the tasks are statically allocateded on procesors.
//    The page table are statically build in the boot phase, and they do not 
//    change during execution. 
//    The GIET_VM uses both small pages (4 Kbytes), and big pages (2 Mbytes). 
//
//    Each page table (one page table per virtual space) is monolithic, and contains 
//    one PT1 (8 Kbytes) and a variable number of PT2s (4 Kbytes each). For each vspace,
//    the number of PT2s is defined by the size of the PTAB vseg in the mapping.
//    The PT1 is indexed by the ix1 field (11 bits) of the VPN. Each entry is 32 bits.
//    A PT2 is indexed the ix2 field (9 bits) of the VPN. Each entry is a double word. 
//    The first word contains the flags, the second word contains the PPN.
//    The page tables are distributed/replicated in all clusters.
///////////////////////////////////////////////////////////////////////////////////////
// Implementation Notes:
//
// 1) The cluster_id variable is a linear index in the mapping_info array of clusters. 
//    The cluster_xy variable is the tological index = x << Y_WIDTH + y
// 
// 2) We set the _tty0_boot_mode variable to force the _printf() function to use
//    the tty0_spin_lock for exclusive access to TTY0.
///////////////////////////////////////////////////////////////////////////////////////

#include <giet_config.h>
#include <hard_config.h>
#include <mapping_info.h>
#include <kernel_malloc.h>
#include <memspace.h>
#include <tty_driver.h>
#include <xcu_driver.h>
#include <bdv_driver.h>
#include <hba_driver.h>
#include <dma_driver.h>
#include <cma_driver.h>
#include <nic_driver.h>
#include <ioc_driver.h>
#include <iob_driver.h>
#include <pic_driver.h>
#include <mwr_driver.h>
#include <ctx_handler.h>
#include <irq_handler.h>
#include <vmem.h>
#include <pmem.h>
#include <utils.h>
#include <tty0.h>
#include <kernel_locks.h>
#include <kernel_barriers.h>
#include <elf-types.h>
#include <fat32.h>
#include <mips32_registers.h>
#include <stdarg.h>

#if !defined(X_SIZE)
# error: The X_SIZE value must be defined in the 'hard_config.h' file !
#endif

#if !defined(Y_SIZE)
# error: The Y_SIZE value must be defined in the 'hard_config.h' file !
#endif

#if !defined(X_WIDTH)
# error: The X_WIDTH value must be defined in the 'hard_config.h' file !
#endif

#if !defined(Y_WIDTH)
# error: The Y_WIDTH value must be defined in the 'hard_config.h' file !
#endif

#if !defined(SEG_BOOT_MAPPING_BASE) 
# error: The SEG_BOOT_MAPPING_BASE value must be defined in the hard_config.h file !
#endif

#if !defined(NB_PROCS_MAX)
# error: The NB_PROCS_MAX value must be defined in the 'hard_config.h' file !
#endif

#if !defined(GIET_NB_VSPACE_MAX)
# error: The GIET_NB_VSPACE_MAX value must be defined in the 'giet_config.h' file !
#endif

#if !defined(GIET_ELF_BUFFER_SIZE) 
# error: The GIET_ELF_BUFFER_SIZE value must be defined in the giet_config.h file !
#endif

////////////////////////////////////////////////////////////////////////////
//      Global variables for boot code
////////////////////////////////////////////////////////////////////////////

extern void boot_entry();

// FAT internal representation for boot code  
__attribute__((section(".kdata")))
fat32_fs_t          fat   __attribute__((aligned(512)));

// Temporaty buffer used to load one complete .elf file  
__attribute__((section(".kdata")))
char                boot_elf_buffer[GIET_ELF_BUFFER_SIZE] __attribute__((aligned(512)));

// Physical memory allocators array (one per cluster)
__attribute__((section(".kdata")))
pmem_alloc_t        boot_pmem_alloc[X_SIZE][Y_SIZE];

// Distributed kernel heap (one per cluster)
// __attribute__((section(".kdata")))
// kernel_heap_t       kernel_heap[X_SIZE][Y_SIZE];

// Schedulers virtual base addresses array (one per processor)
__attribute__((section(".kdata")))
static_scheduler_t* _schedulers[X_SIZE][Y_SIZE][NB_PROCS_MAX];

// Page tables virtual base addresses array (one per vspace)
__attribute__((section(".kdata")))
unsigned int        _ptabs_vaddr[GIET_NB_VSPACE_MAX][X_SIZE][Y_SIZE];

// Page tables physical base addresses (one per vspace and per cluster)
__attribute__((section(".kdata")))
paddr_t             _ptabs_paddr[GIET_NB_VSPACE_MAX][X_SIZE][Y_SIZE];

// Page tables pt2 allocators (one per vspace and per cluster)
__attribute__((section(".kdata")))
unsigned int        _ptabs_next_pt2[GIET_NB_VSPACE_MAX][X_SIZE][Y_SIZE];

// Page tables max_pt2  (same value for all page tables)
__attribute__((section(".kdata")))
unsigned int        _ptabs_max_pt2;

// WTI channel allocator (one per cluster)
__attribute__((section(".kdata")))
unsigned int        _wti_channel_alloc[X_SIZE][Y_SIZE];

// boot code uses a spin lock to protect TTY0
__attribute__((section(".kdata")))
unsigned int        _tty0_boot_mode = 1;

__attribute__((section(".kdata")))
spin_lock_t         _ptabs_spin_lock[GIET_NB_VSPACE_MAX][X_SIZE][Y_SIZE];

// barrier used by boot code for parallel execution
__attribute__((section(".kdata")))
simple_barrier_t    _barrier_all_clusters;

// this variable is defined in the tty0.c file
extern spin_lock_t  _tty0_spin_lock;

//////////////////////////////////////////////////////////////////////////////
// This function registers a new PTE1 in the page table defined
// by the vspace_id argument, and the (x,y) coordinates.
// It updates only the first level PT1.
// As each vseg is mapped by a different processor, the PT1 entry cannot
// be concurrently accessed, and we don't need to take any lock.
//////////////////////////////////////////////////////////////////////////////
void boot_add_pte1( unsigned int vspace_id,
                    unsigned int x,
                    unsigned int y,
                    unsigned int vpn,        // 20 bits right-justified
                    unsigned int flags,      // 10 bits left-justified 
                    unsigned int ppn )       // 28 bits right-justified
{
    // compute index in PT1
    unsigned int    ix1 = vpn >> 9;         // 11 bits for ix1

    // get page table physical base address 
    paddr_t  pt1_pbase = _ptabs_paddr[vspace_id][x][y];

    if ( pt1_pbase == 0 )
    {
        _printf("\n[BOOT ERROR] in boot_add_pte1() : no PTAB in cluster[%d,%d]"
                    " containing processors\n", x , y );
        _exit();
    }

    // compute pte1 : 2 bits V T / 8 bits flags / 3 bits RSVD / 19 bits bppi
    unsigned int    pte1 = PTE_V |
                           (flags & 0x3FC00000) |
                           ((ppn>>9) & 0x0007FFFF);

    // write pte1 in PT1
    _physical_write( pt1_pbase + 4*ix1, pte1 );

    asm volatile ("sync");

}   // end boot_add_pte1()

//////////////////////////////////////////////////////////////////////////////
// This function registers a new PTE2 in the page table defined
// by the vspace_id argument, and the (x,y) coordinates.
// It updates both the first level PT1 and the second level PT2.
// As the set of PT2s is implemented as a fixed size array (no dynamic 
// allocation), this function checks a possible overflow of the PT2 array.
// As a given entry in PT1 can be shared by several vsegs, mapped by 
// different processors, we need to take the lock protecting PTAB[v][x]y].
//////////////////////////////////////////////////////////////////////////////
void boot_add_pte2( unsigned int vspace_id,
                    unsigned int x,
                    unsigned int y,
                    unsigned int vpn,        // 20 bits right-justified
                    unsigned int flags,      // 10 bits left-justified 
                    unsigned int ppn )       // 28 bits right-justified
{
    unsigned int ix1;
    unsigned int ix2;
    paddr_t      pt2_pbase;     // PT2 physical base address
    paddr_t      pte2_paddr;    // PTE2 physical address
    unsigned int pt2_id;        // PT2 index
    unsigned int ptd;           // PTD : entry in PT1

    ix1 = vpn >> 9;             // 11 bits for ix1
    ix2 = vpn & 0x1FF;          //  9 bits for ix2

    // get page table physical base address 
    paddr_t      pt1_pbase = _ptabs_paddr[vspace_id][x][y];

    if ( pt1_pbase == 0 )
    {
        _printf("\n[BOOT ERROR] in boot_add_pte2() : no PTAB for vspace %d "
                "in cluster[%d,%d]\n", vspace_id , x , y );
        _exit();
    }

    // get lock protecting PTAB[vspace_id][x][y]
    _spin_lock_acquire( &_ptabs_spin_lock[vspace_id][x][y] );

    // get ptd in PT1
    ptd = _physical_read( pt1_pbase + 4 * ix1 );

    if ((ptd & PTE_V) == 0)    // undefined PTD: compute PT2 base address, 
                               // and set a new PTD in PT1 
    {
        // get a new pt2_id
        pt2_id = _ptabs_next_pt2[vspace_id][x][y];
        _ptabs_next_pt2[vspace_id][x][y] = pt2_id + 1;

        // check overflow
        if (pt2_id == _ptabs_max_pt2) 
        {
            _printf("\n[BOOT ERROR] in boot_add_pte2() : PTAB[%d,%d,%d]"
                    " contains not enough PT2s\n", vspace_id, x, y );
            _exit();
        }

        pt2_pbase = pt1_pbase + PT1_SIZE + PT2_SIZE * pt2_id;
        ptd = PTE_V | PTE_T | (unsigned int) (pt2_pbase >> 12);

        // set PTD into PT1
        _physical_write( pt1_pbase + 4*ix1, ptd);
    }
    else                       // valid PTD: compute PT2 base address
    {
        pt2_pbase = ((paddr_t)(ptd & 0x0FFFFFFF)) << 12;
    }

    // set PTE in PT2 : flags & PPN in two 32 bits words
    pte2_paddr  = pt2_pbase + 8 * ix2;
    _physical_write(pte2_paddr     , (PTE_V | flags) );
    _physical_write(pte2_paddr + 4 , ppn );

    // release lock protecting PTAB[vspace_id][x][y]
    _spin_lock_release( &_ptabs_spin_lock[vspace_id][x][y] );

    asm volatile ("sync");

}   // end boot_add_pte2()

////////////////////////////////////////////////////////////////////////////////////
// Align the value of paddr or vaddr to the required alignement,
// defined by alignPow2 == L2(alignement).
////////////////////////////////////////////////////////////////////////////////////
paddr_t paddr_align_to( paddr_t paddr, unsigned int alignPow2 ) 
{
    paddr_t mask = (1 << alignPow2) - 1;
    return ((paddr + mask) & ~mask);
}

unsigned int vaddr_align_to( unsigned int vaddr, unsigned int alignPow2 ) 
{
    unsigned int mask = (1 << alignPow2) - 1;
    return ((vaddr + mask) & ~mask);
}

/////////////////////////////////////////////////////////////////////////////////////
// This function map a vseg identified by the vseg pointer.
//
// A given vseg can be mapped in a Big Physical Pages (BPP: 2 Mbytes) or in a
// Small Physical Pages (SPP: 4 Kbytes), depending on the "big" attribute of vseg,
// with the following rules:
// - SPP : There is only one vseg in a small physical page, but a single vseg
//   can cover several contiguous small physical pages.
// - BPP : It can exist several vsegs in a single big physical page, and a single
//   vseg can cover several contiguous big physical pages.
//
// 1) First step: it computes various vseg attributes and checks 
//    alignment constraints.
//
// 2) Second step: it allocates the required number of contiguous physical pages, 
//    computes the physical base address (if the vseg is not identity mapping),
//    and register it in the vseg pbase field.
//    Only the vsegs used by the boot code and the peripheral vsegs
//    can be identity mapping. The first big physical page in cluster[0,0] 
//    is reserved for the boot vsegs.
//
// 3) Third step (only for vseg that have the VSEG_TYPE_PTAB): the M page tables
//    associated to the M vspaces must be packed in the same vseg.
//    We divide this vseg in M sub-segments, and compute the vbase and pbase
//    addresses for M page tables, and register these addresses in the _ptabs_paddr
//    and _ptabs_vaddr arrays.
//  
/////////////////////////////////////////////////////////////////////////////////////
void boot_vseg_map( mapping_vseg_t* vseg,
                    unsigned int    vspace_id )
{
    mapping_header_t*   header  = (mapping_header_t *)SEG_BOOT_MAPPING_BASE;
    mapping_cluster_t*  cluster = _get_cluster_base(header);
    mapping_pseg_t*     pseg    = _get_pseg_base(header);

    //////////// First step : compute vseg attributes 

    // compute destination cluster pointer & coordinates
    pseg    = pseg + vseg->psegid;
    cluster = cluster + pseg->clusterid;
    unsigned int        x_dest     = cluster->x;
    unsigned int        y_dest     = cluster->y;

    // compute the "big" vseg attribute
    unsigned int        big = vseg->big;

    // all vsegs must be aligned on 4Kbytes
    if ( vseg->vbase & 0x00000FFF ) 
    {
        _printf("\n[BOOT ERROR] vseg %s not aligned : vbase = %x\n", 
                vseg->name, vseg->vbase );
        _exit();
    }

    // compute the "is_ram" vseg attribute
    unsigned int        is_ram;
    if ( pseg->type == PSEG_TYPE_RAM )  is_ram = 1;
    else                                is_ram = 0;

    // compute the "is_ptab" attribute
    unsigned int        is_ptab;
    if ( vseg->type == VSEG_TYPE_PTAB ) is_ptab = 1;
    else                                is_ptab = 0;

    // compute actual vspace index
    unsigned int vsid;
    if ( vspace_id == 0xFFFFFFFF ) vsid = 0;
    else                           vsid = vspace_id;

    //////////// Second step : compute ppn and npages  
    //////////// - if identity mapping :  ppn <= vpn 
    //////////// - if vseg is periph   :  ppn <= pseg.base >> 12
    //////////// - if vseg is ram      :  ppn <= physical memory allocator

    unsigned int ppn;          // first physical page index (28 bits = |x|y|bppi|sppi|) 
    unsigned int vpn;          // first virtual page index  (20 bits = |ix1|ix2|)
    unsigned int vpn_max;      // last  virtual page index  (20 bits = |ix1|ix2|)

    vpn     = vseg->vbase >> 12;
    vpn_max = (vseg->vbase + vseg->length - 1) >> 12;

    // compute npages
    unsigned int npages;       // number of required (big or small) pages
    if ( big == 0 ) npages  = vpn_max - vpn + 1;            // number of small pages
    else            npages  = (vpn_max>>9) - (vpn>>9) + 1;  // number of big pages

    // compute ppn
    if ( vseg->ident )           // identity mapping
    {
        ppn = vpn;
    }
    else                         // not identity mapping
    {
        if ( is_ram )            // RAM : physical memory allocation required
        {
            // compute pointer on physical memory allocator in dest cluster
            pmem_alloc_t*     palloc = &boot_pmem_alloc[x_dest][y_dest];

            if ( big == 0 )             // SPP : small physical pages
            {
                // allocate contiguous small physical pages
                ppn = _get_small_ppn( palloc, npages );
            }
            else                            // BPP : big physical pages
            {
 
                // one big page can be shared by several vsegs 
                // we must chek if BPP already allocated 
                if ( is_ptab )   // It cannot be mapped
                {
                    ppn = _get_big_ppn( palloc, npages ); 
                }
                else             // It can be mapped
                {
                    unsigned int ix1   = vpn >> 9;   // 11 bits
                    paddr_t      paddr = _ptabs_paddr[vsid][x_dest][y_dest] + (ix1<<2);
                    unsigned int pte1  = _physical_read( paddr );

                    if ( (pte1 & PTE_V) == 0 )     // BPP not allocated yet
                    {
                        // allocate contiguous big physical pages 
                        ppn = _get_big_ppn( palloc, npages );
                    }
                    else                           // BPP already allocated
                    {
                        // test if new vseg has the same mode bits than
                        // the other vsegs in the same big page
                        unsigned int pte1_mode = 0;
                        if (pte1 & PTE_C) pte1_mode |= C_MODE_MASK;
                        if (pte1 & PTE_X) pte1_mode |= X_MODE_MASK;
                        if (pte1 & PTE_W) pte1_mode |= W_MODE_MASK;
                        if (pte1 & PTE_U) pte1_mode |= U_MODE_MASK;
                        if (vseg->mode != pte1_mode) 
                        {
                            _printf("\n[BOOT ERROR] in boot_vseg_map() : "
                                    "vseg %s has different flags than another vseg "
                                    "in the same BPP\n", vseg->name );
                            _exit();
                        }
                        ppn = ((pte1 << 9) & 0x0FFFFE00);
                    }
                }
                ppn = ppn | (vpn & 0x1FF);
            }
        }
        else                    // PERI : no memory allocation required
        {
            ppn = pseg->base >> 12;
        }
    }

    // update vseg.pbase field and update vsegs chaining
    vseg->pbase     = ((paddr_t)ppn) << 12;
    vseg->mapped    = 1;


    //////////// Third step : (only if the vseg is a page table)
    //////////// - compute the physical & virtual base address for each vspace
    ////////////   by dividing the vseg in several sub-segments.
    //////////// - register it in _ptabs_vaddr & _ptabs_paddr arrays,
    ////////////   and initialize next_pt2 allocators.
    //////////// - reset all entries in first level page tables
    
    if ( is_ptab )
    {
        unsigned int   vs;        // vspace index
        unsigned int   nspaces;   // number of vspaces
        unsigned int   nsp;       // number of small pages for one PTAB
        unsigned int   offset;    // address offset for current PTAB

        nspaces = header->vspaces;
        offset  = 0;

        // each PTAB must be aligned on a 8 Kbytes boundary
        nsp = ( vseg->length >> 12 ) / nspaces;
        if ( (nsp & 0x1) == 0x1 ) nsp = nsp - 1;

        // compute max_pt2
        _ptabs_max_pt2 = ((nsp<<12) - PT1_SIZE) / PT2_SIZE;

        for ( vs = 0 ; vs < nspaces ; vs++ )
        {
            _ptabs_vaddr   [vs][x_dest][y_dest] = (vpn + offset) << 12;
            _ptabs_paddr   [vs][x_dest][y_dest] = ((paddr_t)(ppn + offset)) << 12;
            _ptabs_next_pt2[vs][x_dest][y_dest] = 0;
            offset += nsp;

            // reset all entries in PT1 (8 Kbytes)
            _physical_memset( _ptabs_paddr[vs][x_dest][y_dest], PT1_SIZE, 0 );
        }
    }

    asm volatile ("sync");

#if BOOT_DEBUG_PT
if ( big )
_printf("\n[BOOT] vseg %s : cluster[%d,%d] / "
       "vbase = %x / length = %x / BIG    / npages = %d / pbase = %l\n",
       vseg->name, x_dest, y_dest, vseg->vbase, vseg->length, npages, vseg-> pbase );
else
_printf("\n[BOOT] vseg %s : cluster[%d,%d] / "
        "vbase = %x / length = %x / SMALL / npages = %d / pbase = %l\n",
       vseg->name, x_dest, y_dest, vseg->vbase, vseg->length, npages, vseg-> pbase );
#endif

} // end boot_vseg_map()

/////////////////////////////////////////////////////////////////////////////////////
// For the vseg defined by the vseg pointer, this function register PTEs
// in one or several page tables.
// It is a global vseg (kernel vseg) if (vspace_id == 0xFFFFFFFF).
// The number of involved PTABs depends on the "local" and "global" attributes:
//  - PTEs are replicated in all vspaces for a global vseg.
//  - PTEs are replicated in all clusters containing procs for a non local vseg.
/////////////////////////////////////////////////////////////////////////////////////
void boot_vseg_pte( mapping_vseg_t*  vseg,
                    unsigned int     vspace_id )
{
    // compute the "global" vseg attribute and actual vspace index
    unsigned int        global;
    unsigned int        vsid;    
    if ( vspace_id == 0xFFFFFFFF )
    {
        global = 1;
        vsid   = 0;
    }
    else
    {
        global = 0;
        vsid   = vspace_id;
    }

    // compute the "local" and "big" attributes 
    unsigned int        local  = vseg->local;
    unsigned int        big    = vseg->big;

    // compute vseg flags
    // The three flags (Local, Remote and Dirty) are set to 1 
    // to avoid hardware update for these flags, because GIET_VM 
    // does use these flags.
    unsigned int flags = 0;
    if (vseg->mode & C_MODE_MASK) flags |= PTE_C;
    if (vseg->mode & X_MODE_MASK) flags |= PTE_X;
    if (vseg->mode & W_MODE_MASK) flags |= PTE_W;
    if (vseg->mode & U_MODE_MASK) flags |= PTE_U;
    if ( global )                 flags |= PTE_G;
                                  flags |= PTE_L;
                                  flags |= PTE_R;
                                  flags |= PTE_D;

    // compute VPN, PPN and number of pages (big or small) 
    unsigned int vpn     = vseg->vbase >> 12;
    unsigned int vpn_max = (vseg->vbase + vseg->length - 1) >> 12;
    unsigned int ppn     = (unsigned int)(vseg->pbase >> 12);
    unsigned int npages;
    if ( big == 0 ) npages  = vpn_max - vpn + 1;            
    else            npages  = (vpn_max>>9) - (vpn>>9) + 1; 

    // compute destination cluster coordinates, for local vsegs
    mapping_header_t*   header       = (mapping_header_t *)SEG_BOOT_MAPPING_BASE;
    mapping_cluster_t*  cluster      = _get_cluster_base(header);
    mapping_pseg_t*     pseg         = _get_pseg_base(header);
    mapping_pseg_t*     pseg_dest    = &pseg[vseg->psegid];
    mapping_cluster_t*  cluster_dest = &cluster[pseg_dest->clusterid];
    unsigned int        x_dest       = cluster_dest->x;
    unsigned int        y_dest       = cluster_dest->y;

    unsigned int p;           // iterator for physical page index
    unsigned int x;           // iterator for cluster x coordinate  
    unsigned int y;           // iterator for cluster y coordinate  
    unsigned int v;           // iterator for vspace index 

    // loop on PTEs
    for ( p = 0 ; p < npages ; p++ )
    { 
        if  ( (local != 0) && (global == 0) )         // one cluster  / one vspace
        {
            if ( big )   // big pages => PTE1s
            {
                boot_add_pte1( vsid,
                               x_dest,
                               y_dest,
                               vpn + (p<<9),
                               flags, 
                               ppn + (p<<9) );
            }
            else         // small pages => PTE2s
            {
                boot_add_pte2( vsid,
                               x_dest,
                               y_dest,
                               vpn + p,      
                               flags, 
                               ppn + p );
            }
        }
        else if ( (local == 0) && (global == 0) )     // all clusters / one vspace
        {
            for ( x = 0 ; x < X_SIZE ; x++ )
            {
                for ( y = 0 ; y < Y_SIZE ; y++ )
                {
                    if ( cluster[(x * Y_SIZE) + y].procs )
                    {
                        if ( big )   // big pages => PTE1s
                        {
                            boot_add_pte1( vsid,
                                           x,
                                           y,
                                           vpn + (p<<9),
                                           flags, 
                                           ppn + (p<<9) );
                        }
                        else         // small pages => PTE2s
                        {
                            boot_add_pte2( vsid,
                                           x,
                                           y,
                                           vpn + p,
                                           flags, 
                                           ppn + p );
                        }
                    }
                }
            }
        }
        else if ( (local != 0) && (global != 0) )     // one cluster  / all vspaces 
        {
            for ( v = 0 ; v < header->vspaces ; v++ )
            {
                if ( big )   // big pages => PTE1s
                {
                    boot_add_pte1( v,
                                   x_dest,
                                   y_dest,
                                   vpn + (p<<9),
                                   flags, 
                                   ppn + (p<<9) );
                }
                else         // small pages = PTE2s
                { 
                    boot_add_pte2( v,
                                   x_dest,
                                   y_dest,
                                   vpn + p,
                                   flags, 
                                   ppn + p );
                }
            }
        }
        else if ( (local == 0) && (global != 0) )     // all clusters / all vspaces
        {
            for ( x = 0 ; x < X_SIZE ; x++ )
            {
                for ( y = 0 ; y < Y_SIZE ; y++ )
                {
                    if ( cluster[(x * Y_SIZE) + y].procs )
                    {
                        for ( v = 0 ; v < header->vspaces ; v++ )
                        {
                            if ( big )  // big pages => PTE1s
                            {
                                boot_add_pte1( v,
                                               x,
                                               y,
                                               vpn + (p<<9),
                                               flags, 
                                               ppn + (p<<9) );
                            }
                            else        // small pages -> PTE2s
                            {
                                boot_add_pte2( v,
                                               x,
                                               y,
                                               vpn + p,
                                               flags, 
                                               ppn + p );
                            }
                        }
                    }
                }
            }
        }
    }  // end for pages

    asm volatile ("sync");

}  // end boot_vseg_pte()


///////////////////////////////////////////////////////////////////////////////
// Processor P[x][y][0] computes physical base address for all globals vsegs,
// using the local Page Table, to check page tables initialisation.
///////////////////////////////////////////////////////////////////////////////
void boot_ptab_check( unsigned int x,
                      unsigned int y )
{
    mapping_header_t*   header = (mapping_header_t *)SEG_BOOT_MAPPING_BASE;
    mapping_vseg_t*     vseg   = _get_vseg_base(header);
    page_table_t*       ptab   = (page_table_t*)_ptabs_vaddr[0][x][y];

    unsigned int vseg_id;
    for (vseg_id = 0; vseg_id < header->globals; vseg_id++) 
    {
        unsigned int  vpn   = vseg[vseg_id].vbase >> 12;
        unsigned int  ppn   = 0;  
        unsigned int  flags = 0;
        _v2p_translate( ptab , vpn , &ppn , &flags );
        _printf("@@@ P[%d,%d,0] access vseg %s : vpn = %x / ppn = %x\n",
                x , y , vseg[vseg_id].name , vpn , ppn );  
    }
} 

///////////////////////////////////////////////////////////////////////////////
// This function is executed by  processor[x][y][0] in each cluster 
// containing at least one processor.
// It initialises all page table for all global or private vsegs
// mapped in cluster[x][y], as specified in the mapping.
// In each cluster all page tables for the different vspaces must be
// packed in one vseg occupying one single BPP (Big Physical Page).
// 
// For each vseg, the mapping is done in two steps:
// 1) mapping : the boot_vseg_map() function allocates contiguous BPPs 
//    or SPPs (if the vseg is not associated to a peripheral), and register
//    the physical base address in the vseg pbase field. It initialises the
//    _ptabs_vaddr[] and _ptabs_paddr[] arrays if the vseg is a PTAB.
//
// 2) page table initialisation : the boot_vseg_pte() function initialise 
//    the PTEs (both PTE1 and PTE2) in one or several page tables:
//    - PTEs are replicated in all vspaces for a global vseg.
//    - PTEs are replicated in all clusters for a non local vseg.
//
// We must handle vsegs in the following order
//   1) global vseg containing PTAB mapped in cluster[x][y],
//   2) global vsegs occupying more than one BPP mapped in cluster[x][y], 
//   3) others global vsegs mapped in cluster[x][y],
//   4) all private vsegs in all user spaces mapped in cluster[x][y].
///////////////////////////////////////////////////////////////////////////////
void boot_ptab_init( unsigned int cx,
                     unsigned int cy ) 
{
    mapping_header_t*   header = (mapping_header_t *)SEG_BOOT_MAPPING_BASE;
    mapping_vspace_t*   vspace = _get_vspace_base(header);
    mapping_vseg_t*     vseg   = _get_vseg_base(header);
    mapping_cluster_t*  cluster ;
    mapping_pseg_t*     pseg    ;

    unsigned int vspace_id;
    unsigned int vseg_id;

    unsigned int procid     = _get_procid();
    unsigned int lpid       = procid & ((1<<P_WIDTH)-1);

    if( lpid )
    {
        _printf("\n[BOOT ERROR] in boot_ptab_init() : "
                "P[%d][%d][%d] should not execute it\n", cx, cy, lpid );
        _exit();
    } 

    if ( header->vspaces == 0 )
    {
        _printf("\n[BOOT ERROR] in boot_ptab_init() : "
                "mapping %s contains no vspace\n", header->name );
        _exit();
    }

    ///////// Phase 1 : global vseg containing the PTAB (two barriers required)

    // get PTAB global vseg in cluster(cx,cy)
    unsigned int found = 0;
    for (vseg_id = 0; vseg_id < header->globals; vseg_id++) 
    {
        pseg    = _get_pseg_base(header) + vseg[vseg_id].psegid;
        cluster = _get_cluster_base(header) + pseg->clusterid;
        if ( (vseg[vseg_id].type == VSEG_TYPE_PTAB) && 
             (cluster->x == cx) && (cluster->y == cy) )
        {
            found = 1;
            break;
        }
    }
    if ( found == 0 )
    {
        _printf("\n[BOOT ERROR] in boot_ptab_init() : "
                "cluster[%d][%d] contains no PTAB vseg\n", cx , cy );
        _exit();
    }

    boot_vseg_map( &vseg[vseg_id], 0xFFFFFFFF );

    //////////////////////////////////////////////
    _simple_barrier_wait( &_barrier_all_clusters );
    //////////////////////////////////////////////

    boot_vseg_pte( &vseg[vseg_id], 0xFFFFFFFF );

    //////////////////////////////////////////////
    _simple_barrier_wait( &_barrier_all_clusters );
    //////////////////////////////////////////////

    ///////// Phase 2 : global vsegs occupying more than one BPP 

    for (vseg_id = 0; vseg_id < header->globals; vseg_id++) 
    {
        pseg    = _get_pseg_base(header) + vseg[vseg_id].psegid;
        cluster = _get_cluster_base(header) + pseg->clusterid;
        if ( (vseg[vseg_id].length > 0x200000) &&
             (vseg[vseg_id].mapped == 0) &&
             (cluster->x == cx) && (cluster->y == cy) )
        {
            boot_vseg_map( &vseg[vseg_id], 0xFFFFFFFF );
            boot_vseg_pte( &vseg[vseg_id], 0xFFFFFFFF );
        }
    }

    ///////// Phase 3 : all others global vsegs 

    for (vseg_id = 0; vseg_id < header->globals; vseg_id++) 
    { 
        pseg    = _get_pseg_base(header) + vseg[vseg_id].psegid;
        cluster = _get_cluster_base(header) + pseg->clusterid;
        if ( (vseg[vseg_id].mapped == 0) && 
             (cluster->x == cx) && (cluster->y == cy) )
        {
            boot_vseg_map( &vseg[vseg_id], 0xFFFFFFFF );
            boot_vseg_pte( &vseg[vseg_id], 0xFFFFFFFF );
        }
    }

    ///////// Phase 4 : all private vsegs 

    for (vspace_id = 0; vspace_id < header->vspaces; vspace_id++) 
    {
        for (vseg_id = vspace[vspace_id].vseg_offset;
             vseg_id < (vspace[vspace_id].vseg_offset + vspace[vspace_id].vsegs);
             vseg_id++) 
        {
            pseg    = _get_pseg_base(header) + vseg[vseg_id].psegid;
            cluster = _get_cluster_base(header) + pseg->clusterid;
            if ( (cluster->x == cx) && (cluster->y == cy) )
            {
                boot_vseg_map( &vseg[vseg_id], vspace_id );
                boot_vseg_pte( &vseg[vseg_id], vspace_id );
            }
        }
    }

    //////////////////////////////////////////////
    _simple_barrier_wait( &_barrier_all_clusters );
    //////////////////////////////////////////////

} // end boot_ptab_init()

////////////////////////////////////////////////////////////////////////////////
// This function should be executed by P[0][0][0] only. It complete the
// page table initialisation, taking care of all global vsegs that are
// not mapped in a cluster containing a processor, and have not been
// handled by the boot_ptab_init(x,y) function.
// An example of such vsegs are the external peripherals in TSAR_LETI platform.
////////////////////////////////////////////////////////////////////////////////
void boot_ptab_extend()
{

    mapping_header_t*   header = (mapping_header_t *)SEG_BOOT_MAPPING_BASE;
    mapping_vseg_t*     vseg   = _get_vseg_base(header);

    unsigned int vseg_id;

    for (vseg_id = 0; vseg_id < header->globals; vseg_id++) 
    {
        if ( vseg[vseg_id].mapped == 0 )  
        {
            boot_vseg_map( &vseg[vseg_id], 0xFFFFFFFF );
            boot_vseg_pte( &vseg[vseg_id], 0xFFFFFFFF );
        }
    }
}  // end boot_ptab_extend()

///////////////////////////////////////////////////////////////////////////////
// This function returns in the vbase and length buffers the virtual base 
// address and the length of the  segment allocated to the schedulers array 
// in the cluster defined by the clusterid argument.
///////////////////////////////////////////////////////////////////////////////
void boot_get_sched_vaddr( unsigned int  cluster_id,
                           unsigned int* vbase, 
                           unsigned int* length )
{
    mapping_header_t* header = (mapping_header_t *)SEG_BOOT_MAPPING_BASE;
    mapping_vseg_t*   vseg   = _get_vseg_base(header);
    mapping_pseg_t*   pseg   = _get_pseg_base(header);

    unsigned int vseg_id;
    unsigned int found = 0;

    for ( vseg_id = 0 ; (vseg_id < header->vsegs) && (found == 0) ; vseg_id++ )
    {
        if ( (vseg[vseg_id].type == VSEG_TYPE_SCHED) && 
             (pseg[vseg[vseg_id].psegid].clusterid == cluster_id ) )
        {
            *vbase  = vseg[vseg_id].vbase;
            *length = vseg[vseg_id].length;
            found = 1;
        }
    }
    if ( found == 0 )
    {
        mapping_cluster_t* cluster = _get_cluster_base(header);
        _printf("\n[BOOT ERROR] No vseg of type SCHED in cluster [%d,%d]\n",
                cluster[cluster_id].x, cluster[cluster_id].y );
        _exit();
    }
} // end boot_get_sched_vaddr()

////////////////////////////////////////////////////////////////////////////////////
// This function is executed in parallel by all processors P[x][y][0].
// It initialises all schedulers in cluster [x][y]. The MMU must be activated.
// It is split in two phases separated by a synchronisation barrier.
// - In Step 1, it initialises the _schedulers[x][y][l] pointers array,
//              the idle_task context and the HWI / PTI vectors. 
// - In Step 2, it scan all tasks in all vspaces to complete the tasks contexts, 
//              initialisation as specified in the mapping_info data structure,
//              and set the CP0_SCHED register. 
////////////////////////////////////////////////////////////////////////////////////
void boot_scheduler_init( unsigned int x, 
                          unsigned int y )
{
    mapping_header_t*    header  = (mapping_header_t *)SEG_BOOT_MAPPING_BASE;
    mapping_cluster_t*   cluster = _get_cluster_base(header);
    mapping_vspace_t*    vspace  = _get_vspace_base(header);
    mapping_vseg_t*      vseg    = _get_vseg_base(header);
    mapping_task_t*      task    = _get_task_base(header);
    mapping_periph_t*    periph  = _get_periph_base(header);
    mapping_irq_t*       irq     = _get_irq_base(header);

    unsigned int         periph_id; 
    unsigned int         irq_id;
    unsigned int         vspace_id;
    unsigned int         vseg_id;
    unsigned int         task_id; 

    unsigned int         sched_vbase;          // schedulers array vbase address 
    unsigned int         sched_length;         // schedulers array length
    static_scheduler_t*  psched;               // pointer on processor scheduler

    unsigned int cluster_id = x * Y_SIZE + y;  
    unsigned int nprocs = cluster[cluster_id].procs;
    unsigned int lpid;                       
    
    /////////////////////////////////////////////////////////////////////////
    // Step 1 : initialize the schedulers[] array of pointers,
    //          the idle task context and the HWI and PTI interrupt vectors.
    //          The WTI interrupt vector entries corresponding to interrupts
    //          generated by the PIC component are handled later.

    // get scheduler array virtual base address in cluster[x,y]
    boot_get_sched_vaddr( cluster_id, &sched_vbase, &sched_length );

    if ( sched_length < (nprocs<<13) ) // 8 Kbytes per scheduler
    {
        _printf("\n[BOOT ERROR] Sched segment too small in cluster[%d,%d]\n", x, y );
        _exit();
    }

    // loop on local processors
    for ( lpid = 0 ; lpid < nprocs ; lpid++ )
    {
        // get scheduler pointer and initialise the schedulers pointers array
        psched = (static_scheduler_t*)(sched_vbase + (lpid<<13));
        _schedulers[x][y][lpid] = psched;

        // initialise the "tasks" and "current" variables default values
        psched->tasks   = 0;
        psched->current = IDLE_TASK_INDEX;

        // default values for HWI / PTI / SWI vectors (valid bit = 0)
        unsigned int slot;
        for (slot = 0; slot < 32; slot++)
        {
            psched->hwi_vector[slot] = 0;
            psched->pti_vector[slot] = 0;
            psched->wti_vector[slot] = 0;
        }

        // WTI[lpid] <= ISR_WAKUP / PTI[lpid] <= ISR_TICK 
        psched->wti_vector[lpid] = ISR_WAKUP | 0x80000000;
        psched->pti_vector[lpid] = ISR_TICK  | 0x80000000;

        // initializes the idle_task context:
        // - the SR slot is 0xFF03 because this task run in kernel mode.
        // - it uses the page table of vspace[0]
        // - it uses the kernel TTY terminal
        // - slots containing addresses (SP,RA,EPC) are initialised by kernel_init()

        psched->context[IDLE_TASK_INDEX][CTX_CR_ID]   = 0;
        psched->context[IDLE_TASK_INDEX][CTX_SR_ID]   = 0xFF03;
        psched->context[IDLE_TASK_INDEX][CTX_PTPR_ID] = _ptabs_paddr[0][x][y]>>13;
        psched->context[IDLE_TASK_INDEX][CTX_PTAB_ID] = _ptabs_vaddr[0][x][y];
        psched->context[IDLE_TASK_INDEX][CTX_TTY_ID]  = 0;
        psched->context[IDLE_TASK_INDEX][CTX_LTID_ID] = IDLE_TASK_INDEX;
        psched->context[IDLE_TASK_INDEX][CTX_VSID_ID] = 0;
        psched->context[IDLE_TASK_INDEX][CTX_RUN_ID]  = 1;
    }

    // scan local peripherals to get local XCU 
    mapping_periph_t*  xcu = NULL;

    for ( periph_id = cluster[cluster_id].periph_offset ;
          periph_id < cluster[cluster_id].periph_offset + cluster[cluster_id].periphs;
          periph_id++ )
    {
        if( periph[periph_id].type == PERIPH_TYPE_XCU ) 
        {
            xcu = &periph[periph_id];

            if ( xcu->arg0 < (nprocs * header->irq_per_proc) )
            {
                _printf("\n[BOOT ERROR] Not enough inputs for XCU[%d,%d]\n", x, y );
                _exit();
            }
        }
    } 

    if ( xcu == NULL )
    {         
        _printf("\n[BOOT ERROR] missing XCU in cluster[%d,%d]\n", x , y );
        _exit();
    }

    // scan HWIs connected to local XCU 
    // for round-robin allocation to local processors
    lpid = 0;
    for ( irq_id = xcu->irq_offset ;
          irq_id < xcu->irq_offset + xcu->irqs ;
          irq_id++ )
    {
        unsigned int type    = irq[irq_id].srctype;
        unsigned int srcid   = irq[irq_id].srcid;
        unsigned int isr     = irq[irq_id].isr & 0xFFFF;
        unsigned int channel = irq[irq_id].channel << 16;

        if ( (type != IRQ_TYPE_HWI) || (srcid > 31) )
        {
            _printf("\n[BOOT ERROR] Bad IRQ in cluster[%d,%d]\n", x, y );
            _exit();
        }

        _schedulers[x][y][lpid]->hwi_vector[srcid] = isr | channel | 0x80000000;

        lpid = (lpid + 1) % nprocs; 
    } // end for irqs

    //////////////////////////////////////////////
    _simple_barrier_wait( &_barrier_all_clusters );
    //////////////////////////////////////////////

    ////////////////////////////////////////////////////////////////////////
    // Step 2 : Initialise the tasks context. The context of task placed
    //          on  processor P must be stored in the scheduler of P.
    //          This require two nested loops: loop on the tasks, and loop 
    //          on the local processors. We complete the scheduler when the
    //          required placement fit one local processor.

    for (vspace_id = 0; vspace_id < header->vspaces; vspace_id++) 
    {
        // We must set the PTPR depending on the vspace, because the start_vector 
        // and the stack address are defined in virtual space.
        _set_mmu_ptpr( (unsigned int)(_ptabs_paddr[vspace_id][x][y] >> 13) );

        // loop on the tasks in vspace (task_id is the global index in mapping)
        for (task_id = vspace[vspace_id].task_offset;
             task_id < (vspace[vspace_id].task_offset + vspace[vspace_id].tasks);
             task_id++) 
        {
            // get the required task placement coordinates [x,y,p]
            unsigned int req_x      = cluster[task[task_id].clusterid].x;
            unsigned int req_y      = cluster[task[task_id].clusterid].y;
            unsigned int req_p      = task[task_id].proclocid;                 

            // ctx_sr : value required before an eret instruction
            unsigned int ctx_sr = 0x2000FF13;

            // ctx_ptpr : page table physical base address (shifted by 13 bit)
            unsigned int ctx_ptpr = (_ptabs_paddr[vspace_id][req_x][req_y] >> 13);

            // ctx_ptab : page_table virtual base address
            unsigned int ctx_ptab = _ptabs_vaddr[vspace_id][req_x][req_y];

            // ctx_epc : Get the virtual address of the memory location containing
            // the task entry point : the start_vector is stored by GCC in the seg_data 
            // segment and we must wait the .elf loading to get the entry point value...
            vseg_id = vspace[vspace_id].start_vseg_id;     
            unsigned int ctx_epc = vseg[vseg_id].vbase + (task[task_id].startid)*4;

            // ctx_sp :  Get the vseg containing the stack 
            vseg_id = task[task_id].stack_vseg_id;
            unsigned int ctx_sp = vseg[vseg_id].vbase + vseg[vseg_id].length;

            // get vspace thread index
            unsigned int thread_id = task[task_id].trdid;

            // loop on the local processors
            for ( lpid = 0 ; lpid < nprocs ; lpid++ )
            {
                if ( (x == req_x) && (y == req_y) && (req_p == lpid) )   // fit
                {
                    // pointer on selected scheduler
                    psched = _schedulers[x][y][lpid];

                    // get local task index in scheduler
                    unsigned int ltid = psched->tasks;

                    // update the "tasks" and "current" fields in scheduler:
                    psched->tasks   = ltid + 1;
                    psched->current = 0;

                    // initializes the task context 
                    psched->context[ltid][CTX_CR_ID]     = 0;
                    psched->context[ltid][CTX_SR_ID]     = ctx_sr;
                    psched->context[ltid][CTX_SP_ID]     = ctx_sp;
                    psched->context[ltid][CTX_EPC_ID]    = ctx_epc;
                    psched->context[ltid][CTX_PTPR_ID]   = ctx_ptpr;
                    psched->context[ltid][CTX_PTAB_ID]   = ctx_ptab;
                    psched->context[ltid][CTX_LTID_ID]   = ltid;
                    psched->context[ltid][CTX_GTID_ID]   = task_id;
                    psched->context[ltid][CTX_TRDID_ID]  = thread_id;
                    psched->context[ltid][CTX_VSID_ID]   = vspace_id;
                    psched->context[ltid][CTX_RUN_ID]    = 1;

                    psched->context[ltid][CTX_TTY_ID]    = 0xFFFFFFFF;
                    psched->context[ltid][CTX_CMA_FB_ID] = 0xFFFFFFFF;
                    psched->context[ltid][CTX_CMA_RX_ID] = 0xFFFFFFFF;
                    psched->context[ltid][CTX_CMA_TX_ID] = 0xFFFFFFFF;
                    psched->context[ltid][CTX_NIC_RX_ID] = 0xFFFFFFFF;
                    psched->context[ltid][CTX_NIC_TX_ID] = 0xFFFFFFFF;
                    psched->context[ltid][CTX_TIM_ID]    = 0xFFFFFFFF;
                    psched->context[ltid][CTX_HBA_ID]    = 0xFFFFFFFF;

#if BOOT_DEBUG_SCHED
_printf("\nTask %s in vspace %s allocated to P[%d,%d,%d]\n"
        " - ctx[LTID]  = %d\n"
        " - ctx[SR]    = %x\n"
        " - ctx[SP]    = %x\n"
        " - ctx[EPC]   = %x\n"
        " - ctx[PTPR]  = %x\n"
        " - ctx[PTAB]  = %x\n"
        " - ctx[VSID]  = %d\n"
        " - ctx[TRDID] = %d\n",
        task[task_id].name,
        vspace[vspace_id].name,
        x, y, lpid,
        psched->context[ltid][CTX_LTID_ID],
        psched->context[ltid][CTX_SR_ID],
        psched->context[ltid][CTX_SP_ID],
        psched->context[ltid][CTX_EPC_ID],
        psched->context[ltid][CTX_PTPR_ID],
        psched->context[ltid][CTX_PTAB_ID],
        psched->context[ltid][CTX_VSID_ID],
        psched->context[ltid][CTX_TRDID_ID] );
#endif
                } // end if FIT
            } // end for loop on local procs
        } // end loop on tasks
    } // end loop on vspaces
} // end boot_scheduler_init()


/////////////////////////////////////////////////////////////////////////////
// This function loops on all processors in all clusters to display
// the interrupt vectors for each processor.
/////////////////////////////////////////////////////////////////////////////
void boot_sched_irq_display()
{
    unsigned int         cx;
    unsigned int         cy;
    unsigned int         lpid;
    unsigned int         slot;
    unsigned int         entry;

    mapping_header_t*    header  = (mapping_header_t *)SEG_BOOT_MAPPING_BASE;
    mapping_cluster_t*   cluster = _get_cluster_base(header);

    static_scheduler_t*  psched; 

    for ( cx = 0 ; cx < X_SIZE ; cx++ )
    {
        for ( cy = 0 ; cy < Y_SIZE ; cy++ )
        {
            unsigned int cluster_id = (cx * Y_SIZE) + cy;
            unsigned int nprocs = cluster[cluster_id].procs;

            for ( lpid = 0 ; lpid < nprocs ; lpid++ )
            {
                psched = _schedulers[cx][cy][lpid];
        
                _printf("\n[BOOT] scheduler for proc[%d,%d,%d] : ntasks = %d\n",
                        cx , cy , lpid , psched->tasks );

                for ( slot = 0 ; slot < 32 ; slot++ )
                {
                    entry = psched->hwi_vector[slot];
                    if ( entry & 0x80000000 ) 
                    _printf(" - HWI %d / isrtype = %d / channel = %d\n",
                            slot , (entry & 0xFFFF) , ((entry >> 16) & 0x7FFF) );
                }
                for ( slot = 0 ; slot < 32 ; slot++ )
                {
                    entry = psched->wti_vector[slot];
                    if ( entry & 0x80000000 ) 
                    _printf(" - WTI %d / isrtype = %d / channel = %d\n",
                            slot , (entry & 0xFFFF) , ((entry >> 16) & 0x7FFF) );
                }
                for ( slot = 0 ; slot < 32 ; slot++ )
                {
                    entry = psched->pti_vector[slot];
                    if ( entry & 0x80000000 ) 
                    _printf(" - PTI %d / isrtype = %d / channel = %d\n",
                            slot , (entry & 0xFFFF) , ((entry >> 16) & 0x7FFF) );
                }
            }
        }
    } 
}  // end boot_sched_display()


/////////////////////////////////////////////////////////////////////////////
// This function complete the schedulers initialisation when the platform
// contains a PIC component in the IO cluster.
// It is executed by P[0][0][0] only.
// It scan HWIs connected to PIC for Round Robin allocation to processors,
// as WTI. It allocates one WTI per processor, starting from P[0,0,0], 
// and increments (cluster_id, lpid) as required.
/////////////////////////////////////////////////////////////////////////////
void boot_pic_wti_init()
{
    mapping_header_t*    header  = (mapping_header_t *)SEG_BOOT_MAPPING_BASE;
    mapping_cluster_t*   cluster = _get_cluster_base(header);
    mapping_periph_t*    periph  = _get_periph_base(header);
    mapping_irq_t*       irq     = _get_irq_base(header);

    unsigned int         periph_id;   // peripheral index in mapping_info
    unsigned int         irq_id;      // irq index in mapping_info

    // get cluster_io index in mapping
    unsigned int         x_io       = header->x_io;  
    unsigned int         y_io       = header->y_io;  
    unsigned int         cluster_io = (x_io * Y_SIZE) + y_io; 
    
    // scan peripherals in cluster_io to find PIC 
    mapping_periph_t*    pic = NULL;

    for ( periph_id = cluster[cluster_io].periph_offset ;
          periph_id < cluster[cluster_io].periph_offset + cluster[cluster_io].periphs;
          periph_id++ )
    {
        if ( periph[periph_id].type == PERIPH_TYPE_PIC ) 
        {
            pic = &periph[periph_id];
            break;
        }
    }

    if ( pic == NULL )  return;

    // initialize WTI channel allocators in all clusters
    unsigned int x;
    unsigned int y;
    for ( x = 0 ; x < X_SIZE ; x++ )
    {
        for ( y = 0 ; y < Y_SIZE ; y++ )
        {
            _wti_channel_alloc[x][y] = NB_PROCS_MAX;
        }
    }

    // scan IRQS defined in PIC
    unsigned int  cluster_id = 0;
    unsigned int  lpid       = 0;
    unsigned int  cx         = cluster[cluster_id].x;
    unsigned int  cy         = cluster[cluster_id].y;

    for ( irq_id = pic->irq_offset ;
          irq_id < pic->irq_offset + pic->irqs ;
          irq_id++ )
    {
        // compute next values for cluster_id, lpid, cx, cy
        // if no more WTI allocatable in current cluster
        unsigned int overflow = 0;

        while ( (lpid >= cluster[cluster_id].procs) ||
                (_wti_channel_alloc[cx][cy] >= 32) )
        {
            cluster_id = (cluster_id + 1) % (X_SIZE*Y_SIZE);
            cx         = cluster[cluster_id].x;
            cy         = cluster[cluster_id].y;
            lpid       = 0;

            overflow++;

            if ( overflow > 1024 )
            {
                _printf("\n[BOOT ERROR] Not enough processors for external IRQs\n");
                _exit();
            }
        } 
        // allocate a WTI to processor defined by (cluster_id,lpid)
        unsigned int type    = irq[irq_id].srctype;
        unsigned int srcid   = irq[irq_id].srcid;
        unsigned int isr     = irq[irq_id].isr & 0xFFFF;
        unsigned int channel = irq[irq_id].channel << 16;

        if ( (type != IRQ_TYPE_HWI) || (srcid > 31) )
        {
            _printf("\n[BOOT ERROR] in boot_pic_wti_init() Bad IRQ type\n");
            _exit();
        }

        // get scheduler address for selected processor
        static_scheduler_t* psched = _schedulers[cx][cy][lpid];

        // update WTI vector for selected processor
        unsigned int index            = _wti_channel_alloc[cx][cy];
        psched->wti_vector[index]     = isr | channel | 0x80000000;

        // update IRQ fields in mapping for PIC initialisation
        irq[irq_id].dest_id = index;
        irq[irq_id].dest_xy = (cx << Y_WIDTH) + cy;

        // update pointers
        _wti_channel_alloc[cx][cy] = index + 1;
        lpid                       = lpid + 1;

    }  // end for IRQs

#if BOOT_DEBUG_SCHED
boot_sched_irq_display();
#endif

} // end boot_pic_wti_init()
                
//////////////////////////////////////////////////////////////////////////////////
// This function loads the map.bin file from block device.
//////////////////////////////////////////////////////////////////////////////////
void boot_mapping_init()
{
    // open file "map.bin"
    int fd_id = _fat_open( IOC_BOOT_MODE,
                           "map.bin",
                           0 );         // no creation
    if ( fd_id == -1 )
    {
        _printf("\n[BOOT ERROR] : map.bin file not found \n");
        _exit();
    }

#if BOOT_DEBUG_MAPPING
_printf("\n[BOOT] map.bin file successfully open at cycle %d\n", _get_proctime() );
#endif

    // get "map.bin" file size (from fat) and check it
    unsigned int size    = fat.fd[fd_id].file_size;

    if ( size > SEG_BOOT_MAPPING_SIZE )
    {
        _printf("\n[BOOT ERROR] : allocated segment too small for map.bin file\n");
        _exit();
    }

#if BOOT_DEBUG_MAPPING
_printf("\n[BOOT] map.bin buffer pbase = %x / buffer size = %x / file_size = %x\n",
        SEG_BOOT_MAPPING_BASE , SEG_BOOT_MAPPING_SIZE , size );
#endif

    // load "map.bin" file into buffer
    unsigned int nblocks = size >> 9;
    unsigned int offset  = size & 0x1FF;
    if ( offset ) nblocks++;

    unsigned int ok = _fat_read( IOC_BOOT_MODE,
                                 fd_id, 
                                 (unsigned int*)SEG_BOOT_MAPPING_BASE, 
                                 nblocks,       
                                 0 );      // offset
    if ( ok == -1 )
    {
        _printf("\n[BOOT ERROR] : unable to load map.bin file \n");
        _exit();
    }

#if BOOT_DEBUG_MAPPING
_printf("\n[BOOT] map.bin file successfully loaded at cycle %d\n", _get_proctime() );
#endif

    // check mapping signature, number of clusters, number of vspaces  
    mapping_header_t * header = (mapping_header_t *)SEG_BOOT_MAPPING_BASE;
    if ( (header->signature != IN_MAPPING_SIGNATURE) ||
         (header->x_size    != X_SIZE)               || 
         (header->y_size    != Y_SIZE)               ||
         (header->vspaces   > GIET_NB_VSPACE_MAX)    )
    {
        _printf("\n[BOOT ERROR] Illegal mapping signature: %x\n", header->signature );
        _exit();
    }

#if BOOT_DEBUG_MAPPING
unsigned int  line;
unsigned int* pointer = (unsigned int*)SEG_BOOT_MAPPING_BASE;
_printf("\n[BOOT] First block of mapping\n");
for ( line = 0 ; line < 8 ; line++ )
{
    _printf(" | %x | %x | %x | %x | %x | %x | %x | %x |\n",
            *(pointer + 0),
            *(pointer + 1),
            *(pointer + 2),
            *(pointer + 3),
            *(pointer + 4),
            *(pointer + 5),
            *(pointer + 6),
            *(pointer + 7) );

    pointer = pointer + 8;
}
#endif

#if BOOT_DEBUG_MAPPING
_printf("\n[BOOT] map.bin file checked at cycle %d\n", _get_proctime() );
#endif

    // close file "map.bin"
    _fat_close( fd_id );
    
} // end boot_mapping_init()


/////////////////////////////////////////////////////////////////////////////////////
// This function load all loadable segments for one .elf file, identified 
// by the "pathname" argument. Some loadable segments can be copied in several
// clusters: same virtual address but different physical addresses.  
// - It open the file.
// - It loads the complete file in the dedicated boot_elf_buffer.
// - It copies each loadable segments  at the virtual address defined in 
//   the .elf file, making several copies if the target vseg is not local.
// - It closes the file.
// This function is supposed to be executed by processor[0,0,0].
// Note: 
//   We must use physical addresses to reach the destination buffers that
//   can be located in remote clusters. We use either a _physical_memcpy(), 
//   or a _dma_physical_copy() if DMA is available.
//////////////////////////////////////////////////////////////////////////////////////
void load_one_elf_file( unsigned int is_kernel,     // kernel file if non zero
                        char*        pathname,
                        unsigned int vspace_id )    // to scan the proper vspace
{
    mapping_header_t  * header  = (mapping_header_t *)SEG_BOOT_MAPPING_BASE;
    mapping_vspace_t  * vspace  = _get_vspace_base(header);
    mapping_vseg_t    * vseg    = _get_vseg_base(header);

    unsigned int seg_id;

#if BOOT_DEBUG_ELF
_printf("\n[BOOT] Start searching file %s at cycle %d\n", 
        pathname, _get_proctime() );
#endif

    // open .elf file
    int fd_id = _fat_open( IOC_BOOT_MODE,
                           pathname,
                           0 );      // no creation
    if ( fd_id < 0 )
    {
        _printf("\n[BOOT ERROR] load_one_elf_file() : %s not found\n", pathname );
        _exit();
    }

    // check buffer size versus file size
    if ( fat.fd[fd_id].file_size > GIET_ELF_BUFFER_SIZE )
    {
        _printf("\n[BOOT ERROR] in load_one_elf_file() : %s / size = %x "
                "larger than GIET_ELF_BUFFER_SIZE = %x\n",
                pathname , fat.fd[fd_id].file_size , GIET_ELF_BUFFER_SIZE );
        _exit();
    }

    // compute number of sectors
    unsigned int nbytes   = fat.fd[fd_id].file_size;
    unsigned int nsectors = nbytes>>9;
    if( nbytes & 0x1FF) nsectors++;

    // load file in elf buffer
    if( _fat_read( IOC_BOOT_MODE, 
                   fd_id, 
                   boot_elf_buffer,
                   nsectors,
                   0 ) != nsectors )
    {
        _printf("\n[BOOT ERROR] load_one_elf_file() : unexpected EOF for file %s\n",
                pathname );
        _exit();
    }

    // Check ELF Magic Number in ELF header
    Elf32_Ehdr* elf_header_ptr = (Elf32_Ehdr*)boot_elf_buffer;

    if ( (elf_header_ptr->e_ident[EI_MAG0] != ELFMAG0) ||
         (elf_header_ptr->e_ident[EI_MAG1] != ELFMAG1) ||
         (elf_header_ptr->e_ident[EI_MAG2] != ELFMAG2) ||
         (elf_header_ptr->e_ident[EI_MAG3] != ELFMAG3) )
    {
        _printf("\n[BOOT ERROR] load_elf() : file %s does not use ELF format\n",
                pathname );
        _exit();
    }

    // get program header table pointer
    unsigned int pht_index = elf_header_ptr->e_phoff;
    if( pht_index == 0 )
    {
        _printf("\n[BOOT ERROR] load_one_elf_file() : file %s "
                "does not contain loadable segment\n", pathname );
        _exit();
    }
    Elf32_Phdr* elf_pht_ptr = (Elf32_Phdr*)(boot_elf_buffer + pht_index);

    // get number of segments
    unsigned int nsegments   = elf_header_ptr->e_phnum;

    // Loop on loadable segments in the .elf file
    for (seg_id = 0 ; seg_id < nsegments ; seg_id++)
    {
        if(elf_pht_ptr[seg_id].p_type == PT_LOAD)
        {
            // Get segment attributes
            unsigned int seg_vaddr  = elf_pht_ptr[seg_id].p_vaddr;
            unsigned int seg_offset = elf_pht_ptr[seg_id].p_offset;
            unsigned int seg_filesz = elf_pht_ptr[seg_id].p_filesz;
            unsigned int seg_memsz  = elf_pht_ptr[seg_id].p_memsz;

#if BOOT_DEBUG_ELF
_printf("\n[BOOT] Segment %d : vaddr = %x / size = %x\n",
        seg_id , seg_vaddr , seg_filesz );
#endif

            if( seg_memsz < seg_filesz )
            {
                _printf("\n[BOOT ERROR] load_one_elf_file() : segment at vaddr = %x"
                        " in file %s has memsize < filesize \n", seg_vaddr, pathname );
                _exit();
            }

            // fill empty space with 0 as required
            if( seg_memsz > seg_filesz )
            {
                unsigned int i; 
                for( i = seg_filesz ; i < seg_memsz ; i++ ) 
                   boot_elf_buffer[i+seg_offset] = 0;
            } 

            unsigned int src_vaddr = (unsigned int)boot_elf_buffer + seg_offset;

            // search all vsegs matching the virtual address
            unsigned int vseg_first;
            unsigned int vseg_last;
            unsigned int vseg_id;
            unsigned int found = 0;
            if ( is_kernel )
            {
                vseg_first = 0;
                vseg_last  = header->globals;
            }
            else
            {
                vseg_first = vspace[vspace_id].vseg_offset;
                vseg_last  = vseg_first + vspace[vspace_id].vsegs;
            }

            for ( vseg_id = vseg_first ; vseg_id < vseg_last ; vseg_id++ )
            {
                if ( seg_vaddr == vseg[vseg_id].vbase )  // matching 
                {
                    found = 1;

                    // get destination buffer physical address and size
                    paddr_t      seg_paddr  = vseg[vseg_id].pbase;
                    unsigned int seg_size   = vseg[vseg_id].length;
                    
#if BOOT_DEBUG_ELF
_printf("   loaded into vseg %s at paddr = %l / buffer size = %x\n",
        vseg[vseg_id].name , seg_paddr , seg_size );
#endif
                    // check vseg size
                    if ( seg_size < seg_filesz )
                    {
                        _printf("\n[BOOT ERROR] in load_one_elf_file() : vseg %s "
                                "is to small for loadable segment %x in file %s\n",
                                vseg[vseg_id].name , seg_vaddr , pathname );
                        _exit();
                    }

                    // copy the segment from boot buffer to destination buffer
                    // using DMA channel[0,0,0] if it is available.
                    if( NB_DMA_CHANNELS > 0 )
                    {
                        _dma_physical_copy( 0,                  // DMA in cluster[0,0]
                                            0,                  // DMA channel 0
                                            (paddr_t)seg_paddr, // destination paddr
                                            (paddr_t)src_vaddr, // source paddr
                                            seg_filesz );       // size
                    }
                    else
                    {
                        _physical_memcpy( (paddr_t)seg_paddr,   // destination paddr
                                          (paddr_t)src_vaddr,   // source paddr
                                          seg_filesz );         // size
                    }
                }
            }  // end for vsegs in vspace

            // check at least one matching vseg
            if ( found == 0 )
            {
                _printf("\n[BOOT ERROR] in load_one_elf_file() : vseg for loadable "
                        "segment %x in file %s not found "
                        "check consistency between the .py and .ld files\n",
                        seg_vaddr, pathname );
                _exit();
            }
        }
    }  // end for loadable segments

    // close .elf file
    _fat_close( fd_id );

    _printf("\n[BOOT] File %s loaded at cycle %d\n", 
            pathname , _get_proctime() );

} // end load_one_elf_file()


/////i////////////////////////////////////////////////////////////////////////////////
// This function uses the map.bin data structure to load the "kernel.elf" file
// as well as the various "application.elf" files into memory.
// - The "preloader.elf" file is not loaded, because it has been burned in the ROM.
// - The "boot.elf" file is not loaded, because it has been loaded by the preloader.
// This function scans all vsegs defined in the map.bin data structure to collect
// all .elf files pathnames, and calls the load_one_elf_file() for each .elf file.
// As the code can be replicated in several vsegs, the same code can be copied 
// in one or several clusters by the load_one_elf_file() function.
//////////////////////////////////////////////////////////////////////////////////////
void boot_elf_load()
{
    mapping_header_t* header = (mapping_header_t *)SEG_BOOT_MAPPING_BASE;
    mapping_vspace_t* vspace = _get_vspace_base( header );
    mapping_vseg_t*   vseg   = _get_vseg_base( header );

    unsigned int      vspace_id;
    unsigned int      vseg_id;
    unsigned int      found;

    // Scan all global vsegs to find the pathname to the kernel.elf file
    found = 0;
    for( vseg_id = 0 ; vseg_id < header->globals ; vseg_id++ )
    {
        if(vseg[vseg_id].type == VSEG_TYPE_ELF) 
        {   
            found = 1;
            break;
        }
    }

    // We need one kernel.elf file
    if (found == 0)
    {
        _printf("\n[BOOT ERROR] boot_elf_load() : kernel.elf file not found\n");
        _exit();
    }

    // Load the kernel 
    load_one_elf_file( 1,                           // kernel file
                       vseg[vseg_id].binpath,       // file pathname
                       0 );                         // vspace 0

    // loop on the vspaces, scanning all vsegs in the vspace,
    // to find the pathname of the .elf file associated to the vspace.
    for( vspace_id = 0 ; vspace_id < header->vspaces ; vspace_id++ )
    {
        // loop on the private vsegs
        unsigned int found = 0;
        for (vseg_id = vspace[vspace_id].vseg_offset;
             vseg_id < (vspace[vspace_id].vseg_offset + vspace[vspace_id].vsegs);
             vseg_id++) 
        {
            if(vseg[vseg_id].type == VSEG_TYPE_ELF) 
            {   
                found = 1;
                break;
            }
        }

        // We want one .elf file per vspace
        if (found == 0)
        {
            _printf("\n[BOOT ERROR] boot_elf_load() : "
                    ".elf file not found for vspace %s\n", vspace[vspace_id].name );
            _exit();
        }

        load_one_elf_file( 0,                          // not a kernel file
                           vseg[vseg_id].binpath,      // file pathname
                           vspace_id );                // vspace index

    }  // end for vspaces

} // end boot_elf_load()

////////////////////////////////////////////////////////////////////////////////
// This function intializes the external peripherals in cluster_io.
////////////////////////////////////////////////////////////////////////////////
void boot_peripherals_init() 
{
    mapping_header_t * header   = (mapping_header_t *)SEG_BOOT_MAPPING_BASE;
    mapping_cluster_t * cluster = _get_cluster_base(header);
    mapping_periph_t * periph   = _get_periph_base(header);
    mapping_irq_t * irq         = _get_irq_base(header);

    unsigned int periph_id;
    unsigned int channel_id;

#if BOOT_DEBUG_PERI
_printf("\n[BOOT] External peripherals initialisation in cluster[%d,%d]\n",
        X_IO , Y_IO );
#endif

    // get pointer on cluster_io
    mapping_cluster_t * cluster_io = &cluster[X_IO * Y_SIZE + Y_IO];

    // loop on peripherals
    for (periph_id = cluster_io->periph_offset;
         periph_id < cluster_io->periph_offset + cluster_io->periphs; 
         periph_id++) 
    {
        unsigned int type       = periph[periph_id].type;
        unsigned int subtype    = periph[periph_id].subtype;
        unsigned int channels   = periph[periph_id].channels;

        switch (type) 
        {
            case PERIPH_TYPE_IOC:    // vci_block_device component
            {
                if ( subtype == IOC_SUBTYPE_BDV )
                {
                    _bdv_init();
                }
                else if ( subtype == IOC_SUBTYPE_HBA ) 
                {
                    for (channel_id = 0; channel_id < channels; channel_id++) 
                        _hba_init( channel_id );
                }
                else if ( subtype == IOC_SUBTYPE_SPI ) 
                {
                    //TODO
                }
                break;
            }
            case PERIPH_TYPE_TTY:    // vci_multi_tty component
            {
                for (channel_id = 0; channel_id < channels; channel_id++) 
                {
                    _tty_init( channel_id );
                }
                break;
            }
            case PERIPH_TYPE_NIC:    // vci_multi_nic component
            {
                _nic_global_init( 1,      // broadcast accepted
                                  1,      // bypass activated
                                  0,      // tdm non activated
                                  0 );    // tdm period 
                break;
            }
            case PERIPH_TYPE_IOB:    // vci_io_bridge component
            {
                if (GIET_USE_IOMMU) 
                {
                    // TODO
                    // get the iommu page table physical address
                    // set IOMMU page table address
                    // pseg_base[IOB_IOMMU_PTPR] = ptab_pbase;    
                    // activate IOMMU
                    // pseg_base[IOB_IOMMU_ACTIVE] = 1;        
                }
                break;
            }
            case PERIPH_TYPE_PIC:    // vci_iopic component
            {
                // scan all IRQs defined in mapping for PIC component,
                // and initialises addresses for WTI IRQs
                for ( channel_id = periph[periph_id].irq_offset ;
                      channel_id < periph[periph_id].irq_offset + 
                      periph[periph_id].irqs ; channel_id++ )
                {
                    unsigned int hwi_id     = irq[channel_id].srcid;   // HWI in PIC
                    unsigned int wti_id     = irq[channel_id].dest_id; // WTI in XCU
                    unsigned int cluster_xy = irq[channel_id].dest_xy; // XCU coordinates
                    unsigned int vaddr;

                    _xcu_get_wti_address( wti_id, &vaddr );
                    _pic_init( hwi_id, vaddr, cluster_xy ); 

#if BOOT_DEBUG_PERI
_printf("[BOOT] PIC : hwi_index = %d => wti_index = %d for XCU[%d,%d]\n",
        hwi_id , wti_id , cluster_xy >> Y_WIDTH , cluster_xy & ((1<<Y_WIDTH)-1) ); 
#endif
                }
                break;
            }
        }  // end switch periph type
    } // end loop on peripherals
} // end boot_peripherals_init()

///////////////////////////////////////////////////////////////////////////////////////
// This function is executed in parallel by all processors[x][y][0].
// It initialises the physical memory allocator in each cluster containing a RAM pseg.
///////////////////////////////////////////////////////////////////////////////////////
void boot_pmem_init( unsigned int cx,
                     unsigned int cy ) 
{
    mapping_header_t*  header     = (mapping_header_t *)SEG_BOOT_MAPPING_BASE;
    mapping_cluster_t* cluster    = _get_cluster_base(header);
    mapping_pseg_t*    pseg       = _get_pseg_base(header);

    unsigned int pseg_id;
    unsigned int procid     = _get_procid();
    unsigned int lpid       = procid & ((1<<P_WIDTH)-1);

    if( lpid )
    {
        _printf("\n[BOOT ERROR] boot_pmem_init() : "
        "P[%d][%d][%d] should not execute it\n", cx, cy, lpid );
        _exit();
    }   

    // scan the psegs in local cluster to find  pseg of type RAM
    unsigned int found      = 0;
    unsigned int cluster_id = cx * Y_SIZE + cy;
    unsigned int pseg_min   = cluster[cluster_id].pseg_offset;
    unsigned int pseg_max   = pseg_min + cluster[cluster_id].psegs;
    for ( pseg_id = pseg_min ; pseg_id < pseg_max ; pseg_id++ )
    {
        if ( pseg[pseg_id].type == PSEG_TYPE_RAM )
        {
            unsigned int base = (unsigned int)pseg[pseg_id].base;
            unsigned int size = (unsigned int)pseg[pseg_id].length;
            _pmem_alloc_init( cx, cy, base, size );
            found = 1;

#if BOOT_DEBUG_PT
_printf("\n[BOOT] pmem allocator initialised in cluster[%d][%d]"
        " : base = %x / size = %x\n", cx , cy , base , size );
#endif
            break;
        }
    }

    if ( found == 0 )
    {
        _printf("\n[BOOT ERROR] boot_pmem_init() : no RAM in cluster[%d][%d]\n",
              cx , cy );
        _exit();
    }   
} // end boot_pmem_init()
 
/////////////////////////////////////////////////////////////////////////
// This function is the entry point of the boot code for all processors.
/////////////////////////////////////////////////////////////////////////
void boot_init() 
{

    unsigned int       gpid       = _get_procid();
    unsigned int       cx         = gpid >> (Y_WIDTH + P_WIDTH);
    unsigned int       cy         = (gpid >> P_WIDTH) & ((1<<Y_WIDTH)-1);
    unsigned int       lpid       = gpid & ((1 << P_WIDTH) -1);

    // Phase ONE : only P[0][0][0] execute it
    if ( gpid == 0 )    
    {
        unsigned int cid;  // index for loops 

        // initialises the TTY0 spin lock
        _spin_lock_init( &_tty0_spin_lock );

        _printf("\n[BOOT] P[0,0,0] starts at cycle %d\n", _get_proctime() );

        // initialises the FAT
        _fat_init( IOC_BOOT_MODE );

        _printf("\n[BOOT] FAT initialised at cycle %d\n", _get_proctime() );

        // Load the map.bin file into memory 
        boot_mapping_init();

        mapping_header_t*  header     = (mapping_header_t *)SEG_BOOT_MAPPING_BASE;
        mapping_cluster_t* cluster    = _get_cluster_base(header);

        _printf("\n[BOOT] Mapping %s loaded at cycle %d\n",
                header->name , _get_proctime() );

        // initialises the barrier for all clusters containing processors
        unsigned int nclusters = 0;
        for ( cid = 0 ; cid < X_SIZE*Y_SIZE ; cid++ )
        {
            if ( cluster[cid].procs ) nclusters++ ;
        } 

        _simple_barrier_init( &_barrier_all_clusters , nclusters );

        // wake up all processors P[x][y][0]
        for ( cid = 1 ; cid < X_SIZE*Y_SIZE ; cid++ ) 
        {
            unsigned int x          = cluster[cid].x;
            unsigned int y          = cluster[cid].y;
            unsigned int cluster_xy = (x << Y_WIDTH) + y;

            if ( cluster[cid].procs ) 
            {
                unsigned long long paddr = (((unsigned long long)cluster_xy)<<32) +
                                           SEG_XCU_BASE + XCU_REG( XCU_WTI_REG , 0 );

                _physical_write( paddr , (unsigned int)boot_entry );
            }
        }

        _printf("\n[BOOT] Processors P[x,y,0] start at cycle %d\n", _get_proctime() );
    }

    // Phase TWO : All processors P[x][y][0] execute it in parallel
    if( lpid == 0 )
    {
        // Initializes physical memory allocator in cluster[cx][cy]
        boot_pmem_init( cx , cy );

        // Build page table in cluster[cx][cy]
        boot_ptab_init( cx , cy );

        //////////////////////////////////////////////
        _simple_barrier_wait( &_barrier_all_clusters );
        //////////////////////////////////////////////

        // P[0][0][0] complete page tables with vsegs 
        // mapped in clusters without processors 
        if ( gpid == 0 )   
        {
            // complete page tables initialisation
            boot_ptab_extend();

            _printf("\n[BOOT] Physical memory allocators and page tables"
                    " initialized at cycle %d\n", _get_proctime() );
        }

        //////////////////////////////////////////////
        _simple_barrier_wait( &_barrier_all_clusters );
        //////////////////////////////////////////////

        // All processors P[x,y,0] activate MMU (using local PTAB) 
        _set_mmu_ptpr( (unsigned int)(_ptabs_paddr[0][cx][cy]>>13) );
        _set_mmu_mode( 0xF );
        
        // Each processor P[x,y,0] initialises all schedulers in cluster[x,y]
        boot_scheduler_init( cx , cy );

        // Each processor P[x][y][0] initialises its CP0_SCHED register
        _set_sched( (unsigned int)_schedulers[cx][cy][0] );

        //////////////////////////////////////////////
        _simple_barrier_wait( &_barrier_all_clusters );
        //////////////////////////////////////////////
        
        // Processor P[0,0,0] completes schedulers with PIC-WTI 
        // initialises external peripherals and load .elf files.
        if ( gpid == 0 ) 
        {
            // complete schedulers initialisation
            boot_pic_wti_init();

            _printf("\n[BOOT] Schedulers initialised at cycle %d\n", _get_proctime() );

            // initialize non replicated peripherals
            boot_peripherals_init();

            _printf("\n[BOOT] Peripherals initialised at cycle %d\n", _get_proctime() );

            // Loading all .elf files
            boot_elf_load();
        }

        //////////////////////////////////////////////
        _simple_barrier_wait( &_barrier_all_clusters );
        //////////////////////////////////////////////
        
        // each processor P[x][y][0] wake up other processors in same cluster
        mapping_header_t*  header     = (mapping_header_t *)SEG_BOOT_MAPPING_BASE;
        mapping_cluster_t* cluster    = _get_cluster_base(header);
        unsigned int       cluster_xy = (cx << Y_WIDTH) + cy;
        unsigned int       cluster_id = (cx * Y_SIZE) + cy;
        unsigned int p;
        for ( p = 1 ; p < cluster[cluster_id].procs ; p++ )
        {
            _xcu_send_wti( cluster_xy , p , (unsigned int)boot_entry );
        }

        if ( gpid == 0 )    // only P[0][0][0] makes display
        _printf("\n[BOOT] All processors start at cycle %d\n", _get_proctime() );
    }

    // Other processors than P[x][y][0] activate MMU (using local PTAB)
    if ( lpid != 0 )
    {
        _set_mmu_ptpr( (unsigned int)(_ptabs_paddr[0][cx][cy]>>13) );
        _set_mmu_mode( 0xF );
    }

    // All processors set CP0_SCHED register 
    _set_sched( (unsigned int)_schedulers[cx][cy][lpid] );

    // All processors reset BEV bit in SR to use GIET_VM exception handler 
    _set_sr( 0 );

    // All processors jump to kernel_init 
    unsigned int kernel_entry = (unsigned int)&kernel_init_vbase;
    asm volatile( "jr   %0" ::"r"(kernel_entry) );

} // end boot_init()


// Local Variables:
// tab-width: 4
// c-basic-offset: 4
// c-file-offsets:((innamespace . 0)(inline-open . 0))
// indent-tabs-mode: nil
// End:
// vim: filetype=c:expandtab:shiftwidth=4:tabstop=4:softtabstop=4

