///////////////////////////////////////////////////////////////////////////////////
// File     : mwr_driver.c
// Date     : 27/02/2015
// Author   : alain greiner
// Copyright (c) UPMC-LIP6
///////////////////////////////////////////////////////////////////////////////////

#include <giet_config.h>
#include <hard_config.h>
#include <mapping_info.h>
#include <mwr_driver.h>
#include <utils.h>
#include <kernel_locks.h>
#include <tty0.h>
#include <io.h>

#if !defined(X_SIZE) 
# error: You must define X_SIZE in the hard_config.h file
#endif

#if !defined(Y_SIZE) 
# error: You must define X_SIZE in the hard_config.h file
#endif

#if !defined(X_WIDTH) 
# error: You must define X_WIDTH in the hard_config.h file
#endif

#if !defined(Y_WIDTH) 
# error: You must define X_WIDTH in the hard_config.h file
#endif

#if !defined(SEG_MWR_BASE)
# error: You must define SEG_MWR_BASE in the hard_config.h file
#endif

#if !defined(PERI_CLUSTER_INCREMENT) 
# error: You must define PERI_CLUSTER_INCREMENT in the hard_config.h file
#endif


/////////////////////////////////////////////////////////////////////////////
//      Global variables
/////////////////////////////////////////////////////////////////////////////

__attribute__((section(".kdata")))
simple_lock_t  _coproc_lock[X_SIZE*Y_SIZE];

__attribute__((section(".kdata")))
unsigned int   _coproc_done[X_SIZE*Y_SIZE];

/////////////////////////////////////////////////////////////////////////////
// This function returns the value contained in a private register 
// of the coprocessor contained in cluster "cluster_xy".
/////////////////////////////////////////////////////////////////////////////
unsigned int _mwr_get_coproc_register( unsigned int cluster_xy, // cluster
                                       unsigned int index )     // register 
{
    unsigned int vaddr =
        SEG_MWR_BASE + 
        (cluster_xy * PERI_CLUSTER_INCREMENT) + 
        (index << 2);

    return ioread32( (void*)vaddr );
}

/////////////////////////////////////////////////////////////////////////////
// This function sets a new value in a private register 
// of the coprocessor contained in cluster "clustenr_xy".
/////////////////////////////////////////////////////////////////////////////
void _mwr_set_coproc_register( unsigned int cluster_xy,   // cluster index
                               unsigned int index,        // register index
                               unsigned int value )       // writte value
{
    unsigned int vaddr =
        SEG_MWR_BASE + 
        (cluster_xy * PERI_CLUSTER_INCREMENT) +
        (index << 2);
        
    iowrite32( (void*)vaddr, value );
}

/////////////////////////////////////////////////////////////////////////////
// This function returns the value contained in a channel register 
// defined by the "index" and "channel" arguments, in the MWMR_DMA component
// contained in cluster "cluster_xy".
/////////////////////////////////////////////////////////////////////////////
unsigned int _mwr_get_channel_register( unsigned int cluster_xy, // cluster 
                                        unsigned int channel,    // channel 
                                        unsigned int index )     // register
{
    unsigned int vaddr =
        SEG_MWR_BASE + 
        (cluster_xy * PERI_CLUSTER_INCREMENT) + 
        ((channel + 1) * (CHANNEL_SPAN << 2)) +
        (index << 2);

    return ioread32( (void*)vaddr );
}

/////////////////////////////////////////////////////////////////////////////
// This function sets a new value in a channel register
// defined by the "index" and "channel") arguments, in the MWMR_DMA component
// contained in cluster "cluster_xy".
/////////////////////////////////////////////////////////////////////////////
void _mwr_set_channel_register( unsigned int cluster_xy,  // cluster index
                                unsigned int channel,     // channel index
                                unsigned int index,       // register index
                                unsigned int value )      // written value
{
    unsigned int vaddr =
        SEG_MWR_BASE + 
        (cluster_xy * PERI_CLUSTER_INCREMENT) +
        ((channel + 1) * (CHANNEL_SPAN << 2)) +
        (index << 2);
        
    iowrite32( (void*)vaddr, value );
}

///////////////////////////////////////////////////////
void _mwr_isr( unsigned int irq_type,  // should be HWI 
               unsigned int irq_id,    // index returned by XCU
               unsigned int channel )  // MWMR_DMA channel
{
    unsigned int gpid       = _get_procid();
    unsigned int cluster_xy = gpid >> P_WIDTH;
    unsigned int x          = cluster_xy >> Y_WIDTH;
    unsigned int y          = cluster_xy & ((1<<Y_WIDTH)-1);

    // acknowledge IRQ
    _mwr_set_channel_register( cluster_xy, channel, CHANNEL_RUNNING, 0 );

    // compute coproc_id from cluster coordinates
    unsigned int coproc_id = x * Y_SIZE + y;

    // set synchronisation variable
    _coproc_done[coproc_id] = 1;
}



// Local Variables:
// tab-width: 4
// c-basic-offset: 4
// c-file-offsets:((innamespace . 0)(inline-open . 0))
// indent-tabs-mode: nil
// End:
// vim: filetype=c:expandtab:shiftwidth=4:tabstop=4:softtabstop=4

