//////////////////////////////////////////////////////////////////////////////////
// File     : hba_driver.c
// Date     : 23/11/2013
// Author   : alain greiner
// Copyright (c) UPMC-LIP6
///////////////////////////////////////////////////////////////////////////////////
// Implementation notes:
// All accesses to HBA registers are done by the two
// _hba_set_register() and _hba_get_register() low-level functions,
// that are handling virtual / physical extended addressing.
///////////////////////////////////////////////////////////////////////////////////

#include <giet_config.h>
#include <hard_config.h>
#include <hba_driver.h>
#include <xcu_driver.h>
#include <kernel_locks.h>
#include <utils.h>
#include <tty0.h>
#include <ctx_handler.h>
#include <irq_handler.h>
#include <vmem.h>

///////////////////////////////////////////////////////////////////////////////////
//               Global variables
///////////////////////////////////////////////////////////////////////////////////

// global index ot the task, for each entry in the command list
__attribute__((section(".kdata")))
unsigned int _hba_gtid[32];

// status of the command, for each entry in the command list
__attribute__((section(".kdata")))
unsigned int _hba_status[32];

// command list : up to 32 commands
__attribute__((section(".kdata")))
hba_cmd_desc_t  _hba_cmd_list[32] __attribute__((aligned(0x1000)));   

// command tables array : one command table per entry in command list
__attribute__((section(".kdata")))
hba_cmd_table_t _hba_cmd_table[32] __attribute__((aligned(0x1000))); 

// command list write index : next slot to register a command 
__attribute__((section(".kdata")))
unsigned int     _hba_cmd_ptw;

// command list read index : next slot to poll a completed command 
__attribute__((section(".kdata")))
unsigned int     _hba_cmd_ptr;

//////////////////////////////////////////////////////////////////////////////
// This low level function returns the value of register (index)
//////////////////////////////////////////////////////////////////////////////
unsigned int _hba_get_register( unsigned int index )
{
    unsigned int* vaddr = (unsigned int*)SEG_IOC_BASE + index;
    return _io_extended_read( vaddr );
}

//////////////////////////////////////////////////////////////////////////////
// This low level function set a new value in register (index)  
//////////////////////////////////////////////////////////////////////////////
void _hba_set_register( unsigned int index,
                        unsigned int value )
{
    unsigned int* vaddr = (unsigned int*)SEG_IOC_BASE + index;
    _io_extended_write( vaddr, value );
}

///////////////////////////////////////////////////////////////////////////////
//      Extern functions
///////////////////////////////////////////////////////////////////////////////

///////////////////////////////////////////////////////////////////////////////
// This function register a command in both the command list
// and the command table, and updates the HBA_PXCI register.
// return 0 if success, -1 if error
///////////////////////////////////////////////////////////////////////////////
unsigned int _hba_access( unsigned int       use_irq,
                          unsigned int       to_mem,
                          unsigned int       lba,  
                          unsigned long long buf_paddr,
                          unsigned int       count )   
{
    unsigned int procid  = _get_procid();
    unsigned int x       = procid >> (Y_WIDTH + P_WIDTH);
    unsigned int y       = (procid >> P_WIDTH) & ((1<<Y_WIDTH) - 1);
    unsigned int p       = procid & ((1<<P_WIDTH)-1);

#if GIET_DEBUG_IOC_DRIVER
_printf("\n[HBA DEBUG] P[%d,%d,%d] enters _hba_access at cycle %d\n"
        "  use_irq = %d / to_mem = %d / lba = %x / paddr = %l / count = %d\n",
        x , y , p , _get_proctime() , use_irq , to_mem , lba , buf_paddr, count );
#endif

    unsigned int       pxci;           // HBA_PXCI register value
    unsigned int       ptw;            // command list write pointer
    unsigned int       pxis;           // HBA_PXIS register value
    hba_cmd_desc_t*    cmd_desc;       // command descriptor pointer   
    hba_cmd_table_t*   cmd_table;      // command table pointer

    // check buffer alignment
    if( buf_paddr & 0x1FF )
    {
        _printf("\n[HBA ERROR] in _hba_access() : buffer not block aligned\n");
        return -1;
    }

    // get pointer on the next possible entry in command list 
    ptw = _atomic_increment( &_hba_cmd_ptw , 1 );

    // poll PXCI register until pointed entry empty
    do
    {
        // get PXCI register
        pxci = _hba_get_register( HBA_PXCI );
    } 
    while ( pxci & (1<<ptw) );
    
    // compute pointers on command descriptor and command table    
    cmd_desc  = &_hba_cmd_list[ptw];
    cmd_table = &_hba_cmd_table[ptw];

    // set  buffer descriptor in command table 
    cmd_table->entry[0].dba  = (unsigned int)(buf_paddr);
    cmd_table->entry[0].dbau = (unsigned int)(buf_paddr >> 32);
    cmd_table->entry[0].dbc  = count * 512;

    // initialize command table header
    cmd_table->header.lba0 = (char)lba;
    cmd_table->header.lba1 = (char)(lba>>8);
    cmd_table->header.lba2 = (char)(lba>>16);
    cmd_table->header.lba3 = (char)(lba>>24);
    cmd_table->header.lba4 = 0;
    cmd_table->header.lba5 = 0;

    // initialise command descriptor
    cmd_desc->prdtl[0] = 1;
    cmd_desc->prdtl[1] = 0;
    if( to_mem ) cmd_desc->flag[0] = 0x00;
    else         cmd_desc->flag[0] = 0x40;     
   
    // set command in PXCI[ptw]
    _hba_set_register( HBA_PXCI, pxci + (1<<ptw) );


    /////////////////////////////////////////////////////////////////////
    // In synchronous mode, we poll the PXCI register until completion
    /////////////////////////////////////////////////////////////////////
    if ( use_irq == 0 ) 
    {

#if GIET_DEBUG_IOC_DRIVER
_printf("\n[HBA DEBUG] _hba_access() : P[%d,%d,%d] launch transfer"
        " in polling mode at cycle %d\n",
        x , y , p , _get_proctime() );
#endif
        // disable IRQs in PXIE register
        _hba_set_register( HBA_PXIE , 0 );

        // poll PXCI[ptw] until command completed by HBA
        do
        {
            pxci = _hba_get_register( HBA_PXCI ) & (1<<ptw);

#if GIET_DEBUG_IOC_DRIVER
_printf("\n[HBA DEBUG] _hba_access() : P[%d,%d,%d] wait on HBA_STATUS ...\n",
        x , y , p );
#endif
        }
        while( pxci & (1<<ptw) ); 
             
        // get PXIS register
        pxis = _hba_get_register( HBA_PXIS );

        // reset PXIS register
        _hba_set_register( HBA_PXIS , 0 );
    }

    /////////////////////////////////////////////////////////////////
    // in descheduling mode, we deschedule the task
    // and use an interrupt to reschedule the task.
    // We need a critical section, because we must reset the RUN bit
	// before to launch the transfer, and we don't want to be 
    // descheduled between these two operations. 
    /////////////////////////////////////////////////////////////////
    else
    {

#if GIET_DEBUG_IOC_DRIVER
_printf("\n[HBA DEBUG] _hba_access() : P[%d,%d,%d] launch transfer"
        " in descheduling mode at cycle %d\n",
        x , y , p , _get_proctime() );
#endif
        unsigned int save_sr;
        unsigned int ltid = _get_current_task_id();

        // activates HBA interrupts 
        _hba_set_register( HBA_PXIE , 0x00000001 ); 

        // set _hba_gtid[ptw] 
        _hba_gtid[ptw] = (procid<<16) + ltid;

        // enters critical section
        _it_disable( &save_sr ); 

        // reset runnable 
        _set_task_slot( x, y, p, ltid, CTX_RUN_ID, 0 );  

        // deschedule task
        _ctx_switch();                      

#if GIET_DEBUG_IOC_DRIVER
_printf("\n[HBA DEBUG] _hba_access() : P[%d,%d,%d] resume execution at cycle %d\n",
        x , y , p , _get_proctime() );
#endif

        // restore SR
        _it_restore( &save_sr );

        // get command status
        pxis = _hba_status[ptw];
    }    

#if GIET_DEBUG_IOC_DRIVER
_printf("\n[HBA DEBUG] _hba_access() : P[%d,%d,%d] exit at cycle %d\n",
        x , y , p , _get_proctime() );
#endif

    if ( pxis & 0x40000000 ) return pxis;
    else                     return 0;

} // end _hba_access()


////////////////////////
unsigned int _hba_init()
{
    unsigned int       flags;
    unsigned int       vaddr;
    unsigned long long paddr;
    unsigned int       c;      
    unsigned int       pxclb;
    unsigned int       pxclbu;

    // command list pointers
    _hba_cmd_ptw = 0;
    _hba_cmd_ptr = 0;

    // Command list physical addresse
    vaddr  = (unsigned int)(_hba_cmd_list);
    paddr  = _v2p_translate( vaddr , &flags );
    pxclb  = (unsigned int)paddr;
    pxclbu = (unsigned int)(paddr>>32);

    // Command tables physical addresses
    for( c=0 ; c<32 ; c++ )
    {
        // compute command table physical address 
        // for one entry in the command list
        vaddr = (unsigned int)(&_hba_cmd_table[c]);
        paddr = _v2p_translate( vaddr , &flags );

        // initialise the corresponding command descriptor
        _hba_cmd_list[c].ctba  = (unsigned int)paddr;
        _hba_cmd_list[c].ctbau = (unsigned int)(paddr>>32);
    }

    // set HBA registers 
    _hba_set_register( HBA_PXCLB , pxclb  );
    _hba_set_register( HBA_PXCLBU, pxclbu );
    _hba_set_register( HBA_PXIE  , 0      );
    _hba_set_register( HBA_PXIS  , 0      );
    _hba_set_register( HBA_PXCI  , 0      );
    _hba_set_register( HBA_PXCMD , 1      );

    return 0;
}


/////////////////////////////////////
void _hba_isr( unsigned int irq_type,   // HWI / WTI
               unsigned int irq_id,     // index returned by ICU
               unsigned int channel )   // unused 
{
    // get HBA_PXCI containing commands status
    unsigned int pxci = _hba_get_register( HBA_PXCI );

    // scan active commands from (_hba_cmd_ptr) to (_hba_cmd_ptw-1) 
    unsigned int c;
    for ( c = _hba_cmd_ptr ; 
          c != _hba_cmd_ptw ; 
          c = (c + 1) % 32 )
    {
        if ( (pxci & (1<<c)) == 0 )    // command completed
        {
            // increment read pointer;
            _hba_cmd_ptr++;

            // save PXIS register
            _hba_status[c] = _hba_get_register( HBA_PXIS );

            // reset PXIS register
            _hba_set_register( HBA_PXIS , 0 );
 
            // identify waiting task 
            unsigned int remote_procid  = _hba_gtid[c]>>16;
            unsigned int ltid           = _hba_gtid[c] & 0xFFFF;
            unsigned int remote_cluster = remote_procid >> P_WIDTH;
            unsigned int remote_x       = remote_cluster >> Y_WIDTH;
            unsigned int remote_y       = remote_cluster & ((1<<Y_WIDTH)-1);
            unsigned int remote_p       = remote_procid & ((1<<P_WIDTH)-1);
 
            // re-activates waiting task
            _set_task_slot( remote_x,
                            remote_y,
                            remote_p,
                            ltid,
                            CTX_RUN_ID,
                            1 );

            // send a WAKUP WTI to processor running the waiting task 
            _xcu_send_wti( remote_cluster , 
                           remote_p , 
                           0 );          // don't force context switch

#if GIET_DEBUG_IOC_DRIVER  
unsigned int procid  = _get_procid();
unsigned int x       = procid >> (Y_WIDTH + P_WIDTH);
unsigned int y       = (procid >> P_WIDTH) & ((1<<Y_WIDTH)-1);
unsigned int p       = procid & ((1<<P_WIDTH)-1);
_printf("\n[HBA DEBUG] Processor[%d,%d,%d] executes _hba_isr() :\n"
        "  resume task %d running on P[%d,%d,%d] / status = %x at cyle %d\n",
        x , y , p , 
        ltid , remote_x , remote_y , remote_p , _hba_status[c] , _get_proctime() );
#endif
        }
        else                         // command non completed
        {
            break;
        }
    }
} // end _hba_isr()

// Local Variables:
// tab-width: 4
// c-basic-offset: 4
// c-file-offsets:((innamespace . 0)(inline-open . 0))
// indent-tabs-mode: nil
// End:
// vim: filetype=c:expandtab:shiftwidth=4:tabstop=4:softtabstop=4

