////////////////////////////////////////////////////////////////////////////////
// File     : fat32.h 
// Date     : 01/09/2013
// Author   : Marco Jankovic / Alain Greiner
// Copyright (c) UPMC-LIP6
////////////////////////////////////////////////////////////////////////////////

#ifndef _FAT32_H
#define _FAT32_H

#include "giet_config.h"
#include "kernel_locks.h"

/*************** Partition Boot Sector Format **********************************/
//                                     offset |  length
#define BS_JMPBOOT                          0 ,  3
#define BS_OEMNAME                          3 ,  8
#define BPB_BYTSPERSEC                     11 ,  2
#define BPB_SECPERCLUS                     13 ,  1
#define BPB_RSVDSECCNT                     14 ,  2
#define BPB_NUMFATS                        16 ,  1
#define BPB_ROOTENTCNT                     17 ,  2
#define BPB_TOTSEC16                       19 ,  2
#define BPB_MEDIA                          21 ,  1
#define BPB_FATSZ16                        22 ,  2
#define BPB_SECPERTRK                      24 ,  2
#define BPB_NUMHEADS                       26 ,  2
#define BPB_HIDDSEC                        28 ,  4
#define BPB_TOTSEC32                       32 ,  4
#define BPB_PARTITION_TABLE               446 , 64 

// FAT 32
#define BPB_FAT32_FATSZ32                  36 ,  4
#define BPB_FAT32_EXTFLAGS                 40 ,  2
#define BPB_FAT32_FSVER                    42 ,  2
#define BPB_FAT32_ROOTCLUS                 44 ,  4
#define BPB_FAT32_FSINFO                   48 ,  2
#define BPB_FAT32_BKBOOTSEC                50 ,  2
#define BS_FAT32_DRVNUM                    64 ,  1
#define BS_FAT32_BOOTSIG                   66 ,  1
#define BS_FAT32_VOLID                     67 ,  4
#define BS_FAT32_VOLLAB                    71 , 11
#define BS_FAT32_FILSYSTYPE                82 ,  8

// Partitions
#define FIRST_PARTITION_ACTIVE            446 ,  8
#define FIRST_PARTITION_BEGIN_LBA         454 ,  4
#define FIRST_PARTITION_SIZE              458 ,  4 
#define SECOND_PARTITION_ACTIVE           462 ,  8
#define SECOND_PARTITION_BEGIN_LBA        470 ,  4
#define SECOND_PARTITION_SIZE             474 ,  4
#define THIRD_PARTITION_ACTIVE            478 ,  8
#define THIRD_PARTITION_BEGIN_LBA         486 ,  4
#define THIRD_PARTITION_SIZE              490 ,  4
#define FOURTH_PARTITION_ACTIVE           494 ,  8
#define FOURTH_PARTITION_BEGIN_LBA        502 ,  4
#define FOURTH_PARTITION_SIZE             506 ,  4    
/*******************************************************************************/

#define MBR_SIGNATURE_POSITION            510 , 2
#define MBR_SIGNATURE_VALUE               0xAA55  

/************** FAT_FS_INFO SECTOR  ********************************************/
#define FS_SIGNATURE_VALUE_1              0x52526141
#define FS_SIGNATURE_VALUE_2              0x72724161
#define FS_SIGNATURE_VALUE_3              0x000055AA  
#define FS_SIGNATURE_POSITION_1           0   , 4  
#define FS_SIGNATURE_POSITION_2           484 , 4
#define FS_SIGNATURE_POSITION_3           508 , 4  
#define FS_FREE_CLUSTER                   488 , 4
#define FS_FREE_CLUSTER_HINT              492 , 4
/*******************************************************************************/

#define DIR_ENTRY_SIZE          32
                   
/******* Directory Entry Structure (32 bytes) **********************************/
//                          offset | length
#define DIR_NAME                 0 , 11   // dir_entry name
#define DIR_ATTR                11 ,  1   // attributes
#define DIR_NTRES               12 ,  1   // reserved for the OS        
#define DIR_CRT_TIMES_TENTH     13 ,  1 
#define DIR_FST_CLUS_HI         20 ,  2   // cluster index 16 MSB bits
#define DIR_WRT_TIME            22 ,  2   // time of last write
#define DIR_WRT_DATE            24 ,  2   // date of last write
#define DIR_FST_CLUS_LO         26 ,  2   // cluster index 16 LSB bit
#define DIR_FILE_SIZE           28 ,  4   // file size (up to 4 giga bytes)
/*******************************************************************************/

/******* LFN Directory Entry Structure  (32 bytes) *****************************/
//                          offset | length
#define LDIR_ORD                 0 ,  1   // Sequence number (from 0x01 to 0x0f)    
#define LDIR_NAME_1              1 , 10   // name broken into 3 parts 
#define LDIR_ATTR               11 ,  1   // attributes (must be 0x0F) 
#define LDIR_TYPE               12 ,  1   // directory type (must be 0x00)
#define LDIR_CHKSUM             13 ,  1   // checksum of name in short dir  
#define LDIR_NAME_2             14 , 12 
#define LDIR_RSVD               26 ,  2   // artifact of previous fat (must be 0)
#define LDIR_NAME_3             28 ,  4   
/*******************************************************************************/

/***********************  DIR_ATTR values  (attributes) ************************/
#define ATTR_READ_ONLY          0x01
#define ATTR_HIDDEN             0x02
#define ATTR_SYSTEM             0x04
#define ATTR_VOLUME_ID          0x08
#define ATTR_DIRECTORY          0x10
#define ATTR_ARCHIVE            0x20
#define ATTR_LONG_NAME_MASK     0x0f      // READ_ONLY|HIDDEN|SYSTEM|VOLUME_ID
/*******************************************************************************/

/********************* DIR_ORD special values **********************************/
#define FREE_ENTRY              0xE5     // this entry is free in the directory
#define NO_MORE_ENTRY           0x00     // no more entry in the directory
/*******************************************************************************/

/******************** CLuster Index Special Values *****************************/
#define FREE_CLUSTER            0x00000000
#define RESERVED_CLUSTER        0x00000001
#define BAD_CLUSTER             0x0FFFFFF7
#define END_OF_CHAIN_CLUSTER    0x0ffffff8
/*******************************************************************************/

#define FAT_INITIALISED         0xBABEF00D

/************ This struct defines a file descriptor ****************************/
typedef struct file_descriptor_s
{
   unsigned int  used;                      // descriptor contains an open file
   unsigned int  first_cluster;             // first cluster index on disk
   unsigned int  file_size;                 // number of bytes    
   unsigned int  lba_dir_entry;             // lba of dir_entry for an open file 
   char          name[244];                 // pathname
}  file_desc_t;
/*******************************************************************************/

/************ This struct describes a FAT32 disk *******************************/
typedef struct fat32_fs_s
{
    char            fat_cache[512];          // FAT cache: 1 sector  
    spin_lock_t     fat_lock;                // lock protecting exclusive access
    file_desc_t     fd[GIET_OPEN_FILES_MAX]; // file descriptors array
    unsigned int    initialised;             // 0xBABEF00D when FAT initialised
    unsigned int    sector_size;             // number of bytes (power of 2)
    unsigned int    sectors_per_cluster;     // power of 2 (must be greater than 4)
    unsigned int    cluster_size;            // sector_size * sector_per_cluster 
    unsigned int    fat_sectors;             // number of sectors for 1 FAT copy
    unsigned int    fat_lba;                 // lba of first FAT sector
    unsigned int    data_lba;                // lba of first data sector  
    unsigned int    cache_lba;               // lba of sector loaded in fat_cache
    unsigned int    last_cluster_allocated;  // Last known cluster allocated
    unsigned int    number_free_cluster;     // number of free clusters
    unsigned int    fs_info_lba;             // lba of fs_info
} fat32_fs_t;
/*******************************************************************************/

/*********************** Extern Functions  *************************************/

extern int _fat_init( unsigned int use_irq );       // use IRQ if possible

extern int _fat_open(  unsigned int use_irq,        // use IRQ if possible
                       char*  pathname,             // file pathname from root
                       unsigned int create );       // create new file if non zero

extern int _fat_read(  unsigned int use_irq,        // use IRQ if possible
                       unsigned int fd_id,          // file descriptor index 
                       void*        buffer,         // destination buffer
                       unsigned int count,  	    // number of sectors to read
                       unsigned int offset );       // offset sector in file

extern int _fat_write( unsigned int use_irq,        // use IRQ if possible
                       unsigned int fd_id,          // file descriptor index
                       void*        buffer,		    // source buffer
                       unsigned int count,          // number of sectors to write
                       unsigned int offset );       // offset sector in file

extern int _fat_fstat( unsigned int fd_id );        // file descriptor index

extern int _fat_close( unsigned int fd_id );        // file descriptor index


extern int _fat_user_open( char*        pathname,   // file pathname from root
                           unsigned int flags );    // unused

extern int _fat_user_read(  unsigned int fd_id,     // file descriptor index
                            void*        buffer,    // destination buffer
                            unsigned int count,     // number of sectors to read
                            unsigned int offset );  // sectors to skip in file

extern int _fat_user_write( unsigned int fd_id,     // file descriptor index
                            void*        buffer,    // source buffer
                            unsigned int count,     // number of sectors to write
                            unsigned int offset );  //  sectors to skip in file

extern int _fat_user_lseek( unsigned int fd_id,
                            unsigned int offset,
                            unsigned int whence );  

/*******************************************************************************/


#endif


// Local Variables:
// tab-width: 4
// c-basic-offset: 4
// c-file-offsets:((innamespace . 0)(inline-open . 0))
// indent-tabs-mode: nil
// End:
// vim: filetype=c:expandtab:shiftwidth=4:tabstop=4:softtabstop=4

