///////////////////////////////////////////////////////////////////////////////////
// File     : mmc_driver.c
// Date     : 23/05/2013
// Author   : alain greiner
// Copyright (c) UPMC-LIP6
///////////////////////////////////////////////////////////////////////////////////

#include <hard_config.h>
#include <mmc_driver.h>
#include <tty0.h>
#include <kernel_locks.h>
#include <utils.h>

#include <io.h>

#if !defined(X_SIZE) 
# error: You must define X_SIZE in the hard_config.h file
#endif

#if !defined(Y_SIZE) 
# error: You must define X_SIZE in the hard_config.h file
#endif

#if !defined(X_WIDTH) 
# error: You must define X_WIDTH in the hard_config.h file
#endif

#if !defined(Y_WIDTH) 
# error: You must define X_WIDTH in the hard_config.h file
#endif

#if !defined(SEG_MMC_BASE) 
# error: You must define SEG_MMC_BASE in the hard_config.h file
#endif

#if !defined(PERI_CLUSTER_INCREMENT) 
# error: You must define PERI_CLUSTER_INCREMENT in the hard_config.h file
#endif

///////////////////////////////////////////////////////////////////////////////
// Locks protecting MMC components (one per cluster)
// There are two kinds of lock: the _mmc_lock table contains all the locks and
// and is stored in the kernel data segment, whereas the _mmc_distributed_lock
// contains the addresses of locks which are distributed in every cluster (each
// cluster contains the lock which protects its own mmc_component).
// The global variable mmc_boot_mode define the type of lock which is used,
// and must be defined in both kernel_init.c and boot.c files.
// - the boot code must use a spin_lock because the kernel heap is not set.
// - the kernel code can use a sqt_lock when the kernel heap is set.
///////////////////////////////////////////////////////////////////////////////

extern unsigned int  _mmc_boot_mode;

__attribute__((section(".kdata")))
spin_lock_t           _mmc_lock[X_SIZE][Y_SIZE]  __attribute__((aligned(64)));

__attribute__((section(".kdata")))
spin_lock_t*          _mmc_distributed_lock[X_SIZE][Y_SIZE];

//////////////////////////////////////////////////////////////////////////////
// This function initializes the distributed locks stored in the kernel heaps
//////////////////////////////////////////////////////////////////////////////

void _mmc_init_locks()
{
    unsigned int cx;    // cluster X coordinate
    unsigned int cy;    // cluster Y coordinate
    
    for ( cx = 0 ; cx < X_SIZE ; cx++ )
    {
        for ( cy = 0 ; cy < Y_SIZE ; cy++ )
        {
            _mmc_distributed_lock[cx][cy] = _remote_malloc( sizeof(spin_lock_t), cx, cy );
            _spin_lock_init( _mmc_distributed_lock[cx][cy] );
        }
    }
}

///////////////////////////////////////////////////////////////////////////////
// This low level function returns the value contained in register 
// defined by the ("func" / "index") arguments,
// in the MMC component contained in cluster "cluster_xy"
///////////////////////////////////////////////////////////////////////////////
static
unsigned int _mmc_get_register( unsigned int cluster_xy, // cluster index
                                unsigned int func,       // function index
                                unsigned int index )     // register index
{
    unsigned int vaddr =
        SEG_MMC_BASE + 
        (cluster_xy * PERI_CLUSTER_INCREMENT) +
        (MMC_REG(func, index) << 2);

    return ioread32( (void*)vaddr );
}

///////////////////////////////////////////////////////////////////////////////
// This low level function sets a new value in register
// defined by the ("func" / "index") arguments,
// in the MMC component contained in cluster "cluster_xy"
///////////////////////////////////////////////////////////////////////////////
static
void _mmc_set_register( unsigned int cluster_xy,       // cluster index
                        unsigned int func,             // func index
                        unsigned int index,            // register index
                        unsigned int value )           // value to be written
{
    unsigned int vaddr =
        SEG_MMC_BASE + 
        (cluster_xy * PERI_CLUSTER_INCREMENT) +
        (MMC_REG(func, index) << 2);
        
    iowrite32( (void*)vaddr, value );
}

/////////////////////////////////////////
void _mmc_inval( paddr_t      buf_paddr,
                 unsigned int buf_length )
{
    // compute cluster coordinates
    unsigned int cluster_xy = (unsigned int)(buf_paddr>>(40-X_WIDTH-Y_WIDTH));
    unsigned int x          = cluster_xy >> Y_WIDTH;
    unsigned int y          = cluster_xy & ((1<<Y_WIDTH)-1);

    // parameters checking 
    if ( (x >= X_SIZE) || (y >= Y_SIZE) )
    {
        _puts("\n[GIET ERROR] in _mmc_inval() : illegal cluster coordinates\n");
        _exit();
    }

    if ( buf_paddr & 0x3F )
    {
        _puts("\n[GIET ERROR] in _mmc_inval() : paddr not 64 bytes aligned\n");
        _exit();
    }

    // get the lock protecting exclusive access to MEMC
    if ( _mmc_boot_mode ) _spin_lock_acquire( &_mmc_lock[x][y] );
    else                  _spin_lock_acquire( _mmc_distributed_lock[x][y] );

    // write inval arguments
    _mmc_set_register(cluster_xy, 0, MEMC_ADDR_LO   , (unsigned int)buf_paddr );
    _mmc_set_register(cluster_xy, 0, MEMC_ADDR_HI   , (unsigned int)(buf_paddr>>32) );
    _mmc_set_register(cluster_xy, 0, MEMC_BUF_LENGTH, buf_length );
    _mmc_set_register(cluster_xy, 0, MEMC_CMD_TYPE  , MEMC_CMD_INVAL );

    // release the lock 
    if ( _mmc_boot_mode ) _spin_lock_release( &_mmc_lock[x][y] );
    else                  _spin_lock_release( _mmc_distributed_lock[x][y] );
}

///////////////////////////////////////
void _mmc_sync( paddr_t      buf_paddr,
                unsigned int buf_length )
{
    // compute cluster coordinates
    unsigned int cluster_xy = (unsigned int)(buf_paddr>>(40-X_WIDTH-Y_WIDTH));
    unsigned int x          = cluster_xy >> Y_WIDTH;
    unsigned int y          = cluster_xy & ((1<<Y_WIDTH)-1);

    // parameters checking 
    if ( (x >= X_SIZE) || (y >= Y_SIZE) )
    {
        _puts( "\n[GIET ERROR] in _mmc_sync() : illegal cluster coordinates");
        _exit();
    }

    if ( buf_paddr & 0x3F )
    {
        _puts("\n[GIET ERROR] in _mmc_sync() : paddr not 64 bytes aligned\n");
        _exit();
    }

    // get the lock protecting exclusive access to MEMC
    if ( _mmc_boot_mode ) _spin_lock_acquire( &_mmc_lock[x][y] );
    else                  _spin_lock_acquire( _mmc_distributed_lock[x][y] );

    // write inval arguments
    _mmc_set_register(cluster_xy, 0, MEMC_ADDR_LO   , (unsigned int)buf_paddr);
    _mmc_set_register(cluster_xy, 0, MEMC_ADDR_HI   , (unsigned int)(buf_paddr>>32));
    _mmc_set_register(cluster_xy, 0, MEMC_BUF_LENGTH, buf_length);
    _mmc_set_register(cluster_xy, 0, MEMC_CMD_TYPE  , MEMC_CMD_SYNC);

    // release the lock 
    if ( _mmc_boot_mode ) _spin_lock_release( &_mmc_lock[x][y] );
    else                  _spin_lock_release( _mmc_distributed_lock[x][y] );
}

/////////////////////////////////////////////
unsigned int _mmc_instrument( unsigned int x, 
                              unsigned int y,
                              unsigned int reg )
{
    // parameters checking 
    if ( (x >= X_SIZE) || (y >= Y_SIZE) )
    {
        _puts( "\n[GIET ERROR] in _mmc_instrument() : illegal cluster coordinates");
        _exit();
    }

    unsigned int cluster_xy = (x << Y_WIDTH) + y;
    return( _mmc_get_register(cluster_xy , 1 , reg) );
}

///////////////////////////////////////////////////////
void _mmc_isr( unsigned int irq_type,  // should be HWI 
               unsigned int irq_id,    // index returned by ICU
               unsigned int channel )  // unused
{
    unsigned int gpid       = _get_procid();
    unsigned int cluster_xy = gpid >> P_WIDTH;
    unsigned int x          = cluster_xy >> Y_WIDTH;
    unsigned int y          = cluster_xy & ((1<<Y_WIDTH)-1);
    unsigned int p          = gpid & ((1<<P_WIDTH)-1);

    _puts("[GIET ERROR] MMC IRQ received by processor[");
    _putd( x );
    _puts(",");
    _putd( y );
    _puts(",");
    _putd( p );
    _puts("] but _mmc_isr() not implemented\n");
}



// Local Variables:
// tab-width: 4
// c-basic-offset: 4
// c-file-offsets:((innamespace . 0)(inline-open . 0))
// indent-tabs-mode: nil
// End:
// vim: filetype=c:expandtab:shiftwidth=4:tabstop=4:softtabstop=4

