#include "game.h"
#include "disp.h"
#include "ctrl.h"
#include <math.h>
#include <stdio.h>

//////////////////////////////
// Game Maps
//////////////////////////////

static Map map[] =
{
    { // map0
        .tile =
        {
            {1, 0, 0, 0, 0, 1, 0, 0, 1, 1},
            {0, 0, 0, 0, 0, 1, 0, 0, 0, 2},
            {0, 0, 0, 0, 1, 0, 0, 0, 1, 1},
            {0, 0, 0, 1, 3, 0, 3, 0, 0, 0},
            {0, 0, 0, 1, 3, 0, 3, 0, 0, 1},
            {0, 0, 0, 1, 3, 0, 3, 0, 0, 0},
            {0, 0, 0, 1, 1, 0, 3, 0, 0, 1},
            {4, 0, 0, 0, 0, 0, 1, 0, 0, 0},
            {4, 0, 0, 0, 0, 0, 1, 0, 0, 1},
            {0, 4, 4, 4, 4, 0, 0, 0, 1, 0}
        },
        .w = 10,
        .h = 10,
        .startX = 2.f,
        .startY = 3.f,
        .startDir = 70.f * M_PI / 180.f
    },
    { // map1
        .tile =
        {
            {0, 1, 0, 1, 0, 3, 0, 0, 0, 0},
            {0, 1, 0, 0, 0, 3, 0, 0, 0, 0},
            {0, 0, 0, 1, 0, 3, 0, 0, 0, 0},
            {1, 1, 1, 1, 0, 0, 0, 0, 0, 0},
            {4, 2, 4, 1, 3, 0, 3, 3, 3, 0},
            {4, 0, 0, 1, 3, 3, 3, 0, 0, 0},
            {4, 0, 0, 1, 3, 0, 0, 0, 3, 3},
            {4, 0, 0, 0, 0, 0, 4, 0, 0, 3},
            {4, 0, 0, 0, 0, 0, 4, 0, 0, 0},
            {4, 4, 4, 4, 4, 4, 4, 0, 1, 0}
        },
        .w = 10,
        .h = 10,
        .startX = 0.5f,
        .startY = 0.5f,
        .startDir = 90.f * M_PI / 180.f
    },
    { // map2
        .tile =
        {
            {4, 4, 4, 4, 4, 4, 4, 0, 0, 0},
            {0, 0, 0, 0, 0, 0, 0, 0, 3, 0},
            {3, 0, 0, 0, 4, 4, 4, 0, 0, 0},
            {3, 0, 0, 4, 0, 0, 0, 1, 1, 0},
            {3, 0, 4, 2, 0, 0, 0, 0, 0, 0},
            {3, 0, 4, 2, 0, 0, 0, 0, 0, 0},
            {3, 0, 0, 4, 0, 0, 0, 1, 1, 0},
            {3, 0, 0, 0, 4, 4, 4, 0, 0, 0},
            {0, 0, 0, 0, 0, 0, 0, 0, 3, 0},
            {4, 4, 4, 4, 4, 4, 4, 0, 0, 0}
        },
        .w = 10,
        .h = 10,
        .startX = 4.5f,
        .startY = 5.f,
        .startDir = 0.f * M_PI / 180.f
    },
};

////////////////////////////
// extern variables
////////////////////////////

extern Game         game;

//////////////////////
// Local functions
//////////////////////

////////////////////////////////////
static void gameOnBlockHit(int type)
{
    giet_tty_printf("\n[RAYCAST] Fatal collision, killed...\n");
    game.exit = 1;
}

///////////////////////////////////////////////
static void gameCollision(float opx, float opy)
{
    static unsigned int collided_x = 0;
    static unsigned int collided_y = 0;

    float               px = game.player.x;
    float               py = game.player.y;
    int                 fpx = floor(px);
    int                 fpy = floor(py);
    unsigned int        colliding_x = 0;
    unsigned int        colliding_y = 0;
    int                 collide_type_x = 0;
    int                 collide_type_y = 0;
    int                 type;

    // Check for x axis collisions
    if      ((type = gameLocate(floor(px + COLLIDE_GAP), fpy))) 
    {
        colliding_x = 1, collide_type_x = type;
        game.player.x = fpx - COLLIDE_GAP + 1;
    }
    else if ((type = gameLocate(floor(px - COLLIDE_GAP), fpy))) 
    {
        colliding_x = 1, collide_type_x = type;
        game.player.x = fpx + COLLIDE_GAP;
    }

    // Check for y axis collisions
    if      ((type = gameLocate(fpx, floor(py + COLLIDE_GAP)))) 
    {
        colliding_y = 1, collide_type_y = type;
        game.player.y = fpy - COLLIDE_GAP + 1;
    }
    else if ((type = gameLocate(fpx, floor(py - COLLIDE_GAP)))) 
    {
        colliding_y = 1, collide_type_y = type;
        game.player.y = fpy + COLLIDE_GAP;
    }

    // Check if we're inside a wall
    if ((type = gameLocate(fpx, fpy))) 
    {
        colliding_x   = 1   , collide_type_x = type;
        colliding_y   = 1   , collide_type_y = type;
        game.player.x = opx , game.player.y = opy;
    }

    // Take action when the player hits a block
    if (colliding_x && !collided_x)
        gameOnBlockHit(collide_type_x);
    if (colliding_y && !collided_y)
        gameOnBlockHit(collide_type_y);

    collided_x = colliding_x;
    collided_y = colliding_y;
}

/////////////////////////
// Exported functions
/////////////////////////

///////////////
void gameInit()
{
    game.mapId      = 0;
    game.map        = &map[0];
    game.player.x   = game.map->startX;
    game.player.y   = game.map->startY;
    game.player.dir = game.map->startDir;
    game.timeLeft   = TIME_TOTAL;
    game.exit       = 0;
}

//////////////////
void gameControl()
{
    // Only six commands are accepted:
    // - key Q         : quit game
    // - key UP        : forward move
    // - key DOWN      : backward move
    // - key RIGHT     : right move
    // - key LEFT      : left move
    // - any other key : continue
    // several moves can be made before continue

    char c;
    unsigned int state = 0;
    unsigned int done  = 0;

    // display prompt
    giet_tty_printf("\n[RAYCAST] > " );

    // get one command
    while ( done == 0 )
    {
        // get one character        
        giet_tty_getc( &c );

        if (state == 0) // first character
        {
            if      (c == 0x1B)         // ESCAPE : possible player move  
            {
                state = 1;
            }
            else if (c == 0x71)         // quit game
            {
                game.exit = 1;
                done = 1;
                giet_tty_printf("QUIT\n");
            }
            else                        // continue 
            {
                done = 1;
                giet_tty_printf("\n");
            }
        } 
        else if (state == 1) // previous character was ESCAPE
        {
            if      (c == 0x5B)        // BRAKET : possible player move
            {
                state = 2;
            }
            else                       // continue 
            {
                done = 1;
                giet_tty_printf("\n");
            }
        }
        else  // previous characters were ESCAPE,BRACKET
        {
            if      (c == 0x41)        // UP arrow <=> forward move
            {
                game.player.x += PLAYER_MOVE * cos(game.player.dir);
                game.player.y += PLAYER_MOVE * sin(game.player.dir);
                giet_tty_printf("GO ");
                state = 0;
            }
            else if (c == 0x42)        // DOWN arrow <=> backward move
            {
                game.player.x -= PLAYER_MOVE * cos(game.player.dir);
                game.player.y -= PLAYER_MOVE * sin(game.player.dir);
                giet_tty_printf("BACK ");
                state = 0;
            }
            else if (c == 0x43)        // RIGHT arrow <=> turn right     
            {
                game.player.dir += PLAYER_ROT;
                giet_tty_printf("RIGHT ");
                state = 0;
            }
            else if (c == 0x44)        // LEFT arrow <=> turn left      
            {
                game.player.dir -= PLAYER_ROT;
                giet_tty_printf("LEFT ");
                state = 0;
            }
            else                       // continue 
            {
                done = 1;
                giet_tty_printf("\n");
            }
        }
    }  // end while
    
    // check new position
    int hit = gameLocate( game.player.x , game.player.y );
    if ( hit )  gameOnBlockHit( hit );
}

////////////////////////////
int gameLocate(int x, int y)
{
    if ((x < 0 || x >= game.map->w) ||
        (y < 0 || y >= game.map->h)) 
    {
        // Outside the map bounds
        return 1;
    }

    return game.map->tile[y][x];
}

///////////////
void gameTick()
{
    game.timeLeft--;

    if (game.timeLeft == 0) 
    {
        game.exit = 1;
    }
}

