///////////////////////////////////////////////////////////////////////////////////
// File     : tty_driver.c
// Date     : 23/05/2013
// Author   : alain greiner
// Copyright (c) UPMC-LIP6
///////////////////////////////////////////////////////////////////////////////////

#include <giet_config.h>
#include <hard_config.h>
#include <tty_driver.h>
#include <xcu_driver.h>
#include <ctx_handler.h>
#include <utils.h>
#include <tty0.h>

#if !defined(SEG_TTY_BASE)
# error: You must define SEG_TTY_BASE in the hard_config.h file
#endif

////////////////////////////////////////////////////////////////////////////////////
//               extern variables
////////////////////////////////////////////////////////////////////////////////////

// This variable is allocated in kernel_init.c file
extern static_scheduler_t* _schedulers[X_SIZE][Y_SIZE][NB_PROCS_MAX];

////////////////////////////////////////////////////////////////////////////////////
//               global variables
////////////////////////////////////////////////////////////////////////////////////

__attribute__((section(".kdata")))
tty_fifo_t  _tty_rx_fifo[NB_TTY_CHANNELS];

/*
__attribute__((section(".kdata")))
unsigned int   _tty_rx_buf[NB_TTY_CHANNELS];

__attribute__((section(".kdata")))
unsigned int   _tty_rx_full[NB_TTY_CHANNELS]; 

__attribute__((section(".kdata")))
unsigned int   _tty_rx_trdid[NB_TTY_CHANNELS]; 
*/

////////////////////////////////////////////////////////////////////////////////////
//               access functions
////////////////////////////////////////////////////////////////////////////////////

/////////////////////////////////////////////////////
unsigned int _tty_get_register( unsigned int channel,
                                unsigned int index )
{
    unsigned int* vaddr = (unsigned int*)SEG_TTY_BASE + channel*TTY_SPAN + index;
    return _io_extended_read( vaddr );
}

/////////////////////////////////////////////
void _tty_set_register( unsigned int channel,
                        unsigned int index,
                        unsigned int value )
{
    unsigned int* vaddr = (unsigned int*)SEG_TTY_BASE + channel*TTY_SPAN + index;
    _io_extended_write( vaddr, value );
}

//////////////////////////////////////
void _tty_init( unsigned int channel )
{
    _tty_rx_fifo[channel].sts = 0;
    _tty_rx_fifo[channel].ptr = 0;
    _tty_rx_fifo[channel].ptw = 0;
}

////////////////////////////////////////
void _tty_rx_isr( unsigned int irq_type,   // HWI / WTI
                  unsigned int irq_id,     // index returned by XCU
                  unsigned int channel )   // TTY channel
{
    // get pointer on TTY_RX FIFO
    tty_fifo_t*   fifo = &_tty_rx_fifo[channel];

    // get character from TTY_RX channel and acknowledge IRQ
    unsigned int data = _tty_get_register( channel, TTY_READ ); 

    // transfer character to FIFO if not full
    // discard incoming character if full
    if ( fifo->sts < TTY_FIFO_DEPTH )
    {
        // store data into FIFO 
        fifo->data[fifo->ptw] = (char)data; 

        // avoid race 
        asm volatile( "sync" );

        // update FIFO state
        fifo->ptw = (fifo->ptw + 1) % TTY_FIFO_DEPTH;
        fifo->sts = fifo->sts + 1;
    }

    // get owner thread indexes
    unsigned int trdid          = fifo->trdid;
    unsigned int x              = (trdid >> 24) & 0xFF;
    unsigned int y              = (trdid >> 16) & 0xFF;
    unsigned int p              = (trdid >>  8) & 0xFF;
    unsigned int ltid           = (trdid      ) & 0xFF;

    // Reset NORUN_MASK_TTY bit 
    static_scheduler_t* psched  = (static_scheduler_t*)_schedulers[x][y][p];
    unsigned int*       ptr     = &psched->context[ltid].slot[CTX_NORUN_ID];
    _atomic_and( ptr , ~NORUN_MASK_TTY );
}

/////////////////////////////////////////
void _tty_tx_isr( unsigned int irq_type,   // HWI / WTI
                  unsigned int irq_id,     // index returned by XCU
                  unsigned int channel )   // TTY channel
{
    _printf("\n[GIET ERROR] the _tty_tx_isr() is not implemented\n");
    _exit();
}

// Local Variables:
// tab-width: 4
// c-basic-offset: 4
// c-file-offsets:((innamespace . 0)(inline-open . 0))
// indent-tabs-mode: nil
// End:
// vim: filetype=c:expandtab:shiftwidth=4:tabstop=4:softtabstop=4

