#!/usr/bin/env python

from mapping import *

######################################################################################
#   file   : router.py
#   date   : november 2014
#   author : Alain Greiner
#######################################################################################
#  This file describes the mapping of the multi-threaded "router" 
#  application on a multi-clusters, multi-processors architecture.
#  This application contains N+2 parallel threads communicating through MWMR channels:
#  The mapping of virtual segments on the clusters is the following:
#    - the data vseg is mapped on cluster[0,0].
#    - The code vsegs are replicated on all clusters.
#    - the stack vsegs are distibuted on all clusters.
#    - the heap vsegs are distributed on all clusters
#  The mapping of threads on processors is the following:
#    - the "main" thread      => on proc[0,0,0]
#    - the "producer" thread  => on proc[0,0,0]
#    - the "consume"  thread  => on proc[0,0,1]
#    - N   "compute"  threads => on all others processors
#  This mapping uses 5 platform parameters, (obtained from the "mapping" argument)
#    - x_size    : number of clusters in a row
#    - y_size    : number of clusters in a column
#    - nprocs    : number of processors per cluster
####################################################################################

######################
def extend( mapping ):

    x_size    = mapping.x_size
    y_size    = mapping.y_size
    nprocs    = mapping.nprocs
    x_width   = mapping.x_width
    y_width   = mapping.y_width

    # define vsegs base & size
    code_base    = 0x10000000
    code_size    = 0x00010000     # 64 Kbytes (replicated in each cluster)
    
    data_base    = 0x20000000
    data_size    = 0x00010000     # 64 Kbytes (non replicated)

    stack_base   = 0x30000000 
    stack_size   = 0x00010000     # 64 Kbytes (per thread)

    heap_base    = 0x40000000
    heap_size    = 0x00200000     # 2 Mbytes (per cluster)

    # create vspace
    vspace = mapping.addVspace( name = 'router', 
                                startname = 'router_data',
                                active = False )
    
    # data vseg : shared / in cluster [0,0]
    mapping.addVseg( vspace, 'router_data', data_base , data_size, 
                     'C_WU', vtype = 'ELF', x = 0, y = 0, pseg = 'RAM', 
                     binpath = 'bin/router/appli.elf',
                     local = False )

    # code vsegs : local (one copy in each cluster)
    for x in xrange (x_size):
        for y in xrange (y_size):
            mapping.addVseg( vspace, 'router_code_%d_%d' %(x,y), code_base , code_size,
                             'CXWU', vtype = 'ELF', x = x, y = y, pseg = 'RAM', 
                             binpath = 'bin/router/router.elf',
                             local = True )

    # heap vsegs : shared (one per cluster) 
    for x in xrange (x_size):
        for y in xrange (y_size):
            cluster_id = (x * y_size) + y
            if ( mapping.clusters[cluster_id].procs ):
                size  = heap_size
                base  = heap_base + (cluster_id * size)

                mapping.addVseg( vspace, 'router_heap_%d_%d' %(x,y), base , size, 
                                 'C_WU', vtype = 'HEAP', x = x, y = y, pseg = 'RAM', 
                                 local = False, big = True )

    # stacks vsegs: local (one stack per thread => nprocs stacks per cluster)
    # ... plus main_stack in cluster[0][0]
    mapping.addVseg( vspace, 'main_stack',
                     stack_base, stack_size, 'C_WU', vtype = 'BUFFER', 
                     x = 0 , y = 0 , pseg = 'RAM',
                     local = True )

    for x in xrange (x_size):
        for y in xrange (y_size):
            cluster_id = (x * y_size) + y
            if ( mapping.clusters[cluster_id].procs ):
                for p in xrange( nprocs ):
                    proc_id = (((x * y_size) + y) * nprocs) + p
                    base    = stack_base + (proc_id * stack_size) + stack_size 

                    mapping.addVseg( vspace, 'router_stack_%d_%d_%d' % (x,y,p), 
                                     base, stack_size, 'C_WU', vtype = 'BUFFER', 
                                     x = x , y = y , pseg = 'RAM',
                                     local = True )

    # distributed threads / one thread per processor
    # ... plus main on P[0][0][0]
    mapping.addThread( vspace, 'main', True, 0, 0, 1,
                       'main_stack',
                       'router_heap_0_0',
                       0 )                      # index in start_vector

    for x in xrange (x_size):
        for y in xrange (y_size):
            cluster_id = (x * y_size) + y
            if ( mapping.clusters[cluster_id].procs ):
                for p in xrange( nprocs ):
                    if   (x==0) and (y==0) and (p==0):  # thread producer
                        start_index = 3
                        thread_name  = 'producer_0_0_0'            
                    elif (x==0) and (y==0) and (p==1):  # thread consumer 
                        start_index = 2  
                        thread_name  = 'consumer_0_0_1'
                    else :                              # thread compute
                        start_index = 1
                        thread_name  = 'compute_%d_%d_%d' % (x,y,p)

                    mapping.addThread( vspace, thread_name, False , x, y, p,
                                      'router_stack_%d_%d_%d' % (x,y,p),
                                      'router_heap_%d_%d' %(x,y),
                                      start_index )

    # extend mapping name
    mapping.name += '_router'

    return vspace  # useful for test
            
################################ test ######################################################

if __name__ == '__main__':

    vspace = extend( Mapping( 'test', 2, 2, 4 ) )
    print vspace.xml()


# Local Variables:
# tab-width: 4;
# c-basic-offset: 4;
# c-file-offsets:((innamespace . 0)(inline-open . 0));
# indent-tabs-mode: nil;
# End:
#
# vim: filetype=python:expandtab:shiftwidth=4:tabstop=4:softtabstop=4

