//////////////////////////////////////////////////////////////////////////////////
// File     : mwmr_channel.h         
// Date     : 01/04/2012
// Author   : alain greiner
// Copyright (c) UPMC-LIP6
///////////////////////////////////////////////////////////////////////////////////
// The mwmr_channel.c and mwmr_channel.h files are part of the GIET nano-kernel.
// This  middleware implements a user level Multi-Writers / Multi-Readers
// communication channel, that can be used by parallel multi-tasks applications
// respecting the TCG (Tasks and Communications Graph) formalism.
// 
// The mwmr_read() and mwmr_write() functions do not require a system call.
// The channel must have been allocated in a non cacheable segment,
// if the platform does not provide hardware cache coherence.
//
// An MWMR transaction transfer an integer number of items, and an item is
// an integer number of uint32_t (32 bits words).
// The max number of words that can be stored in a MWMR channel is defined by the
// "depth" parameter, and the "width" parameter define the minimal number of
// word contained in an atomic item. Therefore, the "depth" parameter must be
// a multiple of the "width" parameter.
//
// Both the mwmr_read() and mwmr_write() functions are blocking functions. 
// A queuing lock provides exclusive access to the MWMR channel.
///////////////////////////////////////////////////////////////////////////////////

#ifndef _MWMR_CHANNEL_H_
#define _MWMR_CHANNEL_H_

#include "user_lock.h"
#include "stdint.h"

///////////////////////////////////////////////////////////////////////////////////
//         MWMR channel 
// This structure define a - shared - communication buffer between threads.
// It must be declared as a global variable.
///////////////////////////////////////////////////////////////////////////////////

typedef struct mwmr_channel_s 
{
    user_lock_t lock;         // exclusive access lock
    uint32_t    sts;          // number of words available
    uint32_t    ptr;          // index of the first valid data word
    uint32_t    ptw;          // index of the first empty slot 
    uint32_t    depth;        // max number of words in the channel
    uint32_t    width;        // number of words in an item      
    uint32_t*   data;         // circular buffer base address
} mwmr_channel_t;

///////////////////////////////////////////////////////////////////////////////////
//         MWMR bufio 
// This structure define a - private - input or output buffer, that can be used
// to move data to (or ftom a shared MWMR communication channel. 
// It is a local variable in the stack of the reader/writer thread.
// It can be used to simplify access to a MWMR channel when the transfered
// data have a non fixed format, and must be analysed or produced byte per byte.
// An input buffer is automatically refill when it becomes empty.
// An output buffer is automatically flushed when it becomes full.
///////////////////////////////////////////////////////////////////////////////////

typedef struct mwmr_bufio_s
{
    uint32_t         is_input;    // input buffer if non zero 
    uint32_t         ptr;         // current byte index (0 to max-1)
    uint8_t*         base;        // data buffer base address
    uint32_t         max;         // number of bytes after refill (only for input bufio)
    uint32_t         nitems;      // buffer size (number of items)
    uint32_t         nbytes;      // buffer size (number of bytes)
    mwmr_channel_t*  mwmr;        // associated MWMR channel
} mwmr_bufio_t;

//////////////////////////////////////////////////////////////////////////////
//  MWMR channel access functions
//////////////////////////////////////////////////////////////////////////////

void mwmr_init(  mwmr_channel_t* mwmr,
                 uint32_t*       buffer,
                 uint32_t        width,        // number of words per item
                 uint32_t        nitems );     // max number of items

void mwmr_dump( mwmr_channel_t* mwmr );

void mwmr_read(  mwmr_channel_t* mwmr,
                 uint32_t*       buffer,
                 uint32_t        items );

void mwmr_write( mwmr_channel_t* mwmr,
                 uint32_t*       buffer,
                 uint32_t        items );

uint32_t nb_mwmr_read ( mwmr_channel_t*  mwmr,
                        uint32_t*        buffer,
                        uint32_t         items );

uint32_t nb_mwmr_write( mwmr_channel_t*  mwmr,
                        uint32_t*        buffer,
                        uint32_t         items );

//////////////////////////////////////////////////////////////////////////////
//  MWMR bufio access functions
//////////////////////////////////////////////////////////////////////////////

void mwmr_bufio_init( mwmr_bufio_t*    bufio,  
                      uint8_t*         buffer, 
                      uint32_t         size,       // number of bytes 
                      uint32_t         is_input,
                      mwmr_channel_t*  mwmr );
                    
void mwmr_bufio_dump( mwmr_bufio_t* bufio );

uint8_t mwmr_bufio_read_byte( mwmr_bufio_t* bufio );

void mwmr_bufio_skip( mwmr_bufio_t* bufio,
                      uint32_t      length );

void mwmr_bufio_align( mwmr_bufio_t* bufio );

void mwmr_bufio_write_byte( mwmr_bufio_t* bufio,
                            uint8_t       value );

void mwmr_bufio_flush( mwmr_bufio_t* bufio );

#endif

// Local Variables:
// tab-width: 4
// c-basic-offset: 4
// c-file-offsets:((innamespace . 0)(inline-open . 0))
// indent-tabs-mode: nil
// End:
// vim: filetype=c:expandtab:shiftwidth=4:tabstop=4:softtabstop=4

