///////////////////////////////////////////////////////////////////////////////////////
//  file   : display.c  
//  date   : may 2014
//  author : Alain Greiner
///////////////////////////////////////////////////////////////////////////////////////
//  This file describes the single thread "display" application.
//  It uses the external chained buffer DMA to display a stream
//  of images on the frame buffer.  
///////////////////////////////////////////////////////////////////////////////////////

#include <stdio.h>
#include <hard_config.h>     // To check Frame Buffer size

#define FILENAME    "misc/images_128.raw"
#define NPIXELS     128
#define NLINES      128
#define NIMAGES     10                  

#define INTERACTIVE 0

unsigned char buf0[NPIXELS*NLINES] __attribute__((aligned(64)));
unsigned char buf1[NPIXELS*NLINES] __attribute__((aligned(64)));

unsigned int  sts0[16]  __attribute__((aligned(64)));
unsigned int  sts1[16]  __attribute__((aligned(64)));

////////////////////////////////////////////
__attribute__((constructor)) void main()
////////////////////////////////////////////
{
    // get processor identifiers
    unsigned int    x;
    unsigned int    y; 
    unsigned int    p;
    giet_proc_xyp( &x, &y, &p );

    int             fd;
    unsigned int    image = 0;
    char            byte;
    unsigned int    fbuf_x_size;
    unsigned int    fbuf_y_size;

    // checking frame buffer size
    giet_fbf_size( &fbuf_x_size , &fbuf_y_size );
    giet_pthread_assert( ((NPIXELS == fbuf_x_size) && (NLINES == fbuf_y_size)),
                         "[DISPLAY ERROR] Frame buffer size does not fit image size");

    // get a private TTY 
    giet_tty_alloc(0);

    giet_tty_printf("\n[DISPLAY] P[%d,%d,%d] starts at cycle %d\n"
                    "  - buf0_vaddr = %x\n" 
                    "  - buf1_vaddr = %x\n" 
                    "  - sts0_vaddr = %x\n" 
                    "  - sts1_vaddr = %x\n",
                    x, y, p, giet_proctime(),
                    buf0, buf1, sts0, sts1 );

    // open file
    fd = giet_fat_open( FILENAME , 0 );

    giet_tty_printf("\n[DISPLAY] P[%d,%d,%d] open file %s at cycle %d\n", 
                    x, y, p, FILENAME, giet_proctime() );

    // get Frame Buffer ownership
    giet_fbf_alloc();

    // get a Chained Buffer DMA channel for two user buffers
    giet_fbf_cma_alloc( 2 );

    // register the two user buffers
    giet_fbf_cma_init_buf( 0 , buf0 , sts0 );
    giet_fbf_cma_init_buf( 1 , buf1 , sts1 );

    // start CMA peripheral 
    giet_fbf_cma_start();
    
    giet_tty_printf("\n[DISPLAY] Proc[%d,%d,%d] starts CMA at cycle %d\n", 
                    x, y, p, giet_proctime() );

    // Main loop on images
    while ( 1 )
    {
        //////   handling buf0
        giet_fbf_cma_check( 0 );                       // check buf0 empty
        giet_fat_read( fd, buf0, NPIXELS*NLINES );     // load buf0 from disk
        giet_fbf_cma_display( 0 );                     // display buf0

        giet_tty_printf("\n[DISPLAY] Proc[%d,%d,%d] display image %d at cycle %d\n", 
                        x, y, p, image, giet_proctime() );
        image++;

        if ( image == NIMAGES )
        {
            image = 0;
            giet_fat_lseek( fd , 0 , SEEK_SET );
        }

        if ( INTERACTIVE ) giet_tty_getc( &byte );

        //////   handling buf1
        giet_fbf_cma_check( 1 );                       // check buf1 empty
        giet_fat_read( fd, buf1, NPIXELS*NLINES );     // load buf1 from disk
        giet_fbf_cma_display( 1 );                     // display buf1

        giet_tty_printf("\n[DISPLAY] Proc[%d,%d,%d] display image %d at cycle %d\n", 
                        x, y, p, image, giet_proctime() );
        image++;

        if ( image == NIMAGES )
        {
            image = 0;
            giet_fat_lseek( fd , 0 , SEEK_SET );
        }

        if ( INTERACTIVE ) giet_tty_getc( &byte );
    }

    // stop Chained buffer DMA channel
    giet_fbf_cma_stop();

    giet_pthread_exit("completed");
}
