/////////////////////////////////////////////////////////////////////////////
// File     : stdio.h         
// Date     : 01/04/2010
// Author   : alain greiner & Joel Porquet
// Copyright (c) UPMC-LIP6
/////////////////////////////////////////////////////////////////////////////
// The stdio.c and stdio.h files are part of the GIET_VM nano-kernel.
// This library contains all user-level functions that contain a system call
// to access protected or shared ressources.
/////////////////////////////////////////////////////////////////////////////

#ifndef _STDIO_H
#define _STDIO_H

#include "giet_fat32/fat32_shared.h"
#include "giet_common/mips32_registers.h"

// These define must be synchronised with 
// the _syscall_vector defined in file sys_handler.c

#define SYSCALL_PROC_XYP             0x00
#define SYSCALL_PROC_TIME            0x01
#define SYSCALL_PROCS_NUMBER         0x02
#define SYSCALL_GET_XY               0x03
//                                   0x04
#define SYSCALL_VOBJ_GET_VBASE       0x05
#define SYSCALL_VOBJ_GET_LENGTH      0x06
#define SYSCALL_HEAP_INFO            0x07
#define SYSCALL_FBF_SIZE             0x08
#define SYSCALL_FBF_ALLOC            0x09
#define SYSCALL_FBF_CMA_ALLOC        0x0A
#define SYSCALL_FBF_CMA_INIT_BUF     0x0B
#define SYSCALL_FBF_CMA_START        0x0C
#define SYSCALL_FBF_CMA_DISPLAY      0x0D
#define SYSCALL_FBF_CMA_STOP         0x0E
#define SYSCALL_FBF_CMA_CHECK        0x0F

#define SYSCALL_APPS_STATUS          0x10
#define SYSCALL_FBF_SYNC_WRITE       0x11
#define SYSCALL_FBF_SYNC_READ        0x12
//                                   0x13
#define SYSCALL_TIM_ALLOC            0x14
#define SYSCALL_TIM_START            0x15
#define SYSCALL_TIM_STOP             0x16
#define SYSCALL_KILL_APP             0x17
#define SYSCALL_EXEC_APP             0x18
//                                   0x19
#define SYSCALL_PTHREAD_CONTROL      0x1A
#define SYSCALL_PTHREAD_YIELD        0x1B
#define SYSCALL_PTHREAD_KILL         0x1C
#define SYSCALL_PTHREAD_CREATE       0x1D
#define SYSCALL_PTHREAD_JOIN         0x1E
#define SYSCALL_PTHREAD_EXIT         0x1F

#define SYSCALL_FAT_OPEN             0x20
#define SYSCALL_FAT_READ             0x21
#define SYSCALL_FAT_WRITE            0x22
#define SYSCALL_FAT_LSEEK            0x23
#define SYSCALL_FAT_FINFO            0x24
#define SYSCALL_FAT_CLOSE            0x25
#define SYSCALL_FAT_REMOVE           0x26
#define SYSCALL_FAT_RENAME           0x27
#define SYSCALL_FAT_MKDIR            0x28
#define SYSCALL_FAT_OPENDIR          0x29
#define SYSCALL_FAT_CLOSEDIR         0x2A
#define SYSCALL_FAT_READDIR          0x2B
//                                   0x2C
//                                   0x2D
//                                   0x2E
//                                   0x2F

#define SYSCALL_NIC_ALLOC            0x30
#define SYSCALL_NIC_START            0x31
#define SYSCALL_NIC_MOVE             0x32
#define SYSCALL_NIC_STOP             0x33
#define SYSCALL_NIC_STATS            0x34
#define SYSCALL_NIC_CLEAR            0x35
#define SYSCALL_TTY_WRITE            0x36
#define SYSCALL_TTY_READ             0x37
#define SYSCALL_TTY_ALLOC            0x38
//                                   0x39
//                                   0x3A
#define SYSCALL_COPROC_COMPLETED     0x3B
#define SYSCALL_COPROC_ALLOC         0x3C
#define SYSCALL_COPROC_CHANNEL_INIT  0x3D
#define SYSCALL_COPROC_RUN           0x3E
#define SYSCALL_COPROC_RELEASE       0x3F

////////////////////////////////////////////////////////////////////////////
// Define the error codes for the syscall handlers
// These define must be synchronized with values in the sys_handler.h file
////////////////////////////////////////////////////////////////////////////

#define SYSCALL_OK                               ( 0 )
#define SYSCALL_VSPACE_NOT_FOUND                 (-1 )
#define SYSCALL_THREAD_NOT_FOUND                 (-2 )
#define SYSCALL_NOT_IN_SAME_VSPACE               (-3 )
#define SYSCALL_UNCOHERENT_THREAD_CONTEXT        (-4 )
#define SYSCALL_ILLEGAL_THREAD_COMMAND_TYPE      (-5 )
#define SYSCALL_CANNOT_LOAD_DATA_SEGMENT         (-6 )
#define SYSCALL_THREAD_ALREADY_ACTIVE            (-7 )
#define SYSCALL_MAIN_NOT_FOUND                   (-8 )
#define SYSCALL_APPLI_CANNOT_BE_KILLED           (-9 )
#define SYSCALL_PTHREAD_ARGUMENT_NOT_SUPPORTED   (-10)
#define SYSCALL_ILLEGAL_CLUSTER_COORDINATES      (-11)
#define SYSCALL_VSEG_NOT_FOUND                   (-12)
#define SYSCALL_UNDEFINED_SYSTEM_CALL            (-13)
#define SYSCALL_COPROCESSOR_NOT_FOUND            (-14)
#define SYSCALL_COPROCESSOR_ILLEGAL_MODE         (-15)
#define SYSCALL_COPROCESSOR_NON_ALLOCATED        (-16)
#define SYSCALL_CHANNEL_ALREADY_ALLOCATED        (-17)
#define SYSCALL_NO_CHANNEL_AVAILABLE             (-18)
#define SYSCALL_CHANNEL_NON_ALLOCATED            (-19)
#define SYSCALL_ILLEGAL_XY_ARGUMENTS             (-20)
#define SYSCALL_OUT_OF_KERNEL_HEAP_MEMORY        (-21)
#define SYSCALL_ADDRESS_NON_ALIGNED              (-22)
#define SYSCALL_ADDRESS_NON_USER_ACCESSIBLE      (-23)
#define SYSCALL_MISSING_INITIALISATION           (-24)
#define SYSCALL_SHARED_PERIPHERAL_BUSY           (-25)

////////////////////////////////////////////////////////////////////////////
// Command values for the giet_pthread_control() syscall
// These define must be synchronized with values in the sys_handler.h file
////////////////////////////////////////////////////////////////////////////

#define THREAD_CMD_PAUSE        0
#define THREAD_CMD_RESUME       1
#define THREAD_CMD_CONTEXT      2

////////////////////////////////////////////////////////////////////////////
// NULL pointer definition
////////////////////////////////////////////////////////////////////////////

#define NULL (void *)0

////////////////////////////////////////////////////////////////////////////
// This generic C function is used to implement all system calls.
// It writes the system call arguments in the proper registers,
// and tells GCC what has been modified by system call execution.
// Returns -1 to signal an error.
////////////////////////////////////////////////////////////////////////////
static inline int sys_call( int call_no,
                            int arg_0, 
                            int arg_1, 
                            int arg_2, 
                            int arg_3 ) 
{
    register int reg_no_and_output asm("v0") = call_no;
    register int reg_a0 asm("a0") = arg_0;
    register int reg_a1 asm("a1") = arg_1;
    register int reg_a2 asm("a2") = arg_2;
    register int reg_a3 asm("a3") = arg_3;

    asm volatile(
            "syscall"
            : "+r" (reg_no_and_output), /* input/output argument */
              "+r" (reg_a0),             
              "+r" (reg_a1),
              "+r" (reg_a2),
              "+r" (reg_a3),
              "+r" (reg_no_and_output)
            : /* input arguments */
            : "memory",
            /* These persistant registers will be saved on the stack by the
             * compiler only if they contain relevant data. */
            "at",
            "v1",
            "ra",
            "t0",
            "t1",
            "t2",
            "t3",
            "t4",
            "t5",
            "t6",
            "t7",
            "t8",
            "t9"
               );
    return (volatile int)reg_no_and_output;
}

//////////////////////////////////////////////////////////////////////////
//               MIPS32 related system calls 
//////////////////////////////////////////////////////////////////////////

extern void giet_proc_xyp( unsigned int* cluster_x,
                           unsigned int* cluster_y,
                           unsigned int* lpid );

extern unsigned int giet_proctime();

extern unsigned int giet_rand();

//////////////////////////////////////////////////////////////////////////
//              Threads related system calls
//////////////////////////////////////////////////////////////////////////

typedef unsigned int pthread_t;

typedef unsigned int pthread_attr_t;

extern int giet_pthread_create( pthread_t*       trdid,
                                pthread_attr_t*  attr,
                                void*            function,
                                void*            ptr ); 

extern void giet_pthread_exit( void* string );

extern int giet_pthread_join( pthread_t  trdid,
                              void**     ptr );

extern int giet_pthread_kill( pthread_t thread_id,
                              int       signal );

extern void giet_pthread_yield();

extern void giet_pthread_assert( unsigned int condition, 
                                 char*        string );

extern void giet_pthread_control( unsigned int command,
                                  char*        vspace_name,
                                  char*        thread_name );

//////////////////////////////////////////////////////////////////////////
//               Application related system calls
//////////////////////////////////////////////////////////////////////////

extern int giet_kill_application( char* name );

extern int giet_exec_application( char* name );

extern void giet_applications_status( char* name );

//////////////////////////////////////////////////////////////////////////
//             Coprocessors related system calls 
//////////////////////////////////////////////////////////////////////////

// this structure is used by the giet_coproc_channel_init() 
// system call to specify the communication channel parameters. 
typedef struct giet_coproc_channel
{
    unsigned int  channel_mode;    // MWMR / DMA_IRQ / DMA_NO_IRQ
    unsigned int  buffer_size;     // memory buffer size
    unsigned int  buffer_vaddr;    // memory buffer virtual address
    unsigned int  status_vaddr;    // MWMR status virtual address (12 bytes)
    unsigned int  lock_vaddr;      // MWMR lock virtual address (64 bytes)
} giet_coproc_channel_t;

extern void giet_coproc_alloc( unsigned int   cluster_xy,
                               unsigned int   coproc_type,
                               unsigned int*  coproc_info );

extern void giet_coproc_release( unsigned int cluster_xy,
                                 unsigned int coproc_type );

extern void giet_coproc_channel_init( unsigned int            cluster_xy,
                                      unsigned int            coproc_type,
                                      unsigned int            channel,
                                      giet_coproc_channel_t*  desc );

extern void giet_coproc_run( unsigned int cluster_xy,
                             unsigned int coproc_type );

extern void giet_coproc_completed( unsigned int cluster_xy,
                                   unsigned int coproc_type );

//////////////////////////////////////////////////////////////////////////
//             TTY device related system calls 
//////////////////////////////////////////////////////////////////////////

extern void giet_tty_alloc( unsigned int shared );

extern void giet_tty_printf( char* format, ... );

extern void giet_tty_getc( char* byte );

extern void giet_tty_gets( char* buf, unsigned int bufsize );

extern void giet_tty_getw( unsigned int* val );

//////////////////////////////////////////////////////////////////////////
//                TIMER device related system calls 
//////////////////////////////////////////////////////////////////////////

extern void giet_timer_alloc();

extern void giet_timer_start( unsigned int period );

extern void giet_timer_stop();
 
//////////////////////////////////////////////////////////////////////////
//                Frame buffer device related system calls 
//////////////////////////////////////////////////////////////////////////

extern void giet_fbf_size();

extern void giet_fbf_alloc();

extern void giet_fbf_cma_alloc( unsigned int nbufs );

extern void giet_fbf_cma_init_buf( unsigned int index,
                                   void*        buf_vbase, 
                                   void*        sts_vaddr );

extern void giet_fbf_cma_start();

extern void giet_fbf_cma_check( unsigned int buffer );

extern void giet_fbf_cma_display( unsigned int buffer );

extern void giet_fbf_cma_stop();

extern void giet_fbf_sync_read( unsigned int offset, 
                                void*        buffer, 
                                unsigned int length );

extern void giet_fbf_sync_write( unsigned int offset, 
                                 void*        buffer, 
                                 unsigned int length );

//////////////////////////////////////////////////////////////////////////
//                  NIC related system calls 
//////////////////////////////////////////////////////////////////////////

extern void giet_nic_rx_alloc( unsigned int xmax, unsigned int ymax );

extern void giet_nic_tx_alloc( unsigned int xmax, unsigned int ymax );

extern void giet_nic_rx_start();

extern void giet_nic_tx_start();

extern void giet_nic_rx_move( void* buffer );

extern void giet_nic_tx_move( void* buffer );

extern void giet_nic_rx_stop();

extern void giet_nic_tx_stop();

extern void giet_nic_rx_stats();

extern void giet_nic_tx_stats();

extern void giet_nic_rx_clear();

extern void giet_nic_tx_clear();

//////////////////////////////////////////////////////////////////////////
//               FAT related system calls 
//////////////////////////////////////////////////////////////////////////

extern int giet_fat_open( char*        pathname,
                          unsigned int flags );

extern int giet_fat_close( unsigned int fd_id );

extern int giet_fat_file_info( unsigned int     fd_id,
                               fat_file_info_t* info );

extern int giet_fat_read( unsigned int fd_id,
                          void*        buffer,
                          unsigned int count );

extern int giet_fat_write( unsigned int fd,
                           void*        buffer,
                           unsigned int count );

extern int giet_fat_lseek( unsigned int fd,
                           unsigned int offset,
                           unsigned int whence );

extern int giet_fat_remove( char*        pathname,
                            unsigned int should_be_dir );

extern int giet_fat_rename( char*  old_path,
                            char*  new_path ); 

extern int giet_fat_mkdir( char* pathname );

extern int giet_fat_opendir( char* pathname );

extern int giet_fat_closedir( unsigned int fd_id );

extern int giet_fat_readdir( unsigned int  fd_id,
                             fat_dirent_t* entry );

//////////////////////////////////////////////////////////////////////////
//                    Miscelaneous system calls
//////////////////////////////////////////////////////////////////////////

extern void giet_procs_number( unsigned int* x_size,
                               unsigned int* y_size,
                               unsigned int* nprocs );

extern void giet_vobj_get_vbase( char*         vspace_name, 
                                 char*         vobj_name, 
                                 unsigned int* vobj_vaddr);

extern void giet_vobj_get_length( char*         vspace_name, 
                                  char*         vobj_name, 
                                  unsigned int* vobj_vaddr);

extern void giet_heap_info( unsigned int* vaddr, 
                            unsigned int* length,
                            unsigned int  x,
                            unsigned int  y );

extern void giet_get_xy( void*          ptr, 
                         unsigned int*  px,
                         unsigned int*  py );

#endif

// Local Variables:
// tab-width: 4
// c-basic-offset: 4
// c-file-offsets:((innamespace . 0)(inline-open . 0))
// indent-tabs-mode: nil
// End:
// vim: filetype=c:expandtab:shiftwidth=4:tabstop=4:softtabstop=4

