#!/usr/bin/env python

from mapping import *

###################################################################################
#   file   : mjpeg.py 
#   date   : november 2015
#   author : Alain Greiner
###################################################################################
#  This file describes the mapping of the multi-threaded "mjpeg" 
#  application on a multi-clusters, multi-processors architecture.
#
#  The mapping of threads on processors is the following:
#    - the "main" thread, on P[0,0,0] launches all others threads and exit.
#    - the "tg" thread is only running on P[0,0,0].
#    - the "demux", "iqzz", "idct", "vld", and "libu" threads, implementing
#      a block-level pipe-line, are replicated in all clusters.
#  In each cluster the actual mapping depends on the <nprocs> parameter.
#
#  The mapping of virtual segments is the following:
#    - There is one shared data vseg in cluster[0][0]
#    - The code vsegs are replicated in all clusters.
#    - There is one heap vseg per cluster (containing MWMR channels).
#    - The stacks vsegs are distibuted in all clusters.
##################################################################################

######################
def extend( mapping ):

    x_size    = mapping.x_size
    y_size    = mapping.y_size
    nprocs    = mapping.nprocs

    assert (nprocs >= 1) and (nprocs <= 4)

    # define vsegs base & size
    code_base  = 0x10000000     
    code_size  = 0x00010000     # 64 Kbytes (per cluster)
    
    data_base  = 0x20000000
    data_size  = 0x00010000     # 64 Kbytes (non replicated) 

    heap_base  = 0x30000000
    heap_size  = 0x00200000     # 2 Mbytes (per cluster)      

    stack_base = 0x40000000 
    stack_size = 0x00020000     # 128 Kbytes (per thread)

    # create vspace
    vspace = mapping.addVspace( name = 'mjpeg', 
                                startname = 'mjpeg_data', 
                                active = True )
    
    # data vseg : shared / cluster[0][0]
    mapping.addVseg( vspace, 'mjpeg_data', data_base , data_size, 
                     'C_WU', vtype = 'ELF', x = 0, y = 0, pseg = 'RAM', 
                     binpath = 'bin/mjpeg/appli.elf',
                     local = False )

    # heap vsegs : shared (one per cluster) 
    for x in xrange (x_size):
        for y in xrange (y_size):
            cluster_id = (x * y_size) + y
            if ( mapping.clusters[cluster_id].procs ):
                size  = heap_size
                base  = heap_base + (cluster_id * size)

                mapping.addVseg( vspace, 'mjpeg_heap_%d_%d' %(x,y), base , size, 
                                 'C_WU', vtype = 'HEAP', x = x, y = y, pseg = 'RAM', 
                                 local = False, big = True )

    # code vsegs : local (one copy per cluster)
    for x in xrange (x_size):
        for y in xrange (y_size):
            cluster_id = (x * y_size) + y
            if ( mapping.clusters[cluster_id].procs ):

                mapping.addVseg( vspace, 'mjpeg_code_%d_%d' %(x,y), 
                                 code_base , code_size,
                                 'CXWU', vtype = 'ELF', x = x, y = y, pseg = 'RAM', 
                                 binpath = 'bin/mjpeg/appli.elf',
                                 local = True )

    # stacks vsegs: local (one stack per thread => 5 stacks per cluster)
    # ... plus main_stack and tg_stack in cluster[0][0]
    base = stack_base
    mapping.addVseg( vspace, 'mjpeg_main_stack',
                     base, stack_size, 'C_WU', vtype = 'BUFFER', 
                     x = 0 , y = 0 , pseg = 'RAM',
                     local = True )

    base += stack_size

    mapping.addVseg( vspace, 'mjpeg_tg_stack',
                     base , stack_size, 'C_WU', vtype = 'BUFFER', 
                     x = 0 , y = 0 , pseg = 'RAM',
                     local = True )

    base += stack_size

    for x in xrange (x_size):
        for y in xrange (y_size):
            if ( mapping.clusters[cluster_id].procs ):

                mapping.addVseg( vspace, 'mjpeg_demux_stack_%d_%d' % (x,y), 
                                 base, stack_size, 'C_WU', vtype = 'BUFFER', 
                                 x = x , y = y , pseg = 'RAM',
                                 local = True )

                base += stack_size

                mapping.addVseg( vspace, 'mjpeg_vld_stack_%d_%d' % (x,y), 
                                 base, stack_size, 'C_WU', vtype = 'BUFFER', 
                                 x = x , y = y , pseg = 'RAM',
                                 local = True )

                base += stack_size

                mapping.addVseg( vspace, 'mjpeg_iqzz_stack_%d_%d' % (x,y), 
                                 base, stack_size, 'C_WU', vtype = 'BUFFER', 
                                 x = x , y = y , pseg = 'RAM',
                                 local = True )

                base += stack_size

                mapping.addVseg( vspace, 'mjpeg_idct_stack_%d_%d' % (x,y), 
                                 base, stack_size, 'C_WU', vtype = 'BUFFER', 
                                 x = x , y = y , pseg = 'RAM',
                                 local = True )

                base += stack_size

                mapping.addVseg( vspace, 'mjpeg_libu_stack_%d_%d' % (x,y), 
                                 base, stack_size, 'C_WU', vtype = 'BUFFER', 
                                 x = x , y = y , pseg = 'RAM',
                                 local = True )

                base += stack_size

    # threads mapping: demux, vld, iqzz, idct, libu replicated in all clusters
    # main & tg are mapped in cluster[0,0]
    mapping.addThread( vspace, 'main', True, 0, 0, 0,
                       'mjpeg_main_stack',
                       'mjpeg_heap_0_0',
                       0 )                      # index in start_vector

    if ( nprocs == 1 ):
        p_tg    = 0
        p_demux = 0
        p_vld   = 0
        p_iqzz  = 0
        p_idct  = 0
        p_libu  = 0
    elif ( nprocs == 2 ):
        p_tg    = 0
        p_demux = 1
        p_vld   = 1
        p_iqzz  = 1
        p_idct  = 1
        p_libu  = 1
    elif ( nprocs == 3 ):
        p_tg    = 0
        p_demux = 1
        p_vld   = 2
        p_iqzz  = 1
        p_idct  = 1
        p_libu  = 1
    elif ( nprocs == 4 ):
        p_tg    = 0
        p_demux = 1
        p_vld   = 3
        p_iqzz  = 2
        p_idct  = 3
        p_libu  = 2
    
    mapping.addThread( vspace, 'tg', False, 0, 0, p_tg,
                       'mjpeg_tg_stack',
                       'mjpeg_heap_0_0',
                       1 )                      # index in start_vector

    for x in xrange (x_size):
        for y in xrange (y_size):
            if ( mapping.clusters[cluster_id].procs ):

                mapping.addThread( vspace, 'demux_%d_%d' % (x,y), False , x, y, p_demux,
                                   'mjpeg_demux_stack_%d_%d' % (x,y), 
                                   'mjpeg_heap_%d_%d' % (x,y),
                                   2 )   # start_index  

                mapping.addThread( vspace, 'vld_%d_%d' % (x,y), False , x, y, p_vld,
                                   'mjpeg_vld_stack_%d_%d' % (x,y), 
                                   'mjpeg_heap_%d_%d' % (x,y),
                                   3 )   # start_index  

                mapping.addThread( vspace, 'iqzz_%d_%d' % (x,y), False , x, y, p_iqzz,
                                   'mjpeg_iqzz_stack_%d_%d' % (x,y), 
                                   'mjpeg_heap_%d_%d' % (x,y),
                                   4 )   # start_index  

                mapping.addThread( vspace, 'idct_%d_%d' % (x,y), False , x, y, p_idct,
                                   'mjpeg_idct_stack_%d_%d' % (x,y), 
                                   'mjpeg_heap_%d_%d' % (x,y),
                                   5 )   # start_index  

                mapping.addThread( vspace, 'libu_%d_%d' % (x,y), False , x, y, p_libu,
                                   'mjpeg_libu_stack_%d_%d' % (x,y), 
                                   'mjpeg_heap_%d_%d' % (x,y),
                                   6 )   # start_index  

    # extend mapping name
    mapping.name += '_mjpeg'

    return vspace  # useful for test
            
################################ test ############################################

if __name__ == '__main__':

    vspace = extend( Mapping( 'test', 2, 2, 4 ) )
    print vspace.xml()


# Local Variables:
# tab-width: 4;
# c-basic-offset: 4;
# c-file-offsets:((innamespace . 0)(inline-open . 0));
# indent-tabs-mode: nil;
# End:
#
# vim: filetype=python:expandtab:shiftwidth=4:tabstop=4:softtabstop=4

