/////////////////////////////////////////////////////////////////////////////////////////
// File   : libu.c   
// Date   : octobre 2015
// author : Alain Greiner
/////////////////////////////////////////////////////////////////////////////////////////
// This file define the code of the LIBU (line building) thread for MJPEG application.
// This function execute an infinite loop on a stream of decompressed images.
// The image size is fbf_width * fbf_height pixels. Each pixel is one byte.
// The LIBU[index] thread read all blocks of a given image to fill its private
// bufout[index] buffer, that is part of a FBF_CMA chbuf.
// For each image:
// - it checks that buf_out[index] is empty (released by the CMA peripheral).
// - it move all blocks fom the input MWMR channel to the bufout[index] buffer.
// - it request the CMA peripheral to display the image stored in bufout[index]
// - It register the date of display in the date[] array for intrumentation.
/////////////////////////////////////////////////////////////////////////////////////////

#include <stdio.h>
#include <mwmr_channel.h>
#include <stdint.h>
#include <stdlib.h>
#include "mjpeg.h"

// macro to use a shared TTY
#define PRINTF(...)    lock_acquire( &tty_lock ); \
                       giet_tty_printf(__VA_ARGS__);  \
                       lock_release( &tty_lock );

//////////////////////////////////////////////////////////////
__attribute__ ((constructor)) void libu( unsigned int index )
//////////////////////////////////////////////////////////////
{
    mwmr_channel_t* input = idct_2_libu[index];

    uint32_t time;

    uint8_t  bufin[64];
    uint32_t line;
    uint32_t column;

    // get platform parameters
    uint32_t  x_size;
    uint32_t  y_size;
    uint32_t  nprocs;
    giet_procs_number( &x_size , &y_size , &nprocs );

    // get processor coordinates
    unsigned int x , y , p;
    giet_proc_xyp( &x ,&y , &p );

    // private TTY allocation
    // giet_tty_alloc( 0 );

    PRINTF("\n[MJPEG] thread LIBU[%d] starts on P[%d,%d,%d] / trdid = %x\n",
           index , x , y , p, (uint32_t)trdid_libu[index] )

    uint32_t image = index;

    while( image < MAX_IMAGES )  // one image per iteration
    {
        // check CMA buffer empty
        giet_fbf_cma_check( index );

        // two loops on blocks to build image
        for ( line = 0 ; line < nblocks_h ; ++line ) 
        {
            for ( column = 0 ; column < nblocks_w ; ++column )
            {
                // move one block from input MWMR channel (one byte per pixel
                mwmr_read( input , (uint32_t*)bufin , 16 );

                // copy block to cma_buf
                unsigned int i;
                unsigned int src;
                unsigned int dst;
                for ( i = 0; i < 8 ; ++i ) 
                {
                    src = (i * 8);
                    dst = ((line * 8 + i) * fbf_width) + (column * 8);

                    // copy 8 bytes to cma_buf[index]
                    memcpy( &cma_buf[index][dst] , &bufin[src] , 8 );
                }

#if (DEBUG_LIBU > 1)
if ( (index == DEBUG_CLUSTER_INDEX) || (DEBUG_CLUSTER_INDEX == 0XFFFFFFFF) )
{ PRINTF("\nLIBU[%d] copy block[%d] for image %d\n", 
         index, line * nblocks_w + column , image ) }
#endif 
            }

        } // end loops on blocks

        // request CMA to display image
        giet_fbf_cma_display( index );

        // get date of display
        time = giet_proctime();

#if DEBUG_LIBU
if ( (index == DEBUG_CLUSTER_INDEX) || (DEBUG_CLUSTER_INDEX == 0XFFFFFFFF) )
{ PRINTF("\nLIBU[%d] completes image %d at cycle %d\n", index , image , time ) }
#endif
        // register date of display for instrumentation
        date[image] = time;

        image = image + x_size*y_size;

    }  // end while on images

    giet_pthread_exit( "LIBU completed" );

}  // end libu()

