/* 
	File : core_portme.c
*/
/*
	Author : Shay Gal-On, EEMBC
	Legal : TODO!
*/ 
#include "coremark.h"

#include <stdlib.h>
#include <malloc.h>

#if VALIDATION_RUN
volatile ee_s32 seed1_volatile=0x3415;
volatile ee_s32 seed2_volatile=0x3415;
volatile ee_s32 seed3_volatile=0x66;
#endif

#if PERFORMANCE_RUN
volatile ee_s32 seed1_volatile=0x0;
volatile ee_s32 seed2_volatile=0x0;
volatile ee_s32 seed3_volatile=0x66;
#endif

#if PROFILE_RUN
volatile ee_s32 seed1_volatile=0x8;
volatile ee_s32 seed2_volatile=0x8;
volatile ee_s32 seed3_volatile=0x8;
#endif

/* Define : seed4_volatile
	It defines the number of iterations.

	When set to 0, the number of iterations is automatically computed
        during the benchmark's execution
*/
volatile ee_s32 seed4_volatile=ITERATIONS;


/* Define : seed5_volatile
	It defines which algorithms are executed

	When set to 0, all algorithms are executed (matrix, list_join, state)
*/
volatile ee_s32 seed5_volatile=0;


/* Porting : Timing functions
	How to capture time and convert to seconds must be ported to whatever is supported by the platform.

   Define : TIMER_RES_DIVIDER
	Divider to trade off timer resolution and total time that can be measured.

	Use lower values to increase resolution, but make sure that overflow does not occur.
	If there are issues with the return value overflowing, increase this value.
*/
#define TIMER_RES_DIVIDER 1
#define NSECS_PER_SEC 833000000 /* 833 MHz */
#define EE_TICKS_PER_SEC (NSECS_PER_SEC / TIMER_RES_DIVIDER)
#define GETMYTIME(_t) (*_t=giet_proctime())


/* global time variables.*/
static CORE_TICKS start_time_val, stop_time_val;


/* Function : start_time
	This function will be called right before starting the timed portion of the benchmark.

	Implementation may be capturing a system timer (as implemented in the example code) 
	or zeroing some system parameters - e.g. setting the cpu clocks cycles to 0.
*/
void start_time(void) {
	GETMYTIME(&start_time_val );      
}


/* Function : stop_time
	This function will be called right after ending the timed portion of the benchmark.

	Implementation may be capturing a system timer (as implemented in the example code) 
	or other system parameters - e.g. reading the current value of cpu cycles counter.
*/
void stop_time(void) {
	GETMYTIME(&stop_time_val );      
}


/* Function : get_time
	Return an abstract "ticks" number that signifies time on the system.
	
	Actual value returned may be cpu cycles, milliseconds or any other value,
	as long as it can be converted to seconds by <time_in_secs>.
	This methodology is taken to accomodate any hardware or simulated platform.
	The sample implementation returns millisecs by default, 
	and the resolution is controlled by <TIMER_RES_DIVIDER>
*/
CORE_TICKS get_time(void) {
	CORE_TICKS elapsed=(CORE_TICKS)(stop_time_val - start_time_val);
	return elapsed;
}


/* Function : time_in_secs
	Convert the value returned by get_time to seconds.

	The <secs_ret> type is used to accomodate systems with no support for floating point.
	Default implementation implemented by the EE_TICKS_PER_SEC macro above.
*/
secs_ret time_in_secs(CORE_TICKS ticks) {
	secs_ret retval=((secs_ret)ticks) / (secs_ret)EE_TICKS_PER_SEC;
	return retval;
}

ee_u32 default_num_contexts=MULTITHREAD;


/* Function : portable_init
	Target specific initialization code 
	Test for some common mistakes.
*/
void portable_init(core_portable *p, int *argc, char *argv[])
{
	if (sizeof(ee_ptr_int) != sizeof(ee_u8 *)) {
		ee_printf("ERROR! Please define ee_ptr_int to a type that holds a pointer!\n");
	}
	if (sizeof(ee_u32) != 4) {
		ee_printf("ERROR! Please define ee_u32 to a 32b unsigned type!\n");
	}

	// allocate a shared TTY
	giet_tty_alloc(1);

#if MULTITHREAD>1
	unsigned int x_size, y_size, nprocs;	
	giet_procs_number( &x_size, &y_size, &nprocs );
	
	// initialize distributed heaps
	int x, y;
	for (x=0; x < x_size; ++x) {
		for (y=0; y < y_size; ++y) {
			heap_init(x, y);
		}
	}
#else
	// initialize local heap
	unsigned int lx, ly, lp;	
	giet_proc_xyp(&lx, &ly, &lp);
	heap_init(lx, ly);
#endif

	p->portable_id=1;
}


/* Function : portable_fini
	Target specific final code 
*/
void portable_fini(core_portable *p)
{
	core_results *res = (core_results*)((char*)p - 
			(sizeof(core_results) - sizeof(core_portable)));

	int i;
	for (i=0 ; i<default_num_contexts; i++) {
		const CORE_TICKS total_time = res[i].port.stop_time - res[i].port.start_time;
		printf("thread %d: total ticks = %d\n", i, total_time);
	}

	p->portable_id=0;
}


/* Function : __iterate
	Target specific wrapper for the iterate function, which allows to add
        the iterate function in the constructor list

	This is the entry function for the secondary threads
*/
__attribute__ ((constructor))  
void *__iterate(void *pres) {
	core_results *res = (core_results*)pres; 
	GETMYTIME(&(res->port.start_time));
	iterate(pres);
	GETMYTIME(&(res->port.stop_time));
	giet_pthread_exit(NULL);
	return NULL;
}


/* Function : __main
	Target specific wrapper for the main function, which allows to add
        the main function in the constructor list.

	This is the entry function for the main thread
*/
extern MAIN_RETURN_TYPE main(void);
__attribute__ ((constructor))  
void __main() {
	main();
	giet_pthread_exit("Coremark execution completed");
}


/* Function: portable_malloc
	Provide malloc() functionality in a platform specific way.
*/
void *portable_malloc(ee_size_t size) {
	return malloc(size);
}


/* Function: portable_free
	Provide free() functionality in a platform specific way.
*/
void portable_free(void *p) {
	free(p);
}


/* Function: core_start_parallel
	Start benchmarking in a parallel context.
*/
ee_u8 core_start_parallel(core_results *res) {
	return (ee_u8)giet_pthread_create(&(res->port.thread),NULL,__iterate,(void*)res);
}


/* Function: core_stop_parallel
	Stop a parallel context execution of coremark, and gather the results.
*/
ee_u8 core_stop_parallel(core_results *res) {
	return (ee_u8)giet_pthread_join(res->port.thread, NULL);
}
