#!/usr/bin/env python

from mapping import *

###################################################################################
#   file   : coremark.py
#   date   : january 2016
#   author : Cesar FUGUET TORTOLERO
###################################################################################

######################
def extend( mapping ):

    x_size    = mapping.x_size
    y_size    = mapping.y_size
    nprocs    = mapping.nprocs
    x_width   = mapping.x_width
    y_width   = mapping.y_width

    # define vsegs base & size
    code_base  = 0x10000000
    code_size  = 0x00010000     # 64 Kbytes (per cluster)

    data_base  = 0x20000000
    data_size  = 0x00010000     # 64 Kbytes (non replicated)

    heap_base  = 0x30000000
    heap_size  = 0x00200000     # 2M bytes (per cluster)

    stack_base = 0x40000000
    stack_size = 0x00010000     # 64 Kbytes (per thread)

    # create vspace
    vspace = mapping.addVspace( name = 'coremark',
                                startname = 'coremark_data',
                                active = True )

    # data vseg : shared / cluster[0][0]
    mapping.addVseg( vspace, 'coremark_data', data_base , data_size,
                     'C_WU', vtype = 'ELF', x = 0, y = 0, pseg = 'RAM',
                     binpath = 'bin/coremark/appli.elf',
                     local = False )

    # heap vsegs : shared (one per cluster)
    for x in xrange (x_size):
        for y in xrange (y_size):
            cluster_id = (x * y_size) + y
            if ( mapping.clusters[cluster_id].procs ):
                size  = heap_size
                base  = heap_base + (cluster_id * size)
                mapping.addVseg( vspace, 'coremark_heap_%d_%d' %(x,y), base , size,
                                 'C_WU', vtype = 'HEAP', x = x, y = y, pseg = 'RAM',
                                 local = False, big = True )

    # code vsegs : local (one copy per cluster)
    for x in xrange (x_size):
        for y in xrange (y_size):
            cluster_id = (x * y_size) + y
            if ( mapping.clusters[cluster_id].procs ):
                mapping.addVseg( vspace, 'coremark_code_%d_%d' %(x,y),
                                 code_base , code_size,
                                 'CXWU', vtype = 'ELF', x = x, y = y, pseg = 'RAM',
                                 binpath = 'bin/coremark/appli.elf',
                                 local = True )

    # stacks vsegs: local (one stack per thread => nprocs stacks per cluster)
    # Define also the main_stack in cluster[0][0]. The main_stack cannot be
    # local as the main thread passes to workers a local parameter.
    mapping.addVseg( vspace, 'main_stack',
                     stack_base, stack_size, 'C_WU', vtype = 'BUFFER',
                     x = 0 , y = 0 , pseg = 'RAM',
                     local = False )

    for x in xrange (x_size):
        for y in xrange (y_size):
            cluster_id = (x * y_size) + y
            if ( mapping.clusters[cluster_id].procs ):
                for p in xrange( nprocs ):
                    proc_id = (((x * y_size) + y) * nprocs) + p
                    base    = stack_base + (proc_id * stack_size) + stack_size
                    mapping.addVseg( vspace, 'coremark_stack_%d_%d_%d' % (x,y,p),
                                     base, stack_size, 'C_WU', vtype = 'BUFFER',
                                     x = x , y = y , pseg = 'RAM',
                                     local = True )

    # distributed threads / one thread per processor
    # ... plus main on P[0][0][0]
    IS_MAIN = True
    mapping.addThread( vspace, 'main', IS_MAIN, 0, 0, 0,
                       'main_stack',
                       'coremark_heap_0_0',
                       2 )                      # index in start_vector

    for x in xrange (x_size):
        for y in xrange (y_size):
            cluster_id = (x * y_size) + y
            if ( mapping.clusters[cluster_id].procs ):
                for p in xrange( nprocs ):
                    thread_name = 'iterate_%d_%d_%d' % (x,y,p)
                    mapping.addThread( vspace, thread_name, not IS_MAIN, x, y, p,
                                       'coremark_stack_%d_%d_%d' % (x,y,p),
                                       'coremark_heap_%d_%d' % (x,y),
                                       1 )

    # extend mapping name
    mapping.name += '_coremark'

    return vspace  # useful for test

################################ test ############################################

if __name__ == '__main__':
    vspace = extend( Mapping( 'test', 2, 2, 4 ) )
    print vspace.xml()


# Local Variables:
# tab-width: 4;
# c-basic-offset: 4;
# c-file-offsets:((innamespace . 0)(inline-open . 0));
# indent-tabs-mode: nil;
# End:
#
# vim: filetype=python:expandtab:shiftwidth=4:tabstop=4:softtabstop=4

