///////////////////////////////////////////////////////////////////////////////
// File     : sys_handler.h
// Date     : 01/04/2012
// Author   : alain greiner 
// Copyright (c) UPMC-LIP6
///////////////////////////////////////////////////////////////////////////////
// The sys_handler.c and sys_handler.h files are part of the GIET-VM kernel.
// It define the syscall_vector[] (at the end of this file), as well as the 
// associated syscall handlers.
///////////////////////////////////////////////////////////////////////////////

#ifndef _SYS_HANDLER_H
#define _SYS_HANDLER_H

#include "giet_config.h"
#include "kernel_locks.h"
#include "stdio.h"

///////////////////////////////////////////////////////////////////////////////
// Define the possible command values for the giet_pthread_control() syscall
// These define must be synchronized with values in the stdio.c file
///////////////////////////////////////////////////////////////////////////////

#define THREAD_CMD_PAUSE        0
#define THREAD_CMD_RESUME       1
#define THREAD_CMD_CONTEXT      2

///////////////////////////////////////////////////////////////////////////////
// Define the error codes for the syscall handlers
// These define must be synchronized with values in the stdio.c file
///////////////////////////////////////////////////////////////////////////////

#define SYSCALL_OK                               ( 0 )
#define SYSCALL_VSPACE_NOT_FOUND                 (-1 )
#define SYSCALL_THREAD_NOT_FOUND                 (-2 )
#define SYSCALL_NOT_IN_SAME_VSPACE               (-3 )
#define SYSCALL_UNCOHERENT_THREAD_CONTEXT        (-4 )
#define SYSCALL_ILLEGAL_THREAD_COMMAND_TYPE      (-5 )
#define SYSCALL_CANNOT_LOAD_DATA_SEGMENT         (-6 )
#define SYSCALL_THREAD_ALREADY_ACTIVE            (-7 )
#define SYSCALL_MAIN_NOT_FOUND                   (-8 )
#define SYSCALL_APPLI_CANNOT_BE_KILLED           (-9 )
#define SYSCALL_PTHREAD_ARGUMENT_NOT_SUPPORTED   (-10)
#define SYSCALL_ILLEGAL_CLUSTER_COORDINATES      (-11)
#define SYSCALL_VSEG_NOT_FOUND                   (-12)
#define SYSCALL_UNDEFINED_SYSTEM_CALL            (-13)
#define SYSCALL_COPROCESSOR_NOT_FOUND            (-14)
#define SYSCALL_COPROCESSOR_ILLEGAL_MODE         (-15)
#define SYSCALL_COPROCESSOR_NON_ALLOCATED        (-16)
#define SYSCALL_CHANNEL_ALREADY_ALLOCATED        (-17)
#define SYSCALL_NO_CHANNEL_AVAILABLE             (-18)
#define SYSCALL_CHANNEL_NON_ALLOCATED            (-19)
#define SYSCALL_ILLEGAL_ARGUMENT                 (-20)
#define SYSCALL_OUT_OF_KERNEL_HEAP_MEMORY        (-21)
#define SYSCALL_ADDRESS_NON_ALIGNED              (-22)
#define SYSCALL_ADDRESS_NON_USER_ACCESSIBLE      (-23)
#define SYSCALL_MISSING_INITIALISATION           (-24)

///////////////////////////////////////////////////////////////////////////////
//     Syscall Vector Table (indexed by syscall index)
///////////////////////////////////////////////////////////////////////////////

extern const void * _syscall_vector[64];

///////////////////////////////////////////////////////////////////////////////
// This structure is used by the CMA component to move a stream of images 
// from a set of user buffers to the frame buffer in kernel space.
// It contains two chbuf arrays:
// - The SRC chbuf contains <nbufs> buffer descriptors, in user space,
//   that can be distributed (one buffer per cluster) or not.
// - The DST cbuf contains one single buffer, that is the frame buffer.
// Each buffer is described with a 64 bits buffer descriptor:
// - the 26 LSB bits contain bits[31:6] of the status physical address.
// - the 26 following bits contain bits[31:6] of the buffer physical address.
// - the 12 MSB bits contain the common address extension.
// The actual number of user buffers cannot be larger than 256 (at most
// one user buffer per cluster for a 16*16 mesh).
// NB: The user buffers are mapped in user space, but the chbuf descriptor
// contained in this structure is a protected kernel variable.
// This structure must be 64 bytes aligned.
///////////////////////////////////////////////////////////////////////////////

typedef struct fbf_chbuf_s
{
    unsigned long long  fbf_desc;                    // frame buffer descriptor 
    unsigned long long  usr_desc[256];               // user chbuf descriptor 
    unsigned int        nbufs;                       // number of user buffers
} fbf_chbuf_t;   

///////////////////////////////////////////////////////////////////////////////
// This structure is used by the CMA component to move a stream of containers 
// between the NIC chbuf containing 2 buffers, and a kernel chbuf 
// containing up to (X_SIZE * Y_SIZE) buffers (one buffer per cluster).
// The same structure is used for both TX or RX transfers.
// The number of distributed containers can be smaller than (X_SIZE * YSIZE).
// The actual number of buffers used in the chbuf is defined by (xmax * ymax).
// Each buffer is described with a 64 bits buffer descriptor:
// - the 26 LSB bits contain bits[31:6] of the status physical address.
// - the 26 following bits contain bits[31:6] of the buffer physical address.
// - the 12 MSB bits contain the common address extension.
// The <xmax> and <ymax> fields define the actual mesh size.
// This structure must be 64 bytes aligned.
///////////////////////////////////////////////////////////////////////////////

typedef struct nic_chbuf_s
{
    unsigned long long   buf_desc[X_SIZE*Y_SIZE];    // kernel chbuf descriptor
    unsigned int         xmax;                       // nb clusters in a row
    unsigned int         ymax;                       // nb clusters in a column
} nic_chbuf_t;


//////////////////////////////////////////////////////////////////////////////
//           File system related syscall handlers 
//////////////////////////////////////////////////////////////////////////////

extern int _sys_fat_read( unsigned int fd_id,        // file descriptor index
                          unsigned int vaddr,        // buffer vbase     
                          unsigned int count );      // number of bytes

extern int _sys_fat_pread(unsigned int fd_id,        // file descriptor index
                          unsigned int vaddr,        // buffer vbase
                          unsigned int count,        // number of bytes
                          unsigned int offset );     // bytes to skip in file

extern int _sys_fat_write( unsigned int fd_id,       // file descriptor index
                           unsigned int vaddr,       // buffer vbase
                           unsigned int count );     // number of bytes

extern int _sys_fat_mmap( unsigned int fd_id,        // file descriptor index
                          unsigned int count,        // number of pages 
                          unsigned int offset,       // pages to skip in file
                          unsigned int prot );       // protection modes

extern int _sys_fat_munmap( unsigned int vaddr,      // buffer vbase
                            unsigned int count );    // number of pages

extern int _sys_fat_dump( unsigned int type,         // BS/FS/FAT/FILE/DIR
                          char*        pathname,     // file/dir pathname 
                          unsigned int block_id );   // block index in file/dir/fat


//////////////////////////////////////////////////////////////////////////////
//           Applications related syscall handlers 
//////////////////////////////////////////////////////////////////////////////

extern int _sys_kill_application( char* name );

extern int _sys_exec_application( char* name );

extern int _sys_applications_status( char* name );

/////////////////////////////////////////////////////////////////////////////
//          Threads related syscall handlers
/////////////////////////////////////////////////////////////////////////////

extern int _sys_pthread_create( unsigned int*  buffer,
                                void*          attr,
                                void*          function,
                                void*          arg );

extern int _sys_pthread_join( unsigned int  trdid,
                              void*         ptr );

extern int _sys_pthread_kill( unsigned int  trdid,
                              int           signal );

extern int _sys_pthread_exit( void* string);

extern int _sys_pthread_yield();

extern int _sys_pthread_control( unsigned int  command,
                                 char*         vspace_name,
                                 char*         thread_name );

///////////////////////////////////////////////////////////////////////////////
//          Coprocessors related syscall handlers
///////////////////////////////////////////////////////////////////////////////

extern int _sys_coproc_alloc( unsigned int   cluster_xy,
                              unsigned int   coproc_type,
                              unsigned int*  return_info );

extern int _sys_coproc_release( unsigned int cluster_xy,
                                unsigned int coproc_type );

extern int _sys_coproc_channel_init( unsigned int            cluster_xy,
                                     unsigned int            coproc_type,
                                     unsigned int            channel,
                                     giet_coproc_channel_t*  desc );

extern int _sys_coproc_run( unsigned int cluster_xy,
                            unsigned int coproc_type );

extern int _sys_coproc_completed( unsigned int cluster_xy,
                                  unsigned int coproc_type );

///////////////////////////////////////////////////////////////////////////////
//    TTY related syscall handlers
///////////////////////////////////////////////////////////////////////////////

extern int _sys_tty_alloc( unsigned int shared );

extern int _sys_tty_release();

extern int _sys_tty_write( const char*  buffer,
                    unsigned int length,
                    unsigned int channel );

extern int _sys_tty_read(  char*        buffer,
                    unsigned int length,
                    unsigned int channel );

//////////////////////////////////////////////////////////////////////////////
//    TIM related syscall handlers
//////////////////////////////////////////////////////////////////////////////

extern int _sys_tim_alloc();

extern int _sys_tim_release();

extern int _sys_tim_start( unsigned int period );

extern int _sys_tim_stop();

//////////////////////////////////////////////////////////////////////////////
//    NIC related syscall handlers
//////////////////////////////////////////////////////////////////////////////

extern int _sys_nic_alloc( unsigned int is_rx,
                           unsigned int xmax,
                           unsigned int ymax );

extern int _sys_nic_release( unsigned int is_rx );

extern int _sys_nic_start( unsigned int is_rx );

extern int _sys_nic_move( unsigned int is_rx,
                          void*        buffer );

extern int _sys_nic_stop( unsigned int is_rx );

extern int _sys_nic_clear( unsigned int is_rx );

extern int _sys_nic_stats( unsigned int is_rx );

//////////////////////////////////////////////////////////////////////////////
//    FBF related syscall handlers
//////////////////////////////////////////////////////////////////////////////

extern int _sys_fbf_size( unsigned int* width,
                          unsigned int* height );

extern int _sys_fbf_alloc();

extern int _sys_fbf_release();

extern int _sys_fbf_sync_write( unsigned int offset,
                                void*        buffer,
                                unsigned int length );

extern int _sys_fbf_sync_read(  unsigned int offset,
                                void*        buffer,
                                unsigned int length );

extern int _sys_fbf_cma_alloc( unsigned int nbufs );

extern int _sys_fbf_cma_release();

extern int _sys_fbf_cma_init_buf( unsigned int index, 
                                  void*        buf_vaddr, 
                                  void*        sts_vaddr );

extern int _sys_fbf_cma_start();

extern int _sys_fbf_cma_display( unsigned int index );

extern int _sys_fbf_cma_check( unsigned int index );

extern int _sys_fbf_cma_stop();

//////////////////////////////////////////////////////////////////////////////
//    Miscelaneous syscall handlers
//////////////////////////////////////////////////////////////////////////////

extern int _sys_ukn();

extern int _sys_proc_xyp( unsigned int* x,
                          unsigned int* y,
                          unsigned int* p );

extern int _sys_procs_number( unsigned int* x_size,
                              unsigned int* y_size, 
                              unsigned int* nprocs );

extern int _sys_vseg_get_vbase( char*         vspace_name,
                                char*         vseg_name,
                                unsigned int* vbase );

extern int _sys_vseg_get_length( char*         vspace_name, 
                                 char*         vseg_name,
                                 unsigned int* length );

extern int _sys_xy_from_ptr( void*          ptr,
                             unsigned int*  x,
                             unsigned int*  y );

extern int _sys_heap_info( unsigned int* vaddr, 
                           unsigned int* length,
                           unsigned int  x,
                           unsigned int  y ); 

#endif

// Local Variables:
// tab-width: 4
// c-basic-offset: 4
// c-file-offsets:((innamespace . 0)(inline-open . 0))
// indent-tabs-mode: nil
// End:
// vim: filetype=c:expandtab:shiftwidth=4:tabstop=4:softtabstop=4

