///////////////////////////////////////////////////////////////////////////////////
// File     : exc_handler.c
// Date     : 01/04/2012
// Author   : alain greiner and joel porquet
// Copyright (c) UPMC-LIP6
///////////////////////////////////////////////////////////////////////////////////

#include <exc_handler.h>
#include <ctx_handler.h>
#include <sys_handler.h>
#include <utils.h>
#include <tty0.h>

///////////////////////////////////////////////////////////////////////////////////
// Prototypes of exception handlers.
///////////////////////////////////////////////////////////////////////////////////

static void _cause_ukn(unsigned int * regs_table);
static void _cause_adel(unsigned int * regs_table);
static void _cause_ades(unsigned int * regs_table);
static void _cause_ibe(unsigned int * regs_table);
static void _cause_dbe(unsigned int * regs_table);
static void _cause_bp(unsigned int * regs_table);
static void _cause_ri(unsigned int * regs_table);
static void _cause_cpu(unsigned int * regs_table);
static void _cause_ovf(unsigned int * regs_table);

extern void _int_handler();
extern void _sys_handler();

///////////////////////////////////////////////////////////////////////////////////
// Initialize the exception vector indexed by the CR XCODE field
///////////////////////////////////////////////////////////////////////////////////

__attribute__((section(".kdata")))
const _exc_func_t _cause_vector[16] = 
{
    &_int_handler,  /* 0000 : external interrupt */
    &_cause_ukn,    /* 0001 : undefined exception */
    &_cause_ukn,    /* 0010 : undefined exception */
    &_cause_ukn,    /* 0011 : undefined exception */
    &_cause_adel,   /* 0100 : illegal address read exception */
    &_cause_ades,   /* 0101 : illegal address write exception */
    &_cause_ibe,    /* 0110 : instruction bus error exception */
    &_cause_dbe,    /* 0111 : data bus error exception */
    &_sys_handler,  /* 1000 : system call */
    &_cause_bp,     /* 1001 : breakpoint exception */
    &_cause_ri,     /* 1010 : illegal codop exception */
    &_cause_cpu,    /* 1011 : illegal coprocessor access */
    &_cause_ovf,    /* 1100 : arithmetic overflow exception */
    &_cause_ukn,    /* 1101 : undefined exception */
    &_cause_ukn,    /* 1110 : undefined exception */
    &_cause_ukn,    /* 1111 : undefined exception */
};

///////////////////////////////////////////////
static void _display_cause( unsigned int type, unsigned int * regs_table )
{
    unsigned int gpid       = _get_procid();
    unsigned int cluster_xy = gpid >> P_WIDTH;
    unsigned int x          = cluster_xy >> Y_WIDTH;
    unsigned int y          = cluster_xy & ((1 << Y_WIDTH) - 1);
    unsigned int p          = gpid & ((1 << P_WIDTH) - 1);
    unsigned int trdid      = _get_thread_trdid();
    unsigned int ltid       = _get_thread_ltid();

    static_scheduler_t*  psched = (static_scheduler_t*)_get_sched();

    const char * mips32_exc_str[] = { "strange unknown cause  ",
                                      "illegal read address   ",
                                      "illegal write address  ",
                                      "inst bus error         ",
                                      "data bus error         ",
                                      "breakpoint             ",
                                      "reserved instruction   ",
                                      "illegal coproc access  ",
                                      "arithmetic overflow    " };
/*
    Modify format to have one single printf and avoid interleaving 'AG)

    _printf("\n[GIET] Exception for thread %x on processor[%d,%d,%d] at cycle %d\n"
            " - type      : %s\n"
            " - EPC       : %x\n"
            " - BVAR      : %x\n"
            "...Thread desactivated\n",
            trdid , x , y , p , _get_proctime(),
            mips32_exc_str[type], _get_epc(), _get_bvar() );

    _printf("Registers:\n");
    _printf("CR:   %X\tEPC:  %X\tSR:   %X\tCO_HI %X\tC0_LO  %X\n",
            regs_table[33], regs_table[32], regs_table[34], regs_table[36], regs_table[35]);
    _printf("at_1  %X\tv0_2  %X\tv1_3  %X\ta0_4  %X\ta1_5   %X\n",
            regs_table[1], regs_table[2], regs_table[3], regs_table[4], regs_table[5]);
    _printf("a2_6  %X\ta3_7  %X\tt0_8  %X\tt1_9  %X\tt2_10  %X\n",
            regs_table[6], regs_table[7], regs_table[8], regs_table[9], regs_table[10]);
    _printf("t3_11 %X\tt4_12 %X\tt5_13 %X\tt6_14 %X\tt7_15  %X\n",
            regs_table[11], regs_table[12], regs_table[13], regs_table[14], regs_table[15]);
    _printf("s0_16 %X\ts1_17 %X\ts2_18 %X\ts3_19 %X\ts4_20  %X\n",
            regs_table[16], regs_table[17], regs_table[18], regs_table[19], regs_table[20]);
    _printf("s5_21 %X\ts6_22 %X\ts7_23 %X\tt8_24 %X\tt9_25  %X\n",
            regs_table[21], regs_table[22], regs_table[23], regs_table[24], regs_table[25]);
    _printf("gp_28 %X\tsp_29 %X\tfp_30 %X\tra_31 %X\n\n",
            regs_table[28], regs_table[29], regs_table[30], regs_table[31]);
*/
    _printf("\n[GIET] Exception for thread %x on processor[%d,%d,%d] at cycle %d\n"
            "  type : %s\n"
            "CR    %X\tEPC   %X\tSR    %X\tBVAR  %X\n"
            "HI    %X\tLO    %X\n"
            "zero  %X\tat_1  %X\tv0_2  %X\tv1_3  %X\n"
            "a0_4  %X\ta1_5  %X\ta2_6  %X\ta3_7  %X\n"
            "t0_8  %X\tt1_9  %X\tt2_10 %X\tt3_11 %X\n"
            "t4_12 %X\tt5_13 %X\tt6_14 %X\tt7_15 %X\n"
            "s0_16 %X\ts1_17 %X\ts2_18 %X\ts3_19 %X\n"
            "s4_20 %X\ts5_21 %X\ts6_22 %X\ts7_23 %X\n"
            "t8_24 %X\tt9_25 %X\tk0_26 %s\tk1_27 %s\n"
            "gp_28 %X\tsp_29 %X\tfp_30 %X\tra_31 %X\n",
            trdid , x , y , p , _get_proctime(),
            mips32_exc_str[type],
            regs_table[33] , regs_table[32] , regs_table[34] , regs_table[37],
            regs_table[36] , regs_table[35] ,
            0              , regs_table[1]  , regs_table[2]  , regs_table[3] , 
            regs_table[4]  , regs_table[5]  , regs_table[6]  , regs_table[7] ,
            regs_table[8]  , regs_table[9]  , regs_table[10] , regs_table[11],
            regs_table[12] , regs_table[13] , regs_table[14] , regs_table[15],
            regs_table[16] , regs_table[17] , regs_table[18] , regs_table[19],
            regs_table[20] , regs_table[21] , regs_table[22] , regs_table[23],
            regs_table[24] , regs_table[25] , "undefined "   , "undefined "  ,
            regs_table[28] , regs_table[29] , regs_table[30] , regs_table[31] );

    // register KILL signal
    _atomic_or( &psched->context[ltid].slot[CTX_SIGS_ID] , SIGS_MASK_KILL );

    // deschedule calling thread
    unsigned int save_sr;  
    _it_disable( &save_sr );
    _ctx_switch();

}  // end display_cause()

static void _cause_ukn(unsigned int * regs_table)  { _display_cause(0, regs_table); }
static void _cause_adel(unsigned int * regs_table) { _display_cause(1, regs_table); }
static void _cause_ades(unsigned int * regs_table) { _display_cause(2, regs_table); }
static void _cause_ibe(unsigned int * regs_table)  { _display_cause(3, regs_table); }
static void _cause_dbe(unsigned int * regs_table)  { _display_cause(4, regs_table); }
static void _cause_bp(unsigned int * regs_table)   { _display_cause(5, regs_table); }
static void _cause_ri(unsigned int * regs_table)   { _display_cause(6, regs_table); }
static void _cause_cpu(unsigned int * regs_table)  { _display_cause(7, regs_table); }
static void _cause_ovf(unsigned int * regs_table)  { _display_cause(8, regs_table); }

// Local Variables:
// tab-width: 4
// c-basic-offset: 4
// c-file-offsets:((innamespace . 0)(inline-open . 0))
// indent-tabs-mode: nil
// End:
// vim: filetype=c:expandtab:shiftwidth=4:tabstop=4:softtabstop=4

