//////////////////////////////////////////////////////////////////////////////////
// Date     : 01/06/2015
// Authors  : Alain Greiner
// Copyright (c) UPMC-LIP6
//////////////////////////////////////////////////////////////////////////////////
// The fat32.h and fat32.c files define a library of access functions
// to a FAT32 disk on a block device. It is intended to be used by both 
// the boot code and the kernel code.
//////////////////////////////////////////////////////////////////////////////////
// Implementation notes:
// 1. the "lba" (Logical Block Address) is the physical sector index on
//    the block device. The physical sector size is supposed to be 512 bytes.
// 2. the "cluster" variable is actually a cluster index. A cluster contains
//    8 sectors (4K bytes) and the cluster index is a 32 bits word.
// 3. Each file or directory referenced by the software is represented
//    by an "inode". The set of "inodes" is organised as a tree, that is 
//    a sub-tree of the complete file system existing on the block device.
// 4. A given file can be referenced by several software tasks, and each task
//    will use a private handler, called a "file descriptor", allocated by the OS
//    when the task open the file, that is organised as an indexed array.
// 5. This FAT32 library implements (N+1) caches : one private "File_ Cache" 
//    for each referenced file or directory, and a specific "Fat_Cache" for 
//    the FAT itself. Each cache contain a variable number of clusters that are
//    dynamically allocated when they are accessed, and organised as a 64-Tree.
//////////////////////////////////////////////////////////////////////////////////
// General Debug Policy:
// The global variable GIET_DEBUG_FAT is defined in the giet_config.h file.
// The debug is activated if (proctime > GIET_DEBUG_FAT) && (GIET_DEBUG_FAT != 0)
// The GIET_DEBUG_FAT bit 0 defines the level of debug:
//    if   (GIET_DEBUG_FAT & 0x1)    => detailed debug 
//    else                           => external functions only
//////////////////////////////////////////////////////////////////////////////////

#include <giet_config.h>
#include <hard_config.h>
#include <fat32.h>
#include <utils.h>
#include <vmem.h>
#include <kernel_malloc.h>
#include <ctx_handler.h>
#include <bdv_driver.h>
#include <hba_driver.h>
#include <sdc_driver.h>
#include <rdk_driver.h>
#include <mmc_driver.h>
#include <tty0.h>
#include <stdio.h>

//////////////////////////////////////////////////////////////////////////////////
//               Global variables 
//////////////////////////////////////////////////////////////////////////////////

// Fat-Descriptor
__attribute__((section(".kdata")))
fat_desc_t     _fat __attribute__((aligned(64))); 

// buffer used by boot code as a simple cache when scanning FAT
__attribute__((section(".kdata")))
unsigned char  _fat_buffer_fat[4096] __attribute__((aligned(64)));

// buffer used by boot code as a simple cache when scanning a directory in DATA region
__attribute__((section(".kdata")))
unsigned char  _fat_buffer_data[4096] __attribute__((aligned(64)));

// lba of cluster in fat_buffer_fat
__attribute__((section(".kdata")))
unsigned int   _fat_buffer_fat_lba;

// lba of cluster in fat_buffer_data 
__attribute__((section(".kdata")))
unsigned int   _fat_buffer_data_lba;

//////////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////////
//                  Static functions declaration
//////////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////////


#if GIET_DEBUG_FAT

static void _display_fat_descriptor();

static void _display_clusters_list();

#endif

static void _get_name_from_long( unsigned char* buffer, 
                                 char*          name );

static void _get_name_from_short( unsigned char* buffer,
                                  char*          name );

static inline unsigned int _get_levels_from_size( unsigned int size );

static unsigned int _get_name_from_path( char*          pathname,
                                         char*          name,
                                         unsigned int*  nb_read );

static unsigned int _get_last_name( char*   pathname,
                                    char*   name );

static unsigned int _get_fat_entry( unsigned int  cluster,
                                    unsigned int* value );

static unsigned int _set_fat_entry( unsigned int  cluster,
                                    unsigned int  value );

static void _add_inode_in_tree( fat_inode_t*  child,
                                fat_inode_t*  parent );

static void _remove_inode_from_tree( fat_inode_t* inode );

static unsigned int _update_device_from_cache( unsigned int      levels,
                                               fat_cache_node_t* root,
                                               char*             string );

static unsigned int _set_fs_info();

static unsigned int _update_fs_info();

static unsigned int _read_entry( unsigned int    offset,
                                 unsigned int    size,
                                 unsigned char*  buffer,
                                 unsigned int    little_indian );

static unsigned int _cluster_to_lba( unsigned int cluster );

static unsigned int _get_nb_entries( fat_inode_t*   inode,
                                     unsigned int*  nb_entries );

static unsigned int _get_child_from_parent( fat_inode_t*   parent,
                                            char*          name,
                                            fat_inode_t**  inode ); 

static unsigned int _get_inode_from_path( char*          pathname,
                                          fat_inode_t**  inode );

static unsigned int _is_ancestor( fat_inode_t* a,
                                  fat_inode_t* b);

static unsigned int _get_sfn_name( char*          name,
                                   unsigned int*  length,
                                   unsigned int*  nb_lfn,
                                   char*          sfn,
                                   unsigned char* checksum );

static unsigned int _update_dir_entry( fat_inode_t*  inode );

static unsigned int _add_dir_entry( fat_inode_t* child );

static unsigned int _remove_dir_entry( fat_inode_t*  inode );

static void _add_special_directories( fat_inode_t* child );

static unsigned int _one_cluster_allocate( fat_inode_t*   inode, 
                                           unsigned int*  cluster );

static unsigned int _all_clusters_release( fat_inode_t* inode );

static void _release_cache_memory( fat_cache_node_t*  root,
                                   unsigned int       levels );

static fat_cache_node_t* _allocate_one_cache_node( fat_cache_node_t* first_child );

static fat_inode_t* _allocate_one_inode( char*        name,
                                         unsigned int is_dir,
                                         unsigned int cluster,
                                         unsigned int size,
                                         unsigned int count,
                                         unsigned int dentry,
                                         unsigned int cache_allocate );

static void _allocate_one_buffer( fat_inode_t*    inode,
                                  unsigned int    cluster_id,
                                  unsigned int    cluster );

static unsigned int _get_free_cluster( unsigned int*  cluster );

static unsigned int _remove_node_from_fs( fat_inode_t* inode );

static unsigned int _file_info_no_cache( char*          pathname,
                                         unsigned int*  file_cluster,
                                         unsigned int*  file_size );

static unsigned int _next_cluster_no_cache( unsigned int   cluster,
                                            unsigned int*  next );

static inline int get_length( int offset , int length ) { return length; }

static inline int get_offset( int offset , int length ) { return offset; }



//////////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////////
//                  Static functions definition
//////////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////////


#if GIET_DEBUG_FAT
/////////////////////////////////////
static void _display_fat_descriptor()
{
    _printf("\n###############  FAT DESCRIPTOR  ################################"  
            "\nFAT initialized                  %x"
            "\nBlock Size  (bytes)              %x"
            "\nCluster Size  (bytes)            %x"
            "\nFAT region first lba             %x"
            "\nFAT region size (blocks)         %x"
            "\nDATA region first lba            %x"
            "\nDATA region size (blocks)        %x"
            "\nNumber of free clusters          %x"
            "\nFirst free cluster index         %x" 
            "\nFat_cache_levels                 %d" 
            "\n#################################################################\n",
            _fat.initialized,
            _fat.sector_size,
            _fat.cluster_size,
            _fat.fat_lba,
            _fat.fat_sectors,
            _fat.data_lba,
            _fat.data_sectors,
            _fat.free_clusters_number,
            _fat.free_cluster_hint,
            _fat.fat_cache_levels );

} // end _display_fat_descriptor()
#endif


#if GIET_DEBUG_FAT
////////////////////////////////////////////////////////
static void _display_clusters_list( fat_inode_t* inode )
{
    unsigned int next        = 0;
    unsigned int cluster_id  = 0;
    unsigned int current     = inode->cluster;

    _printf("\n --- clusters for <%s> ---\n",
            inode->name );

    while( current < END_OF_CHAIN_CLUSTER_MIN ) 
    {
        if ( (current < 2) || (cluster_id >= 1024) ) 
        {
            _printf("\n[FAT ERROR] in _display_clusters_list()\n");
            _exit();
        }

        _get_fat_entry( current , &next );
        _printf(" > %X", current );
        cluster_id++;
        if ( (cluster_id & 0x7) == 0 ) _printf("\n");
        current = next;
    }
    _printf("\n");
}  // end _display_clusters_list()
#endif



/////////////////////////////////////////////////////////////////////
static inline unsigned int _get_levels_from_size( unsigned int size )
{ 
    if      ( size <= (1<<18) ) return 1;     // 64 clusters == 256 Kbytes
    else if ( size <= (1<<24) ) return 2;     // 64 * 64 clusters => 16 Mbytes
    else if ( size <= (1<<30) ) return 3;     // 64 * 64 * 64 cluster => 1 Gbytes
    else                        return 4;     // 64 * 64 * 64 * 64 clusters 
}



////////////////////////////////////////////////////////
static unsigned int _read_entry( unsigned int    offset,
                                 unsigned int    size,
                                 unsigned char*  buffer,
                                 unsigned int    little_endian )
{
    unsigned int n;
    unsigned int res  = 0;

    if ( little_endian)
    {
        for( n = size ; n > 0 ; n-- ) res = (res<<8) | buffer[offset+n-1];
    }
    else
    {
        for( n = 0 ; n < size ; n++ ) res = (res<<8) | buffer[offset+n];
    }
    return res;

}  // end _read_entry



//////////////////////////////////////////////////////////////////
static inline unsigned int _cluster_to_lba( unsigned int cluster )       
{
    if ( cluster < 2 )
    { 
        _printf("\n[FAT ERROR] _cluster_to_lba(): cluster smaller than 2\n");
        _exit();
    }

   return  ((cluster - 2) << 3) + _fat.data_lba;
}


//////////////////////////////////////////////////////
static inline unsigned char _to_lower(unsigned char c)
{
   if (c >= 'A' && c <= 'Z') return (c | 0x20);
   else                      return c;
}


//////////////////////////////////////////////////////
static inline unsigned char _to_upper(unsigned char c)
{
   if (c >= 'a' && c <= 'z') return (c & ~(0x20));
   else                      return c;
}



///////////////////////////////////////////////////////////////////////////
static unsigned int _get_name_from_path( char*          pathname,  // input
                                         char*          name,      // output
                                         unsigned int*  nb_read )  // input & output	
{
    // skip leading "/" character
    if ( pathname[*nb_read] == '/' ) *nb_read = *nb_read + 1;

    // initialises current indexes
    unsigned int i = *nb_read;
    unsigned int j = 0;
   
    while ( (pathname[i] != '/') && (pathname[i] != 0) )
    {
        name[j++] = pathname[i++];    
        if ( j > NAME_MAX_SIZE ) return 1;
    }

    // set end of string
    name[j] = 0;

    // skip trailing "/" character
    if ( pathname[i] == '/' ) *nb_read += j+1;
    else                      *nb_read += j;

    return 0;
}



////////////////////////////////////////////////////////////////////
static unsigned int _get_last_name( char*   pathname,       // input
                                    char*   name )          // output
{
    unsigned int nb_read = 0;      
    while ( pathname[nb_read] != 0 )
    {
        if ( _get_name_from_path( pathname, name, &nb_read ) ) return 1;
    }

    return 0;
}   // end _get_last_name()



////////////////////////////////////////////////////////////////////////////////
static void _get_name_from_short( unsigned char* buffer,  // input:  SFN dir_entry
                                  char*          name )   // output: name
{
    unsigned int i;
    unsigned int j = 0;

    // get name
    for ( i = 0; i < 8 && buffer[i] != ' '; i++ )
    {
        name[j] = _to_lower( buffer[i] );
        j++;
    }

    // get extension
    for ( i = 8; i < 8 + 3 && buffer[i] != ' '; i++ )
    {
        // we entered the loop so there is an extension. add the dot
        if ( i == 8 )
        {
            name[j] = '.';
            j++;
        }

        name[j] = _to_lower( buffer[i] );
        j++;
    }

    name[j] = '\0';
}

///////////////////////////////////////////////////////////////////////////////
static void _get_name_from_long( unsigned char*  buffer, // input : LFN dir_entry
                                 char*           name )  // output : name
{
    unsigned int   name_offset         = 0;
    unsigned int   buffer_offset       = get_length(LDIR_ORD);
    unsigned int   l_name_1            = get_length(LDIR_NAME_1);
    unsigned int   l_name_2            = get_length(LDIR_NAME_2);
    unsigned int   l_name_3            = get_length(LDIR_NAME_3);
    unsigned int   l_attr              = get_length(LDIR_ATTR);
    unsigned int   l_type              = get_length(LDIR_TYPE);
    unsigned int   l_chksum            = get_length(LDIR_CHKSUM);
    unsigned int   l_rsvd              = get_length(LDIR_RSVD);

    unsigned int j            = 0;
    unsigned int eof          = 0;

    while ( (buffer_offset != DIR_ENTRY_SIZE)  && (!eof) )
    {
        while (j != l_name_1 && !eof )
        {
            if ( (buffer[buffer_offset] == 0x00) || 
                 (buffer[buffer_offset] == 0xFF) )
            {
                eof = 1;
                continue;
            }
            name[name_offset] = buffer[buffer_offset];
            buffer_offset += 2;
            j += 2;
            name_offset++;
        }

        buffer_offset += (l_attr + l_type + l_chksum);
        j = 0;

        while (j != l_name_2 && !eof )
        {
            if ( (buffer[buffer_offset] == 0x00) || 
                 (buffer[buffer_offset] == 0xFF) )
            {
                eof = 1;
                continue;
            }
            name[name_offset] = buffer[buffer_offset];
            buffer_offset += 2;
            j += 2;
            name_offset++;
        }

        buffer_offset += l_rsvd;
        j = 0;

        while (j != l_name_3 && !eof )
        {
            if ( (buffer[buffer_offset] == 0x00) || 
                 (buffer[buffer_offset] == 0xFF) )
            {
                eof = 1;
                continue;
            }
            name[name_offset] = buffer[buffer_offset];
            buffer_offset += 2;
            j += 2;
            name_offset++;
        }
    }
    name[name_offset] = 0;
} // end get_name_from_long()



//////////////////////////////////////////////////////////////////////////////////
static fat_cache_node_t* _allocate_one_cache_node( fat_cache_node_t* first_child )
{
    fat_cache_node_t* cnode;
    unsigned int i;

    cnode = _malloc( sizeof(fat_cache_node_t) );

    cnode->children[0] = first_child;
    for ( i = 1 ; i < 64 ; i++ )
        cnode->children[i] = NULL;

    return cnode;
}   // end _allocate_one_cache_node()



////////////////////////////////////////////////////////////
static fat_inode_t* _allocate_one_inode( char*        name,
                                         unsigned int is_dir,
                                         unsigned int cluster,
                                         unsigned int size, 
                                         unsigned int count,
                                         unsigned int dentry,
                                         unsigned int cache_allocate )
{
    fat_inode_t* new_inode  = _malloc( sizeof(fat_inode_t) );

    new_inode->parent   = NULL;                 // set by _add_inode_in_tree()
    new_inode->next     = NULL;                 // set by _add_inode_in_tree()
    new_inode->child    = NULL;                 // set by _add_inode_in_tree()
    new_inode->cluster  = cluster;
    new_inode->size     = size; 
    new_inode->cache    = NULL;
    new_inode->levels   = 0;
    new_inode->count    = count;
    new_inode->is_dir   = (is_dir != 0);
    new_inode->dentry   = dentry;             

    _strcpy( new_inode->name , name );  

    if ( cache_allocate )
    {
        new_inode->cache    = _allocate_one_cache_node( NULL );
        new_inode->levels   = _get_levels_from_size( size );
    }

    return new_inode;
}   // end _allocate_one_inode()




////////////////////////////////////////////////////
static void _add_inode_in_tree( fat_inode_t*  child,
                                fat_inode_t*  parent )
{
    child->parent = parent;
    child->next   = parent->child;
    parent->child = child;
}   // end _add_inode-in_tree()




//////////////////////////////////////////////////////////
static void _remove_inode_from_tree( fat_inode_t*  inode )
{
    fat_inode_t*  current;
    fat_inode_t*  prev = inode->parent->child;

    if ( inode == prev )  // removed inode is first in its linked list
    {
        inode->parent->child = inode->next;
    }
    else                  // removed inode is not the first
    {
        for( current = prev->next ; current ; current = current->next )
        {
            if ( current == inode )
            {
                prev->next = current->next;
            }
            prev = current;
        }    
    }    
}  // end _delete_one_inode()



/////////////////////////////////////////////////////////////////
static unsigned int _get_fat_entry( unsigned int  cluster,
                                           unsigned int* value )
{
    // compute cluster_id & entry_id in FAT from cluster index
    // a FAT buffer is an array of 1024 unsigned int entries
    unsigned int       cluster_id = cluster >> 10;       
    unsigned int       entry_id   = cluster & 0x3FF;

    // get pointer on the relevant buffer descriptor in FAT cache
    fat_cache_desc_t*  pdesc;
    unsigned int*      buffer;
    if ( _get_fat_cache_buffer( cluster_id, &pdesc ) ) return 1;

    // get value from FAT slot
    buffer = (unsigned int*)pdesc->buffer;
    *value = buffer[entry_id];

    return 0;
}  // end _get_fat_entry()



////////////////////////////////////////////////////////////////
static inline unsigned int _set_fat_entry( unsigned int cluster, 
                                           unsigned int value  )
{
    // compute cluster_id & entry_id in FAT from cluster index
    // a FAT cluster is an array of 1024 unsigned int entries
    unsigned int cluster_id = cluster >> 10;
    unsigned int entry_id   = cluster & 0x3FF;

    // get pointer on the relevant buffer descriptor in FAT cache
    fat_cache_desc_t*  pdesc;
    unsigned int*      buffer; 
    if ( _get_fat_cache_buffer( cluster_id, &pdesc ) ) return 1;           

    // set value into FAT slot
    buffer           = (unsigned int*)pdesc->buffer;
    buffer[entry_id] = value;
    pdesc->dirty     = 1;

    return 0;
} // end _set_fat_entry()



//////////////////////////////////////////////////////
static void _allocate_one_buffer( fat_inode_t*  inode,
                                  unsigned int  cluster_id,
                                  unsigned int  cluster )
{
    // add cache levels if needed
    while ( _get_levels_from_size( (cluster_id + 1) * 4096 ) > inode->levels )
    {

#if GIET_DEBUG_FAT
if ( _get_proctime() > GIET_DEBUG_FAT )
_printf("\n[DEBUG FAT] _allocate_one_buffer(): adding a cache level for %s\n",
        inode->name );
#endif

        inode->cache = _allocate_one_cache_node( inode->cache );
        inode->levels++;
    }

    // search the 64-tree cache from top to bottom 
    fat_cache_node_t*  node   = inode->cache;
    unsigned int       level;

    for ( level = inode->levels; level != 0; level-- )
    {
        // compute child index
        unsigned int index = (cluster_id >> (6*(level-1))) & 0x3F;

        if ( level == 1 )        // last level => children are cluster descriptors
        {
            fat_cache_desc_t* pdesc = (fat_cache_desc_t*)node->children[index];

            if ( pdesc != NULL )      // slot not empty!!!
            {
                _printf("\n[FAT ERROR] in _allocate_one buffer() : slot not empty "
                        "in File-Cache <%s>\n cluster_id = %d / cache = %x / pdesc[0] = %x\n",
                        inode->name , cluster_id , 
                        (unsigned int)node , (unsigned int)pdesc );
                _exit();
            }

#if GIET_DEBUG_FAT
if ( _get_proctime() > GIET_DEBUG_FAT )
_printf("\n[DEBUG FAT] _allocate_one_buffer(): buffer allocated to <%s> for cluster_id %d\n",
        inode->name, cluster_id );
#endif

            // allocate buffer descriptor
            pdesc = _malloc( sizeof(fat_cache_desc_t) );
            pdesc->lba     = _cluster_to_lba( cluster );
            pdesc->buffer  = _malloc( 4096 );
            pdesc->dirty   = 1;
            node->children[index] = pdesc;
        }
        else                      // not last level => children are 64-tree nodes
        {
            fat_cache_node_t* child = (fat_cache_node_t*)node->children[index];
            if ( child == NULL )  // miss 
            {
                // allocate a cache node if miss
                child = _allocate_one_cache_node( NULL );
                node->children[index] = child;    
            }

            // prepare next iteration
            node  = child;
        }
    } // end for
} // end _allocate_one_buffer




///////////////////////////////////////////////////////////////
static unsigned int _get_free_cluster( unsigned int*  cluster )  
{
    // scan FAT to get next free cluster index
    unsigned int current = _fat.free_cluster_hint;
    unsigned int max     = (_fat.data_sectors >> 3);
    unsigned int value;
    while ( current < max )
    {
        // get FAT entry indexed by current
        if ( _get_fat_entry( current , &value ) ) return 1;

        // return if free
        if ( value == FREE_CLUSTER )
        {
            *cluster = current;
            return 0;
        }

        // increment current
        current++;
    }
        
    // return error if not found  
    return 1;

}  // end _get_free_cluster()




//////////////////////////////////////////////////////////////////////////
static unsigned int _update_device_from_cache( unsigned int        levels,
                                               fat_cache_node_t*   root,
                                               char*               string )
{
    unsigned int index;
    unsigned int ret = 0;

    if ( levels == 1 )  // last level => children are buffer descriptors
    {
        for( index = 0 ; index < 64 ; index++ )
        { 
            fat_cache_desc_t* pdesc = root->children[index];
            if ( pdesc != NULL )
            { 
                // update cluster on device if dirty
                if ( pdesc->dirty )
                {
                    if ( _fat_ioc_access( 1,           // descheduling
                                          0,           // to block device
                                          pdesc->lba,
                                          (unsigned int)pdesc->buffer,
                                          8 ) )
                    {
                        _printf("\n[FAT_ERROR] _update_device from_cache(): "
                                " cannot access lba = %x\n", pdesc->lba );
                        ret = 1;
                    }
                    else
                    {
                        pdesc->dirty = 0;

#if GIET_DEBUG_FAT
if ( _get_proctime() > GIET_DEBUG_FAT )
_printf("\n[DEBUG FAT] _update_device_from_cache(): cluster_id = %d for <%s>\n",
        index , string );
#endif

                    }
                }
            }
        }
    }
    else               // not the last level = recursive call on each children
    {
        for( index = 0 ; index < 64 ; index++ )
        { 
            fat_cache_node_t* pnode = root->children[index];
            if ( pnode != NULL )
            {
                if ( _update_device_from_cache( levels - 1,
                                                root->children[index],
                                                string ) ) ret = 1;
            }    
        }
    }
    return ret;
}  // end _update_device_from_cache()



///////////////////////////////////////////////////////////////////
static void _release_cache_memory( fat_cache_node_t*  root,
                                   unsigned int       levels )
{
    unsigned int i;

    if ( levels == 1 )  // last level => children are cluster descriptors
    {
        for( i = 0 ; i < 64 ; i++ )
        { 
            fat_cache_desc_t* pdesc = root->children[i];

            if ( pdesc != NULL )
            { 
                _free( pdesc->buffer );
                _free( pdesc );
                root->children[i] = NULL;
            }
        }
    }
    else               // not the last level = recursive call on each children
    {
        for( i = 0 ; i < 64 ; i++ )
        { 
            fat_cache_node_t* cnode = root->children[i];

            if ( cnode != NULL )
            {
                _release_cache_memory( cnode, levels - 1 );
                _free( cnode );
                root->children[i] = NULL;
            }
        }
    }
}  // end _release_cache_memory()





////////////////////////////////////////////////////////////////
static unsigned int _one_cluster_allocate( fat_inode_t*   inode,
                                           unsigned int*  cluster ) 
{

#if GIET_DEBUG_FAT
if ( _get_proctime() > GIET_DEBUG_FAT )
_printf("\n[DEBUG FAT] _one_cluster_allocate(): enter for <%s>\n", inode->name );
#endif

    // Check free cluster available
    if ( _fat.free_clusters_number == 0 )
    {
        _printf("\n[FAT ERROR] in _one_cluster_allocate(): no more free clusters\n");
        return 1;
    }

    // scan the Fat-Cache to get last allocated cluster index
    unsigned int nb_current_clusters = 0;
    unsigned int current = inode->cluster;
    unsigned int last    = current;
    unsigned int next    = 0;
    unsigned int new     = 0;
    while ( current < END_OF_CHAIN_CLUSTER_MIN )
    {
        // get next cluster
        if ( _get_fat_entry( current , &next ) ) return 1;
        
        // increment number of allocated clusters
        nb_current_clusters++;

        // update loop variables
        last    = current;
        current = next;
    }  

    // allocate one free cluster from FAT
    if ( _get_free_cluster( &new ) ) 
    {
        _printf("\n[FAT ERROR] in _one_cluster_allocate() : no more free clusters\n");
        return 1;
    }

    // allocate one 4K buffer to File-Cache
    _allocate_one_buffer( inode,
                          nb_current_clusters,
                          new );

    // update allocated FAT slot
    if ( _set_fat_entry( new , END_OF_CHAIN_CLUSTER_MAX ) ) return 1;

    // update FAT descriptor global variables
    _fat.free_clusters_number--;
    _fat.free_cluster_hint = new;

    // update cluster chaining
    if ( nb_current_clusters == 0 )  // first cluster : update cluster field in inode
    {
        inode->cluster = new;
    }
    else                             // not the last : update previous last cluster in FAT
    {
        if ( _set_fat_entry( last , new ) ) return 1;
    }

    // update the FAT on device
    if ( _update_device_from_cache( _fat.fat_cache_levels,
                                    _fat.fat_cache_root,
                                    "FAT" ) ) 
    {
        _printf("\n[FAT ERROR] in _one_cluster_allocate() updating FAT on device\n");
        return 1;
    }

    // update FS-INFO sector on device
    if ( _update_fs_info() )
    {
        _printf("\n[FAT ERROR] in _one_cluster_allocate() updating FS-INFO sector on device\n");
        return 1;
    }

#if GIET_DEBUG_FAT
if ( _get_proctime() > GIET_DEBUG_FAT )
_printf("\n[DEBUG FAT] _one_cluster_allocate(): for <%s> cluster = %x\n",
        inode->name , new );
#endif

    // returns allocated cluster index 
    *cluster = new;
    return 0;

}  // end _one_cluster_allocate()

////////////////////////////////////////////////////////////
// recursive function called by _all_clusters_release()
////////////////////////////////////////////////////////////
static unsigned int _cluster_release( unsigned int cluster )
{
    if ( cluster < END_OF_CHAIN_CLUSTER_MIN )  // non terminal case
    {
        // get next cluster
        unsigned int next;
        if ( _get_fat_entry( cluster , &next ) ) return 1;

        // call _cluster_release() on next cluster
        if ( _cluster_release( next ) ) return 1;

        // release cluster
        if ( _set_fat_entry( cluster , FREE_CLUSTER ) ) return 1;

        // Update free_cluster _hint and free_clusters_number in FAT descriptor
        _fat.free_clusters_number++;
        if ( cluster < _fat.free_cluster_hint ) _fat.free_cluster_hint = cluster;
    }       

    // do nothing if terminal case : cluster == END_OF_CHAIN

    return 0;

}  // end _cluster_release()

///////////////////////////////////////////////////////////////
static unsigned int _all_clusters_release( fat_inode_t* inode )
{
    // release recursively all clusters in FAT chaining reverse order
    // starting from last cluster in chain, ending by first.
 
    if ( _cluster_release( inode->cluster ) )
    {
        _printf("\n[FAT ERROR] in _all_clusters_release() releasing clusters\n");
        return 1;
    }

    // update FAT on device
    if ( _update_device_from_cache( _fat.fat_cache_levels,
                                    _fat.fat_cache_root,
                                    "FAT" ) )
    {
        _printf("\n[FAT ERROR] in _all_clusters_release() updating FAT on device\n");
        return 1;
    }

    // update FS-INFO sector on device
    if ( _update_fs_info() )
    {
        _printf("\n[FAT ERROR] in _all_clusters_release() updating FS_INFO sector\n");
        return 1;
    }

    // update cluster field in inode
    inode->cluster = END_OF_CHAIN_CLUSTER_MIN;

#if GIET_DEBUG_FAT
if ( _get_proctime() > GIET_DEBUG_FAT )
_printf("\n[DEBUG FAT] _all_clusters_release() done for file <%s>\n", inode->name );
#endif

    return 0;
}  // end _all_clusters_release()



///////////////////////////////////////////////////////////
static void _add_special_directories( fat_inode_t*  child )
{
    // get File-Cache buffer for child and cluster_id = 0
    fat_cache_desc_t*   pdesc  = (fat_cache_desc_t*)child->cache->children[0];
    unsigned char*      entry;

    unsigned int i;
    unsigned int cluster;

    // set "." entry (32 bytes)
    entry   = pdesc->buffer;
    cluster = child->cluster;
    
    for ( i = 0 ; i < 32 ; i++ )
    {
        if      (i == 0 )     entry[i] = 0x2E;          // SFN
        else if (i <  11)     entry[i] = 0x20;          // SFN
        else if (i == 11)     entry[i] = 0x10;          // ATTR == dir
        else if (i == 20)     entry[i] = cluster>>16;   // cluster.B2
        else if (i == 21)     entry[i] = cluster>>24;   // cluster.B3
        else if (i == 26)     entry[i] = cluster>>0;    // cluster.B0
        else if (i == 27)     entry[i] = cluster>>8;    // cluster.B1
        else                  entry[i] = 0x00;
    }

    // set ".." entry (32 bytes)
    entry   = pdesc->buffer + 32;
    cluster = child->parent->cluster;

    // handling special case when parent is root directory
    if ( cluster == 2 ) cluster = 0;

    for ( i = 0 ; i < 32 ; i++ )
    {
        if      (i <  2 )     entry[i] = 0x2E;          // SFN
        else if (i <  11)     entry[i] = 0x20;          // SFN
        else if (i == 11)     entry[i] = 0x10;          // ATTR == dir
        else if (i == 20)     entry[i] = cluster>>16;   // cluster.B2
        else if (i == 21)     entry[i] = cluster>>24;   // cluster.B3
        else if (i == 26)     entry[i] = cluster>>0;    // cluster.B0
        else if (i == 27)     entry[i] = cluster>>8;    // cluster.B1
        else                  entry[i] = 0x00;
    }
}  // end _add_special_directories



////////////////////////////////////////////////////////////
static unsigned int _is_ancestor( fat_inode_t* a,
                                  fat_inode_t* b )
{
    while ( b )
    {
        if ( a == b )
            return 1;

        b = b->parent;
    }

    return 0;
} // _is_ancestor()



////////////////////////////////////////////////////////////
static unsigned int _get_sfn_name( char*           name,
                                   unsigned int*   length,
                                   unsigned int*   nb_lfn,
                                   char*           sfn,
                                   unsigned char*  checksum )
{
    // compute name length
    unsigned int name_length = _strlen( name );

    // compute prefix and suffix length
    // only the last '.' is taken into account
    unsigned int suffix_length = 0;
    unsigned int prefix_length = 0;
    unsigned int dot_found     = 0;
    unsigned int i;
    for ( i=0 ; i<name_length ; i++ )
    {
        if (name[i] == '.' )
        {
            if ( dot_found ) 
            {
                prefix_length += suffix_length + 1;
                suffix_length =  0;
            }
            else
            {
                dot_found = 1;
            }
        }
        else
        { 
            if ( dot_found) 
            {
                suffix_length++;
            }
            else
            {
                prefix_length++;
            }
        }
    } 

    // build SFN prefix (8bits)
    if (prefix_length <= 8)
    {
        for( i=0 ; i<8 ; i++)
        {
            if ( i<prefix_length ) sfn[i] = _to_upper( name[i] );
            else                   sfn[i] = 0x20;
        }
    }
    else
    {
        for( i=0 ; i<6 ; i++)
        {
            sfn[i] = _to_upper( name[i] );
        }
        sfn[6] = 0x7E;
        sfn[7] = 0x31;
    }

    // build SFN suffix (3 bits)
    if ( suffix_length == 0 )
    {
        sfn[8]  = 0x20;
        sfn[9]  = 0x20;
        sfn[10] = 0x20;
    }
    else if ( suffix_length == 1 )
    {
        sfn[8]  = _to_upper( name[name_length-1] );
        sfn[9]  = 0x20;
        sfn[10] = 0x20;
    }
    else if ( suffix_length == 2 )
    {
        sfn[8]  = _to_upper( name[name_length-2] );
        sfn[9]  = _to_upper( name[name_length-1] );
        sfn[10] = 0x20;
    }
    else
    {
        sfn[8]  = _to_upper( name[name_length-suffix_length] );
        sfn[9]  = _to_upper( name[name_length-suffix_length+1] );
        sfn[10] = _to_upper( name[name_length-suffix_length+2] );
    }

    // compute 8 bits checksum
    unsigned char sum = 0;
    for ( i=0 ; i<11 ; i++ )
    {
        sum = (((sum & 0x01)<<7) | ((sum & 0xFE)>>1)) + sfn[i];
    }
    *checksum = sum;

    // set nb_lfn and length values
    if      ( name_length <= 13 )
    {
        *length  = name_length;
        *nb_lfn  = 1;
        return 0;
    }
    else if ( name_length <= 26 )
    {
        *length  = name_length;
        *nb_lfn  = 2;
        return 0;
    }
    else if ( name_length <= 31 )
    {
        *length  = name_length;
        *nb_lfn  = 3;
        return 0;
    }
    else
    {
        return 1;
    }
}  // _get_sfn_name()




///////////////////////////////////////////////////////////
static unsigned int _get_nb_entries( fat_inode_t*   inode,
                                     unsigned int*  nb_entries )
{
    // scan directory until "end of directory" with two embedded loops:
    // - scan the clusters allocated to this directory
    // - scan the entries to find NO_MORE_ENTRY
    fat_cache_desc_t*  pdesc;                      // pointer on buffer descriptor
    unsigned char*     buffer;                     // 4 Kbytes buffer (one cluster)
    unsigned int       ord;                        // ORD field in directory entry
    unsigned int       attr;                       // ATTR field in directory entry
    unsigned int       cluster_id = 0;             // cluster index in directory
    unsigned int       offset     = 0;             // position in scanned buffer
    unsigned int       found      = 0;             // NO_MORE_ENTRY found
    unsigned int       count      = 0;             // number of valid NORMAL entries

    // loop on clusters allocated to directory
    while ( found == 0 )
    {
        // get one 4 Kytes buffer from File_Cache  
        if ( _get_file_cache_buffer( inode,
                                     cluster_id,
                                     0,          
                                     &pdesc ) )   return 1;
        buffer = pdesc->buffer;
        
        // loop on directory entries in buffer
        while ( (offset < 4096) && (found == 0) )
        {
            attr = _read_entry( DIR_ATTR , buffer + offset , 0 );   
            ord  = _read_entry( LDIR_ORD , buffer + offset , 0 );

            if ( ord == NO_MORE_ENTRY )
            {
                found = 1;
            }  
            else if ( ord == FREE_ENTRY )             // free entry => skip
            {
                offset = offset + 32;
            }
            else if ( attr == ATTR_LONG_NAME_MASK )   // LFN entry => skip
            {
                offset = offset + 32;
            }
            else                                      // NORMAL entry
            {
                offset = offset + 32;
                count++;
            }
        }  // end loop on directory entries

        cluster_id++;
        offset = 0;

    }  // end loop on clusters

    // return nb_entries
    *nb_entries = count;
    
    return 0;
}  // end _get_nb_entries()



////////////////////////////////////////////////////////////
static unsigned int _update_dir_entry( fat_inode_t*  inode )
{  
    // get Cache-File buffer containing the parent directory entry
    // 128 directories entries in one 4 Kbytes buffer
    fat_cache_desc_t*  pdesc;
    unsigned char*     buffer;   
    unsigned int       cluster_id = inode->dentry>>7;
    unsigned int       offset     = (inode->dentry & 0x7F)<<5;

    if ( _get_file_cache_buffer( inode->parent,
                                 cluster_id,
                                 0,
                                 &pdesc ) )    return 1;
    buffer       = pdesc->buffer;
    pdesc->dirty = 1;

    // update size field
    buffer[offset + 28] = inode->size>>0;       // size.B0 
    buffer[offset + 29] = inode->size>>8;       // size.B1 
    buffer[offset + 30] = inode->size>>16;      // size.B2 
    buffer[offset + 31] = inode->size>>24;      // size.B3 

    // update cluster field
    buffer[offset + 26] = inode->cluster>>0;    // cluster.B0 
    buffer[offset + 27] = inode->cluster>>8;    // cluster.B1 
    buffer[offset + 20] = inode->cluster>>16;   // cluster.B2 
    buffer[offset + 21] = inode->cluster>>24;   // cluster.B3 
    
    return 0;
} // end _update_dir_entry()




//////////////////////////////////////////////////////////
static unsigned int _add_dir_entry( fat_inode_t*   child )
{
    // get child attributes
    unsigned int      is_dir  = child->is_dir;     
    unsigned int      size    = child->size;
    unsigned int      cluster = child->cluster;
    fat_inode_t*      parent  = child->parent;

    if ( parent == NULL ) return 1;

    // compute number of required 32 bytes entries to store 
    // the complete child name and a legal 8.3 SFN name.
    unsigned int    length;
    unsigned int    nb_lfn;
    char            sfn[11];
    unsigned char   checksum;
    if ( _get_sfn_name( child->name, 
                        &length,
                        &nb_lfn,
                        sfn,
                        &checksum ) )  return 1;

#if GIET_DEBUG_FAT
if ( _get_proctime() > GIET_DEBUG_FAT )
_printf("\n[DEBUG FAT] _add_dir_entry(): try to add <%s> in <%s> / nb_lfn = %d\n", 
        child->name , parent->name, nb_lfn );
#endif

    // Find end of directory : two embedded loops:
    // - scan the clusters allocated to this directory
    // - scan the entries to find NO_MORE_ENTRY
    fat_cache_desc_t*  pdesc;                      // pointer on buffer descriptor
    unsigned char*     buffer;                     // 4 Kbytes buffer (one cluster)
    unsigned int       cluster_id = 0;             // cluster index in directory
    unsigned int       offset     = 0;             // position in scanned buffer
    unsigned int       found      = 0;             // NO_MORE_ENTRY found

    // loop on clusters allocated to directory
    while ( found == 0 )
    {
        // get the 4 Kytes buffer from File_Cache  
        if ( _get_file_cache_buffer( parent,
                                     cluster_id,
                                     0,
                                     &pdesc ) )   return 1;

        buffer = pdesc->buffer;
        
        // loop on directory entries in buffer
        while ( (offset < 4096) && (found == 0) )
        {
            if ( _read_entry( LDIR_ORD , buffer + offset , 0 ) == NO_MORE_ENTRY )
            {
                found        = 1;
                pdesc->dirty = 1;
            }  
            else
            {
                offset = offset + 32;
            }
        }  // end loop on entries

        if ( found == 0 )
        {
            cluster_id++;
            offset = 0;
        }
    }  // end loop on clusters

#if GIET_DEBUG_FAT
if ( _get_proctime() > GIET_DEBUG_FAT )
_printf("\n[DEBUG FAT] _add_dir_entry(): get NO_MORE directory entry : "
        " buffer = %x / offset = %x / cluster_id = %d\n",
        (unsigned int)buffer , offset , cluster_id );
#endif

    // enter FSM to modify parent directory:
    // The new child requires to write 3, 4, or 5 directory entries.
    // To actually register the new child, we use a 5 steps FSM
    // (one state per entry to be written), that is traversed as:
    //    LFN3 -> LFN2 -> LFN1 -> NORMAL -> NOMORE 
    // The buffer and first directory entry to be  written are identified 
    // by the variables : buffer / cluster_id / offset 

    unsigned char* name  = (unsigned char*)child->name;

    unsigned int step;          // FSM state

    if      ( nb_lfn == 1 ) step = 3;
    else if ( nb_lfn == 2 ) step = 4;
    else if ( nb_lfn == 3 ) step = 5;
    
    unsigned int   i;           // byte index in 32 bytes directory
    unsigned int   c;           // character index in name
    unsigned char* entry;       // buffer + offset;

    while ( step )   
    {
        // get another buffer if required
        if ( offset >= 4096 )  // new buffer required
        {
            if ( _get_file_cache_buffer( parent,
                                         cluster_id + 1,
                                         0,
                                         &pdesc ) )      return 1;
            buffer       = pdesc->buffer;
            pdesc->dirty = 1;
            offset       = 0;
        }

        // compute directory entry address
        entry = buffer + offset;

#if GIET_DEBUG_FAT
if ( _get_proctime() > GIET_DEBUG_FAT )
_printf("\n[DEBUG FAT] _add_dir_entry(): FSM step = %d /"
        " offset = %x / nb_lfn = %d\n", step, offset, nb_lfn );
#endif

        // write one 32 bytes directory entry per iteration
        switch ( step )
        {
            case 5:   // write LFN3 entry
            {
                c = 26;
                // scan the 32 bytes in dir_entry
                for ( i = 0 ; i < 32 ; i++ )
                {
                    if (i == 0)
                    {
                        if ( nb_lfn == 3) entry[i] = 0x43;
                        else              entry[i] = 0x03;
                    }
                    else if ( ( ((i >= 1 ) && (i<=10) && ((i&1)==1))   ||
                                ((i >= 14) && (i<=25) && ((i&1)==0))   ||
                                ((i >= 28) && (i<=31) && ((i&1)==0)) ) &&
                              ( c < length ) )
                    {
                                          entry[i] = name[c];
                                          c++;
                    }
                    else if (i == 11)     entry[i] = 0x0F;
                    else if (i == 13)     entry[i] = checksum;
                    else                  entry[i] = 0x00;
                }
                step--;
                break;
            }
            case 4:   // write LFN2 entry  
            {
                c = 13;
                // scan the 32 bytes in dir_entry
                for ( i = 0 ; i < 32 ; i++ )
                {
                    if (i == 0)
                    {
                        if ( nb_lfn == 2) entry[i] = 0x42;
                        else              entry[i] = 0x02;
                    }
                    else if ( ( ((i >= 1 ) && (i<=10) && ((i&1)==1))   ||
                                ((i >= 14) && (i<=25) && ((i&1)==0))   ||
                                ((i >= 28) && (i<=31) && ((i&1)==0)) ) &&
                              ( c < length ) )
                    {
                                          entry[i] = name[c];
                                          c++;
                    }
                    else if (i == 11)     entry[i] = 0x0F;
                    else if (i == 13)     entry[i] = checksum;
                    else                  entry[i] = 0x00;
                }
                step--;
                break;
            }
            case 3:   // Write LFN1 entry   
            {
                c = 0;
                // scan the 32 bytes in dir_entry
                for ( i = 0 ; i < 32 ; i++ )
                {
                    if (i == 0)
                    {
                        if ( nb_lfn == 1) entry[i] = 0x41;
                        else              entry[i] = 0x01;
                    }
                    else if ( ( ((i >= 1 ) && (i<=10) && ((i&1)==1))   ||
                                ((i >= 14) && (i<=25) && ((i&1)==0))   ||
                                ((i >= 28) && (i<=31) && ((i&1)==0)) ) &&
                              ( c < length ) )
                    {
                                          entry[i] = name[c];
                                          c++;
                    }
                    else if (i == 11)     entry[i] = 0x0F;
                    else if (i == 13)     entry[i] = checksum;
                    else                  entry[i] = 0x00;
                }
                step--;
                break;
            }
            case 2:   // write NORMAL entry     
            {
                // scan the 32 bytes in dir_entry
                for ( i = 0 ; i < 32 ; i++ )
                {
                    if      ( i < 11 )                              // 8.3 SFN
                    {
                                          entry[i] = sfn[i];
                    }
                    else if (i == 11)                               // ATTR
                    {
                        if (is_dir)       entry[i] = 0x10;
                        else              entry[i] = 0x20;
                    }
                    else if (i == 20)     entry[i] = cluster>>16;   // cluster.B2
                    else if (i == 21)     entry[i] = cluster>>24;   // cluster.B3
                    else if (i == 26)     entry[i] = cluster>>0;    // cluster.B0
                    else if (i == 27)     entry[i] = cluster>>8;    // cluster.B1
                    else if (i == 28)     entry[i] = size>>0;       // size.B0
                    else if (i == 29)     entry[i] = size>>8;       // size.B1
                    else if (i == 30)     entry[i] = size>>16;      // size.B2
                    else if (i == 31)     entry[i] = size>>24;      // size.B3
                    else                  entry[i] = 0x00;
                }

                // update the dentry field in child inode
                child->dentry = ((cluster_id<<12) + offset)>>5;

                step--;
                break;
            }
            case 1:   // write NOMORE entry  
            {
                entry [0] = 0x00;
                step--;
                break;
            }
        } // end switch step
        offset += 32;
    } // exit while => exit FSM    

#if GIET_DEBUG_FAT
if ( _get_proctime() > GIET_DEBUG_FAT )
{
    _printf("\n[DEBUG FAT] _add_dir_entry(): <%s> successfully added in <%s>\n",
            child->name , parent->name );
}
#endif

    return 0;        
} // end _add_dir_entry()



////////////////////////////////////////////////////////////
static unsigned int _remove_dir_entry( fat_inode_t*  inode )
{
    // compute number of LFN entries
    unsigned int nb_lfn;
    unsigned int name_length = _strlen( inode->name );
    if      ( name_length <= 13 ) nb_lfn  = 1;
    else if ( name_length <= 26 ) nb_lfn  = 2;
    else                          nb_lfn  = 3;

    // get cluster_id and offset in parent directory cache
    unsigned int  dentry     = inode->dentry;
    unsigned int  cluster_id = dentry >> 7;
    unsigned int  offset     = (dentry & 0x7F)<<5;

    // get buffer from parent directory cache
    unsigned char*     buffer;
    fat_cache_desc_t*  pdesc;

    if ( _get_file_cache_buffer( inode->parent,
                                 cluster_id,
                                 0,
                                 &pdesc ) ) return 1;
    buffer       = pdesc->buffer;
    pdesc->dirty = 1;

    // invalidate NORMAL entry in directory cache
    buffer[offset] = 0xE5;

    // invalidate LFN entries
    while ( nb_lfn )
    {
        if (offset == 0)  // we must load buffer for (cluster_id - 1)
        {
            if ( cluster_id == 0 )
                break;

            if ( _get_file_cache_buffer( inode->parent,
                                         cluster_id - 1,
                                         0,
                                         &pdesc ) )   return 1;
            buffer       = pdesc->buffer;
            pdesc->dirty = 1;
            offset       = 4096;
        }

        offset = offset - 32;

        // check for LFN entry
        if ( _read_entry( DIR_ATTR , buffer + offset , 0 ) != ATTR_LONG_NAME_MASK )
            break;

        // invalidate LFN entry
        buffer[offset] = 0xE5;

        nb_lfn--;
    }     
         
    return 0;
}  // end _remove_dir_entry




//////////////////////////////////////////////////////////////////
static unsigned int _get_child_from_parent( fat_inode_t*   parent,
                                            char*          name, 
                                            fat_inode_t**  inode )
{
    fat_inode_t*   current;

#if GIET_DEBUG_FAT
if ( _get_proctime() > GIET_DEBUG_FAT )
_printf("\n[DEBUG FAT] _get_child_from_parent(): search <%s> in directory <%s>\n",
        name , parent->name );
#endif
    
    // scan inodes in the parent directory 
    for ( current = parent->child ; current ; current = current->next )
    {
        if ( _strcmp( name , current->name ) == 0 )
        {

#if GIET_DEBUG_FAT
if ( _get_proctime() > GIET_DEBUG_FAT )
_printf("\n[DEBUG FAT] _get_child_from_parent(): found inode for <%s> in <%s>\n", 
        name , parent->name );
#endif
            *inode = current;
            return 0;           // name found
        }
    }

    // not found in Inode-Tree => access the parent directory file_cache.
    // Two embedded loops:
    // - scan the clusters allocated to this directory
    // - scan the directory entries in each 4 Kbytes buffer

    unsigned char*    buffer;           // pointer on one cache buffer
    char              cname[32];        // buffer for one full entry name 
    char              lfn1[16];         // buffer for one partial name
    char              lfn2[16];         // buffer for one partial name
    char              lfn3[16];         // buffer for one partial name
    unsigned int      size;             // searched file/dir size (bytes) 
    unsigned int      cluster;          // searched file/dir cluster index
    unsigned int      is_dir;           // searched file/dir type
    unsigned int      attr;             // directory entry ATTR field
    unsigned int      ord;              // directory entry ORD field
    unsigned int      lfn = 0;          // LFN entries number
    unsigned int      dentry;           // directory entry index 
    unsigned int      offset     = 0;   // byte offset in buffer
    unsigned int      cluster_id = 0;   // cluster index in directory
    int               found      = 0;   // not found (0) / name found (1) / end of dir (-1)

#if GIET_DEBUG_FAT
if ( _get_proctime() > GIET_DEBUG_FAT )
_printf("\n[DEBUG FAT] _get_child_from_parent(): child <%s> in <%s> not found in Inode-Tree\n"
        " search in parent cache\n", name , parent->name );
#endif

    // scan the clusters allocated to parent directory
    while ( found == 0 )
    {
        // get one 4 Kytes buffer from parent File_Cache  
        fat_cache_desc_t*  pdesc;
        if ( _get_file_cache_buffer( parent,
                                     cluster_id,
                                     0,
                                     &pdesc ) )    return 2;
        buffer = pdesc->buffer;

        // scan this buffer until end of directory, end of buffer, or name found
        while( (offset < 4096) && (found == 0) )
        {
            attr = _read_entry( DIR_ATTR , buffer + offset , 0 );   
            ord  = _read_entry( LDIR_ORD , buffer + offset , 0 );

            if (ord == NO_MORE_ENTRY)                 // no more entry => break
            {
                found = -1;
            }
            else if ( ord == FREE_ENTRY )             // free entry => skip
            {
                offset = offset + 32;
            }
            else if ( attr == ATTR_LONG_NAME_MASK )   // LFN entry => get partial name
            {
                unsigned int seq = ord & 0x3;
                lfn = (seq > lfn) ? seq : lfn;   
                if      ( seq == 1 ) _get_name_from_long( buffer + offset, lfn1 );
                else if ( seq == 2 ) _get_name_from_long( buffer + offset, lfn2 );
                else if ( seq == 3 ) _get_name_from_long( buffer + offset, lfn3 );
                offset = offset + 32;
            }
            else                                 // NORMAL entry
            {
                // build the extracted name
                if      ( lfn == 0 )
                {
                    _get_name_from_short( buffer + offset , cname );
                }
                else if ( lfn == 1 )
                {
                    _strcpy( cname      , lfn1 );
                }   
                else if ( lfn == 2 ) 
                {
                    _strcpy( cname      , lfn1 );
                    _strcpy( cname + 13 , lfn2 );
                }
                else if ( lfn == 3 ) 
                {
                    _strcpy( cname      , lfn1 );
                    _strcpy( cname + 13 , lfn2 );
                    _strcpy( cname + 26 , lfn3 );
                }

                // get dentry arguments if extracted name == searched name
                if ( _strcmp( name , cname ) == 0 )
                {
                    cluster = (_read_entry( DIR_FST_CLUS_HI , buffer + offset , 1 ) << 16) |
                              (_read_entry( DIR_FST_CLUS_LO , buffer + offset , 1 )      ) ;
                    dentry  = ((cluster_id<<12) + offset)>>5;
                    is_dir  = ((attr & ATTR_DIRECTORY) == ATTR_DIRECTORY);
                    size    = _read_entry( DIR_FILE_SIZE , buffer + offset , 1 );
                    found   = 1;
                }
                offset = offset + 32;
                lfn    = 0;
            }
        }  // end loop on directory entries
        cluster_id++;
        offset = 0;
    }  // end loop on buffers

    if ( found == -1 )  // found end of directory in parent directory 
    {

#if GIET_DEBUG_FAT
if ( _get_proctime() > GIET_DEBUG_FAT )
_printf("\n[DEBUG FAT] _get_child_from_parent(): found end of directory in <%s>\n",
        parent->name );
#endif
        *inode = NULL;
        return 1;
    }
    else               // found searched name in parent directory
    {
        // allocate a new inode and an empty Cache-File
        *inode = _allocate_one_inode( name,
                                      is_dir,
                                      cluster,
                                      size,
                                      0,             // count
                                      dentry,
                                      1 );           // cache_allocate

        // introduce it in Inode-Tree
        _add_inode_in_tree( *inode , parent );

#if GIET_DEBUG_FAT
if ( _get_proctime() > GIET_DEBUG_FAT )
_printf("\n[DEBUG FAT] _get_child_from_parent(): found <%s> on device\n", name );
#endif
        return 0;
    }
}  // end _get_child_from_parent()




//////////////////////////////////////////////////////////////////
static unsigned int _get_inode_from_path( char*          pathname,
                                          fat_inode_t**  inode )
{
    char                 name[32];         // buffer for one name in pathname
    unsigned int         nb_read;	       // number of characters written in name[] 
    fat_inode_t*         parent;           // parent inode
    fat_inode_t*         child;            // child inode
    unsigned int         last;             // while exit condition 
    unsigned int         code;             // return value

#if GIET_DEBUG_FAT
if ( _get_proctime() > GIET_DEBUG_FAT )
_printf("\n[DEBUG FAT] _get_inode_from_path(): enters for path <%s>\n", pathname );
#endif

    // handle root directory case
    if ( _strcmp( pathname , "/" ) == 0 )
    {

#if GIET_DEBUG_FAT
if ( _get_proctime() > GIET_DEBUG_FAT )
_printf("\n[DEBUG FAT] _get_inode_from_path(): found root inode for <%s>\n", 
        pathname );
#endif
        *inode  = _fat.inode_tree_root;
        return 0;
    }

    // If the pathname is not "/", we traverse the inode tree from the root.
    // We use _get_name_from_path() to scan pathname and extract inode names.
    // We use _get_child_from_parent() to scan each directory in the path. 

    last       = 0;
    nb_read    = 0;                      // number of characters analysed in path
    parent     = _fat.inode_tree_root;   // Inode-Tree root 
    
    while ( !last )
    {
        // get searched file/dir name
        if ( _get_name_from_path( pathname, name, &nb_read ) )
        {
            return 3;   // error : name too long
        }

        // compute last iteration condition
        last = (pathname[nb_read] == 0);

#if GIET_DEBUG_FAT
if ( _get_proctime() > GIET_DEBUG_FAT )
_printf("\n[DEBUG FAT] _get_inode_from_path(): got name <%s>\n", name );
#endif

        if ( _strcmp( name, ".." ) == 0)
        {
            // found special name "..", try to go up
            code = 0;
            if ( parent->parent )
                child = parent->parent;
            else
                child = parent;
        }
        else if ( _strcmp( name, "." ) == 0 )
        {
            // found special name ".", stay on the same level
            code = 0;
            child = parent;
        }
        else
        {
            // get child inode from parent directory
            code = _get_child_from_parent( parent,
                                           name,
                                           &child );

            // we need to find the child inode for all non terminal names
            if ( (code == 2) || ((code == 1 ) && !last) )
            {

    #if GIET_DEBUG_FAT
    if ( _get_proctime() > GIET_DEBUG_FAT )
    _printf("\n[DEBUG FAT] _get_inode_from_path(): neither parent, nor child found for <%s>\n",
            pathname );
    #endif
                return 2;  // error : parent inode not found
            }
        }

        // update parent if not the last iteration
        if ( !last )
            parent = child;
    } // end while

    // returns inode pointer
    if (code == 0 )
    {

#if GIET_DEBUG_FAT
if ( _get_proctime() > GIET_DEBUG_FAT )
_printf("\n[DEBUG FAT] _get_inode_from_path(): found inode for <%s>\n", 
        pathname );
#endif
        *inode  = child;
    }
    else
    {

#if GIET_DEBUG_FAT
if ( _get_proctime() > GIET_DEBUG_FAT )
_printf("\n[DEBUG FAT] _get_inode_from_path(): found only parent inode for <%s>\n",
        pathname );
#endif
        *inode  = parent;
    }

    return code;                 // can be 0 (found) or 1 (not found)

}  // end _get_inode_from_path() 




//////////////////////////////////////////////////////////////
static unsigned int _remove_node_from_fs( fat_inode_t* inode )
{
    // check for root node
    if ( !inode->parent ) return 1;

    // remove entry in parent directory
    if ( _remove_dir_entry( inode ) ) return 1;

    // update parent directory on device
    if ( _update_device_from_cache( inode->parent->levels,
                                    inode->parent->cache,
                                    inode->parent->name ) ) return 1;

    // release clusters allocated to file/dir in DATA region
    if ( _all_clusters_release( inode ) ) return 1;

    // release File-Cache
    _release_cache_memory( inode->cache, inode->levels );
    _free ( inode->cache );

    // remove inode from Inode-Tree
    _remove_inode_from_tree( inode );

    // release inode
    _free ( inode );

    return 0;
}  // end _remove_node_from_fs()


//////////////////////////////////////////////////////////////////
static unsigned int _next_cluster_no_cache( unsigned int   cluster,
                                            unsigned int*  next )
{
    // compute cluster_id and slot_id 
    // each cluster contains 1024 slots (4 bytes per slot)
    unsigned int cluster_id  = cluster >> 10;
    unsigned int slot_id     = cluster & 0x3FF;

    // compute lba of cluster identified by cluster_id
    unsigned int lba = _fat.fat_lba + (cluster_id << 3);

    // get cluster containing the adressed FAT slot in FAT buffer
    if ( _fat_buffer_fat_lba != lba )
    {
        if ( _fat_ioc_access( 0,         // no descheduling
                              1,         // read
                              lba,
                              (unsigned int)_fat_buffer_fat,
                              8 ) )
        {
            _printf("\n[FAT ERROR] _next_cluster_no_cache(): "
                    "cannot load lba = %x into fat_buffer\n", lba );
            return 1;
        }

        _fat_buffer_fat_lba = lba;
    }

    // return next cluster index
    unsigned int* buf = (unsigned int*)_fat_buffer_fat;
    *next = buf[slot_id];
    return 0;
   
}  // end _next_cluster_no_cache()




/////////////////////////////////////////////////////////////////
static unsigned int _file_info_no_cache( char*          pathname,
                                         unsigned int*  file_cluster,
                                         unsigned int*  file_size )
{
    
#if GIET_DEBUG_FAT
if ( _get_proctime() > GIET_DEBUG_FAT )
_printf("\n[DEBUG FAT] _file_info_no_cache(): enters for path <%s>\n", pathname );
#endif

    char            name[32];             // buffer for one name in the analysed pathname
    char            lfn1[16];             // buffer for a partial name in LFN entry
    char            lfn2[16];             // buffer for a partial name in LFN entry
    char            lfn3[16];             // buffer for a partial name in LFN entry
    char            cname[32];            // buffer for a full name in a directory entry
    unsigned int    nb_read;           	  // number of characters analysed in path 
    unsigned int    parent_cluster;       // cluster index for the parent directory 
    unsigned int    child_cluster = 0;    // cluster index for the searched file/dir
    unsigned int    child_size = 0;       // size of the searched file/dir
    unsigned int    child_is_dir;         // type of the searched file/dir
    unsigned int    offset;               // offset in a 4 Kbytes buffer
    unsigned int    ord;                  // ORD field in a directory entry
    unsigned int    attr;                 // ATTR field in a directory entry
    unsigned int    lfn = 0;              // number of lfn entries
    unsigned char*  buf;                  // pointer on a 4 Kbytes buffer 
    unsigned int    found;                // name found in current directory entry

    // Three embedded loops: 
    // - scan pathname to extract file/dir names, 
    // - for each name, scan the clusters of the parent directory
    // - for each cluster, scan the 4 Kbytes buffer to find the file/dir name
    // The starting point is the root directory (cluster 2)

    nb_read        = 0;
    parent_cluster = 2; 

    // scan pathname  
    while ( pathname[nb_read] != 0 )    
    {
        // get searched file/dir name
        if ( _get_name_from_path( pathname, name, &nb_read ) ) return 1;

#if GIET_DEBUG_FAT
if ( _get_proctime() > GIET_DEBUG_FAT )
_printf("\n[DEBUG FAT] _file_info_no_cache(): search name <%s>"
        " in cluster %x\n", name , parent_cluster );
#endif
        found  = 0;

        // scan clusters containing the parent directory
        while ( found == 0 )  
        {
            // compute lba 
            unsigned int lba = _cluster_to_lba( parent_cluster );

            // load one cluster of the parent directory into data_buffer
            if ( _fat_buffer_data_lba != lba )
            {
                if ( _fat_ioc_access( 0,         // no descheduling
                                      1,         // read
                                      lba,
                                      (unsigned int)_fat_buffer_data,
                                      8 ) )
                {
                    _printf("\n[FAT ERROR] _file_info_no_cache(): "
                            "cannot load lba = %x into data_buffer\n", lba );
                    return 1;
                }

                _fat_buffer_data_lba = lba;
            }

            offset = 0;

            // scan this 4 Kbytes buffer 
            while ( (offset < 4096) && (found == 0) )
            {
                buf  = _fat_buffer_data + offset;
                attr = _read_entry( DIR_ATTR , buf , 0 );   
                ord  = _read_entry( LDIR_ORD , buf , 0 );

                if (ord == NO_MORE_ENTRY)               // no more entry => break
                {
                    found = 2;
                }
                else if ( ord == FREE_ENTRY )           // free entry => skip
                {
                    offset = offset + 32;
                }
                else if ( attr == ATTR_LONG_NAME_MASK ) // LFN entry => get partial name
                {
                    unsigned int seq = ord & 0x3;
                    lfn = (seq > lfn) ? seq : lfn;   
                    if      ( seq == 1 ) _get_name_from_long( buf, lfn1 );
                    else if ( seq == 2 ) _get_name_from_long( buf, lfn2 );
                    else if ( seq == 3 ) _get_name_from_long( buf, lfn3 );
                    offset = offset + 32;
                }
                else                                    // NORMAL entry
                {
                    // build the full mame for current directory entry
                    if      ( lfn == 0 )
                    {
                        _get_name_from_short( buf , cname );
                    }
                    else if ( lfn == 1 )
                    {
                        _strcpy( cname      , lfn1 );
                    }   
                    else if ( lfn == 2 ) 
                    {
                        _strcpy( cname      , lfn1 );
                        _strcpy( cname + 13 , lfn2 );
                    }
                    else if ( lfn == 3 ) 
                    {
                        _strcpy( cname      , lfn1 );
                        _strcpy( cname + 13 , lfn2 );
                        _strcpy( cname + 26 , lfn3 );
                    }
                    
                    // test if extracted name == searched name
                    if ( _strcmp( name , cname ) == 0 )
                    {
                        child_cluster = (_read_entry( DIR_FST_CLUS_HI , buf , 1 ) << 16) |
                                        (_read_entry( DIR_FST_CLUS_LO , buf , 1 )      ) ;
                        child_is_dir  = ((attr & ATTR_DIRECTORY) == ATTR_DIRECTORY);
                        child_size    = _read_entry( DIR_FILE_SIZE , buf , 1 );
                        found         = 1;
                    }
                    offset = offset + 32;
                    lfn = 0;
                }
            }  // en loop on directory entries
            
            // compute next cluster index
            unsigned int next;
            if ( _next_cluster_no_cache ( parent_cluster , &next ) ) return 1;
            parent_cluster = next;
        } // end loop on clusters 

        if ( found == 2 )  // found end of directory => error
        { 
            _printf("\n[FAT ERROR] _file_info_no_cache(): <%s> not found\n",
                    name );
            return 1;
        }
 
        // check type
        if ( ((pathname[nb_read] == 0) && (child_is_dir != 0)) ||
             ((pathname[nb_read] != 0) && (child_is_dir == 0)) )
        {
            _printf("\n[FAT ERROR] _file_info_no_cache(): illegal type for <%s>\n", name );
            return 1;
        }

        // update parent_cluster for next name
        parent_cluster = child_cluster;

    }  // end loop on names

#if GIET_DEBUG_FAT
if ( _get_proctime() > GIET_DEBUG_FAT )
_printf("\n[DEBUG FAT] _file_info_no_cache(): success for <%s> / "
        "file_size = %x / file_cluster = %x\n", pathname, child_size, child_cluster );
#endif

    // return file cluster and size
    *file_size    = child_size;
    *file_cluster = child_cluster;
    return 0;

}  // end _file_info_no_cache()



/////////////////////////////
unsigned int _set_fs_info()
{
    // load FS_INFO sector into FAT buffer
    if ( _fat_ioc_access( 0,                                // no descheduling 
                          1,                                // read
                          _fat.fs_info_lba,                 // lba 
                          (unsigned int)_fat.block_buffer,
                          1 ) )                             // one block
    { 
        _printf("\n[FAT ERROR] _set_fs_info(): cannot load FS_INFO Sector\n"); 
        return 1;
    }
    _fat.block_buffer_lba = _fat.fs_info_lba;

    // get general info from FAT descriptor
    unsigned int  data_blocks  = _fat.data_sectors;

    // initialise <free_clusters_number> from FS-INFO sector
    unsigned int free_clusters = _read_entry( FS_FREE_CLUSTERS, _fat.block_buffer, 1);
    if ( free_clusters >= (data_blocks>>3) )
    {
        _printf("\n[FAT ERROR] _set_fs_info(): illegal FS_FREE_CLUSTERS in FS-INFO\n"
                "  fs_free_clusters = %x / total_clusters = %x\n",
                free_clusters , (data_blocks>>3)  ); 
        return 1;
    }

    _fat.free_clusters_number  = free_clusters;

    // initialise <free_cluster_hint> from FS_INFO sector 
    unsigned int free_cluster_hint = _read_entry( FS_FREE_CLUSTER_HINT, _fat.block_buffer, 1);     
    if ( free_cluster_hint > (data_blocks>>3) )
    {
        _printf("\n[FAT ERROR] _set_fs_info(): illegal FS_FREE_CLUSTER_HINT in FS-INFO\n" 
                "  fs_free_cluster_hint = %x / total_clusters = %x\n",
                free_cluster_hint , (data_blocks>>3)  ); 
        return 1;
    }

    _fat.free_cluster_hint  = free_cluster_hint;

#if GIET_DEBUG_FAT
if ( _get_proctime() > GIET_DEBUG_FAT )
_printf("\n[DEBUG FAT] _set_fs_info() : free_clusters = %x / free_cluster_hint = %x\n",
        free_clusters , free_cluster_hint );
#endif

    return 0;
    
}  // end _set_fs_info()




/////////////////////////////////////
static unsigned int _update_fs_info()
{
    // load buffer if miss
    if ( _fat.fs_info_lba != _fat.block_buffer_lba )
    {
        if ( _fat_ioc_access( 1,                 // descheduling
                              1,                 // read
                              _fat.fs_info_lba, 
                              (unsigned int)_fat.block_buffer, 
                              1 ) )              // one block
        {
            _printf("\n[FAT_ERROR] _update_fs_info(): cannot read block\n");
            return 1;
        }
        _fat.block_buffer_lba = _fat.fs_info_lba;
    }

    // update buffer
    unsigned int* ptr;

    ptr  = (unsigned int*)(_fat.block_buffer + get_offset(FS_FREE_CLUSTERS) );
    *ptr = _fat.free_clusters_number;

    ptr  = (unsigned int*)(_fat.block_buffer + get_offset(FS_FREE_CLUSTER_HINT) );
    *ptr = _fat.free_cluster_hint;
   
    // write bloc to FAT
    if ( _fat_ioc_access( 1,                // descheduling
                          0,                // write
                          _fat.fs_info_lba,
                          (unsigned int)_fat.block_buffer, 
                          1 ) )             // one block
    {
        _printf("\n[FAT_ERROR] _update_fs_info(): cannot write block\n");
        return 1;
    }

#if GIET_DEBUG_FAT
if ( _get_proctime() > GIET_DEBUG_FAT )
_printf("\n[DEBUG FAT] _update_fs_info() : free_clusters = %x / free_cluster_hint = %x\n",
        _fat.free_clusters_number , _fat.free_cluster_hint );
#endif

    return 0;
}  // end _update_fs_info()



///////////////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////////
//             Extern functions                                               
///////////////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////////

/////////////////////////////////////////////////////////////////////////////
int _fat_ioc_access( unsigned int use_irq,       // descheduling if non zero
                     unsigned int to_mem,        // read / write
                     unsigned int lba,           // first sector on device
                     unsigned int buf_vaddr,     // memory buffer vaddr
                     unsigned int count )        // number of sectors
{
    // compute memory buffer physical address
    unsigned int       flags;         // for _v2p_translate
    unsigned long long buf_paddr;     // buffer physical address 

    if ( ((_get_mmu_mode() & 0x4) == 0 ) || USE_IOC_RDK )  // identity
    { 
        buf_paddr = (unsigned long long)buf_vaddr;
    }
    else                                // V2P translation required
    {
        buf_paddr = _v2p_translate( buf_vaddr , &flags );
    }

#if (GIET_DEBUG_FAT & 1)
if ( _get_proctime() > GIET_DEBUG_FAT )
_printf("\n[DEBUG FAT] _fat_ioc_access(): enters at cycle %d\n"
        "  to_mem = %d / vaddr = %x / paddr = %l / sectors = %d / lba = %x\n",
        _get_proctime(), to_mem, buf_vaddr, buf_paddr, count, lba );
#endif


#if GIET_NO_HARD_CC     // L1 cache inval (virtual addresses)
    if ( to_mem ) _dcache_buf_invalidate( buf_vaddr, count<<9 );
#endif


#if   ( USE_IOC_BDV )   // call the proper driver
    return( _bdv_access( use_irq , to_mem , lba , buf_paddr , count ) ); 
#elif ( USE_IOC_HBA )
    return( _hba_access( use_irq , to_mem , lba , buf_paddr , count ) );
#elif ( USE_IOC_SDC )
    return( _sdc_access( use_irq , to_mem , lba , buf_paddr , count ) );
#elif ( USE_IOC_SPI )
    return( _spi_access( use_irq , to_mem , lba , buf_paddr , count ) );
#elif ( USE_IOC_RDK )
    return( _rdk_access( use_irq , to_mem , lba , buf_paddr , count ) );
#else
    _printf("\n[FAT ERROR] _fat_ioc_access(): no IOC driver\n");
    _exit();
#endif

}  // end _fat_ioc_access()



/////////////////////////////////////////
int _fat_init( unsigned int kernel_mode )  
{

#if GIET_DEBUG_FAT
if ( _get_proctime() > GIET_DEBUG_FAT )
_printf("\n[DEBUG FAT] _fat_init(): enters at cycle %d\n", _get_proctime() );
#endif

    // FAT initialisation should be done only once
    if ( _fat.initialized == FAT_INITIALIZED )
    {
        _printf("\n[FAT WARNING] _fat_init(): FAT already initialized\n");
        return GIET_FAT32_OK;
    }

    // load Boot sector (VBR) into FAT buffer
    if ( _fat_ioc_access( 0,                                  // no descheduling
                          1,                                  // read
                          0,                                  // block index
                          (unsigned int)_fat.block_buffer,
                          1 ) )                               // one block 
    {
        _printf("\n[FAT ERROR] _fat_init(): cannot load VBR\n");
        return GIET_FAT32_IO_ERROR;
    }

    _fat.block_buffer_lba = 0;
    
#if GIET_DEBUG_FAT 
if ( _get_proctime() > GIET_DEBUG_FAT )
{
    _printf("\n[DEBUG FAT] _fat_init(): Boot sector loaded\n");
}
#endif

    // checking various FAT32 assuptions from boot sector
    if( _read_entry( BPB_BYTSPERSEC, _fat.block_buffer, 1 ) != 512 )
    {
        _printf("\n[FAT ERROR] _fat_init(): The sector size must be 512 bytes\n");
        return GIET_FAT32_INVALID_BOOT_SECTOR;
    }
    if( _read_entry( BPB_SECPERCLUS, _fat.block_buffer, 1 ) != 8 )
    {
        _printf("\n[FAT ERROR] _fat_init(): The cluster size must be 8 blocks\n");
        return GIET_FAT32_INVALID_BOOT_SECTOR;
    }
    if( _read_entry( BPB_NUMFATS, _fat.block_buffer, 1 ) != 1 )
    {
        _printf("\n[FAT ERROR] _fat_init(): The number of FAT copies in FAT region must be 1\n");
        return GIET_FAT32_INVALID_BOOT_SECTOR;
    }
    if( (_read_entry( BPB_FAT32_FATSZ32, _fat.block_buffer, 1 ) & 0xF) != 0 )
    {
        _printf("\n[FAT ERROR] _fat_init(): The FAT region must be multiple of 16 sectors\n");
        return GIET_FAT32_INVALID_BOOT_SECTOR;
    }
    if( _read_entry( BPB_FAT32_ROOTCLUS, _fat.block_buffer, 1 ) != 2 )
    {
        _printf("\n[FAT ERROR] _fat_init(): The root directory must be at cluster 2\n");
        return GIET_FAT32_INVALID_BOOT_SECTOR;
    }

    // initialise Fat-Descriptor from VBR
    _fat.sector_size         = 512;
    _fat.cluster_size        = 4096;
    _fat.fat_sectors         = _read_entry( BPB_FAT32_FATSZ32 , _fat.block_buffer , 1 );
    _fat.fat_lba             = _read_entry( BPB_RSVDSECCNT , _fat.block_buffer , 1 );
    _fat.data_sectors        = _fat.fat_sectors << 10;
    _fat.data_lba            = _fat.fat_lba + _fat.fat_sectors;
    _fat.fs_info_lba         = _read_entry( BPB_FAT32_FSINFO , _fat.block_buffer , 1 );
    _fat_buffer_fat_lba      = 0xFFFFFFFF;
    _fat_buffer_data_lba     = 0xFFFFFFFF;
    _fat.initialized         = FAT_INITIALIZED;

    /////////////////////////////////////////////////////////////////////
    // This is done only when the _fat_init() is called in kernel mode

    if ( kernel_mode )
    {
        // initialise <free_clusters_number> and <first_free_cluster in FAT descriptor
        if ( _set_fs_info() ) return GIET_FAT32_IO_ERROR;

        // create Inode-Tree root
        _fat.inode_tree_root = _allocate_one_inode("/",   // dir name
                                                   1,     // is directory
                                                   2,     // cluster index
                                                   4096,  // at least one buffer
                                                   0,     // no children
                                                   0,     // no dentry
                                                   1);    // allocate cache

        // initialize lock
        _spin_lock_init( &_fat.fat_lock );

        // initialize File Descriptor Array
        unsigned int i;
        for( i = 0 ; i < GIET_OPEN_FILES_MAX ; i++ ) _fat.fd[i].allocated = 0;

        // initialize fat_cache root
        _fat.fat_cache_root   = _allocate_one_cache_node( NULL );
        _fat.fat_cache_levels = _get_levels_from_size( _fat.fat_sectors << 9 );
    }  // end if kernel_mode

#if GIET_DEBUG_FAT
if ( _get_proctime() > GIET_DEBUG_FAT )
_display_fat_descriptor();
#endif

    return GIET_FAT32_OK;
}  // end _fat_init()




////////////////////////////////////////////////////////////////////
int _fat_open( char*        pathname,      // absolute path from root
               unsigned int flags )        // O_CREAT / O_RDONLY / O_TRUNC
{
    unsigned int         fd_id;            // index in File-Descriptor-Array
    unsigned int         code;             // error code
    fat_inode_t*         inode;            // anonymous inode pointer
    fat_inode_t*         child;            // pointer on searched file inode
    fat_inode_t*         parent;           // pointer on parent directory inode
    
    // get flags
    unsigned int create    = ((flags & O_CREAT)  != 0);
    unsigned int read_only = ((flags & O_RDONLY) != 0);
    unsigned int truncate  = ((flags & O_TRUNC)  != 0);
    unsigned int append    = ((flags & O_APPEND) != 0);

#if GIET_DEBUG_FAT
    unsigned int procid  = _get_procid();
    unsigned int x       = procid >> (Y_WIDTH + P_WIDTH);
    unsigned int y       = (procid >> P_WIDTH) & ((1 << Y_WIDTH) - 1);
    unsigned int p       = procid & ((1 << P_WIDTH) - 1);
    if ( _get_proctime() > GIET_DEBUG_FAT )
        _printf("\n[DEBUG FAT] _fat_open(): P[%d,%d,%d] enters for path <%s>\n"
                " create = %d / read_only = %d / truncate = %d\n",
                x, y, p, pathname , create , read_only , truncate );
#endif

    // checking FAT initialized
    if ( _fat.initialized != FAT_INITIALIZED )
    {
        _printf("\n[FAT ERROR] _fat_open(): FAT not initialized\n");
        return GIET_FAT32_NOT_INITIALIZED;
    }

    // takes the FAT lock and register it in thread context
    static_scheduler_t*  psched = _get_sched();
    unsigned int         ltid   = _get_thread_ltid();
    _spin_lock_acquire( &_fat.fat_lock );
    _atomic_or( &psched->context[ltid].slot[CTX_LOCKS_ID] , LOCKS_MASK_FAT ); 

    // get inode pointer
    code = _get_inode_from_path( pathname , &inode );

    if ( code == 2 )                          // parent inode not found
    {
        _spin_lock_release( &_fat.fat_lock );
        _atomic_and( &psched->context[ltid].slot[CTX_LOCKS_ID] , ~LOCKS_MASK_FAT ); 

        _printf("\n[FAT ERROR] _fat_open(): path to parent not found"
                " for file <%s>\n", pathname );
        return GIET_FAT32_FILE_NOT_FOUND;
    }
    else if ( code == 3 )                     // illegal path name
    {
        _spin_lock_release( &_fat.fat_lock );
        _atomic_and( &psched->context[ltid].slot[CTX_LOCKS_ID] , ~LOCKS_MASK_FAT ); 

        _printf("\n[FAT ERROR] _fat_open(): one name in path too long"
                " for file <%s>\n", pathname );
        return GIET_FAT32_NAME_TOO_LONG;
    }
    else if ( (code == 1) && (create == 0) )   // child inode not found
    {
        _spin_lock_release( &_fat.fat_lock );
        _atomic_and( &psched->context[ltid].slot[CTX_LOCKS_ID] , ~LOCKS_MASK_FAT ); 

        _printf("\n[FAT ERROR] _fat_open(): file not found"
                " for file <%s>\n", pathname );
        return GIET_FAT32_FILE_NOT_FOUND;
    }
    else if ( (code == 1) && (create != 0) )   // child inode not found => create 
    {
        // set parent inode pointer
        parent = inode;

#if GIET_DEBUG_FAT
        if ( _get_proctime() > GIET_DEBUG_FAT )
            _printf("\n[DEBUG FAT] _fat_open(): P[%d,%d,%d] create a new file <%s>\n",
                    x , y , p , pathname );
#endif

        // get new file name / error check already done by _get_inode_from_path()
        char name[32];        
        _get_last_name( pathname , name );

        // allocate a new inode and an empty Cache-File 
        child = _allocate_one_inode( name,
                                     0,                         // not a directory
                                     END_OF_CHAIN_CLUSTER_MAX,  // no cluster allocated
                                     0,                         // size : new file is empty
                                     0,                         // count incremented later
                                     0,                         // set by add_dir_entry
                                     1 );                       // cache_allocate

        // introduce inode into Inode-Tree
        _add_inode_in_tree( child , parent );

        // add an entry in the parent directory Cache_file
        // and update the dentry field in child inode
        if ( _add_dir_entry( child ) )
        {
            _spin_lock_release( &_fat.fat_lock );
            _atomic_and( &psched->context[ltid].slot[CTX_LOCKS_ID] , ~LOCKS_MASK_FAT ); 

            _printf("\n[FAT ERROR] _fat_open(): cannot update parent directory"
                    " for file <%s>\n" , pathname );
            return GIET_FAT32_IO_ERROR;
        } 

        // update DATA region on block device for parent directory
        if ( _update_device_from_cache( parent->levels,
                                        parent->cache,
                                        parent->name ) )
        {
            _spin_lock_release( &_fat.fat_lock );
            _atomic_and( &psched->context[ltid].slot[CTX_LOCKS_ID] , ~LOCKS_MASK_FAT ); 

            _printf("\n[FAT ERROR] _fat_open(): cannot update DATA region "
                    " for parent of file <%s>\n", pathname );
            return GIET_FAT32_IO_ERROR;
        }

        // update FAT region on block device
        if ( _update_device_from_cache( _fat.fat_cache_levels,
                                        _fat.fat_cache_root,
                                        "FAT" ) )
        {
            _spin_lock_release( &_fat.fat_lock );
            _atomic_and( &psched->context[ltid].slot[CTX_LOCKS_ID] , ~LOCKS_MASK_FAT ); 

            _printf("\n[FAT ERROR] _fat_open(): cannot update FAT region"
                    " for file <%s>\n", pathname );
            return GIET_FAT32_IO_ERROR;
        }

        // no need to truncate a new file
        truncate = 0;

#if GIET_DEBUG_FAT
        if ( _get_proctime() > GIET_DEBUG_FAT )
        {
            _printf("\n[DEBUG FAT] _fat_open() : new inode created for <%s>\n" 
                    " size = %x / cluster = %x / cache = %x",
                    child->name , child->size , child->cluster , child->cache );
            if ( child->cache != NULL )
            {
                _printf(" / pdesc[0] = %x\n", (unsigned int)(child->cache->children[0]) );
            }
            else
            {
                _printf("\n");
            }
        }
#endif

    }
    else                                    // inode found
    {
        // set searched file inode pointer
        child = inode;

#if GIET_DEBUG_FAT
        if ( _get_proctime() > GIET_DEBUG_FAT )
        {
            _printf("\n[DEBUG FAT] _fat_open(): P[%d,%d,%d] found file <%s>\n"
                    " inode = %x / size = %x\n",
                    x , y , p , pathname , (unsigned int)child , child->size );

            _display_clusters_list( child );
        }
#endif

    }

    // Search an empty slot in file descriptors array
    fd_id = 0;
    while ( (_fat.fd[fd_id].allocated) != 0 && (fd_id < GIET_OPEN_FILES_MAX) )
    {
        fd_id++;
    }

    // check if an empty slot has been found
    if ( fd_id >= GIET_OPEN_FILES_MAX )
    {
        _spin_lock_release( &_fat.fat_lock );
        _atomic_and( &psched->context[ltid].slot[CTX_LOCKS_ID] , ~LOCKS_MASK_FAT ); 

        _printf("\n[FAT ERROR] _fat_open(): File-Descriptors-Array full\n");
        return GIET_FAT32_TOO_MANY_OPEN_FILES;
    }

    // truncate the file if requested
    if ( truncate && !read_only && !child->is_dir && child->size != 0 )
    {
        // release File-Cache (keep root node)
        _release_cache_memory( child->cache, child->levels );

        // release clusters allocated to file/dir in DATA region
        if ( _all_clusters_release( child ) )
        {
            _spin_lock_release( &_fat.fat_lock );
            _atomic_and( &psched->context[ltid].slot[CTX_LOCKS_ID] , ~LOCKS_MASK_FAT ); 

            _printf("\n[FAT ERROR] _fat_open(): can't truncate file\n");
            return GIET_FAT32_IO_ERROR;
        }

        // update parent directory entry (size and cluster index)
        if ( _update_dir_entry( child ) )
        {
            _spin_lock_release( &_fat.fat_lock );
            _atomic_and( &psched->context[ltid].slot[CTX_LOCKS_ID] , ~LOCKS_MASK_FAT ); 

            _printf("\n[FAT ERROR] _fat_open(): can't truncate file\n");
            return GIET_FAT32_IO_ERROR;
        }

        // update inode
        child->size   = 0;
        child->levels = 1;
    }

    // update file descriptor
    _fat.fd[fd_id].allocated  = 1;
    _fat.fd[fd_id].read_only  = read_only;
    _fat.fd[fd_id].inode      = child;
    _fat.fd[fd_id].seek       = ( append ) ? child->size : 0;

    // increment the refcount
    child->count = child->count + 1;

#if GIET_DEBUG_FAT
    if ( _get_proctime() > GIET_DEBUG_FAT )
        _printf("\n[DEBUG FAT] _fat_open(): P[%d,%d,%d] get fd = %d for <%s>\n"
                " inode = %x / offset = %x / read_only = %d / size = %x / cluster = %x\n",
                x , y , p , fd_id , pathname , 
                (unsigned int)_fat.fd[fd_id].inode,
                _fat.fd[fd_id].seek,
                _fat.fd[fd_id].read_only,
                _fat.fd[fd_id].inode->size,
                _fat.fd[fd_id].inode->cluster );
#endif

    // releases the lock
    _spin_lock_release( &_fat.fat_lock );
    _atomic_and( &psched->context[ltid].slot[CTX_LOCKS_ID] , ~LOCKS_MASK_FAT ); 

    return fd_id;
} // end _fat_open()




////////////////////////////////////
int _fat_close( unsigned int fd_id )
{
    // checking FAT initialized
    if( _fat.initialized != FAT_INITIALIZED )
    {
        _printf("\n[FAT ERROR] _fat_close(): FAT not initialized\n");
        return GIET_FAT32_NOT_INITIALIZED;
    }

    if( (fd_id >= GIET_OPEN_FILES_MAX) )
    {
        _printf("\n[FAT ERROR] _fat_close(): illegal file descriptor index\n");
        return GIET_FAT32_INVALID_FD;
    } 

    // takes the FAT lock and register it in thread context
    static_scheduler_t*  psched = _get_sched();
    unsigned int         ltid   = _get_thread_ltid();
    _spin_lock_acquire( &_fat.fat_lock );
    _atomic_or( &psched->context[ltid].slot[CTX_LOCKS_ID] , LOCKS_MASK_FAT ); 

    if( _fat.fd[fd_id].allocated == 0 )
    {
        _spin_lock_release( &_fat.fat_lock );
        _atomic_and( &psched->context[ltid].slot[CTX_LOCKS_ID] , ~LOCKS_MASK_FAT ); 

        _printf("\n[FAT ERROR] _fat_close(): file not open\n");
        return GIET_FAT32_NOT_OPEN;
    }

    // get the inode pointer 
    fat_inode_t*  inode = _fat.fd[fd_id].inode;

#if GIET_DEBUG_FAT
if ( _get_proctime() > GIET_DEBUG_FAT )
_printf("\n[FAT DEBUG] _fat_close() for file <%s> : refcount = %d"
        " / size = %x / cluster = %x\n",
        inode->name , inode->count , inode->size , inode->cluster );
#endif

    // decrement reference count
    inode->count = inode->count - 1;
    
    // update block device and release File-Cache if no more references
    if ( inode->count == 0 )
    {
        // update all dirty clusters for closed file
        if ( _update_device_from_cache( inode->levels, 
                                        inode->cache,
                                        inode->name ) ) 
        {
            _spin_lock_release( &_fat.fat_lock );
            _atomic_and( &psched->context[ltid].slot[CTX_LOCKS_ID] , ~LOCKS_MASK_FAT ); 

            _printf("\n[FAT ERROR] _fat_close(): cannot write dirty clusters "
                    "for file <%s>\n", inode->name );
            return GIET_FAT32_IO_ERROR;
        }

#if GIET_DEBUG_FAT
if ( _get_proctime() > GIET_DEBUG_FAT )
_printf("\n[FAT DEBUG] _fat_close() updated device for file <%s>\n", inode->name );
#endif

        // update dirty clusters for parent directory
        if ( inode->parent &&
             _update_device_from_cache( inode->parent->levels,
                                        inode->parent->cache,
                                        inode->parent->name ) )
        {
            _spin_lock_release( &_fat.fat_lock );
            _atomic_and( &psched->context[ltid].slot[CTX_LOCKS_ID] , ~LOCKS_MASK_FAT ); 

            _printf("\n[FAT ERROR] _fat_close(): cannot write dirty clusters "
                    "for directory <%s>\n", inode->parent->name );
            return GIET_FAT32_IO_ERROR;
        }

#if GIET_DEBUG_FAT
if ( _get_proctime() > GIET_DEBUG_FAT )
_printf("\n[FAT DEBUG] _fat_close() updated device for parent directory <%s>\n",
        inode->parent->name );
#endif

        // release memory allocated to File-Cache (keep cache root node)
        _release_cache_memory( inode->cache, inode->levels );

#if GIET_DEBUG_FAT
if ( _get_proctime() > GIET_DEBUG_FAT )
_printf("\n[FAT DEBUG] _fat_close() release memory for File-Cache <%s>\n",
        inode->name );
#endif

    }  // end if (refcount == 0)


    // release fd_id entry in file descriptor array
    _fat.fd[fd_id].allocated = 0;

    // release lock
    _spin_lock_release( &_fat.fat_lock );
    _atomic_and( &psched->context[ltid].slot[CTX_LOCKS_ID] , ~LOCKS_MASK_FAT ); 

    return GIET_FAT32_OK;
} // end fat_close()




////////////////////////////////////////////
int _fat_file_info( unsigned int     fd_id,
                    fat_file_info_t* info )
{
    if ( _fat.initialized != FAT_INITIALIZED )
    {
        _printf("\n[FAT ERROR] _fat_file_info(): FAT not initialized\n");
        return GIET_FAT32_NOT_INITIALIZED;
    }

    if ( fd_id >= GIET_OPEN_FILES_MAX )
    {
        _printf("\n[FAT ERROR] _fat_file_info(): illegal file descriptor index\n");
        return GIET_FAT32_INVALID_FD;
    } 

    if ( _fat.fd[fd_id].allocated == 0 )
    {
        _printf("\n[FAT ERROR] _fat_file_info(): file not open\n");
        return GIET_FAT32_NOT_OPEN;
    }

    info->size   = _fat.fd[fd_id].inode->size;
    info->offset = _fat.fd[fd_id].seek;
    info->is_dir = _fat.fd[fd_id].inode->is_dir;

    return GIET_FAT32_OK;
} // end _fat_file_info()




/////////////////////////////////////////////////////////////////////
int _fat_read( unsigned int fd_id,          // file descriptor index
               unsigned int vaddr,          // destination buffer vaddr
               unsigned int count,          // number of bytes to read
               unsigned int extend,         // physical address extension
               unsigned int offset,         // forced file offset
               unsigned int modes )         // special modes
{

#if GIET_DEBUG_FAT
unsigned int procid  = _get_procid();
unsigned int x       = procid >> (Y_WIDTH + P_WIDTH);
unsigned int y       = (procid >> P_WIDTH) & ((1<<Y_WIDTH)-1);
unsigned int p       = procid & ((1<<P_WIDTH)-1);
if ( _get_proctime() > GIET_DEBUG_FAT )
_printf("\n[DEBUG FAT] _fat_read(): P[%d,%d,%d] enters at cycle %d\n"
        "  fd = %d / vaddr = %x / bytes = %x / extend = %x / forced_offset = %x\n",
        x , y , p , _get_proctime(),
        fd_id , vaddr , count , extend , offset );
#endif

    // checking FAT initialized
    if( _fat.initialized != FAT_INITIALIZED )
    {
        _printf("\n[FAT ERROR] in _fat_read(): FAT not initialized\n");
        return GIET_FAT32_NOT_INITIALIZED;
    }

    // check fd_id overflow
    if ( fd_id >= GIET_OPEN_FILES_MAX )
    {
        _printf("\n[FAT ERROR] in _fat_read(): illegal file descriptor\n");
        return GIET_FAT32_INVALID_FD;
    }

    // check file open
    if ( _fat.fd[fd_id].allocated == 0 )
    {
        _printf("\n[FAT ERROR] in _fat_read(): file not open\n");
        return GIET_FAT32_NOT_OPEN;
    }

    // takes the FAT lock and register it in thread context
    static_scheduler_t*  psched = _get_sched();
    unsigned int         ltid   = _get_thread_ltid();
    _spin_lock_acquire( &_fat.fat_lock );
    _atomic_or( &psched->context[ltid].slot[CTX_LOCKS_ID] , LOCKS_MASK_FAT ); 
           
    // get special modes
    unsigned int physical_addressing = modes & FAT_PADDR_MODE;
    unsigned int forced_offset       = modes & FAT_FORCED_OFFSET;

    // get file inode pointer and offset
    fat_inode_t* inode  = _fat.fd[fd_id].inode;
    unsigned int seek   = forced_offset ? offset : _fat.fd[fd_id].seek;

    // check seek versus file size
    if ( (seek >= inode->size) && !inode->is_dir )
    {
        _spin_lock_release( &_fat.fat_lock );
        _atomic_and( &psched->context[ltid].slot[CTX_LOCKS_ID] , ~LOCKS_MASK_FAT ); 

        _printf("\n[FAT ERROR] in _fat_read(): offset larger than file size"
                " / seek = %x / file_size = %x\n",
                seek , inode->size );
        return GIET_FAT32_IO_ERROR;
    }

    // check and ajust count argument for a file
    if ( (count > (inode->size - seek)) && !inode->is_dir ) count = inode->size - seek;

    // compute first_cluster_id and first_byte_to_move 
    unsigned int first_cluster_id   = seek >> 12;
    unsigned int first_byte_to_move = seek & 0xFFF;   

    // compute last_cluster and last_byte_to_move 
    unsigned int last_cluster_id   = (seek + count - 1) >> 12;   
    unsigned int last_byte_to_move = (seek + count - 1) & 0xFFF;

#if GIET_DEBUG_FAT
if ( _get_proctime() > GIET_DEBUG_FAT )
_printf("\n[DEBUG FAT] _fat_read(): P[%d,%d,%d] search file <%s> with seek = %x\n "
        " first_cluster_id = %x / first_byte_to_move = %x"
        " / last_cluster_id = %x / last_byte_to_move = %x\n",
        x , y , p , inode->name , seek ,
        first_cluster_id , first_byte_to_move , last_cluster_id , last_byte_to_move );
#endif

    // loop on all cluster covering the requested transfer
    unsigned int cluster_id;
    unsigned int done = 0;
    for ( cluster_id = first_cluster_id ; cluster_id <= last_cluster_id ; cluster_id++ )
    {
        // get pointer on the cluster_id buffer in cache 
        unsigned char*     cbuf;
        fat_cache_desc_t*  pdesc;
        if ( _get_file_cache_buffer( inode, 
                                     cluster_id,
                                     0,
                                     &pdesc ) )
        {
            _spin_lock_release( &_fat.fat_lock );
            _atomic_and( &psched->context[ltid].slot[CTX_LOCKS_ID] , ~LOCKS_MASK_FAT ); 

            _printf("\n[FAT ERROR] in _fat_read(): cannot load file <%s>\n",
                    inode->name );
            return GIET_FAT32_IO_ERROR;
        }
        cbuf = pdesc->buffer;

#if GIET_DEBUG_FAT
if ( _get_proctime() > GIET_DEBUG_FAT )
_printf("\n[DEBUG FAT] _fat_read(): P[%d,%d,%d] moves cluster_id %d from Cache-File <%s>\n",
        x , y , p , cluster_id, inode->name );
#endif

        // compute memcpy arguments
        unsigned char*  source;
        unsigned int    nbytes;

        if ( (cluster_id == first_cluster_id) && (cluster_id == last_cluster_id) )
        {
            source = cbuf + first_byte_to_move; 
            nbytes = last_byte_to_move - first_byte_to_move + 1;
        }
        else if ( cluster_id == first_cluster_id )
        {
            source = cbuf + first_byte_to_move; 
            nbytes = 4096 - first_byte_to_move;
        }
        else if ( cluster_id == last_cluster_id )
        {
            source = cbuf; 
            nbytes = last_byte_to_move + 1;
        }
        else  // not first / not last
        {
            source = cbuf; 
            nbytes = 4096;
        }

        // move data 
        if ( physical_addressing == 0 )           // no physical addressing
        {
            char* dest = (char*)(vaddr + done);
            memcpy( dest , source , nbytes );
        }
        else                                      // physical addressing required
        {
            unsigned int flags;
            paddr_t pdest    = (((paddr_t)extend)<<32) + vaddr + done;
            paddr_t psource  = _v2p_translate( (unsigned int)source, &flags );
            _physical_memcpy( pdest , psource , nbytes );
        }

        done = done + nbytes;
    }

#if GIET_DEBUG_FAT
if ( _get_proctime() > GIET_DEBUG_FAT )
_printf("\n[DEBUG FAT] _fat_read(): P[%d,%d,%d] loaded file <%s> from Cache-File\n",
        x , y , p , inode->name );
#endif

    // update seek if required
    if ( forced_offset == 0 ) _fat.fd[fd_id].seek += done;

    // release lock
    _spin_lock_release( &_fat.fat_lock );
    _atomic_and( &psched->context[ltid].slot[CTX_LOCKS_ID] , ~LOCKS_MASK_FAT ); 

    return done;
} // end _fat_read()




////////////////////////////////////////////////////////////////
int _fat_write( unsigned int fd_id,    // file descriptor index
                unsigned int vaddr,    // source buffer vaddr
                unsigned int count,    // number of bytes to write
                unsigned int extend,   // physical address extension
                unsigned int modes )   // special modes
{
    // checking FAT initialized
    if( _fat.initialized != FAT_INITIALIZED )
    {
        _printf("\n[FAT ERROR] _fat_write(): FAT not initialized\n");
        return GIET_FAT32_NOT_INITIALIZED;
    }

    // takes the FAT lock and register it in thread context
    static_scheduler_t*  psched = _get_sched();
    unsigned int         ltid   = _get_thread_ltid();
    _spin_lock_acquire( &_fat.fat_lock );
    _atomic_or( &psched->context[ltid].slot[CTX_LOCKS_ID] , LOCKS_MASK_FAT ); 

           
    // check fd_id overflow
    if ( fd_id >= GIET_OPEN_FILES_MAX )
    {
        _spin_lock_release( &_fat.fat_lock );
        _atomic_and( &psched->context[ltid].slot[CTX_LOCKS_ID] , ~LOCKS_MASK_FAT ); 

        _printf("\n[FAT ERROR] _fat_write(): illegal file descriptor\n");
        return GIET_FAT32_INVALID_FD;
    }

    // check file open
    if ( _fat.fd[fd_id].allocated == 0 )
    {
        _spin_lock_release( &_fat.fat_lock );
        _atomic_and( &psched->context[ltid].slot[CTX_LOCKS_ID] , ~LOCKS_MASK_FAT ); 

        _printf("\n[FAT ERROR] _fat_write(): file not open\n" );
        return GIET_FAT32_NOT_OPEN;
    }

    // check file writable
    if ( _fat.fd[fd_id].read_only )
    {
        _spin_lock_release( &_fat.fat_lock );
        _atomic_and( &psched->context[ltid].slot[CTX_LOCKS_ID] , ~LOCKS_MASK_FAT ); 

        _printf("\n[FAT ERROR] _fat_write(): file <%s> is read-only\n",
                _fat.fd[fd_id].inode->name );
        return GIET_FAT32_READ_ONLY;
    }

    // get special modes
    unsigned int physical_addressing = modes & FAT_PADDR_MODE;

    // get file inode pointer and seek 
    fat_inode_t* inode  = _fat.fd[fd_id].inode;
    unsigned int seek   = _fat.fd[fd_id].seek;

#if GIET_DEBUG_FAT
unsigned int procid  = _get_procid();
unsigned int x       = procid >> (Y_WIDTH + P_WIDTH);
unsigned int y       = (procid >> P_WIDTH) & ((1<<Y_WIDTH)-1);
unsigned int p       = procid & ((1<<P_WIDTH)-1);
if ( _get_proctime() > GIET_DEBUG_FAT )
_printf("\n[DEBUG FAT] _fat_write(): P[%d,%d,%d] enters for file <%s> "
        " / bytes = %x / seek = %x\n",
        x , y , p , inode->name , count , seek );
#endif

    // check if file size must be incremented
    // and allocate new clusters from FAT if required
    unsigned int old_size = inode->size;
    unsigned int new_size = seek + count;
    if ( new_size > old_size )
    {
        // compute current and required numbers of clusters
        unsigned old_clusters = old_size >> 12;
        if ( old_size & 0xFFF ) old_clusters++;

        unsigned new_clusters = new_size >> 12;
        if ( new_size & 0xFFF ) new_clusters++;

        // allocate new clusters from FAT if required
        if ( new_clusters > old_clusters )
        {

#if GIET_DEBUG_FAT
if ( _get_proctime() > GIET_DEBUG_FAT )
_printf("\n[DEBUG FAT] _fat_write(): P[%d,%d,%d] allocates new clusters for file <%s>"
        " / current = %d / required = %d\n",
        x , y , p , inode->name , old_clusters , new_clusters );
#endif
            // allocate missing clusters
            unsigned int cid;
            unsigned int index;  // unused
            for ( cid = 0 ; cid < (new_clusters - old_clusters) ; cid++ )
            {
                if ( _one_cluster_allocate( inode , &index ) )
                {
                    _spin_lock_release( &_fat.fat_lock );
                    _atomic_and( &psched->context[ltid].slot[CTX_LOCKS_ID] ,
                                 ~LOCKS_MASK_FAT ); 

                    _printf("\n[FAT ERROR] in _fat_write(): no free cluster"
                            " for file <%s>\n", _fat.fd[fd_id].inode->name );
                    return GIET_FAT32_NO_FREE_SPACE;
                }
            }
        }
         
        // update size in inode
        inode->size = new_size;
 
        // update parent directory entry (size and cluster index)
        if ( _update_dir_entry( inode ) )
        {
            _spin_lock_release( &_fat.fat_lock );
            _atomic_and( &psched->context[ltid].slot[CTX_LOCKS_ID] , ~LOCKS_MASK_FAT ); 

            _printf("\n[FAT ERROR] _fat_write(): cannot update parent directory entry"
                    " for file <%s>\n", _fat.fd[fd_id].inode->name );
            return GIET_FAT32_IO_ERROR;
        }
            

#if GIET_DEBUG_FAT
if ( _get_proctime() > GIET_DEBUG_FAT )
_printf("\n[DEBUG FAT] _fat_write(): P[%d,%d,%d] updates size for file <%s> / size = %x\n",
        x , y , p , inode->name , (new_size - old_size) );
#endif

    }

    // compute first_cluster_id and first_byte_to_move 
    unsigned int first_cluster_id   = seek >> 12;
    unsigned int first_byte_to_move = seek & 0xFFF;   

    // compute last_cluster and last_byte_to_move 
    unsigned int last_cluster_id   = (seek + count - 1) >> 12;   
    unsigned int last_byte_to_move = (seek + count - 1) & 0xFFF;

#if GIET_DEBUG_FAT
if ( _get_proctime() > GIET_DEBUG_FAT )
_printf("\n[DEBUG FAT] _fat_write(): P[%d,%d,%d] starts loop on clusters for file <%s>\n"
        "  first_cluster_id = %d / first_byte_to_move = %x"
        " / last_cluster_id = %d / last_byte_to_move = %x\n",
        x , y , p , inode->name ,
        first_cluster_id , first_byte_to_move , last_cluster_id , last_byte_to_move );
#endif

    // loop on all clusters covering the requested transfer
    unsigned int cluster_id;
    unsigned int done = 0;
    for ( cluster_id = first_cluster_id ; cluster_id <= last_cluster_id ; cluster_id++ )
    {
        // get pointer on one 4K buffer in File-Cache 
        unsigned char*     cbuf;
        fat_cache_desc_t*  pdesc;
        if ( _get_file_cache_buffer( inode,   
                                     cluster_id,  
                                     0,
                                     &pdesc ) )   
        {
            _spin_lock_release( &_fat.fat_lock );
            _atomic_and( &psched->context[ltid].slot[CTX_LOCKS_ID] , ~LOCKS_MASK_FAT ); 

            _printf("\n[FAT ERROR] _fat_write(): cannot load file <%s>\n",
                    inode->name );
            return GIET_FAT32_IO_ERROR;
        }
        
        cbuf         = pdesc->buffer;
        pdesc->dirty = 1;
   
#if GIET_DEBUG_FAT
if ( _get_proctime() > GIET_DEBUG_FAT )
_printf("\n[DEBUG FAT] _fat_write(): P[%d,%d,%d] move cluster_id %d to Cache-file <%s>\n",
        x , y , p , cluster_id, inode->name );
#endif

        // compute memcpy arguments
        unsigned char* dest;
        unsigned int   nbytes;
        if ( (cluster_id == first_cluster_id) && (cluster_id == last_cluster_id) )
        {
            dest   = cbuf + first_byte_to_move; 
            nbytes = last_byte_to_move - first_byte_to_move + 1;
        }
        else if ( cluster_id == first_cluster_id )
        {
            dest   = cbuf + first_byte_to_move; 
            nbytes = 4096 - first_byte_to_move;
        }
        else if ( cluster_id == last_cluster_id )
        {
            dest   = cbuf; 
            nbytes = last_byte_to_move + 1;
        }
        else
        {
            dest   = cbuf; 
            nbytes = 4096;
        }

        // move data
        if ( physical_addressing == 0 )     // no physical addressing
        {
            char* source = (char*)(vaddr + done);
            memcpy( dest , source , nbytes ); 
        }
        else                                  // physical addressing required
        {
            unsigned int flags;
            paddr_t      psource = (((paddr_t)extend)<<32) + vaddr + done;
            paddr_t      pdest   = _v2p_translate( (unsigned int)dest , &flags );
            _physical_memcpy( pdest , psource , nbytes );
        }

        done = done + nbytes;

    } // end for clusters

    // update seek
    _fat.fd[fd_id].seek += done;

#if GIET_DEBUG_FAT
if ( _get_proctime() > GIET_DEBUG_FAT )
_printf("\n[DEBUG FAT] _fat_write(): P[%d,%d,%d] store file <%s> into Cache-File\n",
        x , y , p , inode->name );
#endif

    // release lock
    _spin_lock_release( &_fat.fat_lock );
    _atomic_and( &psched->context[ltid].slot[CTX_LOCKS_ID] , ~LOCKS_MASK_FAT ); 

    return done;
} // end _fat_write()



///////////////////////////////////
int _fat_lseek( unsigned int fd_id,
                unsigned int seek,
                unsigned int whence )
{
    // checking FAT initialized
    if( _fat.initialized != FAT_INITIALIZED )
    {
        _printf("\n[FAT ERROR] _fat_lseek(): FAT not initialized\n");
        return GIET_FAT32_NOT_INITIALIZED;
    }

    // check fd_id overflow
    if ( fd_id >= GIET_OPEN_FILES_MAX )
    {
        _printf("\n[FAT ERROR] _fat_lseek(): illegal file descriptor\n");
        return GIET_FAT32_INVALID_FD;
    }

    // takes the FAT lock and register it in thread context
    static_scheduler_t*  psched = _get_sched();
    unsigned int         ltid   = _get_thread_ltid();
    _spin_lock_acquire( &_fat.fat_lock );
    _atomic_or( &psched->context[ltid].slot[CTX_LOCKS_ID] , LOCKS_MASK_FAT ); 


    // check file open
    if ( _fat.fd[fd_id].allocated == 0 )
    {
        _spin_lock_release( &_fat.fat_lock );
        _atomic_and( &psched->context[ltid].slot[CTX_LOCKS_ID] , ~LOCKS_MASK_FAT ); 

        _printf("\n[FAT ERROR] _fat_lseek(): file not open\n");
        return GIET_FAT32_NOT_OPEN;
    }

    unsigned int  new_seek;

    // compute new seek
    if      ( whence == SEEK_CUR ) new_seek = _fat.fd[fd_id].seek + seek;
    else if ( whence == SEEK_SET ) new_seek = seek;
    else if ( whence == SEEK_END ) new_seek = _fat.fd[fd_id].inode->size + seek;
    else
    {
        _spin_lock_release( &_fat.fat_lock );
        _atomic_and( &psched->context[ltid].slot[CTX_LOCKS_ID] , ~LOCKS_MASK_FAT ); 

        _printf("\n[FAT ERROR] _fat_lseek(): illegal whence value\n");
        return GIET_FAT32_INVALID_ARG;
    }

    // update file descriptor offset 
    _fat.fd[fd_id].seek = new_seek;

#if GIET_DEBUG_FAT
unsigned int procid  = _get_procid();
unsigned int x       = procid >> (Y_WIDTH + P_WIDTH);
unsigned int y       = (procid >> P_WIDTH) & ((1<<Y_WIDTH)-1);
unsigned int p       = procid & ((1<<P_WIDTH)-1);
if ( _get_proctime() > GIET_DEBUG_FAT )
_printf("\n[DEBUG FAT] _fat_lseek(): P[%d,%d,%d] set seek = %x for file <%s>\n",
        x , y , p , new_seek , _fat.fd[fd_id].inode->name );
#endif

    // release lock
    _spin_lock_release( &_fat.fat_lock );
    _atomic_and( &psched->context[ltid].slot[CTX_LOCKS_ID] , ~LOCKS_MASK_FAT ); 

    return new_seek;
}  // end _fat_lseek()



///////////////////////////////////////
int _fat_remove( char*        pathname,
                 unsigned int should_be_dir )
{
    fat_inode_t*  inode;            // searched file inode pointer

#if GIET_DEBUG_FAT
unsigned int procid  = _get_procid();
unsigned int x       = procid >> (Y_WIDTH + P_WIDTH);
unsigned int y       = (procid >> P_WIDTH) & ((1<<Y_WIDTH)-1);
unsigned int p       = procid & ((1<<P_WIDTH)-1);
if ( _get_proctime() > GIET_DEBUG_FAT )
_printf("\n[DEBUG FAT] _fat_remove(): P[%d,%d,%d] enters for path <%s>\n",
        x, y, p, pathname );
#endif

    // checking FAT initialized
    if( _fat.initialized != FAT_INITIALIZED )
    {
        _printf("\n[FAT ERROR] _fat_remove(): FAT not initialized\n");
        return GIET_FAT32_NOT_INITIALIZED;
    }

    // takes the FAT lock and register it in thread context
    static_scheduler_t*  psched = _get_sched();
    unsigned int         ltid   = _get_thread_ltid();
    _spin_lock_acquire( &_fat.fat_lock );
    _atomic_or( &psched->context[ltid].slot[CTX_LOCKS_ID] , LOCKS_MASK_FAT ); 


    // get searched file inode 
    unsigned int code = _get_inode_from_path( pathname , &inode );

#if GIET_DEBUG_FAT
if ( _get_proctime() > GIET_DEBUG_FAT )
_printf("\n[DEBUG FAT] _fat_remove(): P[%d,%d,%d] found inode %x for <%s> / code = %d\n",
        x , y , p , (unsigned int)inode , pathname , code );
#endif

    if ( (code == 1) || (code == 2) )
    {
        _spin_lock_release( &_fat.fat_lock );
        _atomic_and( &psched->context[ltid].slot[CTX_LOCKS_ID] , ~LOCKS_MASK_FAT ); 

        _printf("\n[FAT ERROR] _fat_remove(): file <%s> not found\n", 
                pathname );
        return GIET_FAT32_FILE_NOT_FOUND;
    }
    else if ( code == 3 )
    {
        _spin_lock_release( &_fat.fat_lock );
        _atomic_and( &psched->context[ltid].slot[CTX_LOCKS_ID] , ~LOCKS_MASK_FAT ); 

        _printf("\n[FAT ERROR] _fat_remove(): name too long in <%s>\n",
                pathname );
        return GIET_FAT32_NAME_TOO_LONG;
    }

    // check inode type
    if ( (inode->is_dir != 0) && (should_be_dir == 0) ) 
    {
        _spin_lock_release( &_fat.fat_lock );
        _atomic_and( &psched->context[ltid].slot[CTX_LOCKS_ID] , ~LOCKS_MASK_FAT ); 

        _printf("\n[FAT ERROR] _fat_remove(): <%s> is a directory\n",
                pathname );
        return GIET_FAT32_IS_DIRECTORY;
    }
    if ( (inode->is_dir == 0) && (should_be_dir != 0) )
    {
        _spin_lock_release( &_fat.fat_lock );
        _atomic_and( &psched->context[ltid].slot[CTX_LOCKS_ID] , ~LOCKS_MASK_FAT ); 

        _printf("\n[FAT ERROR] _fat_remove(): <%s> is not a directory\n", 
                pathname );
        return GIET_FAT32_NOT_A_DIRECTORY;
    }

#if GIET_DEBUG_FAT
if ( _get_proctime() > GIET_DEBUG_FAT )
_printf("\n[DEBUG FAT] _fat_remove(): P[%d,%d,%d] checked inode type for <%s>\n",
        x , y , p , pathname );
#endif
    
    // check references count for a file
    if ( (inode->is_dir == 0) && (inode->count != 0) )
    {
        _spin_lock_release( &_fat.fat_lock );
        _atomic_and( &psched->context[ltid].slot[CTX_LOCKS_ID] , ~LOCKS_MASK_FAT ); 

        _printf("\n[FAT ERROR] _fat_remove(): file <%s> still referenced\n",
                pathname );
        return GIET_FAT32_IS_OPEN;
    }

    //  check empty for a directory
    if ( inode->is_dir )
    {
        unsigned int entries;
        if ( _get_nb_entries( inode , &entries ) )
        {
            _spin_lock_release( &_fat.fat_lock );
            _atomic_and( &psched->context[ltid].slot[CTX_LOCKS_ID] , ~LOCKS_MASK_FAT ); 

            _printf("\n[FAT ERROR] _fat_remove(): cannot scan directory <%s>\n", 
                    pathname );
            return GIET_FAT32_IO_ERROR;
        }
        else if ( entries > 2 )
        {
            _spin_lock_release( &_fat.fat_lock );
            _atomic_and( &psched->context[ltid].slot[CTX_LOCKS_ID] , ~LOCKS_MASK_FAT ); 

            _printf("\n[FAT ERROR] _fat_remove(): directory <%s> not empty\n", 
                    pathname );
            return GIET_FAT32_DIRECTORY_NOT_EMPTY;
        }
    }

#if GIET_DEBUG_FAT
if ( _get_proctime() > GIET_DEBUG_FAT )
_printf("\n[DEBUG FAT] _fat_remove(): P[%d,%d,%d] checked remove condition OK for <%s>\n",
        x , y , p , pathname );
#endif
    
    // remove the file or directory from the file system
    if ( _remove_node_from_fs( inode ) )
    {
        _spin_lock_release( &_fat.fat_lock );
        _atomic_and( &psched->context[ltid].slot[CTX_LOCKS_ID] , ~LOCKS_MASK_FAT ); 

        _printf("\n[FAT ERROR] _fat_remove(): cannot remove <%s> from FS\n",
                pathname );
        return GIET_FAT32_IO_ERROR;
    }

    // release lock and return success
    _spin_lock_release( &_fat.fat_lock );
    _atomic_and( &psched->context[ltid].slot[CTX_LOCKS_ID] , ~LOCKS_MASK_FAT ); 

#if GIET_DEBUG_FAT
if ( _get_proctime() > GIET_DEBUG_FAT )
_printf("\n[DEBUG FAT] _fat_remove(): P[%d,%d,%d] removed  <%s> from FS\n",
        x, y, p, pathname );
#endif
    
    return GIET_FAT32_OK;
        
}  // end _fat_remove()





/////////////////////////////////
int _fat_rename( char*  old_path,
                 char*  new_path )
{
    fat_inode_t*  inode;        // anonymous inode pointer
    fat_inode_t*  old;          // inode identified by old_path      => to be deleted
    fat_inode_t*  new;          // inode identified by new_path      => to be created
    fat_inode_t*  old_parent;   // parent inode  in old_path         => to be modified
    fat_inode_t*  new_parent;   // parent inode  in new_path         => to be modified
    fat_inode_t*  to_remove;    // previouly identified by new_path  => to be removed
    unsigned int  code;

#if GIET_DEBUG_FAT
unsigned int procid  = _get_procid();
unsigned int x       = procid >> (Y_WIDTH + P_WIDTH);
unsigned int y       = (procid >> P_WIDTH) & ((1<<Y_WIDTH)-1);
unsigned int p       = procid & ((1<<P_WIDTH)-1);
if ( _get_proctime() > GIET_DEBUG_FAT )
_printf("\n[DEBUG FAT] _fat_rename(): P[%d,%d,%d] enters to move <%s> to <%s>\n",
        x , y , p , old_path , new_path );
#endif

    // checking FAT initialized
    if( _fat.initialized != FAT_INITIALIZED )
    {
        _printf("\n[FAT ERROR] _fat_rename(): FAT not initialized\n");
        return GIET_FAT32_NOT_INITIALIZED;
    }

    // takes the FAT lock and register it in thread context
    static_scheduler_t*  psched = _get_sched();
    unsigned int         ltid   = _get_thread_ltid();
    _spin_lock_acquire( &_fat.fat_lock );
    _atomic_or( &psched->context[ltid].slot[CTX_LOCKS_ID] , LOCKS_MASK_FAT ); 


    // get "old" and "old_parent" inode pointers
    if ( _get_inode_from_path( old_path , &inode ) )
    {
        _spin_lock_release( &_fat.fat_lock );
        _atomic_and( &psched->context[ltid].slot[CTX_LOCKS_ID] , ~LOCKS_MASK_FAT ); 

        _printf("\n[FAT ERROR] _fat_rename(): <%s> not found\n", old_path );
        return GIET_FAT32_FILE_NOT_FOUND;
    }
    else
    {
        old        = inode;
        old_parent = inode->parent;
    }

    // get "to_removed" and "new_parent" inode pointers
    code = _get_inode_from_path( new_path , &inode );

    if ( code == 0 )       // new_path inode already exist 
    {
        if ( inode == old )  // the file will replace itself, do nothing
        {
            _spin_lock_release( &_fat.fat_lock );
            _atomic_and( &psched->context[ltid].slot[CTX_LOCKS_ID] , ~LOCKS_MASK_FAT ); 

            return GIET_FAT32_OK;
        }

        to_remove        = inode;
        new_parent       = inode->parent;
    }
    else if ( code == 1 )  // to_remove does not exist but parent exist
    {
        to_remove        = NULL;
        new_parent       = inode;
    }
    else                   // parent directory in new_path not found
    {
        _spin_lock_release( &_fat.fat_lock );
        _atomic_and( &psched->context[ltid].slot[CTX_LOCKS_ID] , ~LOCKS_MASK_FAT ); 

        _printf("\n[FAT ERROR] _fat_rename(): <%s> not found\n", new_path );
        return GIET_FAT32_FILE_NOT_FOUND;
    }

    // check for move into own subdirectory
    if ( _is_ancestor( old, new_parent ) )
    {
        _spin_lock_release( &_fat.fat_lock );
        _atomic_and( &psched->context[ltid].slot[CTX_LOCKS_ID] , ~LOCKS_MASK_FAT ); 

        _printf("\n[FAT ERROR] _fat_rename(): can't move %s into  own directory\n", old_path );
        return GIET_FAT32_MOVE_INTO_SUBDIR;
    }

#if GIET_DEBUG_FAT
if ( _get_proctime() > GIET_DEBUG_FAT )
{
if ( to_remove )
_printf("\n[DEBUG FAT] _fat_rename(): old_parent = %s / old = %s / new_parent = %s "
        "/ to_remove = %s\n",
        old_parent->name , old->name , new_parent->name , to_remove->name );
else
_printf("\n[DEBUG FAT] _fat_rename(): old_parent = %s / old = %s / new_parent = %s "
        "/ no remove\n", 
        old_parent->name , old->name , new_parent->name );
}
#endif

    // check remove condition for "to_remove" inode
    if ( to_remove )
    {
        if ( to_remove->is_dir )   // it's a directory
        {
            unsigned int entries;
            if ( _get_nb_entries( to_remove , &entries ) )
            {
                _spin_lock_release( &_fat.fat_lock );
                _atomic_and( &psched->context[ltid].slot[CTX_LOCKS_ID] , ~LOCKS_MASK_FAT ); 

                _printf("\n[FAT ERROR] _fat_rename(): cannot scan directory <%s>\n", 
                        to_remove->name );
                return GIET_FAT32_IO_ERROR;
            }
            else if ( entries > 2 )
            {
                _spin_lock_release( &_fat.fat_lock );
                _atomic_and( &psched->context[ltid].slot[CTX_LOCKS_ID] , ~LOCKS_MASK_FAT ); 

                _printf("\n[FAT ERROR] _fat_rename(): directory <%s> not empty\n", 
                        to_remove->name );
                return GIET_FAT32_DIRECTORY_NOT_EMPTY;
            }
        }
        else                       // it's a file
        {
            if ( to_remove->count ) 
            {
                _spin_lock_release( &_fat.fat_lock );
                _atomic_and( &psched->context[ltid].slot[CTX_LOCKS_ID] , ~LOCKS_MASK_FAT ); 

                _printf("\n[FAT ERROR] _fat_rename(): file <%s> still referenced\n", 
                        to_remove->name );
                return GIET_FAT32_IS_OPEN;
            }
        }
    }

#if GIET_DEBUG_FAT
if ( _get_proctime() > GIET_DEBUG_FAT )
_printf("\n[FAT DEBUG] _fat_rename(): P[%d,%d,%d] checked remove condition OK\n",
        x , y , p );
#endif

    // get new last name / error checking already done by _get_inode_from_path()
    char  new_name[32];
    _get_last_name( new_path , new_name );

    // allocate "new" inode
    new = _allocate_one_inode( new_name,
                               old->is_dir,
                               old->cluster,
                               old->size,
                               0,              // count
                               0,              // dentry set by _add_dir_entry()
                               0 );            // no cache_allocate
 
    // attach the "old" File-Cache to the "new" inode
    new->levels = old->levels;
    new->cache  = old->cache;

    // add "new" to "new_parent" directory in Inode-Tree
    _add_inode_in_tree( new , new_parent );
    
    // add "new" to "new_parent" directory File-Cache
    // and update the dentry field in new inode
    if ( _add_dir_entry( new ) )
    {
        _spin_lock_release( &_fat.fat_lock );
        _atomic_and( &psched->context[ltid].slot[CTX_LOCKS_ID] , ~LOCKS_MASK_FAT ); 

        _printf("\n[FAT ERROR] _fat_rename(): cannot add <%s> into <%s>\n",
                new->name , new_parent->name );
        return GIET_FAT32_IO_ERROR;
    }

    // updates "new_parent" directory on device
    if ( _update_device_from_cache( new_parent->levels,
                                    new_parent->cache,
                                    new_parent->name ) )
    {
        _spin_lock_release( &_fat.fat_lock );
        _atomic_and( &psched->context[ltid].slot[CTX_LOCKS_ID] , ~LOCKS_MASK_FAT ); 

        _printf("\n[FAT ERROR] _fat_rename(): cannot update <%s> on device\n",
                    new_parent->name );
        return GIET_FAT32_IO_ERROR;
    }

    // remove "old" from "old_parent" File-Cache 
    if ( _remove_dir_entry( old ) )
    {
        _spin_lock_release( &_fat.fat_lock );
        _atomic_and( &psched->context[ltid].slot[CTX_LOCKS_ID] , ~LOCKS_MASK_FAT ); 

        _printf("\n[FAT ERROR] _fat_rename(): cannot remove <%s> from <%s>\n",
                old->name , old_parent->name );
        return GIET_FAT32_IO_ERROR;
    }
 
    // remove "old" inode from Inode-Tree
    _remove_inode_from_tree( old );

    // release "old" inode
    _free( old );

    // updates "old_parent" directory on device
    if ( _update_device_from_cache( old_parent->levels,
                                    old_parent->cache,
                                    old_parent->name ) )
    {
        _spin_lock_release( &_fat.fat_lock );
        _atomic_and( &psched->context[ltid].slot[CTX_LOCKS_ID] , ~LOCKS_MASK_FAT ); 

        _printf("\n[FAT ERROR] _fat_rename(): cannot update <%s> on device\n",
                    old_parent->name );
        return GIET_FAT32_IO_ERROR;
    }

    // remove "to_remove" from File System (if required)
    if ( to_remove )
    {
        if ( _remove_node_from_fs( to_remove ) )
        {
            _spin_lock_release( &_fat.fat_lock );
            _atomic_and( &psched->context[ltid].slot[CTX_LOCKS_ID] , ~LOCKS_MASK_FAT ); 

            _printf("\n[FAT ERROR] _fat_rename(): cannot remove <%s> from FS\n",
                    to_remove->name );
            return GIET_FAT32_IO_ERROR;
        }
    }

    // release lock
    _spin_lock_release( &_fat.fat_lock );
    _atomic_and( &psched->context[ltid].slot[CTX_LOCKS_ID] , ~LOCKS_MASK_FAT ); 

    return GIET_FAT32_OK;
}  // end _fat_rename()




////////////////////////////////
int _fat_mkdir( char* pathname )
{
    fat_inode_t*         inode;            // anonymous inode pointer
    fat_inode_t*         child;            // searched directory inode pointer
    fat_inode_t*         parent;           // parent directory inode pointer

#if GIET_DEBUG_FAT
unsigned int procid  = _get_procid();
unsigned int x       = procid >> (Y_WIDTH + P_WIDTH);
unsigned int y       = (procid >> P_WIDTH) & ((1<<Y_WIDTH)-1);
unsigned int p       = procid & ((1<<P_WIDTH)-1);
if ( _get_proctime() > GIET_DEBUG_FAT )
_printf("\n[DEBUG FAT] _fat_mkdir(): P[%d,%d,%d] enters for path <%s>\n",
        x, y, p, pathname );
#endif

    // checking FAT initialized
    if( _fat.initialized != FAT_INITIALIZED )
    {
        _printf("\n[FAT ERROR] _fat_mkdir(): FAT not initialized\n");
        return GIET_FAT32_NOT_INITIALIZED;
    }

    // takes the FAT lock and register it in thread context
    static_scheduler_t*  psched = _get_sched();
    unsigned int         ltid   = _get_thread_ltid();
    _spin_lock_acquire( &_fat.fat_lock );
    _atomic_or( &psched->context[ltid].slot[CTX_LOCKS_ID] , LOCKS_MASK_FAT ); 

    // get inode
    unsigned int code = _get_inode_from_path( pathname , &inode );

    if ( code == 2 )  
    {
        _spin_lock_release( &_fat.fat_lock );
        _atomic_and( &psched->context[ltid].slot[CTX_LOCKS_ID] , ~LOCKS_MASK_FAT ); 

        _printf("\n[FAT ERROR] _fat_mkdir(): path to parent not found"
                " for directory <%s>\n", pathname );
        return GIET_FAT32_FILE_NOT_FOUND;
    }
    else if ( code == 3 )  
    {
        _spin_lock_release( &_fat.fat_lock );
        _atomic_and( &psched->context[ltid].slot[CTX_LOCKS_ID] , ~LOCKS_MASK_FAT ); 

        _printf("\n[FAT ERROR] _fat_mkdir(): one name in path too long"
                " for directory  <%s>\n", pathname );
        return GIET_FAT32_NAME_TOO_LONG;
    }
    else if ( code == 0 )
    {
        _spin_lock_release( &_fat.fat_lock );
        _atomic_and( &psched->context[ltid].slot[CTX_LOCKS_ID] , ~LOCKS_MASK_FAT ); 

        _printf("\n[FAT ERROR] _fat_mkdir(): directory <%s> already exist\n",
                pathname );
        return GIET_FAT32_FILE_EXISTS;
    }
    else if ( code == 1 )   // directory not found => create 
    {
        parent = inode;

#if GIET_DEBUG_FAT
if ( _get_proctime() > GIET_DEBUG_FAT )
_printf("\n[DEBUG FAT] _fat_mkdir(): P[%d,%d,%d] create new directory <%s>\n",
        x , y , p , pathname );
#endif

        // get directory name / error check already done by _get_inode_from_path()
        char name[32];       
        _get_last_name( pathname , name );

        // allocate a new inode and an empty Cache-File 
        child = _allocate_one_inode( name,
                                     1,                         // it's a directory
                                     END_OF_CHAIN_CLUSTER_MAX,  // cluster set later
                                     0,                         // size = 0 for directory
                                     0,                         // count
                                     0,                         // dentry set later
                                     1 );                       // cache_allocate

        // introduce inode in Inode-Tree
        _add_inode_in_tree( child , parent );
 
        // allocate one cluster from FAT for child
        unsigned int cluster;
        if ( _one_cluster_allocate( child , &cluster ) )
        {
            _spin_lock_release( &_fat.fat_lock );
            _atomic_and( &psched->context[ltid].slot[CTX_LOCKS_ID] , ~LOCKS_MASK_FAT ); 

            _printf("\n[FAT ERROR] _fat_mkdir(): no free cluster"
                    " for directory <%s>\n" , pathname );
            return GIET_FAT32_NO_FREE_SPACE;
        }

        // update cluster index in inode
        child->cluster = cluster;

        // add new entry in parent directory File-Cache
        // and update dentry field in child inode
        if ( _add_dir_entry( child ) )
        {
            _spin_lock_release( &_fat.fat_lock );
            _atomic_and( &psched->context[ltid].slot[CTX_LOCKS_ID] , ~LOCKS_MASK_FAT ); 

            _printf("\n[FAT ERROR] _fat_mkdir(): cannot update parent directory"
                    " for directory <%s>\n" , pathname );
            return GIET_FAT32_IO_ERROR;
        } 

        // add "." and ".." directories in child directory
        _add_special_directories( child );

        // update DATA region on block device for parent directory
        if ( _update_device_from_cache( parent->levels,
                                        parent->cache,
                                        parent->name ) )
        {
            _spin_lock_release( &_fat.fat_lock );
            _atomic_and( &psched->context[ltid].slot[CTX_LOCKS_ID] , ~LOCKS_MASK_FAT ); 

            _printf("\n[FAT ERROR] _fat_mkdir(): cannot update DATA region "
                    " for parent of directory <%s>\n", pathname );
            return GIET_FAT32_IO_ERROR;
        }

        // update FAT region on block device
        if ( _update_device_from_cache( _fat.fat_cache_levels,
                                        _fat.fat_cache_root,
                                        "FAT" ) )
        {
            _spin_lock_release( &_fat.fat_lock );
            _atomic_and( &psched->context[ltid].slot[CTX_LOCKS_ID] , ~LOCKS_MASK_FAT ); 

            _printf("\n[FAT ERROR] _fat_mkdir(): cannot update FAT region"
                    " for directory <%s>\n", pathname );
            return GIET_FAT32_IO_ERROR;
        }

        // update DATA region on block device for the new directory
        if ( _update_device_from_cache( child->levels,   
                                        child->cache,
                                        child->name ) )
        {
            _spin_lock_release( &_fat.fat_lock );
            _atomic_and( &psched->context[ltid].slot[CTX_LOCKS_ID] , ~LOCKS_MASK_FAT ); 

            _printf("\n[FAT ERROR] _fat_mkdir(): cannot update DATA region"
                    " for directory <%s>\n", pathname );
            return GIET_FAT32_IO_ERROR;
        }
    }  // end create directory

    // release lock
    _spin_lock_release( &_fat.fat_lock );
    _atomic_and( &psched->context[ltid].slot[CTX_LOCKS_ID] , ~LOCKS_MASK_FAT ); 

    return GIET_FAT32_OK;
}  // end _fat_mkdir()




/////////////////////////////////////////
extern int _fat_opendir( char* pathname )
{
    int fd_id = _fat_open( pathname, O_RDONLY );

    if ( fd_id < 0 )
        return fd_id;

    if ( !_fat.fd[fd_id].inode->is_dir )
    {
        _printf("\n[FAT ERROR] _fat_opendir(): <%s> is not a directory\n",
                pathname );
        return GIET_FAT32_NOT_A_DIRECTORY;
    }

    return fd_id;
}




//////////////////////////////////////////////
extern int _fat_closedir( unsigned int fd_id )
{
    return _fat_close( fd_id );
}




/////////////////////////////////////////////
extern int _fat_readdir( unsigned int  fd_id,
                         fat_dirent_t* entry )
{
    unsigned int  lfn   = 0;            // lfn entries count
    unsigned int  attr;                 // ATTR field value
    unsigned int  ord;                  // ORD field value
    char          lfn1[16];             // temporary buffer for string in LFN1
    char          lfn2[16];             // temporary buffer for string in LFN2
    char          lfn3[16];             // temporary buffer for string in LFN3
    unsigned char buf[DIR_ENTRY_SIZE];  // raw entry buffer
    fat_file_info_t info;

    // check for directory
    int ret = _fat_file_info( fd_id, &info );
    if (ret < 0)
    {
        return ret;
    }
    else if ( !info.is_dir )
    {
        _printf("\n[FAT ERROR] in _fat_readdir(): not a directory\n" );
        return GIET_FAT32_NOT_A_DIRECTORY;
    }


#if GIET_DEBUG_FAT
unsigned int procid  = _get_procid();
unsigned int x       = procid >> (Y_WIDTH + P_WIDTH);
unsigned int y       = (procid >> P_WIDTH) & ((1<<Y_WIDTH)-1);
unsigned int p       = procid & ((1<<P_WIDTH)-1);
if ( _get_proctime() > GIET_DEBUG_FAT )
_printf("\n[DEBUG FAT] _fat_readdir(): P[%d,%d,%d] enter for <%s>\n",
        x , y , p , _fat.fd[fd_id].inode->name );
#endif

    while ( 1 )
    {
        if ( _fat_read( fd_id, 
                        (unsigned int)&buf, 
                        DIR_ENTRY_SIZE,
                        0, 0, 0 )  != sizeof(buf) )
        {
            _printf("\n[FAT ERROR] in _fat_readdir(): can't read entry\n" );
            return GIET_FAT32_IO_ERROR;
        }

        attr = _read_entry( DIR_ATTR, buf, 0 );
        ord  = _read_entry( LDIR_ORD, buf, 0 );

        if (ord == NO_MORE_ENTRY)               // no more entry in directory => stop
        {
            // seek back to this entry
            _atomic_increment( &_fat.fd[fd_id].seek , -DIR_ENTRY_SIZE );

            return GIET_FAT32_NO_MORE_ENTRIES;
        }
        else if ( ord == FREE_ENTRY )           // free entry => skip
        {
            continue;
        }
        else if ( attr == ATTR_LONG_NAME_MASK ) // LFN entry => get partial names
        {
            unsigned int seq = ord & 0x3;
            lfn = (seq > lfn) ? seq : lfn;
            if      ( seq == 1 ) _get_name_from_long( buf, lfn1 );
            else if ( seq == 2 ) _get_name_from_long( buf, lfn2 );
            else if ( seq == 3 ) _get_name_from_long( buf, lfn3 );
            continue;
        }
        else                                    // NORMAL entry => stop
        {
            break;
        }
    }

    // TODO handle is_vid
    entry->cluster = (_read_entry( DIR_FST_CLUS_HI, buf, 1 ) << 16) |
                     (_read_entry( DIR_FST_CLUS_LO, buf, 1 )      ) ;
    entry->size    = (_read_entry( DIR_FILE_SIZE  , buf, 1 )      ) ;
    entry->is_dir  = ((attr & ATTR_DIRECTORY) == ATTR_DIRECTORY);

    if      ( lfn == 0 )
    {
        _get_name_from_short( buf, entry->name );
    }
    else if ( lfn == 1 )
    {
        _strcpy( entry->name     , lfn1 );
    }
    else if ( lfn == 2 )
    {
        _strcpy( entry->name     , lfn1 );
        _strcpy( entry->name + 13, lfn2 );
    }
    else if ( lfn == 3 )
    {
        _strcpy( entry->name     , lfn1 );
        _strcpy( entry->name + 13, lfn2 );
        _strcpy( entry->name + 26, lfn3 );
    }

    return GIET_FAT32_OK;
}  // end _fat_readdir()




///////////////////////////////////////////////
int _fat_load_no_cache( char*        pathname,
                        unsigned int buffer_vbase,  
                        unsigned int buffer_size ) 
{
    // checking FAT initialized
    if( _fat.initialized != FAT_INITIALIZED )
    {
        _printf("\n[FAT ERROR] _fat_load_no_cache(): FAT not initialized\n");
        return GIET_FAT32_NOT_INITIALIZED;
    }

    unsigned int  file_size;
    unsigned int  cluster;

#if GIET_DEBUG_FAT
unsigned int procid  = _get_procid();
unsigned int x       = procid >> (Y_WIDTH + P_WIDTH);
unsigned int y       = (procid >> P_WIDTH) & ((1<<Y_WIDTH)-1);
unsigned int p       = procid & ((1<<P_WIDTH)-1);
if ( _get_proctime() > GIET_DEBUG_FAT )
_printf("\n[DEBUG FAT] _fat_load_no_cache(): P[%d,%d,%d] enters for file <%s>\n",
        x , y , p , pathname );
#endif

    // get file size, and cluster index in FAT
    if ( _file_info_no_cache( pathname,
                              &cluster,
                              &file_size ) )
    {
        _printf("\n[FAT ERROR] _fat_load_no_cache(): file <%s> not found\n",
        pathname );
        return GIET_FAT32_FILE_NOT_FOUND;
    }

    // check buffer size
    if ( file_size > buffer_size )
    {
        _printf("\n[FAT ERROR] _fat_load_no_cache(): buffer too small : "
                "file_size = %x / buffer_size = %x", file_size , buffer_size );
        return GIET_FAT32_BUFFER_TOO_SMALL;
    }

    // compute total number of clusters to read
    unsigned int nb_clusters = file_size >> 12;
    if ( file_size & 0xFFF ) nb_clusters++;

    // initialise buffer address
    unsigned int dst = buffer_vbase;

    // loop on the clusters containing the file
    while ( nb_clusters > 0 )
    {
        unsigned int lba = _cluster_to_lba( cluster );

        if( _fat_ioc_access( 0,         // no descheduling
                             1,         // read
                             lba, 
                             dst, 
                             8 ) )      // 8 blocks
        {
            _printf("\n[FAT ERROR] _fat_load_no_cache(): cannot load lba %x", lba );
            return GIET_FAT32_IO_ERROR;
        }
         

        // compute next cluster index
        unsigned int next;
        if ( _next_cluster_no_cache( cluster , &next ) )
        {
            _printf("\n[FAT ERROR] _fat_load_no_cache(): cannot get next cluster "
                    " for cluster = %x\n", cluster );
            return GIET_FAT32_IO_ERROR;
        }
        
        // update variables for next iteration
        nb_clusters = nb_clusters - 1;
        dst         = dst + 4096;
        cluster     = next;
    }
         
#if GIET_DEBUG_FAT
if ( _get_proctime() > GIET_DEBUG_FAT )
_printf("\n[DEBUG FAT] _fat_load_no_cache(): P[%d,%d,%d] loaded <%s> at vaddr = %x"
        " / size = %x\n", x , y , p , pathname , buffer_vbase , file_size );
#endif

    return GIET_FAT32_OK;
}  // end _fat_load_no_cache() 




//////////////////////////////////////////////////////////
int _get_fat_cache_buffer( unsigned int        cluster_id,
                           fat_cache_desc_t**  desc )
{
    // get cache pointer and number of levels
    fat_cache_node_t*   node   = _fat.fat_cache_root;    
    unsigned int        level  = _fat.fat_cache_levels;  

    if ( _get_levels_from_size( (cluster_id + 1) * 4096 ) > level )
    {
        _printf("\n[FAT ERROR] in _get_fat_cache_buffer() : "
                "cluster_id %d too large\n", cluster_id );
        return GIET_FAT32_IO_ERROR;
    }

    // search the 64-tree cache from top to bottom 
    while ( level )
    {
        // compute child index at each level
        unsigned int index = (cluster_id >> (6*(level-1))) & 0x3F;

        if ( level == 1 )        // last level => children are buffer descriptors
        {
            fat_cache_desc_t*   pdesc = (fat_cache_desc_t*)node->children[index];

            if ( pdesc == NULL )      // miss 
            {

#if GIET_DEBUG_FAT
if ( _get_proctime() > GIET_DEBUG_FAT )
_printf("\n[DEBUG FAT] _get_fat_cache_buffer(): miss for cluster_id %d\n", cluster_id );
#endif
                // compute missing cluster lba
                unsigned int lba = _fat.fat_lba + (cluster_id << 3);

                // allocate a 4 Kbytes buffer and a buffer descriptor
                void* buf      = _malloc( 4096 );
                pdesc          = _malloc( sizeof(fat_cache_desc_t) );
                pdesc->lba     = lba;
                pdesc->buffer  = buf;
                pdesc->dirty   = 0;
                node->children[index] = pdesc;

                // load cluster from device
                if ( _fat_ioc_access( 1,         // descheduling
                                      1,         // to memory
                                      lba,
                                      (unsigned int)buf,
                                      8 ) )
                {
                    _free( buf );
                    _free( pdesc );
                    _printf("\n[FAT ERROR] in _get_fat_cache_buffer() : "
                            ": cannot access block device for lba = %x\n", lba );
                    return GIET_FAT32_IO_ERROR;
                }

            }

            // return pdesc pointer
            *desc = pdesc;

            // prepare next iteration
            level--;
        }
        else                      // not last level => children are 64-tree nodes
        {
            fat_cache_node_t* child = (fat_cache_node_t*)node->children[index];
            if ( child == NULL )  // miss 
            {
                // allocate a cache node if miss
                child = _allocate_one_cache_node( NULL );
                node->children[index] = child;    
            }

            // prepare next iteration
            node = child;
            level--;
        }
    } // end while

    return 0;
}  // end _get_fat_cache_buffer()




//////////////////////////////////////////////////////
int _get_file_cache_buffer( fat_inode_t*        inode,
                            unsigned int        cluster_id,
                            unsigned int        writable,
                            fat_cache_desc_t**  desc )
{

#if GIET_DEBUG_FAT
if ( _get_proctime() > GIET_DEBUG_FAT )
_printf("\n[DEBUG FAT] _get_file_cache_buffer() : enters in File-Cache <%s>"
        " for cluster_id = %d\n size = %x / cache = %x / desc[%d] = %x\n", 
        inode->name , cluster_id ,
        inode->size , (unsigned int)inode->cache , cluster_id ,
        (unsigned int)inode->cache->children[cluster_id] );
#endif

    // checking FAT initialized
    if( _fat.initialized != FAT_INITIALIZED )
    {
        _printf("\n[FAT ERROR] in _get_file_cache_buffer() : FAT not initialized\n");
        return GIET_FAT32_NOT_INITIALIZED;
    }

    // checking arguments
    if ( inode == NULL )   // illegal inode argument
    {
        _printf("\n[FAT ERROR] in _get_file_cache_buffer() : illegal inode argument\n");
        return GIET_FAT32_INVALID_ARG;
    }

    // add cache levels if needed
    while ( _get_levels_from_size( (cluster_id + 1) * 4096 ) > inode->levels )
    {

#if GIET_DEBUG_FAT
if ( _get_proctime() > GIET_DEBUG_FAT )
_printf("\n[DEBUG FAT] _get_file_cache_buffer() : add a File-Cache level\n" );
#endif

        inode->cache = _allocate_one_cache_node( inode->cache );
        inode->levels++;
    }

    // get inode type, size, and File-Cache
    unsigned int       size   = inode->size;
    unsigned int       is_dir = inode->is_dir;
    fat_cache_node_t*  node   = inode->cache;  
    unsigned int       level  = inode->levels;

    // search the 64-tree cache from top to bottom 
    while ( level )
    {
        // compute child index at each level
        unsigned int index = (cluster_id >> (6*(level-1))) & 0x3F;

        if ( level == 1 )        // last level => children are buffer descriptors
        {
            fat_cache_desc_t*   pdesc   = (fat_cache_desc_t*)node->children[index];
            unsigned int        next    = 0;
            unsigned int        prev    = 0;
            unsigned int        current;
            unsigned int        cid;
            unsigned int        lba;
            unsigned int        one_cluster_allocated;

            // File-Cache miss handling:
            // In case of miss, the missing buffer is allocated,
            // and the missing cluster is loaded from block device.
            // A new cluster is allocated from FAT if required, when
            // the writable argument is set.
            if ( pdesc == NULL )    
            {

#if GIET_DEBUG_FAT
if ( _get_proctime() > GIET_DEBUG_FAT )
_printf("\n[DEBUG FAT] _get_file_cache_buffer() : miss in File-Cache <%s> "
        " for cluster_id = %d\n"
        " cluster = %x / size = %x / is_dir = %d / cache = %x / desc[%d] = %x\n",
        inode->name , cluster_id , inode->cluster , inode->size , inode->is_dir ,
        (unsigned int)inode->cache , cluster_id , 
        (unsigned int)inode->cache->children[cluster_id] );
#endif
                // compute one_cluster_allocated condition, depending on file / dir type
                if ( is_dir )  one_cluster_allocated = ( cluster_id < is_dir );
                else           one_cluster_allocated = ( (cluster_id<<12) < size );

                if ( one_cluster_allocated )  // cluster already allocated => allocate buffer
                {
                    // scan the FAT to find the cluster index for cluster_id
                    current = inode->cluster;
                    for ( cid = 0 ; cid < cluster_id ; cid++ )
                    {
                        // get next cluster index from FAT
                        if ( _get_fat_entry( current , &next ) ) return 1;
                        current = next;
                    }

                    // compute lba
                    lba = _cluster_to_lba( current );

                    // allocate a 4 Kbytes buffer and a buffer descriptor 
                    // the selected heap depends on the calling thread
                    void* buf      = _malloc( 4096 );
                    pdesc          = _malloc( sizeof(fat_cache_desc_t) );

                    // set buffer descriptor
                    pdesc->lba     = lba;
                    pdesc->buffer  = buf;
                    pdesc->dirty   = writable;
                    node->children[index] = pdesc;

                    // load cluster from device
                    if ( _fat_ioc_access( 1,         // descheduling
                                          1,         // to memory
                                          lba,
                                          (unsigned int)buf,
                                      8 ) )
                    {
                         
                        _free( buf );
                        _free( pdesc );
                        _printf("\n[FAT ERROR] in _get_file_cache_buffer() : "
                                "cannot access block device for lba = %x\n", lba );
                        return GIET_FAT32_IO_ERROR;
                    }
                }
                else if ( writable == 0 ) // not writable and cluster not allocated in FAT
                {
                    _printf("\n[FAT ERROR] in _get_file_cache_buffer() : "
                            " file size too small for <%s>\n"
                            " size = %x / cluster_id = %d / procid = %x\n",
                            inode->name , inode->size , cluster_id , _get_procid() );
                    return GIET_FAT32_IO_ERROR;
                }
                else   // writable and cluster NOT allocated in FAT => allocate cluster & buffer
                {
                    // scan the FAT to allocate all required clusters 
                    current = inode->cluster;
                    for ( cid = 0 ; cid <= cluster_id ; cid++ )
                    {
                        if ( current >= END_OF_CHAIN_CLUSTER_MIN ) // non allocated
                        {
                            // allocate one cluster on device
                            if ( _one_cluster_allocate( inode , &current ) )
                            {
                                _printf("\n[FAT ERROR] in _get_file_cache_buffer() : "
                                        "cannot allocate new cluster for file <%s>\n",
                                        inode->name );
                                return GIET_FAT32_IO_ERROR;
                            }
                        }
 
                        // get next cluster index from FAT
                        if ( _get_fat_entry( current , &next ) )
                        {
                            _printf("\n[FAT ERROR] in _get_file_cache_buffer() : "
                                    "cannot get next cluster for file <%s>\n",
                                    inode->name );
                            return GIET_FAT32_IO_ERROR;
                        }
                        prev    = current;
                        current = next;
                    }

                    // update size or is_dir attributes in inode
                    if ( is_dir )    inode->is_dir = cluster_id;
                    else             inode->size   = (cluster_id + 1)<<12;

                    // update directory entry from inode
                    _update_dir_entry( inode );

                    // compute lba
                    lba = _cluster_to_lba( current );

                    // allocate a 4 Kbytes buffer and a buffer descriptor 
                    // the selected heap depends on the calling thread
                    void* buf      = _malloc( 4096 );
                    pdesc          = _malloc( sizeof(fat_cache_desc_t) );

                    // set buffer descriptor
                    pdesc->lba     = lba;
                    pdesc->buffer  = buf;
                    pdesc->dirty   = writable;
                    node->children[index] = pdesc;
                }
            }  // end File-Cache miss handling

            // return pdesc pointer
            *desc = pdesc;

#if GIET_DEBUG_FAT
if ( _get_proctime() > GIET_DEBUG_FAT )
_printf("\n[DEBUG FAT] _get_file_cache_buffer(): found buffer = %x "
        " in file <%s> for cluster_id %d\n",
        (unsigned int)pdesc->buffer , inode->name , cluster_id );
#endif
            // prepare next iteration
            level--;
        }
        else                      // not last level => children are 64-tree nodes
        {
            fat_cache_node_t* child = (fat_cache_node_t*)node->children[index];
            if ( child == NULL )  // miss 
            {
                // allocate a cache node if miss
                child = _allocate_one_cache_node( NULL );
                node->children[index] = child;    
            }

            // prepare next iteration
            node = child;
            level--;
        }
    } // end while

    return GIET_FAT32_OK;

}  // end _get_file_cache_buffer()






// Local Variables:
// tab-width: 4
// c-basic-offset: 4
// c-file-offsets:((innamespace . 0)(inline-open . 0))
// indent-tabs-mode: nil
// End:
// vim: filetype=c:expandtab:shiftwidth=4:tabstop=4:softtabstop=4

