#!/usr/bin/env python

from mapping import *

##################################################################################
#   file   : rosenfeld.py
#   date   : January 2016
#   author : Quentin Meunier
##################################################################################
#  This file describes the mapping of the single-threaded "rosenfeld".
#  This include both the mapping of virtual segments on the clusters,
#  and the mapping of tasks on processor.
#  The mapping of virtual segments is the following:
#    - There is one shared data vseg in cluster[0][0]
#    - The code vsegs are replicated on all clusters containing processors.
#    - There is one heap vseg per cluster containing processors.
#    - The stacks vsegs are distibuted on all clusters containing processors.
#  This mapping uses 5 platform parameters, (obtained from the "mapping" argument)
#    - x_size    : number of clusters in a row
#    - y_size    : number of clusters in a column
#    - x_width   : number of bits coding x coordinate
#    - y_width   : number of bits coding y coordinate
#    - nprocs    : number of processors per cluster
##################################################################################

#########################
def extend(mapping):

    x_size    = mapping.x_size
    y_size    = mapping.y_size
    nprocs    = mapping.nprocs
    x_width   = mapping.x_width
    y_width   = mapping.y_width

    # define vsegs base & size
    code_base  = 0x10000000
    code_size  = 0x00030000     # 192 Kbytes (replicated in each cluster)
    
    data_base  = 0x20000000
    data_size  = 0x00010000     # 64 Kbytes (non replicated)

    # QM warning: if less than 8M per cluster (i.e. 2M per thread), small pages are required
    stack_base = 0x40000000 
    stack_size = 0x00080000     # 512K (per cluster) => 128K per thread
    #stack_size = 0x00400000     # 4M (per cluster) => 1M per thread

    heap_base  = 0x60000000 
    #heap_size  = 0x00800000     # 8 Mbytes (per cluster) 
    heap_size  = 0x10000000     # 256 Mbytes (total) 

    # create vspace
    vspace = mapping.addVspace( name = 'rosenfeld', startname = 'rosen_data', active = True )
    
    # data vseg : shared (only in cluster[0,0])
    mapping.addVseg( vspace, 'rosen_data', data_base, data_size, 
                     'C_WU', vtype = 'ELF', x = 0, y = 0, pseg = 'RAM', 
                     binpath = 'bin/rosenfeld/appli.elf',
                     local = False )

    # code vsegs : local (one copy in each cluster)
    for x in xrange (x_size):
        for y in xrange (y_size):
            cluster_id = (x * y_size) + y
            if (mapping.clusters[cluster_id].procs):

                mapping.addVseg( vspace, 'rosen_code_%d_%d' %(x,y), 
                                 code_base , code_size,
                                 'CXWU', vtype = 'ELF', x = x, y = y, pseg = 'RAM', 
                                 binpath = 'bin/rosenfeld/appli.elf',
                                 local = True )

    # stacks vsegs: local (one stack per processor => nprocs stacks per cluster)
    for x in xrange (x_size):
        for y in xrange (y_size):
            cluster_id = (x * y_size) + y
            if (mapping.clusters[cluster_id].procs):
                for p in xrange(nprocs):
                    proc_id = (((x * y_size) + y) * nprocs) + p
                    size    = (stack_size / nprocs) & 0xFFFFF000
                    base    = stack_base + (proc_id * size)

                    mapping.addVseg( vspace, 'rosen_stack_%d_%d_%d' % (x,y,p), 
                                     base, size, 'C_WU', vtype = 'BUFFER', 
                                     x = x , y = y , pseg = 'RAM',
                                     local = True, big = False )

    # heap vsegs: distributed non local (all heap vsegs can be accessed by all tasks)
    for x in xrange (x_size):
        for y in xrange (y_size):
            cluster_id = (x * y_size) + y
            if (mapping.clusters[cluster_id].procs):
                nclusters = x_size * y_size
                if x == 0 and y == 0:
                    size = heap_size / 2
                    base = heap_base
                else:
                    size = heap_size / (2 * nclusters)
                    base = heap_base + heap_size / 2 + ((y * x_size) + x - 1) * size
                mapping.addVseg(vspace, 'rosen_heap_%d_%d' % (x, y), base, size, 
                        'C_WU', vtype = 'HEAP', x = x, y = y, pseg = 'RAM',
                        local = False, big = True )

    # distributed tasks / one task per processor
    for x in xrange (x_size):
        for y in xrange (y_size):
            cluster_id = (x * y_size) + y
            if (mapping.clusters[cluster_id].procs):
                for p in xrange(nprocs):
                    trdid = (((x * y_size) + y) * nprocs) + p
                    if trdid == 0:
                        startid = 0
                        is_main = 1
                    else:
                        startid = 1
                        is_main = 0
                    #startid = trdid == 0 and 0 or 1
                    #is_main = startid == 0

                    mapping.addThread(vspace, 'rosen_%d_%d_%d' % (x, y, p),
                                    is_main, x, y, p,
                                    'rosen_stack_%d_%d_%d' % (x,y,p),
                                    'rosen_heap_%d_%d' % (x,y), startid)


    # extend mapping name
    mapping.name += '_rosenfeld'

    return vspace  # useful for test
            
################################ test ##################################################

if __name__ == '__main__':

    vspace = extend( Mapping( 'test', 2, 2, 4 ) )
    print vspace.xml()


# Local Variables:
# tab-width: 4;
# c-basic-offset: 4;
# c-file-offsets:((innamespace . 0)(inline-open . 0));
# indent-tabs-mode: nil;
# End:
#
# vim: filetype=python:expandtab:shiftwidth=4:tabstop=4:softtabstop=4

