/* ------------- */
/* --- MCA.h --- */
/* ------------- */

/*
 * Copyright (c) 2016 Lionel Lacassagne, LIP6, UPMC, CNRS
 * Init  : 2016/03/03
 */

// Multi/Many Cores Connected Component Computation en Analysis
// extension of pixel-based and run-based algorithm to manycores with distributed memory

#ifndef __MCA_H__
#define __MCA_H__

#include "ecc_features.h"

#include "nrc_os_config.h"

#if TARGET_OS == GIETVM
    #include <user_lock.h>
    #include <user_barrier.h>
#elif TARGET_OS == LINUX
    #include <pthread.h>
#endif


// QM : using mutex lock instead of spinlock,
// because apparently spinlocks cause a bug in valgrind
// (solved but the installed version is not recent enough)
// cf. https://bugs.kde.org/show_bug.cgi?id=336435
pthread_mutex_t print_lock;




#if MCA_VERBOSE_LEVEL >= 1
    #define MCA_VERBOSE1(X) ({             \
        pthread_mutex_lock(&print_lock);   \
        X;                                 \
        pthread_mutex_unlock(&print_lock); \
        })
#else
    #define MCA_VERBOSE1(X)
#endif

#if MCA_VERBOSE_LEVEL >= 2
    #define MCA_VERBOSE2(X) ({             \
        pthread_mutex_lock(&print_lock);   \
        X;                                 \
        pthread_mutex_unlock(&print_lock); \
        })
#else
    #define MCA_VERBOSE2(X)
#endif

#if MCA_VERBOSE_LEVEL >= 3
    #define MCA_VERBOSE3(X) ({             \
        pthread_mutex_lock(&print_lock);   \
        X;                                 \
        pthread_mutex_unlock(&print_lock); \
        })
#else
    #define MCA_VERBOSE3(X)
#endif


typedef struct sMCA {
    int p, np;         // numero du processeur et nb total de processeurs
    
    uint8  ** X; // image source
    uint32 ** E; // image d'etiquette 32 bits
    
    int width;
    int height;
    
    int i0, i1;
    int j0, j1;
    
    uint32 e0, e1; // indice pour chaque bande
    uint32 ne;     // indice max d'etiquettes utilise par bande

    int alpha;     // puissance de 2 >= a la taille d'un bloc
    uint32  * T;   // table d'quivalence table (Rosenfeld) ou d'indices (Warp)
    uint32 ** D;   // distributed table (instanciee dans chaque worker)
    
    RegionStats * stats;
    RegionStats ** F;
    
    struct sMCA * mca;   // pointeur vers le maitre (pour les esclaves)
    struct sMCA ** mcas; // tableau de pointeurs vers les workers

    // For pyramidal barriers
    int nb_level;
    pthread_barrier_t * barriers;
} MCA;
 
void MCA_Error(char * msg);

MCA * MCA_pConstructor_Empty(void);
MCA * MCA_pDestructor(MCA * mca);

void MCA_Set_ImageX(MCA * mca, uint8  ** X);
void MCA_Set_ImageL(MCA * mca, uint32 ** E);

void MCA_Set_Size(MCA * mca, int width, int height);
void MCA_Set_NP(MCA * mca, int np);

uint32 MCA_CalcMaxLabels(int connection, uint32 height, uint32 width);

void MCA_Display_Parameters(MCA *mca);

void MCA_Initialize(MCA * mca);
void MCA_Finalize  (MCA * mca);


// master to workers
void MCA_Scatter_ImageX(MCA * mca);

// workers to master
void MCA_Gather_ImageL(MCA * mca);

// CC run
void MCA_Rosenfeld(MCA * mca);
void MCA_MPar(MCA * mca);
void MCA_Warp(MCA * mca);


#endif // __MCA_H__

