/* ---------------- */
/* --- nrio2f.c --- */
/* ---------------- */

/*
 * Copyright (c) 2000-2014, Lionel Lacassagne, All rights reserved
 * Univ Paris Sud XI, CNRS
 * 
 * Distributed under the Boost Software License, Version 1.0
 * see accompanying file LICENSE.txt or copy it at
 * http://www.boost.org/LICENSE_1_0.txt
 */

#include <stdio.h>
#include <stddef.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>

#include "nrc_os_config.h"
#include "mypredef.h"
#include "nrtype.h"
#include "nrdef.h"
#include "nrmacro.h"
#include "nrkernel.h"

#include "nralloc1.h"
#include "nralloc2.h"
#include "nrio0.h"
#include "nrio1.h"
#include "nrio2.h"

/*
 * --------------------
 * --- write_matrix ---
 * --------------------
 */

#undef write_type_matrix
#define write_type_matrix(t) \
void short_name(t,write_,matrix)(t ** m, int32_t nrl, int32_t nrh, int32_t ncl, int32_t nch, char * format, char * filename) \
{                                                                \
    FILE * f = fopen(filename, "wt");                            \
    if (f == NULL) {                                             \
        nrerror("Can't open file %s in %s", filename, __func__); \
    }                                                            \
    for (int32_t i = nrl; i <= nrh; i++) {                       \
        for (int32_t j = ncl; j <= nch; j++) {                   \
            fprintf(f, format, elem);                            \
        }                                                        \
        fprintf(f, "\n");                                        \
    }                                                            \
    fclose(f);                                                   \
}

#undef elem
#define elem m[i][j]
write_type_matrix(int8_t);
write_type_matrix(uint8_t);
write_type_matrix(int16_t);
write_type_matrix(uint16_t);
write_type_matrix(int32_t);
write_type_matrix(uint32_t);
write_type_matrix(int64_t);
write_type_matrix(uint64_t);
write_type_matrix(float);
write_type_matrix(double);
#undef elem
#define elem m[i][j].r, m[i][j].g, m[i][j].b
write_type_matrix(rgb8);
#undef elem
#define elem m[i][j].r, m[i][j].g, m[i][j].b, m[i][j].x
write_type_matrix(rgbx8);


/*
 * ----------------------
 * --- write_matrix_T ---
 * ----------------------
 */


#undef write_type_matrix_T
#define write_type_matrix_T(t) \
void short_name(t,write_,matrix_T)(t ** m, int32_t nrl, int32_t nrh, int32_t ncl, int32_t nch, char * format, char * filename) \
{                                                                \
    FILE * f = fopen(filename, "wt");                            \
    if (f == NULL) {                                             \
        nrerror("Can't open file %s in %s", filename, __func__); \
    }                                                            \
    for (int32_t j = ncl; j <= nch; j++) {                       \
        for (int32_t i = nrl; i <= nrh; i++) {                   \
            fprintf(f, format, elem);                            \
        }                                                        \
        fprintf(f, "\n");                                        \
    }                                                            \
    fclose(f);                                                   \
}

#undef elem
#define elem m[i][j]
write_type_matrix_T(int8_t);
write_type_matrix_T(uint8_t);
write_type_matrix_T(int16_t);
write_type_matrix_T(uint16_t);
write_type_matrix_T(int32_t);
write_type_matrix_T(uint32_t);
write_type_matrix_T(int64_t);
write_type_matrix_T(uint64_t);
write_type_matrix_T(float);
write_type_matrix_T(double);
#undef elem
#define elem m[i][j].r, m[i][j].g, m[i][j].b
write_type_matrix_T(rgb8);
#undef elem
#define elem m[i][j].r, m[i][j].g, m[i][j].b, m[i][j].x
write_type_matrix_T(rgbx8);



/*
 * ---------------------------
 * --- write_matrix_number ---
 * ---------------------------
 */


#undef write_type_matrix_number
#define write_type_matrix_number(t) \
void short_name(t,write_,matrix_number)(t ** m, int32_t nrl, int32_t nrh, int32_t ncl, int32_t nch, char * format, char * filename) \
{                                                                \
    FILE * f = fopen(filename, "wt");                            \
    if (f == NULL) {                                             \
        nrerror("Can't open file %s in %s", filename, __func__); \
    }                                                            \
    fprintf(f, "%5c", '#');                                      \
    for (int32_t j = ncl; j <= nch; j++) {                       \
        fprintf(f, format, j);                                   \
    }                                                            \
    fprintf(f, "\n");                                            \
    for (int32_t i = nrl; i <= nrh; i++) {                       \
        fprintf(f, "[%3d]", i);                                  \
        for (int32_t j = ncl; j <= nch; j++) {                   \
            fprintf(f, format, m[i][j]);                         \
        }                                                        \
        fprintf(f, "\n");                                        \
    }                                                            \
    fclose(f);                                                   \
}


#undef elem
#define elem m[i][j]
write_type_matrix_number(int8_t);
write_type_matrix_number(uint8_t);
write_type_matrix_number(int16_t);
write_type_matrix_number(uint16_t);
write_type_matrix_number(int32_t);
write_type_matrix_number(uint32_t);
write_type_matrix_number(int64_t);
write_type_matrix_number(uint64_t);
write_type_matrix_number(float);
write_type_matrix_number(double);
#undef elem
#define elem m[i][j].r, m[i][j].g, m[i][j].b
write_type_matrix_number(rgb8);
#undef elem
#define elem m[i][j].r, m[i][j].g, m[i][j].b, m[i][j].x
write_type_matrix_number(rgbx8);


/*
 * -----------------------------
 * --- write_matrix_T_number ---
 * -----------------------------
 */


#undef write_type_matrix_T_number
#define write_type_matrix_T_number(t) \
void short_name(t,write_,matrix_T_number)(t ** m, int32_t nrl, int32_t nrh, int32_t ncl, int32_t nch, char * format, char * filename) \
{                                                                \
    FILE * f = fopen(filename, "wt");                            \
    if (f == NULL) {                                             \
        nrerror("Can't open file %s in %s", filename, __func__); \
    }                                                            \
    fprintf(f, "%5c", '#');                                      \
    for (int32_t i = nrl; i <= nrh; i++) {                       \
        fprintf(f, format, i);                                   \
    }                                                            \
    fprintf(f, "\n");                                            \
    for (int32_t j = ncl; j <= nch; j++) {                       \
        fprintf(f, "[%3d]", j);                                  \
        for (int32_t i = nrl; i <= nrh; i++) {                   \
            fprintf(f, format, m[i][j]);                         \
        }                                                        \
        fprintf(f, "\n");                                        \
    }                                                            \
    fclose(f);                                                   \
}


#undef elem
#define elem m[i][j]
write_type_matrix_T_number(int8_t);
write_type_matrix_T_number(uint8_t);
write_type_matrix_T_number(int16_t);
write_type_matrix_T_number(uint16_t);
write_type_matrix_T_number(int32_t);
write_type_matrix_T_number(uint32_t);
write_type_matrix_T_number(int64_t);
write_type_matrix_T_number(uint64_t);
write_type_matrix_T_number(float);
write_type_matrix_T_number(double);
#undef elem
#define elem m[i][j].r, m[i][j].g, m[i][j].b
write_type_matrix_T_number(rgb8);
#undef elem
#define elem m[i][j].r, m[i][j].g, m[i][j].b, m[i][j].x
write_type_matrix_T_number(rgbx8);


/*
 * ---------------------
 * --- fwrite_matrix ---
 * ---------------------
 */

#undef fwrite_type_matrix
#define fwrite_type_matrix(t) \
void short_name(t,fwrite_,matrix)(t ** m, int32_t nrl, int32_t nrh, int32_t ncl, int32_t nch, char * filename) \
{                                                                \
    int32_t ncol = nch - ncl + 1;                                \
    FILE * f = fopen(filename, "wb");                            \
    if (f == NULL) {                                             \
        nrerror("Can't open file %s in %s", filename, __func__); \
    }                                                            \
    for (int32_t i = nrl; i <= nrh; i++) {                       \
        fwrite(m[i] + nrl, sizeof(t), ncol, f);                  \
    }                                                            \
    fclose(f);                                                   \
}

fwrite_type_matrix(int8_t);
fwrite_type_matrix(uint8_t);
fwrite_type_matrix(int16_t);
fwrite_type_matrix(uint16_t);
fwrite_type_matrix(int32_t);
fwrite_type_matrix(uint32_t);
fwrite_type_matrix(int64_t);
fwrite_type_matrix(uint64_t);
fwrite_type_matrix(float);
fwrite_type_matrix(double);
fwrite_type_matrix(rgb8);
fwrite_type_matrix(rgbx8);



/*
 * --------------------
 * --- fread_matrix ---
 * --------------------
 */

#undef fread_type_matrix
#define fread_type_matrix(t) \
void short_name(t,fread_,matrix)(char * filename, t ** m, int32_t nrl, int32_t nrh, int32_t ncl, int32_t nch)  \
{                                                                \
    int32_t ncol = nch - ncl + 1;                                \
    int32_t nread;                                               \
    FILE * f = fopen(filename, "rb");                            \
    if (f == NULL) {                                             \
        nrerror("Can't open file %s in %s", filename, __func__); \
    }                                                            \
    for (int32_t i = nrl; i <= nrh; i++) {                       \
        nread = fread(m[i] + ncl, sizeof(t), ncol, f);           \
        if (nread != ncol) {                                     \
            nrerror("%s : can't read data", __func__);           \
        }                                                        \
    }                                                            \
    fclose(f);                                                   \
}

fread_type_matrix(int8_t);
fread_type_matrix(uint8_t);
fread_type_matrix(int16_t);
fread_type_matrix(uint16_t);
fread_type_matrix(int32_t);
fread_type_matrix(uint32_t);
fread_type_matrix(int64_t);
fread_type_matrix(uint64_t);
fread_type_matrix(float);
fread_type_matrix(double);
fread_type_matrix(rgb8);
fread_type_matrix(rgbx8);



#undef write_type_matrix_positive
#define write_type_matrix_positive(t) \
void short_name(t,write_,matrix_positive)(t ** m, int32_t i0, int32_t i1, int32_t j0, int32_t j1, int32_t iformat, char * filename) \
{                                                                \
    char * format;                                               \
    char * str;                                                  \
    FILE * f;                                                    \
    select_display_positive_parameters(iformat, &format, &str);  \
    f = fopen(filename, "wt");                                   \
    if (f == NULL) {                                             \
        nrerror("Can't open file %s in %s", filename, __func__); \
    }                                                            \
    for (int32_t i = i0; i <= i1; i++) {                         \
        for (int32_t j = j0; j <= j1; j++) {                     \
            if (m[i][j] != 0) {                                  \
                fprintf(f, format, m[i][j]);                     \
            }                                                    \
            else {                                               \
                fprintf(f, "%s", str);                           \
            }                                                    \
        }                                                        \
        fprintf(f, "\n");                                        \
    }                                                            \
    fclose(f);                                                   \
}

write_type_matrix_positive(uint8_t);
write_type_matrix_positive(uint16_t);
write_type_matrix_positive(uint32_t);


/* ------------------------ */
/* -- PGM IO for bmatrix -- */
/* ------------------------ */


/* ------------------------------------------ */
static char * readitem(FILE * file, char * buffer)
/* ------------------------------------------ */
    // read a word
    // public domain function: author is unknown
{
    char * aux;
    int32_t k;

    k = 0;
    aux = buffer;
    while (!feof(file)) {
        *aux = fgetc(file);
        switch (k) {
            case 0:
                if (*aux == '#') {
                    k = 1;
                }
                if (isalnum(*aux)) {
                    k = 2;
                    aux++;
                }
                break;
            case 1:
                if (*aux == 0xA) {
                    k = 0;
                }
                break;
            case 2:
                if (!isalnum(*aux)) {
                    *aux = 0;
                    return buffer;
                }
                aux++;
                break;
        }
    }
    *aux = 0;
    return buffer;
}

/* ----------------------------------------------------- */
static void ReadPGMrow(FILE * file, int32_t width, uint8_t * line)
/* ----------------------------------------------------- */
{
    // file is already open (in read) and will not be closed at the end
    fread(&(line[0]), sizeof(uint8_t), width, file);
}


/* ------------------------------------------------------- */
static void WritePGMrow(uint8_t * line, int32_t width, FILE  * file)
/* ------------------------------------------------------- */
{
    // file is already open (in read) and will not be closed at the end
    fwrite(&(line[0]), sizeof(uint8_t), width, file);
}


/* --------------------------------------------------------------------------------------- */
uint8_t ** LoadPGM_ui8matrix(char * filename, int32_t * nrl, int32_t * nrh, int32_t * ncl, int32_t * nch)
/* --------------------------------------------------------------------------------------- */
{
    // only for P5 binary type, not for text type

    int32_t height, width, gris;
    uint8_t ** m;
    FILE * file;

    char * buffer;
    char msg[1024];
    (void) gris;

    buffer = (char *) calloc(80, sizeof(char));

    // open file
    file = fopen(filename, "rb");
    if (file == NULL) {
        snprintf(msg, 1024, "ouverture du fichier %s impossible dans %s\n", filename, __func__);
        nrerror(msg);
    }

    // read PGM header
    readitem(file, buffer);
    if (strcmp(buffer, "P5") != 0) {
        nrerror("Entete du fichier %s invalide\n", filename);
    }

    width  = atoi(readitem(file, buffer));
    height = atoi(readitem(file, buffer));
    gris   = atoi(readitem(file, buffer));

    *nrl = 0;
    *nrh = height - 1;
    *ncl = 0;
    *nch = width - 1;
    m = ui8matrix(*nrl, *nrh, *ncl, *nch);

    for(int32_t i = 0; i < height; i++) {
        ReadPGMrow(file, width, m[i]);
    }

    fclose(file);
    free(buffer);

    return m;
}


/* ----------------------------------------------------------------------------------------------- */
void LoadPGM_ui8matrix2(char * filename, int32_t * nrl, int32_t * nrh, int32_t * ncl, int32_t * nch, uint8_t ** m)
/* ----------------------------------------------------------------------------------------------- */
{
    // only for P5 binary type, not for text type
    int32_t height, width, gris;
    FILE * file;

    char * buffer;
    char msg[1024];
    (void) gris;

    buffer = (char *) calloc(80, sizeof(char));
    /* ouverture du fichier */
    file = fopen(filename, "rb");
    if (file == NULL) {
        snprintf(msg, 1024, "ouverture du fichier %s impossible dans %s\n", filename, __func__);
        nrerror(msg);
    }

    /* lecture de l'entete du fichier pgm */
    readitem(file, buffer);
    if (strcmp(buffer, "P5") != 0) {
        nrerror("Entete du fichier %s invalide\n", filename);
    }

    width  = atoi(readitem(file, buffer));
    height = atoi(readitem(file, buffer));
    gris   = atoi(readitem(file, buffer));

    *nrl = 0;
    *nrh = height - 1;
    *ncl = 0;
    *nch = width - 1;

    for (int32_t i = 0; i < height; i++) {
        ReadPGMrow(file, width, m[i]);
    }

    fclose(file);
    free(buffer);
}


/* -------------------------------------------------------------------------------------------- */
void MLoadPGM_ui8matrix(char * filename, int32_t nrl, int32_t nrh, int32_t ncl, int32_t nch, uint8_t ** m)
/* -------------------------------------------------------------------------------------------- */
{
    // only for P5 binary type, not for text type
    int32_t height, width, gris;
    FILE * file;

    char * buffer;
    char msg[1024];
    (void) gris;

    buffer = (char *) calloc(80, sizeof(char));
    /* ouverture du fichier */
    file = fopen(filename, "rb");
    if (file == NULL) {
        snprintf(msg, 1024, "ouverture du fichier %s impossible dans %s\n", filename, __func__);
        nrerror(msg);
    }

    /* lecture de l'entete du fichier pgm */
    readitem(file, buffer);
    if (strcmp(buffer, "P5") != 0) {
        nrerror("Entete du fichier %s invalide\n", filename);
    }

    width  = atoi(readitem(file, buffer));
    height = atoi(readitem(file, buffer));
    gris   = atoi(readitem(file, buffer));

    for (int32_t i = 0; i < height; i++) {
        ReadPGMrow(file, width, m[i]);
    }

    fclose(file);
    free(buffer);
}


/* ------------------------------------------------------------------------------------------ */
void SavePGM_ui8matrix(uint8_t ** m, int32_t nrl, int32_t nrh, int32_t ncl, int32_t nch, char * filename)
/* ------------------------------------------------------------------------------------------ */
{
    int nrow = nrh - nrl + 1;
    int ncol = nch - ncl + 1;

    char buffer[80];
    char msg[1024];

    FILE * file;

    file = fopen(filename, "wb");
    if (file == NULL) {
        snprintf(msg, 1024, "ouverture du fichier %s impossible dans %s\n", filename, __func__);
        nrerror(msg);
    }

    /* enregistrement de l'image au format rpgm */
    snprintf(buffer, 80, "P5\n%d %d\n255\n", ncol, nrow);
    fwrite(buffer, strlen(buffer), 1, file);
    for (int32_t i = nrl; i <= nrh; i++) {
        WritePGMrow(m[i], ncol, file);
    }

    /* fermeture du fichier */
    fclose(file);
}


/* --------------------------- */
/* -- PNM IO for rgb8matrix -- */
/* --------------------------- */

/* ------------------------------------------------------ */
static void ReadPNMrow(FILE * file, int32_t width, uint8_t * line)
/* ------------------------------------------------------ */
{
    /* Le fichier est ouvert (en lecture) et ne sera pas ferme a la fin */
    fread(&(line[0]), sizeof(uint8_t), 3 * sizeof(uint8_t) * width, file);
}


/* ------------------------------------------------------- */
static void WritePNMrow(uint8_t * line, int32_t width, FILE * file)
/* ------------------------------------------------------- */
{
    /* Le fichier est deja ouvert et ne sera pas ferme a la fin */
    fwrite(&(line[0]), sizeof(uint8_t), 3 * sizeof(uint8_t) * width, file);
}


/* ------------------------------------------------------------------------------------------ */
rgb8 ** LoadPPM_rgb8matrix(char * filename, int32_t * nrl, int32_t * nrh, int32_t * ncl, int32_t * nch)
/* ------------------------------------------------------------------------------------------ */
{
    /* cette version ne lit plus que le type P6 */
    int32_t height, width, gris;
    rgb8 ** m;
    FILE * file;

    char * buffer;
    char msg[1024];
    (void) gris;

    buffer = (char *) calloc(80, sizeof(char));
    /* ouverture du fichier */
    file = fopen(filename,"rb");
    if (file == NULL) {
        snprintf(msg, 1024, "ouverture du fichier %s impossible dans %s\n", filename, __func__);
        nrerror(msg);
    }

    /* lecture de l'entete du fichier pgm */
    readitem(file, buffer);
    if (strcmp(buffer, "P6") != 0) {
        nrerror("Entete du fichier %s invalide\n", filename);
    }

    width  = atoi(readitem(file, buffer));
    height = atoi(readitem(file, buffer));
    gris   = atoi(readitem(file, buffer));

    *nrl = 0;
    *nrh = height - 1;
    *ncl = 0;
    *nch = width - 1;
    m = rgb8matrix(*nrl, *nrh, *ncl, *nch);

    for (int32_t i = 0; i < height; i++) {
        ReadPNMrow(file, width, (uint8_t *) m[i]);
    }

    fclose(file);
    free(buffer);

    return m;
}


/* -------------------------------------------------------------------------------------------------- */
void LoadPPM_rgb8matrix2(char * filename, int * nrl, int * nrh, int * ncl, int * nch, rgb8 ** m)
    /* -------------------------------------------------------------------------------------------------- */
{
    /* cette version ne lit plus que le type P6 */
    int height, width, gris;
    FILE * file;

    char * buffer;
    char msg[1024];
    (void) gris;

    buffer = (char *) calloc(80, sizeof(char));
    /* ouverture du fichier */
    file = fopen(filename, "rb");
    if (file == NULL) {
        snprintf(msg, 1024, "ouverture du fichier %s impossible dans LoadPPM_rgb8matrix2\n", filename);
        nrerror(msg);
    }

    /* lecture de l'entete du fichier pgm */
    readitem(file, buffer);
    if (strcmp(buffer, "P6") != 0) {
        nrerror("Entete du fichier %s invalide\n", filename);
    }
    //nrerror("entete du fichier %s invalide\n", filename);

    width  = atoi(readitem(file, buffer));
    height = atoi(readitem(file, buffer));
    gris   = atoi(readitem(file, buffer));

    *nrl = 0;
    *nrh = height - 1;
    *ncl = 0;
    *nch = width - 1;

    for (int32_t i = 0; i < height; i++) {
        ReadPNMrow(file, width, (uint8_t *) m[i]);
    }
    fclose(file);
    free(buffer);
}


/* ------------------------------------------------------------------------------------------------- */
void SavePPM_rgb8matrix(rgb8 ** m, int32_t nrl, int32_t nrh, int32_t ncl, int32_t nch, char * filename)
/* ------------------------------------------------------------------------------------------------- */
{
    int32_t nrow = nrh - nrl + 1;
    int32_t ncol = nch - ncl + 1;

    char buffer[80];
    char msg[1024];

    FILE * file;

    file = fopen(filename, "wb");
    if (file == NULL) {
        snprintf(msg, 1024, "ouverture du fichier %s impossible dans %s\n", filename, __func__);
        nrerror(msg);
    }

    /* enregistrement de l'image au format rpgm */
    snprintf(buffer, 80, "P6\n%d %d\n255\n", ncol, nrow);
    fwrite(buffer, strlen(buffer), 1, file);
    for (int32_t i = nrl; i <= nrh; i++) {
        WritePNMrow((uint8_t *) m[i], ncol, file);
    }

    /* fermeture du fichier */
    fclose(file);
}


// Local Variables:
// tab-width: 4
// c-basic-offset: 4
// c-file-offsets:((innamespace . 0)(inline-open . 0))
// indent-tabs-mode: nil
// End:

// vim: filetype=cpp:expandtab:shiftwidth=4:tabstop=4:softtabstop=4

