#include "game.h"
#include "disp.h"
#include <stdio.h>
#include <malloc.h>
#include <user_lock.h>

///////////////////////
// Global variables
///////////////////////

unsigned char*           buf;                // one image buffer
void *                   sts;                // buffer status
volatile unsigned int    slice_x;            // slice index (shared)
volatile unsigned int    slice_count;        // slice count (shared)
sqt_lock_t               slice_get_lock;     // slice index lock
sqt_lock_t               slice_done_lock;    // slice index lock
pthread_t                trdid[1024];        // thread identifiers array
Game                     game;               // Game state
unsigned int             fbuf_x_size;        // FBF width
unsigned int             fbuf_y_size;        // FBF height

// Textures
unsigned char*  g_tex[] =
{
    NULL, // 0
    NULL, // rock
    NULL, // door
    NULL, // handle
    NULL, // wood
};

//////////////////////////
// Exported functions
//////////////////////////

//////////////////////////////////////////
__attribute__((constructor)) void render()
{
    unsigned int slice;

    while ( game.exit == 0 ) 
    {
        dispRenderSlice( &slice );
    }
    giet_pthread_exit( NULL );
}

////////////////////////////////////////
__attribute__((constructor)) void main()
{
    unsigned int w, h, p;   // platform parameters
 
    unsigned int i, j, n;   // indexes for loops

    // get private TTY
    giet_tty_alloc(0);

    giet_tty_printf("\n[RAYCAST] enters main at cycle %d\n", 
                    giet_proctime() );

    // get and check platform parameters
    giet_procs_number(&w, &h, &p);

    giet_pthread_assert( (w<=16) , "[RAYCAST ERROR] check hardware config" );
    giet_pthread_assert( (h<=16) , "[RAYCAST ERROR] check hardware config" );
    giet_pthread_assert( (p<= 4) , "[RAYCAST ERROR] check hardware config" );

    // check frame buffer availability
    giet_fbf_size( &fbuf_x_size , &fbuf_y_size );
    giet_pthread_assert( ((fbuf_x_size) && (fbuf_y_size)) ,
                         "[RAYCAST ERROR] no frame buffer available" );

    // compute total number of threads
    unsigned int nthreads = w * h * p;

    // Initialize heap for each cluster
    for (i = 0; i < w; i++)
        for (j = 0; j < h; j++)
            heap_init(i, j);

    // Initialize lock protecting slice allocator 
    sqt_lock_init( &slice_get_lock,  w , h , p );
    sqt_lock_init( &slice_done_lock, w , h , p );

    // Allocate buffer and status for CMA
    buf = malloc(fbuf_x_size * fbuf_x_size);
    sts = malloc(64);

    // Get frame buffer ownership
    giet_fbf_alloc();
    
    // Get a CMA channel for one single user buffer
    giet_fbf_cma_alloc( 1 );
    
    // Register the user buffer and status
    giet_fbf_cma_init_buf( 0 , buf , sts );

    // Start Chained buffer DMA
    giet_fbf_cma_start();

    // Load textures
    g_tex[1] = dispLoadTexture("misc/rock_32.raw");
    g_tex[2] = dispLoadTexture("misc/door_32.raw");
    g_tex[3] = dispLoadTexture("misc/handle_32.raw");
    g_tex[4] = dispLoadTexture("misc/wood_32.raw");

    // Initialise game
    gameInit();

    giet_tty_printf("\n[RAYCAST] initialisation completed at cycle %d\n",
                    giet_proctime() );

    // launch other threads to speed render
    for ( n = 1 ; n < nthreads ; n++ )
    {
        if ( giet_pthread_create( &trdid[n],
                                  NULL,                  // no attribute
                                  &render,
                                  NULL ) )               // no argument
        {
            giet_tty_printf("\n[RAYCAST ERROR] creating thread %d\n", n );
            giet_pthread_exit( NULL );
        }
    }

    // Game main loop : display one frame 
    // and get one player move per iteration
    while ( game.exit == 0 ) 
    {
        // check user buffer empty
        giet_fbf_cma_check( 0 );

        // re-initialise synchronisation variables 
        // to start parallel synthesis
        slice_count = 0;
        slice_x     = 0;

        // contribute to synthesis 
        unsigned int slice;
        while ( dispRenderSlice( &slice ) );

        // Wait last slice completion
        while (slice_count < fbuf_x_size)  giet_tty_printf(" ");

        // display image
        giet_fbf_cma_display( 0 );

        // get new player position [x,y,dir]
        gameControl();
    }

    giet_pthread_exit( NULL );

}  // end main()
