
/*************************************************************************************
 * File : caches_interfaces.h
 * Date : 27/06/2005
 * Authors : F.Arzel
 * Release ; V0.0
 * It is released under the GNU Public License (hopefully).
 * 
 * This file defines a standard "non-blocking cache interface"; separate interfaces
 * are defined for non-blocking instruction cache and non-blocking data cache.
 *************************************************************************************/

#ifndef CACHES_INTERFACES_H
#define CACHES_INTERFACES_H

#include <systemc.h>

/*************************************************************************************
 ********************************        Type           ******************************
 *************************************************************************************/

// ===== Type Icache =====
#define ITYPE_READ           0  // 00
#define ITYPE_LOCK           1  // 01
#define ITYPE_INVALIDATE     2  // 10
#define ITYPE_PREFETCH       3  // 11

// ===== Type Dcache =====
#define DTYPE_READ           0  // 000
#define DTYPE_LOCK           1  // 001
#define DTYPE_INVALIDATE     2  // 010
#define DTYPE_PREFETCH       3  // 011
#define DTYPE_WRITE          4  // 100
#define DTYPE_WRITE_ACK      5  // 101
#define DTYPE_FLUSH          6  // 110
                                // 111

// ===== Size access =====
#define ACCESS_SIZE          2
#define ACCESS_8             0
#define ACCESS_16            1
#define ACCESS_32            2
#define ACCESS_64            3

// ===== Error       =====
#define ERR_NO               0  // 00
#define ERR_BUS              1  // 01
#define ERR_TLB              2  // 10
#define ERR_UNKNOW           3  // 11

// Return the number of bytes of a access
unsigned int access_nb_bytes(sc_uint<ACCESS_SIZE> access)
{
  return 1<<access;
}

/*************************************************************************************
 ********************************  ICACHE interface     ******************************
 *************************************************************************************/

/***  ICACHE signals  ***/

template < unsigned short TRDID_SIZE, 
           unsigned short PKTID_SIZE,
           unsigned short ADDR_SIZE ,
	   unsigned short INS_SIZE  ,
           unsigned short NB_ICACHE  >

struct ICACHE_SIGNALS
{
  sc_signal<bool>                   REQ_VAL;
  sc_signal<bool>                   REQ_ACK;
  sc_signal<sc_uint<TRDID_SIZE> >   REQ_TRDID;
  sc_signal<sc_uint<PKTID_SIZE> >   REQ_PKTID;
  sc_signal<sc_uint<ADDR_SIZE> >    REQ_ADDR;
  sc_signal<sc_uint<2> >            REQ_TYPE;
  
  sc_signal<bool>                   RSP_VAL;
  sc_signal<bool>                   RSP_ACK;
  sc_signal<sc_uint<TRDID_SIZE> >   RSP_TRDID;
  sc_signal<sc_uint<PKTID_SIZE> >   RSP_PKTID;
  sc_signal<sc_uint<INS_SIZE> >     RSP_INS  [NB_ICACHE];
  sc_signal<sc_uint<2> >            RSP_ERR;

#ifdef NONAME_RENAME
  void rename (char * prefixe)
  {
    char rename [100];

    sprintf(rename,"%s_REQ_VAL"      ,prefixe);
    REQ_VAL               .rename(rename);
    sprintf(rename,"%s_REQ_ACK"      ,prefixe);
    REQ_ACK               .rename(rename);
    sprintf(rename,"%s_REQ_TRDID"    ,prefixe);
    REQ_TRDID             .rename(rename);
    sprintf(rename,"%s_REQ_PKTID"    ,prefixe);
    REQ_PKTID             .rename(rename);
    sprintf(rename,"%s_REQ_ADDR"     ,prefixe);
    REQ_ADDR              .rename(rename);
    sprintf(rename,"%s_REQ_TYPE"     ,prefixe);
    REQ_TYPE              .rename(rename);
    
    sprintf(rename,"%s_RSP_VAL"      ,prefixe);
    RSP_VAL               .rename(rename);
    sprintf(rename,"%s_RSP_ACK"      ,prefixe);
    RSP_ACK               .rename(rename);
    sprintf(rename,"%s_RSP_TRDID"    ,prefixe);
    RSP_TRDID             .rename(rename);
    sprintf(rename,"%s_RSP_PKTID"    ,prefixe);
    RSP_PKTID             .rename(rename);
    for (unsigned int it_nb_icache = 0; it_nb_icache < NB_ICACHE; it_nb_icache ++)
      {
	sprintf(rename,"%s_RSP_INS  [%d]",prefixe,it_nb_icache);
	RSP_INS [it_nb_icache].rename(rename);
      }//it_nb_icache
    sprintf(rename,"%s_RSP_ERR"      ,prefixe);
    RSP_ERR               .rename(rename);
  }
#endif //NONAME_RENAME
};

/***  ICACHE cache ports  ***/

template< unsigned short TRDID_SIZE, 
          unsigned short PKTID_SIZE,
          unsigned short ADDR_SIZE ,
	  unsigned short INS_SIZE  ,
          unsigned short NB_ICACHE >

struct ICACHE_CACHE_PORTS
{

  sc_in <bool>                      REQ_VAL;
  sc_out<bool>                      REQ_ACK;
  sc_in <sc_uint<TRDID_SIZE> >      REQ_TRDID;
  sc_in <sc_uint<PKTID_SIZE> >      REQ_PKTID;
  sc_in <sc_uint<ADDR_SIZE> >       REQ_ADDR;
  sc_in <sc_uint<2> >               REQ_TYPE;
  
  sc_out<bool>                      RSP_VAL;
  sc_in <bool>                      RSP_ACK;
  sc_out<sc_uint<TRDID_SIZE> >      RSP_TRDID;
  sc_out<sc_uint<PKTID_SIZE> >      RSP_PKTID;
  sc_out<sc_uint<INS_SIZE> >        RSP_INS [NB_ICACHE];
  sc_out<sc_uint<2> >               RSP_ERR;

  void operator ()
    (ICACHE_SIGNALS<TRDID_SIZE,PKTID_SIZE,ADDR_SIZE,INS_SIZE,NB_ICACHE> &signals)
  {
    REQ_VAL    (signals.REQ_VAL);
    REQ_ACK    (signals.REQ_ACK);
    REQ_TRDID  (signals.REQ_TRDID);
    REQ_PKTID  (signals.REQ_PKTID);     
    REQ_ADDR   (signals.REQ_ADDR);
    REQ_TYPE   (signals.REQ_TYPE);
    
    RSP_VAL    (signals.RSP_VAL);
    RSP_ACK    (signals.RSP_ACK);
    RSP_TRDID  (signals.RSP_TRDID);
    RSP_PKTID  (signals.RSP_PKTID);
    for (unsigned int i = 0; i < NB_ICACHE; i++)
      RSP_INS[i]    (signals.RSP_INS[i]);
    RSP_ERR    (signals.RSP_ERR);
  };
  
  void operator ()
    (ICACHE_CACHE_PORTS<TRDID_SIZE,PKTID_SIZE,ADDR_SIZE,INS_SIZE,NB_ICACHE> &ports)
  {
    
    REQ_VAL    (ports.REQ_VAL);
    REQ_ACK    (ports.REQ_ACK);
    REQ_TRDID  (ports.REQ_TRDID);
    REQ_PKTID  (ports.REQ_PKTID);     
    REQ_ADDR   (ports.REQ_ADDR);
    REQ_TYPE   (ports.REQ_TYPE);
    
    RSP_VAL    (ports.RSP_VAL);
    RSP_ACK    (ports.RSP_ACK);
    RSP_TRDID  (ports.RSP_TRDID);
    RSP_PKTID  (ports.RSP_PKTID);
    for (unsigned int i = 0; i < NB_ICACHE; i++)
      RSP_INS[i]    (ports.RSP_INS[i]);
    RSP_ERR    (ports.RSP_ERR);
  };

#ifdef NONAME_RENAME
  void rename (char * prefixe)
  {
    char rename [100];

    sprintf(rename,"%s_REQ_VAL"      ,prefixe);
    REQ_VAL               .rename(rename);
    sprintf(rename,"%s_REQ_ACK"      ,prefixe);
    REQ_ACK               .rename(rename);
    sprintf(rename,"%s_REQ_TRDID"    ,prefixe);
    REQ_TRDID             .rename(rename);
    sprintf(rename,"%s_REQ_PKTID"    ,prefixe);
    REQ_PKTID             .rename(rename);
    sprintf(rename,"%s_REQ_ADDR"     ,prefixe);
    REQ_ADDR              .rename(rename);
    sprintf(rename,"%s_REQ_TYPE"     ,prefixe);
    REQ_TYPE              .rename(rename);
    
    sprintf(rename,"%s_RSP_VAL"      ,prefixe);
    RSP_VAL               .rename(rename);
    sprintf(rename,"%s_RSP_ACK"      ,prefixe);
    RSP_ACK               .rename(rename);
    sprintf(rename,"%s_RSP_TRDID"    ,prefixe);
    RSP_TRDID             .rename(rename);
    sprintf(rename,"%s_RSP_PKTID"    ,prefixe);
    RSP_PKTID             .rename(rename);
    for (unsigned int it_nb_icache = 0; it_nb_icache < NB_ICACHE; it_nb_icache ++)
      {
	sprintf(rename,"%s_RSP_INS  [%d]",prefixe,it_nb_icache);
	RSP_INS [it_nb_icache].rename(rename);
      }//it_nb_icache
    sprintf(rename,"%s_RSP_ERR"      ,prefixe);
    RSP_ERR               .rename(rename);
  }
#endif //NONAME_RENAME  
};

/***  ICACHE processor ports  ***/

template< unsigned short TRDID_SIZE, 
          unsigned short PKTID_SIZE,
          unsigned short ADDR_SIZE ,
	  unsigned short INS_SIZE  ,
          unsigned short NB_ICACHE >

struct ICACHE_PROCESSOR_PORTS
{

  sc_out<bool>                   REQ_VAL;
  sc_in <bool>                   REQ_ACK;
  sc_out<sc_uint<TRDID_SIZE> >   REQ_TRDID;
  sc_out<sc_uint<PKTID_SIZE> >   REQ_PKTID;
  sc_out<sc_uint<ADDR_SIZE> >    REQ_ADDR;
  sc_out<sc_uint<2> >            REQ_TYPE;
  
  sc_in <bool>                   RSP_VAL;
  sc_out<bool>                   RSP_ACK;
  sc_in <sc_uint<TRDID_SIZE> >   RSP_TRDID;
  sc_in <sc_uint<PKTID_SIZE> >   RSP_PKTID;
  sc_in <sc_uint<INS_SIZE> >     RSP_INS [NB_ICACHE];
  sc_in <sc_uint<2> >            RSP_ERR;

  void operator ()
    (ICACHE_SIGNALS<TRDID_SIZE,PKTID_SIZE,ADDR_SIZE,INS_SIZE,NB_ICACHE> &signals)
  {
    
    REQ_VAL    (signals.REQ_VAL);
    REQ_ACK    (signals.REQ_ACK);
    REQ_TRDID  (signals.REQ_TRDID);
    REQ_PKTID  (signals.REQ_PKTID);     
    REQ_ADDR   (signals.REQ_ADDR);
    REQ_TYPE   (signals.REQ_TYPE);
    
    RSP_VAL    (signals.RSP_VAL);
    RSP_ACK    (signals.RSP_ACK);
    RSP_TRDID  (signals.RSP_TRDID);
    RSP_PKTID  (signals.RSP_PKTID);
    for (unsigned int i = 0; i < NB_ICACHE; i++)
      RSP_INS[i]    (signals.RSP_INS[i]);
    RSP_ERR    (signals.RSP_ERR);
    
  };
  
  void operator ()
    (ICACHE_PROCESSOR_PORTS<TRDID_SIZE,PKTID_SIZE,ADDR_SIZE,INS_SIZE,NB_ICACHE> &ports)
  {
    
    REQ_VAL    (ports.REQ_VAL);
    REQ_ACK    (ports.REQ_ACK);
    REQ_TRDID  (ports.REQ_TRDID);
    REQ_PKTID  (ports.REQ_PKTID);     
    REQ_ADDR   (ports.REQ_ADDR);
    REQ_TYPE   (ports.REQ_TYPE);
    
    RSP_VAL    (ports.RSP_VAL);
    RSP_ACK    (ports.RSP_ACK);
    RSP_TRDID  (ports.RSP_TRDID);
    RSP_PKTID  (ports.RSP_PKTID);
    for (unsigned int i = 0; i < NB_ICACHE; i++)
      RSP_INS[i]    (ports.RSP_INS[i]);
    RSP_ERR    (ports.RSP_ERR);
    
  };
#ifdef NONAME_RENAME
  void rename (char * prefixe)
  {
    char rename [100];

    sprintf(rename,"%s_REQ_VAL"      ,prefixe);
    REQ_VAL               .rename(rename);
    sprintf(rename,"%s_REQ_ACK"      ,prefixe);
    REQ_ACK               .rename(rename);
    sprintf(rename,"%s_REQ_TRDID"    ,prefixe);
    REQ_TRDID             .rename(rename);
    sprintf(rename,"%s_REQ_PKTID"    ,prefixe);
    REQ_PKTID             .rename(rename);
    sprintf(rename,"%s_REQ_ADDR"     ,prefixe);
    REQ_ADDR              .rename(rename);
    sprintf(rename,"%s_REQ_TYPE"     ,prefixe);
    REQ_TYPE              .rename(rename);
    
    sprintf(rename,"%s_RSP_VAL"      ,prefixe);
    RSP_VAL               .rename(rename);
    sprintf(rename,"%s_RSP_ACK"      ,prefixe);
    RSP_ACK               .rename(rename);
    sprintf(rename,"%s_RSP_TRDID"    ,prefixe);
    RSP_TRDID             .rename(rename);
    sprintf(rename,"%s_RSP_PKTID"    ,prefixe);
    RSP_PKTID             .rename(rename);
    for (unsigned int it_nb_icache = 0; it_nb_icache < NB_ICACHE; it_nb_icache ++)
      {
	sprintf(rename,"%s_RSP_INS  [%d]",prefixe,it_nb_icache);
	RSP_INS [it_nb_icache].rename(rename);
      }//it_nb_icache
    sprintf(rename,"%s_RSP_ERR"      ,prefixe);
    RSP_ERR               .rename(rename);
  }
#endif //NONAME_RENAME 
};

/*************************************************************************************
 ********************************  DCACHE interface     ******************************
 *************************************************************************************/

/***  DCACHE signals  ***/

template< unsigned short TRDID_SIZE, 
          unsigned short PKTID_SIZE,
          unsigned short ADDR_SIZE ,
          unsigned short DATA_SIZE >

struct DCACHE_SIGNALS
{

  sc_signal<bool>                     REQ_VAL;
  sc_signal<bool>                     REQ_ACK;
  sc_signal<sc_uint<TRDID_SIZE> >     REQ_TRDID;
  sc_signal<sc_uint<PKTID_SIZE> >     REQ_PKTID;
  sc_signal<sc_uint<ADDR_SIZE> >      REQ_ADDR;
  sc_signal<sc_uint<3> >              REQ_TYPE;
  sc_signal<bool>                     REQ_UNC;
  sc_signal<sc_uint<ACCESS_SIZE> >    REQ_ACCESS;
  sc_signal<sc_uint<DATA_SIZE> >      REQ_WDATA;
  
  sc_signal<bool>                     RSP_VAL;
  sc_signal<bool>                     RSP_ACK;
  sc_signal<sc_uint<TRDID_SIZE> >     RSP_TRDID;
  sc_signal<sc_uint<PKTID_SIZE> >     RSP_PKTID;
  sc_signal<sc_uint<DATA_SIZE> >      RSP_RDATA;
  sc_signal<sc_uint<2> >              RSP_ERR;

#ifdef NONAME_RENAME
  void rename (char * prefixe)
  {
    char rename [100];

    sprintf(rename,"%s_REQ_VAL"      ,prefixe);
    REQ_VAL               .rename(rename);
    sprintf(rename,"%s_REQ_ACK"      ,prefixe);
    REQ_ACK               .rename(rename);
    sprintf(rename,"%s_REQ_TRDID"    ,prefixe);
    REQ_TRDID             .rename(rename);
    sprintf(rename,"%s_REQ_PKTID"    ,prefixe);
    REQ_PKTID             .rename(rename);
    sprintf(rename,"%s_REQ_ADDR"     ,prefixe);
    REQ_ADDR              .rename(rename);
    sprintf(rename,"%s_REQ_TYPE"     ,prefixe);
    REQ_TYPE              .rename(rename);
    sprintf(rename,"%s_REQ_UNC"      ,prefixe);
    REQ_UNC               .rename(rename);
    sprintf(rename,"%s_REQ_ACCESS"   ,prefixe);
    REQ_ACCESS            .rename(rename);
    sprintf(rename,"%s_REQ_WDATA"    ,prefixe);
    REQ_WDATA             .rename(rename);
    
    sprintf(rename,"%s_RSP_VAL"      ,prefixe);
    RSP_VAL               .rename(rename);
    sprintf(rename,"%s_RSP_ACK"      ,prefixe);
    RSP_ACK               .rename(rename);
    sprintf(rename,"%s_RSP_TRDID"    ,prefixe);
    RSP_TRDID             .rename(rename);
    sprintf(rename,"%s_RSP_PKTID"    ,prefixe);
    RSP_PKTID             .rename(rename);
    sprintf(rename,"%s_RSP_RDATA"    ,prefixe);
    RSP_RDATA             .rename(rename);
    sprintf(rename,"%s_RSP_ERR"      ,prefixe);
    RSP_ERR               .rename(rename);
  }
#endif //NONAME_RENAME
};

/***  DCACHE cache ports  ***/

template< unsigned short TRDID_SIZE, 
          unsigned short PKTID_SIZE,
          unsigned short ADDR_SIZE,
          unsigned short DATA_SIZE >

struct DCACHE_CACHE_PORTS
{
  sc_in <bool>                     REQ_VAL;
  sc_out<bool>                     REQ_ACK;
  sc_in <sc_uint<TRDID_SIZE> >     REQ_TRDID;
  sc_in <sc_uint<PKTID_SIZE> >     REQ_PKTID;
  sc_in <sc_uint<ADDR_SIZE> >      REQ_ADDR;
  sc_in <sc_uint<3> >              REQ_TYPE;
  sc_in <bool>                     REQ_UNC;
  sc_in <sc_uint<ACCESS_SIZE> >    REQ_ACCESS;
  sc_in <sc_uint<DATA_SIZE> >      REQ_WDATA;
  
  sc_out<bool>                     RSP_VAL;
  sc_in <bool>                     RSP_ACK;
  sc_out<sc_uint<TRDID_SIZE> >     RSP_TRDID;
  sc_out<sc_uint<PKTID_SIZE> >     RSP_PKTID;
  sc_out<sc_uint<DATA_SIZE> >      RSP_RDATA;
  sc_out<sc_uint<2> >              RSP_ERR;

  void operator ()
    (DCACHE_SIGNALS<TRDID_SIZE,PKTID_SIZE,ADDR_SIZE,DATA_SIZE> &signals)
  {
    
    REQ_VAL    (signals.REQ_VAL);
    REQ_ACK    (signals.REQ_ACK);
    REQ_TRDID  (signals.REQ_TRDID);
    REQ_PKTID  (signals.REQ_PKTID);     
    REQ_ADDR   (signals.REQ_ADDR);
    REQ_TYPE   (signals.REQ_TYPE);
    REQ_UNC    (signals.REQ_UNC);
    REQ_ACCESS (signals.REQ_ACCESS);
    REQ_WDATA  (signals.REQ_WDATA);
    
    RSP_VAL    (signals.RSP_VAL);
    RSP_ACK    (signals.RSP_ACK);
    RSP_TRDID  (signals.RSP_TRDID);
    RSP_PKTID  (signals.RSP_PKTID);
    RSP_RDATA  (signals.RSP_RDATA);
    RSP_ERR    (signals.RSP_ERR);
    
  };
  
  void operator ()
    (DCACHE_CACHE_PORTS<TRDID_SIZE,PKTID_SIZE,ADDR_SIZE,DATA_SIZE> &ports)
  {
    
    REQ_VAL    (ports.REQ_VAL);
    REQ_ACK    (ports.REQ_ACK);
    REQ_TRDID  (ports.REQ_TRDID);
    REQ_PKTID  (ports.REQ_PKTID);     
    REQ_ADDR   (ports.REQ_ADDR);
    REQ_TYPE   (ports.REQ_TYPE);
    REQ_UNC    (ports.REQ_UNC);
    REQ_ACCESS (ports.REQ_ACCESS);
    REQ_WDATA  (ports.REQ_WDATA);
    
    RSP_VAL    (ports.RSP_VAL);
    RSP_ACK    (ports.RSP_ACK);
    RSP_TRDID  (ports.RSP_TRDID);
    RSP_PKTID  (ports.RSP_PKTID);
    RSP_RDATA  (ports.RSP_RDATA);
    RSP_ERR    (ports.RSP_ERR);
    
  };

#ifdef NONAME_RENAME
  void rename (char * prefixe)
  {
    char rename [100];

    sprintf(rename,"%s_REQ_VAL"      ,prefixe);
    REQ_VAL               .rename(rename);
    sprintf(rename,"%s_REQ_ACK"      ,prefixe);
    REQ_ACK               .rename(rename);
    sprintf(rename,"%s_REQ_TRDID"    ,prefixe);
    REQ_TRDID             .rename(rename);
    sprintf(rename,"%s_REQ_PKTID"    ,prefixe);
    REQ_PKTID             .rename(rename);
    sprintf(rename,"%s_REQ_ADDR"     ,prefixe);
    REQ_ADDR              .rename(rename);
    sprintf(rename,"%s_REQ_TYPE"     ,prefixe);
    REQ_TYPE              .rename(rename);
    sprintf(rename,"%s_REQ_UNC"      ,prefixe);
    REQ_UNC               .rename(rename);
    sprintf(rename,"%s_REQ_ACCESS"   ,prefixe);
    REQ_ACCESS            .rename(rename);
    sprintf(rename,"%s_REQ_WDATA"    ,prefixe);
    REQ_WDATA             .rename(rename);
    
    sprintf(rename,"%s_RSP_VAL"      ,prefixe);
    RSP_VAL               .rename(rename);
    sprintf(rename,"%s_RSP_ACK"      ,prefixe);
    RSP_ACK               .rename(rename);
    sprintf(rename,"%s_RSP_TRDID"    ,prefixe);
    RSP_TRDID             .rename(rename);
    sprintf(rename,"%s_RSP_PKTID"    ,prefixe);
    RSP_PKTID             .rename(rename);
    sprintf(rename,"%s_RSP_RDATA"    ,prefixe);
    RSP_RDATA             .rename(rename);
    sprintf(rename,"%s_RSP_ERR"      ,prefixe);
    RSP_ERR               .rename(rename);
  }
#endif //NONAME_RENAME  
};

/***  DCACHE processor ports  ***/

template< unsigned short TRDID_SIZE, 
          unsigned short PKTID_SIZE,
          unsigned short ADDR_SIZE ,
          unsigned short DATA_SIZE >

struct DCACHE_PROCESSOR_PORTS
{

  sc_out<bool>                     REQ_VAL;
  sc_in <bool>                     REQ_ACK;
  sc_out<sc_uint<TRDID_SIZE> >     REQ_TRDID;
  sc_out<sc_uint<PKTID_SIZE> >     REQ_PKTID;
  sc_out<sc_uint<ADDR_SIZE> >      REQ_ADDR;
  sc_out<sc_uint<3> >              REQ_TYPE;
  sc_out<bool>                     REQ_UNC;
  sc_out<sc_uint<ACCESS_SIZE> >    REQ_ACCESS;
  sc_out<sc_uint<DATA_SIZE> >      REQ_WDATA;
  
  sc_in <bool>                     RSP_VAL;
  sc_out<bool>                     RSP_ACK;
  sc_in <sc_uint<TRDID_SIZE> >     RSP_TRDID;
  sc_in <sc_uint<PKTID_SIZE> >     RSP_PKTID;
  sc_in <sc_uint<DATA_SIZE> >      RSP_RDATA;
  sc_in <sc_uint<2> >              RSP_ERR;

  void operator ()
    (DCACHE_SIGNALS<TRDID_SIZE,PKTID_SIZE,ADDR_SIZE,DATA_SIZE> &signals)
  {
    
    REQ_VAL    (signals.REQ_VAL);
    REQ_ACK    (signals.REQ_ACK);
    REQ_TRDID  (signals.REQ_TRDID);
    REQ_PKTID  (signals.REQ_PKTID);     
    REQ_ADDR   (signals.REQ_ADDR);
    REQ_TYPE   (signals.REQ_TYPE);
    REQ_UNC    (signals.REQ_UNC);
    REQ_ACCESS (signals.REQ_ACCESS);
    REQ_WDATA  (signals.REQ_WDATA);
    
    RSP_VAL    (signals.RSP_VAL);
    RSP_ACK    (signals.RSP_ACK);
    RSP_TRDID  (signals.RSP_TRDID);
    RSP_PKTID  (signals.RSP_PKTID);
    RSP_RDATA  (signals.RSP_RDATA);
    RSP_ERR    (signals.RSP_ERR);
    
  };
  
  void operator ()
    (DCACHE_PROCESSOR_PORTS<TRDID_SIZE,PKTID_SIZE,ADDR_SIZE,DATA_SIZE> &ports)
  {
    
    REQ_VAL    (ports.REQ_VAL);
    REQ_ACK    (ports.REQ_ACK);
    REQ_TRDID  (ports.REQ_TRDID);
    REQ_PKTID  (ports.REQ_PKTID);     
    REQ_ADDR   (ports.REQ_ADDR);
    REQ_TYPE   (ports.REQ_TYPE);
    REQ_UNC    (ports.REQ_UNC);
    REQ_ACCESS (ports.REQ_ACCESS);
    REQ_WDATA  (ports.REQ_WDATA);
    
    RSP_VAL    (ports.RSP_VAL);
    RSP_ACK    (ports.RSP_ACK);
    RSP_TRDID  (ports.RSP_TRDID);
    RSP_PKTID  (ports.RSP_PKTID);
    RSP_RDATA  (ports.RSP_RDATA);
    RSP_ERR    (ports.RSP_ERR);
    
  };

#ifdef NONAME_RENAME
  void rename (char * prefixe)
  {
    char rename [100];

    sprintf(rename,"%s_REQ_VAL"      ,prefixe);
    REQ_VAL               .rename(rename);
    sprintf(rename,"%s_REQ_ACK"      ,prefixe);
    REQ_ACK               .rename(rename);
    sprintf(rename,"%s_REQ_TRDID"    ,prefixe);
    REQ_TRDID             .rename(rename);
    sprintf(rename,"%s_REQ_PKTID"    ,prefixe);
    REQ_PKTID             .rename(rename);
    sprintf(rename,"%s_REQ_ADDR"     ,prefixe);
    REQ_ADDR              .rename(rename);
    sprintf(rename,"%s_REQ_TYPE"     ,prefixe);
    REQ_TYPE              .rename(rename);
    sprintf(rename,"%s_REQ_UNC"      ,prefixe);
    REQ_UNC               .rename(rename);
    sprintf(rename,"%s_REQ_ACCESS"   ,prefixe);
    REQ_ACCESS            .rename(rename);
    sprintf(rename,"%s_REQ_WDATA"    ,prefixe);
    REQ_WDATA             .rename(rename);
    
    sprintf(rename,"%s_RSP_VAL"      ,prefixe);
    RSP_VAL               .rename(rename);
    sprintf(rename,"%s_RSP_ACK"      ,prefixe);
    RSP_ACK               .rename(rename);
    sprintf(rename,"%s_RSP_TRDID"    ,prefixe);
    RSP_TRDID             .rename(rename);
    sprintf(rename,"%s_RSP_PKTID"    ,prefixe);
    RSP_PKTID             .rename(rename);
    sprintf(rename,"%s_RSP_RDATA"    ,prefixe);
    RSP_RDATA             .rename(rename);
    sprintf(rename,"%s_RSP_ERR"      ,prefixe);
    RSP_ERR               .rename(rename);
  }
#endif //NONAME_RENAME  
};

#endif
