#ifndef morpheo_behavioural_Signal_h
#define morpheo_behavioural_Signal_h

/*
 * $Id: Signal.h 137 2010-02-16 12:35:48Z rosiere $
 *
 * [ Description ]
 * 
 */

#ifdef SYSTEMC
#include "systemc.h"
#endif

#include <stdint.h>
#include <iostream>

#ifdef VHDL
#include "Behavioural/include/Vhdl.h"
#endif

#include "Behavioural/include/Parameters.h"
#include "Behavioural/include/Direction.h"
#include "Behavioural/include/XML.h"
#include "Behavioural/include/Debug_signal.h"
#include "Common/include/ErrorMorpheo.h"
#include "Common/include/ToBase2.h"
#include "Common/include/ToString.h"
#include "Common/include/Debug.h"

namespace morpheo              {
namespace behavioural          {

  typedef enum {UNKNOW                     ,
		BOOL                       ,
		UINT8_T                    ,
		UINT16_T                   ,
		UINT32_T                   ,
		UINT64_T                   } type_info_t;

  typedef enum {PORT_SYSTEMC_NO            , // also, vhdl_no and testbench_no
                PORT_VHDL_YES_TESTBENCH_YES,
		PORT_VHDL_YES_TESTBENCH_NO ,
		PORT_VHDL_NO_TESTBENCH_YES ,
		PORT_VHDL_NO_TESTBENCH_NO  ,
		CLOCK_VHDL_YES             ,
		CLOCK_VHDL_NO              ,
		RESET_VHDL_YES             ,
		RESET_VHDL_NO              } presence_port_t;

  class Signal
  {
    friend class Interface;

    // -----[fields ]----------------------------------------------------
  private   : const std::string     _name          ;
  private   : const direction_t     _direction     ;
  private   : const presence_port_t _presence_port ;
  private   :       uint32_t        _size          ;

  private   : Signal *              _connect_to_signal       ;   // the actual implementaion, this signal link with one signal (but if signal is an output, it can be connect with many signal ...)
  private   : Signal *              _connect_from_signal     ; // producter of signal. If NULL, then producteur is the current entity
  private   : bool                  _is_allocate             ; // Have allocate a sc_in or sc_out port
  private   : void *                _sc_signal               ; // sc_in or sc_out associated at this signal
  private   : void *                _sc_signal_map           ; // sc_out generated this signal
  private   : bool                  _is_map_as_toplevel_dest ; 
  private   : bool                  _is_map_as_component_src ; 
  private   : bool                  _is_map_as_component_dest;
  private   : type_info_t           _type_info               ;

#ifdef VHDL_TESTBENCH
  private   : std::list<std::string> * _list_value    ;
#endif

    // -----[methods ]---------------------------------------------------
  public    :                   Signal                  (std::string     name          ,
						         direction_t     direction     ,
						         uint32_t        size          ,
						         presence_port_t presence_port = PORT_VHDL_YES_TESTBENCH_YES);
  public    :                   Signal                  (const Signal &);
  public    :                   ~Signal                 ();

  public    : std::string       get_name                (void);
  public    : uint32_t          get_size                (void);
  public    : void              set_size                (uint32_t size);
  public    : void              set_size_max            (uint32_t size);

  public    : Signal *          get_connect_to_signal   (void);
  public    : Signal *          get_connect_from_signal (void);
  public    : direction_t       get_direction           (void);
  public    : type_info_t       get_type_info           (void);

  public    : void *            get_sc_signal           (void);

  public    : bool              presence_vhdl           (void);
  public    : bool              presence_testbench      (void);

  public    : bool              test_map                (uint32_t depth, bool top_level, bool is_behavioural);
//public    : bool              test_equi               (uint32_t depth);

  public    : void              link                    (Signal * signal_dest,
							 bool     signal_dest_is_port);

  public    : void              connect                 (Signal * signal_dest);

#ifdef SYSTEMC         
  public    :template <typename T>
             T                  read (void)
    {
      switch (_direction)
	{
	case IN  : {return read_in  <T>();}
	case OUT : {return read_out <T>();}
	default  : throw (ErrorMorpheo ("Signal \""+_name+"\" : direction unknow.\n"));
	}
    }

  public    :template <typename T>
             T                  read_in (void)
    {
      switch (_type_info)
	{
	case BOOL     : return (static_cast<sc_in  <bool    > *>(_sc_signal_map)->read()); 
	case UINT8_T  : return (static_cast<sc_in  <uint8_t > *>(_sc_signal_map)->read()); 
	case UINT16_T : return (static_cast<sc_in  <uint16_t> *>(_sc_signal_map)->read());
	case UINT32_T : return (static_cast<sc_in  <uint32_t> *>(_sc_signal_map)->read()); 
	case UINT64_T : return (static_cast<sc_in  <uint64_t> *>(_sc_signal_map)->read()); 
	default       : throw (ErrorMorpheo ("Signal \""+_name+"\" : type unknow.\n"));
	}
    }

  public    :template <typename T>
             T                  read_out(void)
    {
      switch (_type_info)
	{
	case BOOL     : return (static_cast<sc_out <bool    > *>(_sc_signal_map)->read()); 
	case UINT8_T  : return (static_cast<sc_out <uint8_t > *>(_sc_signal_map)->read()); 
	case UINT16_T : return (static_cast<sc_out <uint16_t> *>(_sc_signal_map)->read());
	case UINT32_T : return (static_cast<sc_out <uint32_t> *>(_sc_signal_map)->read()); 
	case UINT64_T : return (static_cast<sc_out <uint64_t> *>(_sc_signal_map)->read()); 
	default       : throw (ErrorMorpheo ("Signal \""+_name+"\" : type unknow.\n"));
	}
    }

#undef  FUNCTION
#define FUNCTION "Signal::alloc"
  public    : template <typename T>
  void              alloc           (void * sc_signal)
    {
      log_printf(FUNC,Behavioural,FUNCTION,"Begin");

      if (_type_info != UNKNOW)
	throw (ErrorMorpheo (toString(_("Signal \"%s\" : already allocate.\n"),_name.c_str())));

      if (test<T>(_size) == false)
	throw (ErrorMorpheo (toString(_("Signal \"%s\" : size is too small (%d bits) to the associate type (%d bits).\n"),_name.c_str(),_size,8*sizeof(T))));

      _is_allocate    = true;
      _sc_signal      = sc_signal;
      _sc_signal_map  = sc_signal;

      DEBUG_SIGNAL_ADD(sc_signal,_name);

      if (typeid(T) == typeid(bool    ))
	_type_info = BOOL;
      else
      if (typeid(T) == typeid(uint8_t ))
	_type_info = UINT8_T;
      else
      if (typeid(T) == typeid(uint16_t))
	_type_info = UINT16_T;
      else
      if (typeid(T) == typeid(uint32_t))
	_type_info = UINT32_T;
      else
      if (typeid(T) == typeid(uint64_t))
	_type_info = UINT64_T;
      else
	_type_info = UNKNOW;
      
      log_printf(TRACE,Behavioural,FUNCTION, "Allocation of %s (%s, 0x%.8x)", _name.c_str(),toString(_type_info).c_str(), static_cast<uint32_t>(reinterpret_cast<uint64_t>(_sc_signal_map)));

      log_printf(FUNC,Behavioural,FUNCTION,"End");
    }
#endif

#ifdef VHDL
  public    : void              set_port        (Vhdl * & vhdl);
#  ifdef VHDL_TESTBENCH
  public    : Signal *          get_clock       (void);
  public    : Signal *          get_reset       (void);
  public    : uint32_t          get_reset_cycle (bool active_low);

  public    : void              set_signal      (Vhdl * & vhdl);
  public    : void              get_name_vhdl   (std::list<std::string> * & list_signal);

  public    : void              testbench        (void);
  public    : void              testbench_body   (Vhdl           * & vhdl          ,
						  std::string             counter_name  ,
						  std::string             reset_name    );
  public    : void              testbench_test_ok(Vhdl           * & vhdl          );
#  endif
#endif
  public    : XML               toXML           (void);

  public    : friend std::ostream&   operator<<      (std::ostream& output_stream,
						      morpheo::behavioural::Signal & x);

  };
}; // end namespace behavioural          




  template<>           inline std::string toString<morpheo::behavioural::presence_port_t>(const morpheo::behavioural::presence_port_t& x)
  {
    switch (x)
      {
      case morpheo::behavioural::PORT_VHDL_YES_TESTBENCH_YES : return "Port  is in VHDL's model and TestBench's model" ; break;
      case morpheo::behavioural::PORT_VHDL_YES_TESTBENCH_NO  : return "Port  is in VHDL's model                      " ; break;
      case morpheo::behavioural::PORT_VHDL_NO_TESTBENCH_YES  : return "Port  is in                  TestBench's model" ; break;
      case morpheo::behavioural::PORT_VHDL_NO_TESTBENCH_NO   : return "Port  is in none model                        " ; break;
      case morpheo::behavioural::CLOCK_VHDL_YES              : return "Clock is in VHDL's model                     " ; break;
      case morpheo::behavioural::CLOCK_VHDL_NO               : return "Clock is not in VHDL's model                 " ; break;
      case morpheo::behavioural::RESET_VHDL_YES              : return "Reset is in VHDL's model                     " ; break;
      case morpheo::behavioural::RESET_VHDL_NO               : return "Reset is not in VHDL's model                 " ; break;
      default                                                : return "";                                               break;
      }
  }

  typedef enum {UNKNOW                     ,
		BOOL                       ,
		UINT8_T                    ,
		UINT16_T                   ,
		UINT32_T                   ,
		UINT64_T                   } type_info_t;

  template<>           inline std::string toString<morpheo::behavioural::type_info_t>(const morpheo::behavioural::type_info_t& x)
  {
    switch (x)
      {
      case morpheo::behavioural::BOOL     : return "bool"    ; break;
      case morpheo::behavioural::UINT8_T  : return "uint8_t" ; break;
      case morpheo::behavioural::UINT16_T : return "uint16_t"; break;
      case morpheo::behavioural::UINT32_T : return "uint32_t"; break;
      case morpheo::behavioural::UINT64_T : return "uint64_t"; break;
      case morpheo::behavioural::UNKNOW   :
      default                             : return "unknow"  ; break;
      }
  }

}; // end namespace morpheo              

#endif
