#ifndef morpheo_behavioural_Constants_h
#define morpheo_behavioural_Constants_h

namespace morpheo {
namespace behavioural {

  //=========================================================[ Type ]=====
#  define TYPE_ALU                                 0x0        // 000000 - unit multiple
#  define TYPE_BRANCH                              0x1        // 000001 - unit multiple, to a special routing
#  define TYPE_MEMORY                              0x2        // 000010 - unit uniq
#  define TYPE_MAC                                 0x4        // 000100 - unit uniq
#  define TYPE_TIMER                               0x8        // 001000 - unit uniq
#  define TYPE_CUSTOM                              0x10       // 010000 - unit uniq

  //====================================================[ Operation ]=====

  //-------------------------------------------------------[ Memory ]-----
#  define OPERATION_MEMORY_LOAD                    0x0        // 000_0000
#  define OPERATION_MEMORY_LOAD_8_Z                0x0        // 000_0000
#  define OPERATION_MEMORY_LOAD_16_Z               0x20       // 010_0000
#  define OPERATION_MEMORY_LOAD_32_Z               0x40       // 100_0000
#  define OPERATION_MEMORY_LOAD_64_Z               0x60       // 110_0000
#  define OPERATION_MEMORY_LOAD_8_S                0x10       // 001_0000
#  define OPERATION_MEMORY_LOAD_16_S               0x30       // 011_0000
#  define OPERATION_MEMORY_LOAD_32_S               0x50       // 101_0000
#  define OPERATION_MEMORY_LOAD_64_S               0x70       // 111_0000

#  define OPERATION_MEMORY_STORE_8                 0x8        // 000_1000
#  define OPERATION_MEMORY_STORE_16                0x9        // 000_1001
#  define OPERATION_MEMORY_STORE_32                0xa        // 000_1010
#  define OPERATION_MEMORY_STORE_64                0xb        // 000_1011
#  define OPERATION_MEMORY_STORE_HEAD_OK           0xc        // 000_1100
#  define OPERATION_MEMORY_STORE_HEAD_KO           0xd        // 000_1101

#  define OPERATION_MEMORY_LOCK                    0x1        // 000_0001
#  define OPERATION_MEMORY_INVALIDATE              0x2        // 000_0010
#  define OPERATION_MEMORY_PREFETCH                0x3        // 000_0011
#  define OPERATION_MEMORY_FLUSH                   0x6        // 000_0110
#  define OPERATION_MEMORY_SYNCHRONIZATION         0x7        // 000_0111
  
#define is_operation_memory_load(x)             \
  ((x == OPERATION_MEMORY_LOAD_8_Z ) or		\
   (x == OPERATION_MEMORY_LOAD_16_Z) or		\
   (x == OPERATION_MEMORY_LOAD_32_Z) or		\
   (x == OPERATION_MEMORY_LOAD_64_Z) or		\
   (x == OPERATION_MEMORY_LOAD_8_S ) or		\
   (x == OPERATION_MEMORY_LOAD_16_S) or		\
   (x == OPERATION_MEMORY_LOAD_32_S) or		\
   (x == OPERATION_MEMORY_LOAD_64_S) )
  
#define is_operation_memory_store(x)		\
  ((x == OPERATION_MEMORY_STORE_8      ) or	\
   (x == OPERATION_MEMORY_STORE_16     ) or	\
   (x == OPERATION_MEMORY_STORE_32     ) or	\
   (x == OPERATION_MEMORY_STORE_64     ) or	\
   (x == OPERATION_MEMORY_STORE_HEAD_OK) or	\
   (x == OPERATION_MEMORY_STORE_HEAD_KO))

#define is_operation_memory_store_head(x)       \
  ((x == OPERATION_MEMORY_STORE_HEAD_OK) or	\
   (x == OPERATION_MEMORY_STORE_HEAD_KO))
  
#define is_operation_memory_load_signed(x)      \
  ((x == OPERATION_MEMORY_LOAD_8_S ) or		\
   (x == OPERATION_MEMORY_LOAD_16_S) or		\
   (x == OPERATION_MEMORY_LOAD_32_S) or		\
   (x == OPERATION_MEMORY_LOAD_64_S) )

#  define MEMORY_ACCESS_8                          0x0
#  define MEMORY_ACCESS_16                         0x1
#  define MEMORY_ACCESS_32                         0x2
#  define MEMORY_ACCESS_64                         0x3

#  define MEMORY_SIZE_8                            8 
#  define MEMORY_SIZE_16                           16
#  define MEMORY_SIZE_32                           32
#  define MEMORY_SIZE_64                           64

#  define MASK_MEMORY_ACCESS_8                     0x0
#  define MASK_MEMORY_ACCESS_16                    0x1
#  define MASK_MEMORY_ACCESS_32                    0x3
#  define MASK_MEMORY_ACCESS_64                    0x7

#define memory_size(x)							\
  (((x==OPERATION_MEMORY_LOAD_16_Z)or					\
    (x==OPERATION_MEMORY_LOAD_16_S)or					\
    (x==OPERATION_MEMORY_STORE_16 ))?MEMORY_SIZE_16:			\
   (((x==OPERATION_MEMORY_LOAD_32_Z)or					\
     (x==OPERATION_MEMORY_LOAD_32_S)or					\
     (x==OPERATION_MEMORY_STORE_32 ))?MEMORY_SIZE_32:			\
    (((x==OPERATION_MEMORY_LOAD_64_Z)or					\
      (x==OPERATION_MEMORY_LOAD_64_S)or					\
      (x==OPERATION_MEMORY_STORE_64 ))?MEMORY_SIZE_64:MEMORY_SIZE_8)))
  
#define memory_access(x)						\
  (((x==OPERATION_MEMORY_LOAD_16_Z)or					\
    (x==OPERATION_MEMORY_LOAD_16_S)or					\
    (x==OPERATION_MEMORY_STORE_16 ))?MEMORY_ACCESS_16:			\
   (((x==OPERATION_MEMORY_LOAD_32_Z)or					\
     (x==OPERATION_MEMORY_LOAD_32_S)or					\
     (x==OPERATION_MEMORY_STORE_32 ))?MEMORY_ACCESS_32:			\
    (((x==OPERATION_MEMORY_LOAD_64_Z)or					\
      (x==OPERATION_MEMORY_LOAD_64_S)or					\
      (x==OPERATION_MEMORY_STORE_64 ))?MEMORY_ACCESS_64:MEMORY_ACCESS_8)))
  
#define mask_memory_access(x)						\
  (((x==OPERATION_MEMORY_LOAD_16_Z)or					\
    (x==OPERATION_MEMORY_LOAD_16_S)or					\
    (x==OPERATION_MEMORY_STORE_16 ))?MASK_MEMORY_ACCESS_16:		\
   (((x==OPERATION_MEMORY_LOAD_32_Z)or					\
     (x==OPERATION_MEMORY_LOAD_32_S)or					\
     (x==OPERATION_MEMORY_STORE_32 ))?MASK_MEMORY_ACCESS_32:		\
    (((x==OPERATION_MEMORY_LOAD_64_Z)or					\
      (x==OPERATION_MEMORY_LOAD_64_S)or					\
      (x==OPERATION_MEMORY_STORE_64 ))?MASK_MEMORY_ACCESS_64:MASK_MEMORY_ACCESS_8)))

  //---------------------------------------------[ Functionnal Unit ]-----
#  define OPERATION_ALU_L_ADD                      0x0        // 000_0000 l.add   , l.addi
#  define OPERATION_ALU_L_ADDC                     0x1        // 000_0000 l.addc  , l.addic
#  define OPERATION_ALU_L_SUB                      0x2        // 000_0000 l.sub
#  define OPERATION_ALU_L_MUL                      0x3        // 000_0000 l.mul   , l.muli
#  define OPERATION_ALU_L_MULU                     0x4        // 000_0000 l.mulu
#  define OPERATION_ALU_L_DIV                      0x5        // 000_0000 l.div
#  define OPERATION_ALU_L_DIVU                     0x6        // 000_0000 l.divu
#  define OPERATION_ALU_L_AND                      0x7        // 000_0000 l.and   , l.andi
#  define OPERATION_ALU_L_OR                       0x8        // 000_0000 l.or    , l.ori
#  define OPERATION_ALU_L_XOR                      0x9        // 000_0000 l.xor   , l.xori
#  define OPERATION_ALU_L_TEST_F                   0xa        // 000_0000 l.bf
#  define OPERATION_ALU_L_TEST_NF                  0xb        // 000_0000 l.bnf
#  define OPERATION_ALU_L_JALR                     0xc        // 000_0000 l.jal   , l.jalr , l.jr
#  define OPERATION_ALU_L_SLL                      0xd        // 000_0000 l.sll   , l.slli
#  define OPERATION_ALU_L_SRA                      0xe        // 000_0000 l.sra   , l.srai
#  define OPERATION_ALU_L_SRL                      0xf        // 000_0000 l.srl   , l.srli
#  define OPERATION_ALU_L_ROR                      0x10       // 000_0000 l.ror   , l.rori
#  define OPERATION_ALU_L_MOVHI                    0x11       // 000_0000 l.movhi
#  define OPERATION_ALU_L_EXTEND_S                 0x12       // 000_0000 l.extbs , l.exths, l.extws
#  define OPERATION_ALU_L_EXTEND_Z                 0x13       // 000_0000 l.extbz , l.exthz, l.extwz
#  define OPERATION_ALU_L_CMOV                     0x14       // 000_0000 l.cmov
#  define OPERATION_ALU_L_FF1                      0x15       // 000_0000 l.ff1
#  define OPERATION_ALU_L_FL1                      0x16       // 000_0000 l.fl1
#  define OPERATION_ALU_L_MFSPR                    0x17       // 000_0000 l.mfspr
#  define OPERATION_ALU_L_MTSPR                    0x18       // 000_0000 l.mtspr
#  define OPERATION_ALU_L_SFGES                    0x19       // 000_0000 l.sfges , l.sfges
#  define OPERATION_ALU_L_SFGEU                    0x1a       // 000_0000 l.sfgeu , l.sfgeu
#  define OPERATION_ALU_L_SFGTS                    0x1b       // 000_0000 L.sfgts , l.sfgts
#  define OPERATION_ALU_L_SFGTU                    0x1c       // 000_0000 l.sfgtu , l.sfgtu
#  define OPERATION_ALU_L_SFLES                    0x1d       // 000_0000 l.sfles , l.sfles
#  define OPERATION_ALU_L_SFLEU                    0x1e       // 000_0000 l.sfleu , l.sfleu
#  define OPERATION_ALU_L_SFLTS                    0x1f       // 000_0000 l.sflts , l.sflts
#  define OPERATION_ALU_L_SFLTU                    0x20       // 000_0000 l.sfltu , l.sfltu
#  define OPERATION_ALU_L_SFEQ                     0x21       // 000_0000 l.sfeq  , l.sfeqi
#  define OPERATION_ALU_L_SFNE                     0x22       // 000_0000 l.sfne  , l.sfnei
#  define OPERATION_ALU_L_MAC                      0x23       // 000_0000 l.mac   , l.maci
#  define OPERATION_ALU_L_MACRC                    0x24       // 000_0000 l.macrc
#  define OPERATION_ALU_L_MSB                      0x25       // 000_0000 l.msb

  //-------------------------------------------------------[ Custom ]-----

#  define OPERATION_CUSTOM_L_1                     0x40       // 100_0000
#  define OPERATION_CUSTOM_L_2                     0x41       // 100_0001
#  define OPERATION_CUSTOM_L_3                     0x42       // 100_0010
#  define OPERATION_CUSTOM_L_4                     0x43       // 100_0011
#  define OPERATION_CUSTOM_L_5                     0x44       // 100_0100
#  define OPERATION_CUSTOM_L_6                     0x45       // 100_0101
#  define OPERATION_CUSTOM_L_7                     0x46       // 100_0110
#  define OPERATION_CUSTOM_L_8                     0x47       // 100_0111
#  define OPERATION_CUSTOM_LF_1_D                  0x48       // 100_1000
#  define OPERATION_CUSTOM_LF_1_S                  0x49       // 100_1001
#  define OPERATION_CUSTOM_LV_1                    0x4c       // 100_1100
#  define OPERATION_CUSTOM_LV_2                    0x4d       // 100_1101
#  define OPERATION_CUSTOM_LV_3                    0x4e       // 100_1110
#  define OPERATION_CUSTOM_LV_4                    0x4f       // 100_1111

#  define MAX_OPERATION                            0x80

  //====================================================[ Exception ]=====
  // Exception - OpenRISC

#  define SIZE_EXCEPTION                           5

#  define EXCEPTION_NONE                           0x00       // none exception
#  define EXCEPTION_RESET                          0x01       // software or hardware reset
#  define EXCEPTION_BUS_ERROR                      0x02       // Access at a invalid physical adress
#  define EXCEPTION_DATA_PAGE                      0x03       // No matching or page violation protection in pages tables
#  define EXCEPTION_INSTRUCTION_PAGE               0x04       // No matching or page violation protection in pages tables
#  define EXCEPTION_TICK_TIMER                     0x05       // Tick timer interruption
#  define EXCEPTION_ALIGNMENT                      0x06       // Load/Store access is not aligned
#  define EXCEPTION_ILLEGAL_INSTRUCTION            0x07       // Instruction is illegal (no implemented)
#  define EXCEPTION_INTERRUPT                      0x08       // External interruption 
#  define EXCEPTION_DATA_TLB                       0x09       // DTLB miss
#  define EXCEPTION_INSTRUCTION_TLB                0x0a       // ITLB miss
#  define EXCEPTION_RANGE                          0x0b       // Overflow or access at a unimplemented register or context
#  define EXCEPTION_SYSCALL                        0x0c       // System Call
#  define EXCEPTION_FLOATING_POINT                 0x0d       // Caused by a floating instruction
#  define EXCEPTION_TRAP                           0x0e       // L.trap or debug unit
#  define EXCEPTION_RESERVED_0                     0x0f       // Reserved for a futur usage
#  define EXCEPTION_RESERVED_1                     0x10       // Reserved for a futur usage
#  define EXCEPTION_RESERVED_2                     0x11       // Reserved for a futur usage
#  define EXCEPTION_RESERVED_3                     0x12       // Reserved for a futur usage
#  define EXCEPTION_RESERVED_4                     0x13       // Reserved for a futur usage
#  define EXCEPTION_RESERVED_5                     0x14       // Reserved for a futur usage
#  define EXCEPTION_RESERVED_6                     0x15       // Reserved for implemented specific exceptions
#  define EXCEPTION_RESERVED_7                     0x16       // Reserved for implemented specific exceptions
#  define EXCEPTION_RESERVED_8                     0x17       // Reserved for implemented specific exceptions
#  define EXCEPTION_RESERVED_9                     0x18       // Reserved for implemented specific exceptions
#  define EXCEPTION_CUSTOM_0                       0x19       // Reserved for custom exceptions
#  define EXCEPTION_CUSTOM_1                       0x1a       // Reserved for custom exceptions
#  define EXCEPTION_CUSTOM_2                       0x1b       // Reserved for custom exceptions
#  define EXCEPTION_CUSTOM_3                       0x1c       // Reserved for custom exceptions
#  define EXCEPTION_CUSTOM_4                       0x1d       // Reserved for custom exceptions
#  define EXCEPTION_CUSTOM_5                       0x1e       // Reserved for custom exceptions
#  define EXCEPTION_CUSTOM_6                       0x1f       // Reserved for custom exceptions

#define exception_to_address(x) (x<<8)

  // Exception Execution
#  define EXCEPTION_MEMORY_NONE                    0x0        // Load/Store generate none exception
#  define EXCEPTION_MEMORY_ALIGNMENT               0x1        // Load/Store access is not aligned
#  define EXCEPTION_MEMORY_DATA_TLB                0x2        // DTLB miss
#  define EXCEPTION_MEMORY_DATA_PAGE               0x3        // No matching or page violation protection in pages tables
#  define EXCEPTION_MEMORY_BUS_ERROR               0x4        // Access at a invalid physical address
#  define EXCEPTION_MEMORY_MISS_SPECULATION        0x5        // Load miss speculation
#  define EXCEPTION_MEMORY_LOAD_SPECULATIVE        0x6        // The load is speculative : write in register file, but don't commit

#  define EXCEPTION_CUSTOM_NONE                    0x1        // Custom unit generate none exception
#  define EXCEPTION_CUSTOM_CUST_0                  0x1        // Reserved for custom exceptions
#  define EXCEPTION_CUSTOM_CUST_1                  0x2        // Reserved for custom exceptions
#  define EXCEPTION_CUSTOM_CUST_2                  0x3        // Reserved for custom exceptions
#  define EXCEPTION_CUSTOM_CUST_3                  0x4        // Reserved for custom exceptions
#  define EXCEPTION_CUSTOM_CUST_4                  0x5        // Reserved for custom exceptions
#  define EXCEPTION_CUSTOM_CUST_5                  0x6        // Reserved for custom exceptions
#  define EXCEPTION_CUSTOM_CUST_6                  0x7        // Reserved for custom exceptions

#  define EXCEPTION_ALU_NONE                       0x0        // Functionnal unit generate none exception
#  define EXCEPTION_ALU_RANGE                      0x1        // 
#  define EXCEPTION_ALU_SPR_ACCESS_INVALID         0x2        // SPR     present in ALU but not compatible privilege
#  define EXCEPTION_ALU_SPR_ACCESS_NOT_COMPLETE    0x3        // SPR not present in ALU

  //=======================================================[ dcache ]=====

  //--------------------------------------------------[ dcache_type ]-----

#  define SIZE_DCACHE_TYPE                              4

#  define DCACHE_TYPE_LOAD                              0x0        // 0000
#  define DCACHE_TYPE_LOCK                              0x1        // 0001
#  define DCACHE_TYPE_INVALIDATE                        0x2        // 0010
#  define DCACHE_TYPE_PREFETCH                          0x3        // 0011
//#define DCACHE_TYPE_                                  0x4        // 0100
//#define DCACHE_TYPE_                                  0x5        // 0101
#  define DCACHE_TYPE_FLUSH                             0x6        // 0110
#  define DCACHE_TYPE_SYNCHRONIZATION                   0x7        // 0111
#  define DCACHE_TYPE_STORE_8                           0x8        // 1000
#  define DCACHE_TYPE_STORE_16                          0x9        // 1001
#  define DCACHE_TYPE_STORE_32                          0xa        // 1010
#  define DCACHE_TYPE_STORE_64                          0xb        // 1011
//#define DCACHE_TYPE_                                  0xc        // 1100
//#define DCACHE_TYPE_                                  0xd        // 1101
//#define DCACHE_TYPE_                                  0xe        // 1110
//#define DCACHE_TYPE_                                  0xf        // 1111

// just take the 4 less significative bits.
#define operation_to_dcache_type(x) (x&0xf)

  //-------------------------------------------------[ dcache_error ]-----

#  define SIZE_DCACHE_ERROR                             1

#  define DCACHE_ERROR_NONE                             0x0
#  define DCACHE_ERROR_BUS_ERROR                        0x1

  //=================================================[ special_data ]=====

#  define SIZE_SPECIAL_DATA                             2

// Position of flag in "rename register SR" (NOT IN "SR")
#  define FLAG_POSITION_F                               0x0         // Conditionnal branch flag
#  define FLAG_POSITION_CY                              0x1         // Carry was produced by last arithmetic operation
#  define FLAG_POSITION_OV                              0x0         // Overflow occured during last arithmetic operation

#  define FLAG_F                                        (1<<FLAG_POSITION_F ) // Conditionnal branch flag
#  define FLAG_CY                                       (1<<FLAG_POSITION_CY) // Carry was produced by last arithmetic operation
#  define FLAG_OV                                       (1<<FLAG_POSITION_OV) // Overflow occured during last arithmetic operation

  //==========================================================[ spr ]=====

  enum
    {
      GROUP_SYSTEM_AND_CONTROL,  // 0
      GROUP_DMMU,                // 1
      GROUP_IMMU,                // 2
      GROUP_DCACHE,              // 3
      GROUP_ICACHE,              // 4
      GROUP_MAC,                 // 5
      GROUP_DEBUG,               // 6
      GROUP_PERFORMANCE_COUNTER, // 7
      GROUP_POWER_MANAGEMENT,    // 8
      GROUP_PIC,                 // 9
      GROUP_TICK_TIMER,          // 10
      GROUP_FLOATING_POINT,      // 11
      GROUP_RESERVED_1,          // 12     
      GROUP_RESERVED_2,          // 13     
      GROUP_RESERVED_3,          // 14     
      GROUP_RESERVED_4,          // 15     
      GROUP_RESERVED_5,          // 16     
      GROUP_RESERVED_6,          // 17     
      GROUP_RESERVED_7,          // 18     
      GROUP_RESERVED_8,          // 19     
      GROUP_RESERVED_9,          // 20     
      GROUP_RESERVED_10,         // 21     
      GROUP_RESERVED_11,         // 22     
      GROUP_RESERVED_12,         // 23
      GROUP_CUSTOM_1,            // 24     
      GROUP_CUSTOM_2,            // 25     
      GROUP_CUSTOM_3,            // 26     
      GROUP_CUSTOM_4,            // 27     
      GROUP_CUSTOM_5,            // 28     
      GROUP_CUSTOM_6,            // 29     
      GROUP_CUSTOM_7,            // 30     
      GROUP_CUSTOM_8             // 31     
    };

#  define NB_GROUP                                 32
#  define NB_REG_GROUP_SYSTEM_AND_CONTROL          1536
#  define NB_REG_GROUP_DMMU                        1536
#  define NB_REG_GROUP_IMMU                        1536
#  define NB_REG_GROUP_DCACHE                      6
#  define NB_REG_GROUP_ICACHE                      4
#  define NB_REG_GROUP_MAC                         3
#  define NB_REG_GROUP_DEBUG                       22
#  define NB_REG_GROUP_PERFORMANCE_COUNTER         16
#  define NB_REG_GROUP_POWER_MANAGEMENT            1
#  define NB_REG_GROUP_PIC                         3
#  define NB_REG_GROUP_TICK_TIMER                  2
#  define NB_REG_GROUP_FLOATING_POINT              0
#  define NB_REG_GROUP_RESERVED_1                  0
#  define NB_REG_GROUP_RESERVED_2                  0
#  define NB_REG_GROUP_RESERVED_3                  0
#  define NB_REG_GROUP_RESERVED_4                  0
#  define NB_REG_GROUP_RESERVED_5                  0
#  define NB_REG_GROUP_RESERVED_6                  0
#  define NB_REG_GROUP_RESERVED_7                  0
#  define NB_REG_GROUP_RESERVED_8                  0
#  define NB_REG_GROUP_RESERVED_9                  0
#  define NB_REG_GROUP_RESERVED_10                 0
#  define NB_REG_GROUP_RESERVED_11                 0
#  define NB_REG_GROUP_RESERVED_12                 0
#  define NB_REG_GROUP_CUSTOM_1                    0
#  define NB_REG_GROUP_CUSTOM_2                    0
#  define NB_REG_GROUP_CUSTOM_3                    0
#  define NB_REG_GROUP_CUSTOM_4                    0
#  define NB_REG_GROUP_CUSTOM_5                    0
#  define NB_REG_GROUP_CUSTOM_6                    0
#  define NB_REG_GROUP_CUSTOM_7                    0
#  define NB_REG_GROUP_CUSTOM_8                    0
					           
					           
  // GROUP_MAC				           
#  define SPR_MACLO                                1          // MAC Low
#  define SPR_MACHI                                2          // MAC High

  //----------------------------------------------[ spr_mode_access ]-----

#  define SPR_ACCESS_MODE_NONE                     0x0        // 000
#  define SPR_ACCESS_MODE_READ_ONLY                0x1        // 001
#  define SPR_ACCESS_MODE_WRITE_ONLY               0x2        // 010
#  define SPR_ACCESS_MODE_READ_WRITE               0x3        // 011
#  define SPR_ACCESS_MODE_READ_ONLY_COND           0x5        // 101 special read

  /*
#define                 _size_instruction             32
#define                 _size_instruction_log2        5

  //----------------------------------------------------[ Operation ]-----
// #define                 _nb_operation                 32
// #define                 _size_operation               5

#define                 _operation_none               0x0
#define                 _operation_l_adds             0x1
#define                 _operation_l_addu             0x2
#define                 _operation_l_subs             0x3
#define                 _operation_l_and              0x4
#define                 _operation_l_or               0x5
#define                 _operation_l_xor              0x6
#define                 _operation_l_cmove            0x7
#define                 _operation_l_read_imm         0x8
#define                 _operation_l_movhi            0x9
#define                 _operation_l_muls             0xa
#define                 _operation_l_mulu             0xb
#define                 _operation_l_divs             0xc
#define                 _operation_l_divu             0xd
#define                 _operation_l_exts             0xe
#define                 _operation_l_extz             0xf
#define                 _operation_l_ff1              0x10
#define                 _operation_l_fl1              0x11
#define                 _operation_l_sll              0x12
#define                 _operation_l_sla              0x13
#define                 _operation_l_srl              0x14
#define                 _operation_l_ror              0x15
#define                 _operation_l_cmp_eq           0x16
#define                 _operation_l_cmp_ne           0x17
#define                 _operation_l_cmp_ges          0x18
#define                 _operation_l_cmp_geu          0x19
#define                 _operation_l_cmp_gts          0x1a
#define                 _operation_l_cmp_gtu          0x1b
#define                 _operation_l_cmp_les          0x1c
#define                 _operation_l_cmp_leu          0x1d
#define                 _operation_l_cmp_lts          0x1e
#define                 _operation_l_cmp_ltu          0x1f

  //--------------------------------------------------[ destination ]-----
#define                 _size_destination1            4

#define                  cst_DESTINATION1_NONE        0x0
#define                 mask_DESTINATION1_GPR         0x1
#define                 mask_DESTINATION1_MEMORY      0x2
#define                 mask_DESTINATION1_SPR         0x4
#define                 mask_DESTINATION1_MAC_UNIT    0x8
  
#define                 _size_destination2  3

#define                  cst_DESTINATION2_NONE        0x0
#define                 mask_DESTINATION2_COMMIT      0x1
#define                 mask_DESTINATION2_MEMORY      0x2
#define                 mask_DESTINATION2_SPR         0x4
  
  //----------------------------------------------------[ exec_flag ]-----
#define                 _size_exec_flag               2

#define                 mask_EXEC_FLAG_NONE           0x1
#define                 mask_EXEC_FLAG_CARRY          0x1
#define                 mask_EXEC_FLAG_FLAG           0x1
#define                 mask_EXEC_FLAG_OVERFLOW       0x2

  //---------------------------------------------------[ exec_excep ]-----
#define                  _size_exec_excep             1

#define                 mask_EXEC_EXCEP_NONE          0x0
#define                 mask_EXEC_EXCEP_RANGE         0x1
  
  //----------------------------------------------------[ Condition ]-----
#define                 _size_condition               3
    
#define                  cst_CONDITION_UNCONDITIONAL  0x0           // None condition (jump)
#define                 mask_CONDITION_CONDITIONAL    0x2           
#define                 mask_CONDITION_CONDITIONAL_NF 0x0           // Branch if Flag is clear 
#define                 mask_CONDITION_CONDITIONAL_F  0x1           // Branch if Flag is set   
#define                 mask_CONDITION_REG            0x4           // Branch if a register is read
#define                 mask_CONDITION_STACK          0x8           // Branch with pop  in stack pointer

  //-------------------------------------------------[ branch_state ]-----
#define                 cst_BRANCH_STATE_NONE         0x0           // 0 0
#define                 cst_BRANCH_STATE_NSPEC_TAKE   0x1           // 0 1  -> incondionnal
#define                 cst_BRANCH_STATE_SPEC_NTAKE   0x2           // 1 0
#define                 cst_BRANCH_STATE_SPEC_TAKE    0x3           // 1 1
  */

  /*
#define M_CPU_SIZE_INST       32

//----------------------------------------------------
// Exception type
//----------------------------------------------------

#define M_CPU_LOG2_NB_EXCP    5
#define M_CPU_NB_EXCP         32

#define EXCP_NO               0x00          // none exception
#define EXCP_RESET            0x01          // software or hardware reset
#define EXCP_BERR             0x02          // Access at a invalid physical adress
#define EXCP_D_PAGE           0x03          // No matching or page violation protection in pages tables
#define EXCP_I_PAGE           0x04          // No matching or page violation protection in pages tables
#define EXCP_TICK_TIMER       0x05          // Tick timer interruption
#define EXCP_ALIGNMENT        0x06          // Load/Store access is not aligned
#define EXCP_ILL_INST         0x07          // Instruction is illegal (no implemented)
#define EXCP_IRQ              0x08          // External interruption 
#define EXCP_D_TLB            0x09          // DTLB miss
#define EXCP_I_TLB            0x0a          // ITLB miss
#define EXCP_RANGE            0x0b          // Overflow or access at a unimplemented register or context
#define EXCP_SYSCALL          0x0c          // System Call
#define EXCP_FP               0x0d          // Caused by a floating instruction
#define EXCP_TRAP             0x0e          // L.trap or debug unit
#define EXCP_RES0             0x0f          // Reserved for a futur usage
#define EXCP_RES1             0x10          // Reserved for a futur usage
#define EXCP_RES2             0x11          // Reserved for a futur usage
#define EXCP_RES3             0x12          // Reserved for a futur usage
#define EXCP_RES4             0x13          // Reserved for a futur usage
#define EXCP_RES5             0x14          // Reserved for a futur usage
#define EXCP_RES6             0x15          // Reserved for implemented specific exceptions
#define EXCP_RES7             0x16          // Reserved for implemented specific exceptions
#define EXCP_RES8             0x17          // Reserved for implemented specific exceptions
#define EXCP_RES9             0x18          // Reserved for implemented specific exceptions
#define EXCP_CUST0            0x19          // Reserved for custom exceptions
#define EXCP_CUST1            0x1a          // Reserved for custom exceptions
#define EXCP_CUST2            0x1b          // Reserved for custom exceptions
#define EXCP_CUST3            0x1c          // Reserved for custom exceptions
#define EXCP_CUST4            0x1d          // Reserved for custom exceptions
#define EXCP_CUST5            0x1e          // Reserved for custom exceptions
#define EXCP_CUST6            0x1f          // Reserved for custom exceptions

//----------------------------------------------------
// Flags
//----------------------------------------------------

#define M_CPU_NB_FLAG         3

// Integer flags
#define FLAG_F                0x1           // Conditionnal branch flag
#define FLAG_CY               0x2           // Carry was produced by last arithmtic operation
#define FLAG_OV               0x4           // Overflow occured during last arithmetic operation

// Floating flags
#define FLAG_OVF              0x004         // Overflow occured during last arithmetic operation
#define FLAG_UNF              0x008         // Underflow flags
#define FLAG_SNF              0x010         // Result SNAN
#define FLAG_QNF              0x020         // Result QNAN
#define FLAG_ZF               0x040         // Result is nul
#define FLAG_IXF              0x080         // Result is inexact
#define FLAG_IVF              0x100         // Result is invalid
#define FLAG_INF              0x200         // Result is infinite
#define FLAG_DZF              0x400         // Division by zero

// Position of flag in "rename register SR" (NOT IN "SR")
#define FLAG_POS_F            0x0           // Conditionnal branch flag
#define FLAG_POS_CY           0x1           // Carry was produced by last arithmtic operation
#define FLAG_POS_OV           0x0           // Overflow occured during last arithmetic operation

//----------------------------------------------------
// Instruction type
//----------------------------------------------------

#define M_CPU_LOG2_NB_TYPE    4

#define TYPE_NOP              0x0
#define TYPE_ALU_F            0x1           // Instruction ALU with    flag using (ADD, SUB, ADDC ...)
#define TYPE_ALU_NF           0x2           // Instruction ALU without flag using (AND, OR ...)
#define TYPE_MAC              0x3           // Instruction ALU with    utilisation of register HI/LO
#define TYPE_J                0x4           // Branch instruction
#define TYPE_SPR_READ         0x5           // Instruction special : l.mfspr
#define TYPE_SPR_WRITE        0x6           // Instruction special : l.mtspr
#define TYPE_SPECIAL          0x7           // Instruction execute in decode stage
#define TYPE_CUSTOM           0x8           // Instruction Custom

#define TYPE_LOAD_Z           0x9           // Load  access (extended by zero)
#define TYPE_LOAD_S           0xa           // Load  access (sign extended)
#define TYPE_STORE            0xc           // Store access

//----------------------------------------------------
// Condition to branch
//----------------------------------------------------

#define M_CPU_LOG2_NB_COND    4

#define COND_NONE             0x0           // None condition (jump)
#define COND_F                0x2           // Branch if Flag is set
#define COND_NF               0x3           // Branch if Flag is clear
#define COND_REG              0x4           // Branch if a register is read
#define COND_STACK            0x8           // Branch with pop  in stack pointer

//----------------------------------------------------
// Event : State and Type
//----------------------------------------------------

#define EVENT_STATE_NO_EVENT  0             // no event : current case
#define EVENT_STATE_EVENT     1             // Have a event : make necessary to manage the event
#define EVENT_STATE_WAITEND   2             // Wait end of manage event (restaure a good context)
#define EVENT_STATE_END       3             // CPU can continue

#define EVENT_TYPE_MISS       0             // miss of speculation
#define EVENT_TYPE_EXCP       1             // exception or interruption occure

// SPEC? TAKE?
#define BRANCH_STATE_NONE         0  // 0     0
#define BRANCH_STATE_NSPEC_TAKE   1  // 0     1  -> incondionnal
#define BRANCH_STATE_SPEC_NTAKE   2  // 1     0
#define BRANCH_STATE_SPEC_TAKE    3  // 1     1

//----------------------------------------------------
// Name to particular register
//----------------------------------------------------

//~~~~~~~~~~~~~~~~~~~~~~~~~~
// GENERAL PURPOSE REGISTER
//~~~~~~~~~~~~~~~~~~~~~~~~~~
#define M_CPU_LOG2_NB_GPR_LOG 5
#define M_CPU_NB_GPR_LOG      (1<<M_CPU_LOG2_NB_GPR_LOG)

#define GPR_LOG_LR            0x09          // Link register
#define REG_PHY_SR            0x00          // Status register

//~~~~~~~~~~~~~~~~~~~~~~~~~~
// SPECIAL PURPOSE REGISTER
//~~~~~~~~~~~~~~~~~~~~~~~~~~
#define M_CPU_LOG2_NB_SPR_LOG 1
#define M_CPU_NB_SPR_LOG      (1<<M_CPU_LOG2_NB_SPR_LOG)
#define M_CPU_SPR_SIZE_DATA   2             // Size of the most great register 

#define SPR_LOG_SR_F          0x00          // Status register bit F                   (size = 1)
#define SPR_LOG_SR_CY_OV      0x01          // Status register bit overflow and carry  (size = 2)
//#define SPR_LOG_SR_LO         0x02          // MAC LSB                                 (size = 32)
//#define SPR_LOG_SR_HI         0x03          // MAC MSB                                 (size = 32)
  */

  /*
//----------------------------------------------------
// Code Operation (before decode)
//----------------------------------------------------

// Codop                        - [31:26]      Instructions with immediat
#define OPCOD_L_J             0x00          // 000_000
#define OPCOD_L_JAL           0x01          // 000_001
#define OPCOD_L_BNF           0x03          // 000_011
#define OPCOD_L_BF            0x04          // 000_100
#define OPCOD_L_RFE           0x09          // 001_001
#define OPCOD_L_JR            0x11          // 010_001
#define OPCOD_L_JALR          0x12          // 010_010
#define OPCOD_L_MACI          0x13          // 010_011
#define OPCOD_L_CUST1         0x1c          // 011_100
#define OPCOD_L_CUST2         0x1d          // 011_101
#define OPCOD_L_CUST3         0x1e          // 011_110
#define OPCOD_L_CUST4         0x1f          // 011_111
#define OPCOD_L_CUST5         0x3c          // 111_100
#define OPCOD_L_CUST6         0x3d          // 111_101
#define OPCOD_L_CUST7         0x3e          // 111_110
#define OPCOD_L_CUST8         0x3f          // 111_111
#define OPCOD_L_LD            0x20          // 100_000
#define OPCOD_L_LWZ           0x21          // 100_001
#define OPCOD_L_LWS           0x22          // 100_010
#define OPCOD_L_LBZ           0x23          // 100_011
#define OPCOD_L_LBS           0x24          // 100_100
#define OPCOD_L_LHZ           0x25          // 100_101
#define OPCOD_L_LHS           0x26          // 100_110
#define OPCOD_L_ADDI          0x27          // 100_111
#define OPCOD_L_ADDIC         0x28          // 101_000
#define OPCOD_L_ANDI          0x29          // 101_001
#define OPCOD_L_ORI           0x2a          // 101_010
#define OPCOD_L_XORI          0x2b          // 101_011
#define OPCOD_L_MULI          0x2c          // 101_100
#define OPCOD_L_MFSPR         0x2d          // 101_101
#define OPCOD_L_MTSPR         0x30          // 110_000
#define OPCOD_L_SD            0x32          // 110_010
#define OPCOD_L_SW            0x35          // 110_101
#define OPCOD_L_SB            0x36          // 110_110
#define OPCOD_L_SH            0x37          // 110_111

#define OPCOD_INST_LV         0x0a          // 001_010         // Instruction ORVDX64
#define OPCOD_INST_LF         0x33          // 110_011         // Instruction ORFPX32/64

#define OPCOD_SPECIAL         0x38          // 111_000         // Instructions Register-Register
#define OPCOD_SPECIAL_1       0x39          // 111_001         // Instructions "set flag" with register
#define OPCOD_SPECIAL_2       0x2f          // 101_111         // Instructions "set flag" with immediat
#define OPCOD_SPECIAL_6       0x2e          // 101_110         // Instruction Shift/Rotate with immediat
#define OPCOD_SPECIAL_7       0x31          // 110_001         // Instructions multiply with HI-LO
#define OPCOD_SPECIAL_8       0x06          // 000_110         // Instructions acces at HI-LO 

// OPCOD_SPECIAL   instructions - [9:8] [3:0]  Instructions Register-Register
#define OPCOD_L_ADD           0x00          // 00_0000
#define OPCOD_L_ADDC          0x01          // 00_0001
#define OPCOD_L_SUB           0x02          // 00_0010
#define OPCOD_L_AND           0x03          // 00_0011
#define OPCOD_L_OR            0x04          // 00_0100
#define OPCOD_L_XOR           0x05          // 00_0101
#define OPCOD_L_CMOV          0x0e          // 00_1110
#define OPCOD_L_FF1           0x0f          // 00_1111
#define OPCOD_L_FL1           0x1f          // 01_1111
#define OPCOD_L_MUL           0x36          // 11_0110
#define OPCOD_L_DIV           0x39          // 11_1001
#define OPCOD_L_DIVU          0x3a          // 11_1010
#define OPCOD_L_MULU          0x3b          // 11_1011

#define OPCOD_SPECIAL_3       0xc           // 1100          // Instructions extend
#define OPCOD_SPECIAL_4       0xd           // 1101          // Instructions extend (64b)
#define OPCOD_SPECIAL_5       0x8           // 1000          // Instruction Shift/Rotate with register

// OPCOD_SPECIAL_1 instructions - [25:21]      Instructions "set flag" with register
#define OPCOD_L_SFEQ          0x00          // 00000
#define OPCOD_L_SFNE          0x01          // 00001
#define OPCOD_L_SFGTU         0x02          // 00010
#define OPCOD_L_SFGEU         0x03          // 00011
#define OPCOD_L_SFLTU         0x04          // 00100
#define OPCOD_L_SFLEU         0x05          // 00101
#define OPCOD_L_SFGTS         0x0a          // 01010
#define OPCOD_L_SFGES         0x0b          // 01011
#define OPCOD_L_SFLTS         0x0c          // 01100
#define OPCOD_L_SFLES         0x0d          // 01101

// OPCOD_SPECIAL_2 instructions - [25:21]      Instructions "set flag" with immediat
#define OPCOD_L_SFEQI         0x00          // 00000
#define OPCOD_L_SFNEI         0x01          // 00001
#define OPCOD_L_SFGTUI        0x02          // 00010
#define OPCOD_L_SFGEUI        0x03          // 00011
#define OPCOD_L_SFLTUI        0x04          // 00100
#define OPCOD_L_SFLEUI        0x05          // 00101
#define OPCOD_L_SFGTSI        0x0a          // 01010
#define OPCOD_L_SFGESI        0x0b          // 01011
#define OPCOD_L_SFLTSI        0x0c          // 01100
#define OPCOD_L_SFLESI        0x0d          // 01101

// OPCOD_SPECIAL_3 instructions - [9:6]          Instructions extend
#define OPCOD_L_EXTHS         0x0           // 0000
#define OPCOD_L_EXTHZ         0x2           // 0010
#define OPCOD_L_EXTBS         0x1           // 0001
#define OPCOD_L_EXTBZ         0x3           // 0011

// OPCOD_SPECIAL_4 instructions - [9:6]        Instructions extend (64b)
#define OPCOD_L_EXTWS         0x0           // 0000
#define OPCOD_L_EXTWZ         0x1           // 0001

// OPCOD_SPECIAL_5 instructions - [7:6]        Instruction Shift/Rotate with register
#define OPCOD_L_SLL           0x0           // 00
#define OPCOD_L_SRL           0x1           // 01
#define OPCOD_L_SRA           0x2           // 10
#define OPCOD_L_ROR           0x3           // 11

// OPCOD_SPECIAL_6 instructions - [7:6]        Instruction Shift/Rotate with immediat
#define OPCOD_L_SLLI          0x0           // 00
#define OPCOD_L_SRLI          0x1           // 01
#define OPCOD_L_SRAI          0x2           // 10
#define OPCOD_L_RORI          0x3           // 11

// OPCOD_SPECIAL_7 instructions - [3:0]        Instructions multiply with HI-LO
#define OPCOD_L_MAC           0x1           // 0001
#define OPCOD_L_MSB           0x2           // 0010

// OPCOD_SPECIAL_8 instructions - [17]         Instructions acces at HI-LO 
#define OPCOD_L_MOVHI         0x0           // 0
#define OPCOD_L_MACRC         0x1           // 1

// Particular case                             Instructions systems
#define OPCOD_L_MSYNC         0x22000000
#define OPCOD_L_CSYNC         0x23000000
#define OPCOD_L_PSYNC         0x22800000
#define OPCOD_L_NOP           0x1500
#define OPCOD_L_SYS           0x2000
#define OPCOD_L_TRAP          0x2100

//----------------------------------------------------
// Code Operation (after decode)
//----------------------------------------------------

typedef enum
  {
    // ##### WARNING : This opcode must be the first#####
    INST_L_NO_IMPLEMENTED ,         // Operation is not implemented

    INST_L_ADD            ,         // L.ADD    , L.ADDI   , L.ADDC   , L.ADDIC
    INST_L_AND            ,         // L.AND    , L.ANDI
    INST_L_OR             ,         // L.OR     , L.ORI
    INST_L_XOR            ,         // L.XOR    , L.XORI
    INST_L_CMOV           ,         // L.CMOV
    INST_L_SUB            ,         // L.SUB
    INST_L_FF1            ,         // L.FF1
    INST_L_EXTBS          ,         // L.EXTBS
    INST_L_EXTBZ          ,         // L.EXTBZ
    INST_L_EXTHS          ,         // L.EXTHS
    INST_L_EXTHZ          ,         // L.EXTHZ
    INST_L_EXTWS          ,         // L.EXTWS
    INST_L_EXTWZ          ,         // L.EXTWZ
    INST_L_e              ,         // 
    INST_L_f              ,         // 
    INST_L_MUL            ,         // L.MUL    , L.MULI
    INST_L_MULU           ,         // L.MULU
    INST_L_DIV            ,         // L.DIV
    INST_L_DIVU           ,         // L.DIVU
    INST_L_SLL            ,         // L.SLL    , L.SLLI
    INST_L_SRL            ,         // L.SRL    , L.SRLI 
    INST_L_SRA            ,         // L.SRA    , L.SRAI
    INST_L_ROR            ,         // L.ROR    , L.RORI
    INST_L_SFGES          ,         // L.SFGES  , L.SFGESI
    INST_L_SFGEU          ,         // L.SFGEU  , L.SFGEUI
    INST_L_SFGTS          ,         // L.SFGTS  , L.SFGTSI
    INST_L_SFGTU          ,         // L.SFGTU  , L.SFGTUI
    INST_L_SFLES          ,         // L.SFLES  , L.SFLESI
    INST_L_SFLEU          ,         // L.SFLEU  , L.SFLEUI
    INST_L_SFLTS          ,         // L.SFLTS  , L.SFLTSI
    INST_L_SFLTU          ,         // L.SFLTU  , L.SFLTUI
    INST_L_SFEQ           ,         // L.SFEQ   , L.SFEQI
    INST_L_SFNE           ,         // L.SFNE   , L.SFNEI
    INST_L_READ           ,         // L.BNF    , L.BF     , L.JR
    INST_L_MOVHI          ,         // L.MOVI
    INST_L_CSYNC          ,         // L.CSYNC
    INST_L_MSYNC          ,         // L.MSYNC
    INST_L_PSYNC          ,         // L.PSYNC
    INST_L_RFE            ,         // L.RFE
    INST_L_MAC            ,         // L.MAC    , L.MACI
    INST_L_MSB            ,         // L.MSB
    INST_L_MACRC          ,         // L.MACRC
    INST_L_2b             ,         //
    INST_L_MEMB           ,         // L.LBS    , L.LBZ    , L.SB  
    INST_L_MEMH           ,         // L.LHS    , L.LHZ    , L.SH 
    INST_L_MEMW           ,         // L.LWS    , L.LWZ    , L.SW  
    INST_L_MEMD           ,         // L.LD                , L.SD  
    INST_L_CUST1          ,         // L.CUST1
    INST_L_CUST2          ,         // L.CUST2
    INST_L_CUST3          ,         // L.CUST3
    INST_L_CUST4          ,         // L.CUST4
    INST_L_CUST5          ,         // L.CUST5
    INST_L_CUST6          ,         // L.CUST6 
    INST_L_CUST7          ,         // L.CUST7
    INST_L_CUST8          ,         // L.CUST8
    INST_L_38             ,         //
    INST_L_39             ,         //
    INST_L_3a             ,         //
    INST_L_3b             ,         //
    INST_L_3c             ,         //  
    INST_L_3d             ,         //  
    INST_L_3e             ,         //  
    INST_NOP                        // L.NOP
  } opcod_t;

#define LOG2_NB_INST_L        6
#define      NB_INST_L        64 // +1 -> INST_L_NO_IMPLEMENTED
//#define      NB_INST_L        (INST_L_NO_IMPLEMENTED+1)
  */

}; // end namespace behavioural
}; // end namespace morpheo              

#endif
