/*
    This file is part of MutekH.

    MutekH is free software; you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    MutekH is distributed in the hope that it will be useful, but
    WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with MutekH; if not, write to the Free Software Foundation,
    Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA

    Copyright Alexandre Becoulet <alexandre.becoulet@lip6.fr> (c) 2006

*/

/**
 * @file
 * @module{Device drivers}
 * @short Timer device driver API
 */

#ifndef __DEVICE_TIMER_H__
#define __DEVICE_TIMER_H__

#ifdef __DRIVER_H__
# error This header must not be included after "device/driver.h"
#endif

#include <hexo/types.h>
#include <hexo/error.h>
#include <hexo/gpct_platform_hexo.h>
#include <gpct/cont_clist.h>

struct device_s;
struct driver_s;
struct dev_timer_rq_s;

/** timer absolute value type */
typedef uint64_t dev_timer_value_t;
/** timer relative value type */
typedef uint32_t dev_timer_delay_t;

/** timer device class callback function template */
#define DEVTIMER_CALLBACK(n)	bool_t (n) (struct device_s *dev, struct dev_timer_rq_s *rq)
/** Timer device request callback. This function is called when the
    timer deadline is reached. The request is rescheduled if the
    function return true and the @tt rq->delay field was not zero when registered. */
typedef DEVTIMER_CALLBACK(devtimer_callback_t);

/** Timer request @csee devtimer_request_t */
struct dev_timer_rq_s
{
  dev_timer_value_t		deadline;    //< absolute timer deadline
  dev_timer_delay_t             delay;       //< timer delay
  devtimer_callback_t		*callback;   //< callback function
  void				*pvdata;     //< pv data for callback
  void				*drvdata;    //< driver private data
  CONTAINER_ENTRY_TYPE(CLIST)	queue_entry; //< used by driver to enqueue requests
};


CONTAINER_TYPE(dev_timer_queue, CLIST, struct dev_timer_rq_s, queue_entry);
CONTAINER_FUNC(dev_timer_queue, CLIST, static inline, dev_timer_queue);

CONTAINER_KEY_TYPE(dev_timer_queue, PTR, SCALAR, deadline);
CONTAINER_KEY_FUNC(dev_timer_queue, CLIST, static inline, dev_timer_queue, deadline);


/** Timer device class request() function template. */
#define DEVTIMER_REQUEST(n)	error_t  (n) (struct device_s *dev, struct dev_timer_rq_s *rq)

/** Timer device class request() methode shortcut */
#define dev_timer_request(dev, ...) (dev)->drv->f.timer.f_request(dev, __VA_ARGS__ )


/**
   Timer device class request function. Enqueue a timer request.

   @param dev pointer to device descriptor
   @param rq pointer to request. @tt rq->delay and @tt rq->callback fields
     must be initialized. If @tt rq->delay is zero, @tt rq->deadline must be
     initialized with absolute deadline timer value.

   Request callback will be called immediately from within this
   function if the deadline has already been reached.

   @csee #DEVTIMER_REQUEST @csee #dev_timer_request
*/
typedef DEVTIMER_REQUEST(devtimer_request_t);


/** Timer device class cancel() function template. */
#define DEVTIMER_CANCEL(n)	error_t  (n) (struct device_s *dev, struct dev_timer_rq_s *rq)

/** Timer device class cancel() methode shortcut */
#define dev_timer_cancel(dev, ...) (dev)->drv->f.timer.f_cancel(dev, __VA_ARGS__ )

/**
   Timer device class cancel function. Cancel a queued timer request.

   @param dev pointer to device descriptor
   @param rq pointer to cancel.

   @return @tt -ENOENT if the request was not found (already reached).

   @csee #DEVTIMER_CANCEL @csee #dev_timer_cancel
*/
typedef DEVTIMER_CANCEL(devtimer_cancel_t);



/** Timer device class getvalue() function template. */
#define DEVTIMER_GETVALUE(n)	void (n) (struct device_s *dev, dev_timer_value_t *value)

/** Timer device class getvalue() methode shortcut */
#define dev_timer_getvalue(dev, ...) (dev)->drv->f.timer.f_getvalue(dev, __VA_ARGS__ )

/**
   Timer device class value read.

   @param dev pointer to device descriptor
   @param value pointer to value storage.

   @csee #DEVTIMER_GETVALUE @csee #dev_timer_getvalue
*/
typedef DEVTIMER_GETVALUE(devtimer_getvalue_t);



/** @see dev_timer_info_s */
enum dev_timer_flags_e {
  DEV_TIMER_CPUCYCLES       = 1,
  DEV_TIMER_USECONDS        = 2,
};

struct dev_timer_info_s
{
  uint_fast8_t      flags;       //< timer flags described in @ref dev_timer_flags_e
  dev_timer_delay_t resolution;  //< timer resolution
};

/** Timer device class getinfo() function template. */
#define DEVTIMER_GETINFO(n)	void (n) (struct device_s *dev, struct dev_timer_info_s *info)

/** Timer device class getinfo() methode shortcut */
#define dev_timer_getinfo(dev, ...) (dev)->drv->f.timer.f_getinfo(dev, __VA_ARGS__ )

/**
   Timer device class info read.

   @param dev pointer to device descriptor
   @param info pointer to info storage.

   @csee #DEVTIMER_GETINFO @csee #dev_timer_getinfo
   @csee dev_timer_info_s
*/
typedef DEVTIMER_GETINFO(devtimer_getinfo_t);



/** TIMER device class methods */

struct dev_class_timer_s
{
  devtimer_request_t			*f_request;
  devtimer_cancel_t			*f_cancel;
  devtimer_getvalue_t			*f_getvalue;
  devtimer_getinfo_t			*f_getinfo;
};

#endif

