/* $NetBSD: $ */

/*-
  * Copyright (c) 2009 UPMC/LIP6
  * All rights reserved.
  * This software is distributed under the following condiions
  * compliant with the NetBSD foundation policy.
  *
  * Redistribution and use in source and binary forms, with or without
  * modification, are permitted provided that the following conditions
  * are met:
  * 1. Redistributions of source code must retain the above copyright
  *    notice, this list of conditions and the following disclaimer.
  * 2. Redistributions in binary form must reproduce the above copyright
  *    notice, this list of conditions and the following disclaimer in the
  *    documentation and/or other materials provided with the distribution.
  *
  * THIS SOFTWARE IS PROVIDED BY THE NETBSD FOUNDATION, INC. AND CONTRIBUTORS
  * ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED
  * TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
  * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE FOUNDATION OR CONTRIBUTORS
  * BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
  * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
  * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
  * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
  * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
  * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
  * POSSIBILITY OF SUCH DAMAGE.
  */

#include <sys/cdefs.h>
__KERNEL_RCSID(0, "$NetBSD: cpu.c,v 1.24 2005/12/11 12:18:39 christos Exp $");

#include "opt_multiprocessor.h"

#include <sys/atomic.h>
#include <sys/param.h>
#include <sys/cpu.h>
#include <sys/idle.h>
#include <sys/device.h>
#include <sys/systm.h>
#include <sys/malloc.h>

#include <uvm/uvm.h>

#include <machine/locore.h>

#include <machine/autoconf.h>
#include <machine/vcache.h>
#include <mips/asm.h>

static int	cpu_match(device_t, cfdata_t, void *);
static void	cpu_attach(device_t, device_t, void *);
void	tsarmips_cpu_attach_common(struct cpu_info *);

volatile uint32_t cputramp_cpunum; /* CPU to bootstrap */
struct cputramp_args cputramp_args; /* secondary CPUs bootstrap arguments */

#ifdef MULTIPROCESSOR
static struct cpu_info *cpu_info_last = &cpu_info_store;
#endif

CFATTACH_DECL_NEW(cpu, 0, 
    cpu_match, cpu_attach, NULL, NULL);

static int
cpu_match(device_t parent, cfdata_t match, void *aux)
{
	struct tsardevs_attach_args *tsd = aux;
	if (strcmp(tsd->tsd_devtype, "cpu") != 0)
		return 0;
#ifndef MULTIPROCESSOR
	if (tsd->tsd_reg.reg_addr != -1 &&
	    (tsd->tsd_reg.reg_addr != (mips_cp0_ebase_read() & 0x1ff)))
		return 0;
#endif
	return 1;
}

static void
cpu_attach(device_t parent, device_t self, void *aux)
{
#ifdef MULTIPROCESSOR
	struct cpu_info *ci;
#endif
	struct tsardevs_attach_args *tsd = aux;

	if (tsd->tsd_reg.reg_addr == -1 ||
	    tsd->tsd_reg.reg_addr == (mips_cp0_ebase_read() & 0x1ff)) {
		/* boot CPU: attach directly */
		cpu_info_store.ci_dev = self;
		self->dv_private = &cpu_info_store;
		tsarmips_cpu_attach_common(&cpu_info_store);
		return;
	}
#ifdef MULTIPROCESSOR
	/*
	 * other CPUs: alocate a cpu_info and wake it up. Let the CPU
	 * do the printf
	 */
	ci = malloc(sizeof(struct cpu_info), M_DEVBUF, M_WAITOK);
	if (ci == NULL) {
		aprint_error(": failed to allocate cpu_info\n");
		return;
	}
	memset(ci, 0, sizeof(struct cpu_info));

	ci->ci_cpuid = tsd->tsd_reg.reg_addr;
	ci->ci_dev = self;
	self->dv_private = ci;
	KASSERT(ci->ci_idepth == 0);
	mi_cpu_attach(ci);
	pmap_reference(pmap_kernel());
	ci->ci_pmap = pmap_kernel();
	LIST_INIT(&ci->ci_freepmaps);
	cputramp_args.ct_bootpd = ci->ci_pmap->pm_pdpa;
	printf(" init ct_bootpd 0x%x ", cputramp_args.ct_bootpd);
	cputramp_args.ct_stack =
	    (uintptr_t)ci->ci_data.cpu_idlelwp->l_addr + USPACE
	    - sizeof(struct trapframe) - CALLFRAME_SIZ;
	cputramp_args.ct_ci = ci;
	wbflush();
	/* wake up CPU */
	printf(" waking up %d %d ", (int)tsd->tsd_reg.reg_addr, (int)ci->ci_cpuid);
	cputramp_cpunum = tsd->tsd_reg.reg_addr;
	wbflush();
	/* and wait for cputramp_cpunum to go back to -1 */
	while (cputramp_cpunum != 0xffffffff)
		;
	if ((tsar_cpus_hatched[cpu_index(ci) / 32] & (1 << (cpu_index(ci) % 32)))
	    == 0) {
		printf("cpu %d didn't wake up\n", (int)ci->ci_cpuid);
	} else {
		/* add it to the list */
		KASSERT(ci->ci_next == NULL);
		KASSERT(cpu_info_last->ci_next == NULL);
		cpu_info_last->ci_next = ci;
		cpu_info_last = ci;
	}
#else
	while (1);
#endif /* MULTIPROCESSOR */
}

void
tsarmips_cpu_attach_common(struct cpu_info *ci)
{
	uint32_t versrev;
	int sets, ways, line, colors, tcolors;
	aprint_normal(": TSAR mips id 0x%x", mips_options.mips_cpu_id);
	versrev = tsar_vcache_read(VC_RELEASE);
	aprint_normal(", L1 cache version %d revision %d\n",
	    (versrev >> 16) & 0xffff, versrev & 0xffff);
	versrev = tsar_vcache_read(VC_PARAMS);
	line = ci->ci_l1_cls = 1 << VC_PARAMS_NBL(versrev);
	sets = ci->ci_l1_ics = 1 << VC_PARAMS_SCI(versrev);
	ways = ci->ci_l1_icw = 1 << VC_PARAMS_WCI(versrev);
	aprint_normal_dev(ci->ci_dev, "I-cache %dK %d-way", line * sets * ways / 1024, ways);
	sets = ci->ci_l1_its = 1 << VC_PARAMS_STI(versrev);
	ways = ci->ci_l1_itw = 1 << VC_PARAMS_WTI(versrev);
	aprint_normal(", I-tlb %d entries %d-way\n", sets * ways, ways);
	sets = ci->ci_l1_dcs = 1 << VC_PARAMS_SCD(versrev);
	ways = ci->ci_l1_dcw = 1 << VC_PARAMS_WCD(versrev);
	aprint_normal_dev(ci->ci_dev, "D-cache %dK %d-way", line * sets * ways / 1024, ways);
	sets = ci->ci_l1_dts = 1 << VC_PARAMS_STD(versrev);
	ways = ci->ci_l1_dtw = 1 << VC_PARAMS_WTD(versrev);
	aprint_normal(", D-tlb %d entries %d-way\n", sets * ways, ways);

	/* compute number of cache colors */
	colors = 2;
	tcolors = ci->ci_l1_dts * ci->ci_l1_cls / PAGE_SIZE;
	if (tcolors > colors)
		colors = tcolors;
	tcolors = ci->ci_l1_its * ci->ci_l1_cls / PAGE_SIZE;
	if (tcolors > colors)
		colors = tcolors;
	if (colors & (colors - 1)) {
		aprint_error_dev(ci->ci_dev,
		    "number of colors %d not a power of 2\n", colors);
	} else {
		if (colors > uvmexp.ncolors) {
			aprint_debug_dev(ci->ci_dev,
			    "%d page colors\n", colors);
			uvm_page_recolor(colors);
		}
	}
	mips_cp0_ebase_write(cp0_ebase);
	aprint_debug_dev(ci->ci_dev, "wrote ebase 0x%x\n",
	    mips_cp0_ebase_read() & (~0xfff));
	/* Clear BEV in SR so we start handling our own exceptions */
	mips_cp0_status_write(mips_cp0_status_read() & ~MIPS_SR_BEV);
#ifndef NOFPU
        /*
	 * Indicate the CPU's FPU is owned by the CPU's idlelwp.
	 * (which really means it's free).
	 */
	ci->ci_fpcurlwp = ci->ci_data.cpu_idlelwp;
#endif
}

#ifdef MULTIPROCESSOR
void
cpu_hatch(struct cpu_info *ci)
{
	int index = cpu_index(ci) / 32;
	uint32_t mask = 1 << cpu_index(ci) % 32;
	/*
	 * indicate that we're up and hatched so boot cpu can wake up
	 * next one
	 */
	atomic_or_32(&tsar_cpus_hatched[index], mask);
	wbflush();
	cputramp_cpunum = 0xffffffff;
	wbflush();
	/* and wait */
	while ((tsar_cpus_running[index] & mask) == 0) {
		/* wait */
	}
	/*
	 * initialize the MIPS count/compare clock
	 */
	mips3_cp0_count_write(ci->ci_data.cpu_cc_skew);
	ci->ci_data.cpu_cc_skew = 0;
	/* announce the good new */
	KASSERT(ci->ci_curlwp == curlwp);
	aprint_debug_dev(ci->ci_dev, "running lwp %p ci %p stack %p cause 0x%x status 0x%x\n",
	    curlwp, curcpu(), &index, mips_cp0_cause_read(), mips_cp0_status_read());
	/* turn on interrupts */
	spl0();
	/* And do a tail call to idle_loop */
	idle_loop(NULL);
}

void
cpu_boot_secondary_processors(void)
{
	for (struct cpu_info *ci = cpu_info_store.ci_next;
	     ci != NULL;
	     ci = ci->ci_next) {
		aprint_debug_dev(ci->ci_dev, "cpu_boot_secondary_processors ");
		int index = cpu_index(ci) / 32;
		uint32_t mask = 1 << cpu_index(ci) % 32;
		KASSERT(!CPU_IS_PRIMARY(ci));
		KASSERT(ci->ci_data.cpu_idlelwp);

		/*
		 * Skip this CPU if it didn't sucessfully hatch.
		 */
		if ((tsar_cpus_hatched[index] & mask) == 0) {
			printf("not hatchedn\n");
			continue;
		}

		printf("waking up\n");
		ci->ci_data.cpu_cc_skew = mips3_cp0_count_read();
		atomic_or_ulong(&ci->ci_flags, CPUF_RUNNING);
		atomic_or_32(&tsar_cpus_running[index], mask);
	}
}
#endif /* MULTIPROCESSOR */
