/* $NetBSD: $ */

/*-
  * Copyright (c) 2009 UPMC/LIP6
  * All rights reserved.
  * This software is distributed under the following condiions
  * compliant with the NetBSD foundation policy.
  *
  * Redistribution and use in source and binary forms, with or without
  * modification, are permitted provided that the following conditions
  * are met:
  * 1. Redistributions of source code must retain the above copyright
  *    notice, this list of conditions and the following disclaimer.
  * 2. Redistributions in binary form must reproduce the above copyright
  *    notice, this list of conditions and the following disclaimer in the
  *    documentation and/or other materials provided with the distribution.
  *
  * THIS SOFTWARE IS PROVIDED BY THE NETBSD FOUNDATION, INC. AND CONTRIBUTORS
  * ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED
  * TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
  * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE FOUNDATION OR CONTRIBUTORS
  * BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
  * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
  * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
  * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
  * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
  * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
  * POSSIBILITY OF SUCH DAMAGE.
  */

/* driver for soclib's multitty device */

#include <sys/cdefs.h>
__KERNEL_RCSID(0, "$NetBSD: machdep.c,v 1.223 2008/07/02 17:28:56 ad Exp $");

#include <sys/param.h>
#include <sys/systm.h>
#include <sys/device.h>
#include <sys/conf.h>

#include <machine/autoconf.h>
#include <machine/bus.h>

#include "vcifb_logo.c"

static int vcifb_match(device_t, cfdata_t, void *);
static void vcifb_attach(device_t, device_t, void *);

struct vcifb_softc {
	device_t sc_dev;
	paddr_t sc_p; /* mapping informations */
	vaddr_t sc_psize;
	bus_space_tag_t sc_tag;
	bus_space_handle_t sc_handle;
	/* framebuffer-related information */
	int sc_width, sc_height, sc_depth;
};

CFATTACH_DECL_NEW(vcifb, sizeof(struct vcifb_softc),
    vcifb_match, vcifb_attach, NULL, NULL);

int
vcifb_match(device_t parent, cfdata_t match, void *aux)
{
	struct tsardevs_attach_args *tsd = aux;
	if (strcmp(tsd->tsd_devtype, "soclib:frame_buffer") != 0)
		return 0;
	return 1;
}

void
vcifb_attach(device_t parent, device_t self, void *aux)
{
	struct vcifb_softc *sc = device_private(self);
	struct tsardevs_attach_args *tsd = aux;
	const struct boot_fdt_prop *prop;
	int x, y;
	int start_x;
	int start_y;
	uint16_t p;
	const uint8_t *buf;

	aprint_normal(": SocLib Framebuffer Driver\n");

	sc->sc_dev = self;
	sc->sc_p = tsd->tsd_reg.reg_addr;
	sc->sc_psize = tsd->tsd_reg.reg_size;
	sc->sc_tag = tsd->tsd_tag;

	/* map the device */
	if (bus_space_map(sc->sc_tag, sc->sc_p, sc->sc_psize,
	    BUS_SPACE_MAP_LINEAR | BUS_SPACE_MAP_CACHEABLE, &sc->sc_handle)) {
		aprint_error_dev(self, "couldn't map framebuffer\n");
		return;
	}
	prop = fdt_find_prop(tsd->tsd_node->fdt_node_data, "width");
	if (prop == NULL) {
		aprint_error_dev(self, "no width\n");
		return;
	}
	sc->sc_width = be32toh(prop->prop_value[0]);

	prop = fdt_find_prop(tsd->tsd_node->fdt_node_data, "height");
	if (prop == NULL) {
		aprint_error_dev(self, "no height\n");
		return;
	}
	sc->sc_height = be32toh(prop->prop_value[0]);

	prop = fdt_find_prop(tsd->tsd_node->fdt_node_data, "mode");
	if (prop == NULL) {
		aprint_error_dev(self, "no mode\n");
		return;
	}
	switch(be32toh(prop->prop_value[0]))
	{
	case 16:
		sc->sc_depth = 16;
		break;
	default:
		aprint_error_dev(self, "unknown mode %d\n",
		    be32toh(prop->prop_value[0]));
		return;
	}
	aprint_normal_dev(self, ": %dx%d %d bits\n", sc->sc_width,
	    sc->sc_height, sc->sc_depth);
	start_x = sc->sc_width / 2 - logo_width / 2;
	start_y = sc->sc_height / 2 - logo_height / 2;
	buf = logo;
	for (y = 0; y < logo_height; y++) {
		for (x = 0; x < logo_width; x++) {
			p = ((*buf) >> 3 ) << 11; /* R */
			buf++;
			p |= ((*buf) >> 2) << 5; /* V */
			buf++;
			p |= ((*buf) >> 3);      /* B */
			buf++;
			bus_space_write_2(sc->sc_tag, sc->sc_handle,
			    ((y + start_y) * sc->sc_width + (x + start_x)) * 2,
			    p);
		}
	}
}	

