/* $NetBSD: $ */
/*-
  * Copyright (c) 2009 UPMC/LIP6
  * All rights reserved.
  * This software is distributed under the following condiions
  * compliant with the NetBSD foundation policy.
  *
  * Redistribution and use in source and binary forms, with or without
  * modification, are permitted provided that the following conditions
  * are met:
  * 1. Redistributions of source code must retain the above copyright
  *    notice, this list of conditions and the following disclaimer.
  * 2. Redistributions in binary form must reproduce the above copyright
  *    notice, this list of conditions and the following disclaimer in the
  *    documentation and/or other materials provided with the distribution.
  *
  * THIS SOFTWARE IS PROVIDED BY THE NETBSD FOUNDATION, INC. AND CONTRIBUTORS
  * ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED
  * TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
  * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE FOUNDATION OR CONTRIBUTORS
  * BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
  * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
  * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
  * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
  * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
  * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
  * POSSIBILITY OF SUCH DAMAGE.
  */

/* see bus_space(9) */

#include <sys/cdefs.h>
__KERNEL_RCSID(0, "$NetBSD: bus_space.c,v 1.5 2008/04/28 20:23:31 martin Exp $");

#include <sys/param.h>
#include <sys/systm.h>
#include <uvm/uvm_extern.h>

#include <machine/bus.h>

int tsarmips_addr_map(paddr_t, vaddr_t, vaddr_t *, int);
void tsarmips_addr_unmap(vaddr_t, vaddr_t);

int
tsarmips_addr_map(paddr_t addr, vaddr_t size, vaddr_t *mapping, int cacheable)
{
	paddr_t pa, endpa;
	vaddr_t va;

	pa = mips_trunc_ppage(addr);
	endpa = mips_round_ppage(addr + size);

	KASSERT(pa < endpa);
	va = uvm_km_alloc(kernel_map, endpa - pa, 0,
	    UVM_KMF_VAONLY | UVM_KMF_NOWAIT);
	if (va == 0)
		return ENOMEM;
	*mapping = va;
	for (; pa < endpa; pa += PAGE_SIZE, va += PAGE_SIZE) {
		tsarmips_pmap_kenter_pa(va, pa,
		    VM_PROT_READ | VM_PROT_WRITE, cacheable);
	}
	return 0;
}

void
tsarmips_addr_unmap(vaddr_t va, vaddr_t size)
{
	KASSERT((va & ~(PAGE_SIZE - 1)) == 0);
	size = mips_round_page(size);
	pmap_kremove(va, size);
	uvm_km_free(kernel_map, va, size, UVM_KMF_VAONLY);
}

/* ARGSUSED */
int
bus_space_map(t, bpa, size, flags, bshp)
	bus_space_tag_t t;
	bus_addr_t bpa;
	bus_size_t size;
	int flags;
	bus_space_handle_t *bshp;
{
	int cacheable = (flags & BUS_SPACE_MAP_CACHEABLE) ? 1 : 0;
	/* all mappings are linear */
	int error;

	error = tsarmips_addr_map(bpa, size, bshp, cacheable);
	return error;
}

/* ARGSUSED */
int
bus_space_alloc(t, rstart, rend, size, alignment, boundary, flags,
    bpap, bshp)
	bus_space_tag_t t;
	bus_addr_t rstart, rend;
	bus_size_t size, alignment, boundary;
	int flags;
	bus_addr_t *bpap;
	bus_space_handle_t *bshp;
{

	return (EINVAL);
}

/* ARGSUSED */
void
bus_space_free(t, bsh, size)
	bus_space_tag_t t;
	bus_space_handle_t bsh;
	bus_size_t size;
{

	panic("bus_space_free: shouldn't be here");
}

void
bus_space_unmap(t, bsh, size)
	bus_space_tag_t t;
	bus_space_handle_t bsh;
	bus_size_t size;
{
	tsarmips_addr_unmap(bsh, size);
}

paddr_t
bus_space_mmap(bus_space_tag_t t, bus_addr_t addr, off_t off, int prot,
    int flags)
{
	return (trunc_page(addr + off));
}


/* ARGSUSED */
int
bus_space_subregion(t, bsh, offset, size, nbshp)
	bus_space_tag_t t;
	bus_space_handle_t bsh;
	bus_size_t offset, size;
	bus_space_handle_t *nbshp;
{

	*nbshp = bsh + offset;
	return (0);
}
