/* $NetBSD: $ */
/*-
  * Copyright (c) 2009 UPMC/LIP6
  * All rights reserved.
  * This software is distributed under the following condiions
  * compliant with the NetBSD foundation policy.
  *
  * Redistribution and use in source and binary forms, with or without
  * modification, are permitted provided that the following conditions
  * are met:
  * 1. Redistributions of source code must retain the above copyright
  *    notice, this list of conditions and the following disclaimer.
  * 2. Redistributions in binary form must reproduce the above copyright
  *    notice, this list of conditions and the following disclaimer in the
  *    documentation and/or other materials provided with the distribution.
  *
  * THIS SOFTWARE IS PROVIDED BY THE NETBSD FOUNDATION, INC. AND CONTRIBUTORS
  * ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED
  * TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
  * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE FOUNDATION OR CONTRIBUTORS
  * BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
  * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
  * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
  * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
  * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
  * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
  * POSSIBILITY OF SUCH DAMAGE.
  */

#include <sys/cdefs.h>
__KERNEL_RCSID(0, "$NetBSD: cpu.c,v 1.24 2005/12/11 12:18:39 christos Exp $");

#include "opt_multiprocessor.h"
#include "opt_cputype.h"        /* which mips CPUs do we support? */
#include "opt_ddb.h"
#include "opt_kgdb.h"

#define __INTR_PRIVATE
#include <sys/malloc.h>
#include <sys/types.h>
#include <machine/proc.h>
#include <sys/cpu.h>
#include <machine/cpuset.h>
#include <machine/intr.h>
#include <machine/xicu.h>
#include <machine/db_machdep.h>
#include <ddb/db_output.h>


/*
 * map IPL to mask for cp0:status. bit 1 means interrupt is disabled
 * line 6 is unused and always masked.
 */

#define MIPS_SPL_NONE		(MIPS_INT_MASK_5)
#define MIPS_SPL_SOFTCLOCK	(MIPS_SPL_NONE | MIPS_SOFT_INT_MASK_0)
#define MIPS_SPL_SOFTBIO	(MIPS_SPL_SOFTCLOCK)
#define MIPS_SPL_SOFTNET	(MIPS_SPL_SOFTCLOCK | MIPS_SOFT_INT_MASK_1)
#define MIPS_SPL_SOFTSERIAL	(MIPS_SPL_SOFTNET)
#define MIPS_SPL_VM		(MIPS_SPL_SOFTSERIAL | MIPS_INT_MASK_3)
#define	MIPS_SPL_SCHED		(MIPS_SPL_VM | MIPS_INT_MASK_2)
#define MIPS_SPL_DDB		(MIPS_SPL_SCHED | MIPS_INT_MASK_1)
#define MIPS_SPL_HIGH		(MIPS_INT_MASK)

static const struct ipl_sr_map tsarmips_ipl_sr_map = {
    .sr_bits = {
	[IPL_NONE] = MIPS_SPL_NONE,
	[IPL_SOFTCLOCK] = MIPS_SPL_SOFTCLOCK,
	[IPL_SOFTBIO] = MIPS_SPL_SOFTBIO,
	[IPL_SOFTNET] = MIPS_SPL_SOFTNET,
	[IPL_SOFTSERIAL] = MIPS_SPL_SOFTSERIAL,
	[IPL_VM] = MIPS_SPL_VM,
	[IPL_SCHED] = MIPS_SPL_SCHED,
	[IPL_DDB] = MIPS_SPL_DDB,
	[IPL_HIGH] = MIPS_SPL_HIGH
    }
};

void
intr_init()
{
	ipl_sr_map = tsarmips_ipl_sr_map;
}

#ifdef MULTIPROCESSOR

volatile __cpuset_t cpus_running = { {0} };
volatile __cpuset_t cpus_hatched = { {0} };
volatile __cpuset_t cpus_paused = { {0} };
volatile __cpuset_t cpus_resumed = { {0} };
volatile __cpuset_t cpus_halted = { {0} };

static int  cpu_ipi_wait(volatile __cpuset_t *, __cpuset_t);
static void cpu_ipi_error(const char *, __cpuset_t, __cpuset_t);

void
cpu_broadcast_ipi(int tag)
{
	__cpuset_t running_notme;

	CPUSET_ASSIGN(running_notme, cpus_running);
	CPUSET_DEL(running_notme, cpu_index(curcpu()));
	(void)cpu_multicast_ipi(running_notme, tag);
}

void
cpu_multicast_ipi(__cpuset_t cpuset, int tag)
{
	CPU_INFO_ITERATOR cii;
	struct cpu_info *ci;

	CPUSET_DEL(cpuset, cpu_index(curcpu()));
	if (CPUSET_EMPTY_P(cpuset))
		return;

	for (CPU_INFO_FOREACH(cii, ci)) {
		if (CPUSET_HAS_P(cpuset, cpu_index(ci))) {
			CPUSET_DEL(cpuset, cpu_index(ci));
			(void)cpu_send_ipi(ci, tag);
		}
	}
}

int
cpu_send_ipi(struct cpu_info *ci, int tag)
{
	return xicu_send_ipi(ci, tag);
}

static void
cpu_ipi_error(const char *s, __cpuset_t succeeded, __cpuset_t expected)
{
	CPUSET_SUB(expected, succeeded);
	if (!CPUSET_EMPTY_P(expected)) {
		printf("Failed to %s:", s);
		do {
			int index = CPUSET_NEXT(expected);
			CPUSET_DEL(expected, index);
			printf(" cpu%d", index);
		} while (!CPUSET_EMPTY_P(expected));
		printf("\n");
	}
}

static int
cpu_ipi_wait(volatile __cpuset_t *watchset, __cpuset_t mask)
{
	u_long limit = curcpu()->ci_cpu_freq;   /* some finite amount of time */

	while (limit--)
		if (CPUSET_EQUAL_P(*watchset, mask))
			return 0;	       /* success */

	return 1;			       /* timed out */
}

/*
 * Halt this cpu
 */
void
cpu_halt(void)
{
	int index = cpu_index(curcpu());

	printf("cpu%d: shutting down\n", index);
	CPUSET_ADD(cpus_halted, index);
	spl0();	 /* allow interrupts e.g. further ipi ? */
	for (;;) ;      /* spin */

	/* NOTREACHED */
}

/*
 * Halt all running cpus, excluding current cpu.
 */
void
cpu_halt_others(void)
{
	__cpuset_t cpumask, cpuset;

	CPUSET_ASSIGN(cpuset, cpus_running);
	CPUSET_DEL(cpuset, cpu_index(curcpu()));
	CPUSET_ASSIGN(cpumask, cpuset);
	CPUSET_SUB(cpuset, cpus_halted);

	if (CPUSET_EMPTY_P(cpuset))
		return;

	cpu_multicast_ipi(cpuset, IPI_HALT);
	if (cpu_ipi_wait(&cpus_halted, cpumask))
		cpu_ipi_error("halt", cpumask, cpus_halted);

	/*
	 * TBD
	 * Depending on available firmware methods, other cpus will
	 * either shut down themselfs, or spin and wait for us to
	 * stop them.
	 */
}

/*
 * Pause this cpu
 */
void
cpu_pause(struct reg *regsp)
{
	int s = splhigh();
	int index = cpu_index(curcpu());
	for (;;) {
		CPUSET_ADD(cpus_paused, index);
		do {
			;
		} while (CPUSET_HAS_P(cpus_paused, index));
		CPUSET_ADD(cpus_resumed, index);

#if defined(DDB)
		if (ddb_running_on_this_cpu_p())
			cpu_Debugger();
		if (ddb_running_on_any_cpu_p())
			continue;
#endif
		break;
	}

	splx(s);
}

/*
 * Pause all running cpus, excluding current cpu.
 */
void
cpu_pause_others(void)
{
	__cpuset_t cpuset;

	CPUSET_ASSIGN(cpuset, cpus_running);
	CPUSET_DEL(cpuset, cpu_index(curcpu()));

	if (CPUSET_EMPTY_P(cpuset))
		return;

	cpu_multicast_ipi(cpuset, IPI_SUSPEND);
	if (cpu_ipi_wait(&cpus_paused, cpuset))
		cpu_ipi_error("pause", cpus_paused, cpuset);
}

/*
 * Resume a single cpu
 */
void
cpu_resume(int index)
{
	__cpuset_t me;
	CPUSET_CLEAR(me);
	CPUSET_ADD(me, index);
	CPUSET_CLEAR(cpus_resumed);
	CPUSET_DEL(cpus_paused, index);

	if (cpu_ipi_wait(&cpus_resumed, me))
		cpu_ipi_error("resume", cpus_resumed, me);
}

/*
 * Resume all paused cpus.
 */
void
cpu_resume_others(void)
{
	__cpuset_t cpuset;

	CPUSET_CLEAR(cpus_resumed);
	CPUSET_ASSIGN(cpuset, cpus_paused);
	CPUSET_CLEAR(cpus_paused);

	/* CPUs awake on cpus_paused clear */
	if (cpu_ipi_wait(&cpus_resumed, cpuset))
		cpu_ipi_error("resume", cpus_resumed, cpuset);
}

int
cpu_is_paused(int index)
{

	return CPUSET_HAS_P(cpus_paused, index);
}

#ifdef DDB
void
cpu_debug_dump(void)
{
	CPU_INFO_ITERATOR cii;
	struct cpu_info *ci;
	char running, hatched, paused, resumed, halted;

	db_printf("CPU CPUID STATE CPUINFO	    CPL INT MTX IPIS\n");
	for (CPU_INFO_FOREACH(cii, ci)) {
		hatched = (CPUSET_HAS_P(cpus_hatched, cpu_index(ci)) ? 'H' : '-');
		running = (CPUSET_HAS_P(cpus_running, cpu_index(ci)) ? 'R' : '-');
		paused  = (CPUSET_HAS_P(cpus_paused,  cpu_index(ci)) ? 'P' : '-');
		resumed = (CPUSET_HAS_P(cpus_resumed, cpu_index(ci)) ? 'r' : '-');
		halted  = (CPUSET_HAS_P(cpus_halted,  cpu_index(ci)) ? 'h' : '-');
		db_printf("%3d 0x%03lx %c%c%c%c%c %p "
			"%3d %3d %3d "
			"0x%02lx/0x%02lx\n",
			cpu_index(ci), ci->ci_cpuid,
			running, hatched, paused, resumed, halted,
			ci, ci->ci_cpl, ci->ci_idepth, ci->ci_mtx_count,
			ci->ci_active_ipis, ci->ci_request_ipis);
	}
}
#endif
#endif /* MULTIPROCESSOR */

void
cpu_intr(int pri, vaddr_t pc, u_int32_t status)
{
	xicu_intr(pri, pc, status);
}

irq_t
intr_establish(device_t xicu, irq_type_t type, irq_line_t irq, ipl_t ipl,
    const char *name, char *intrstr, int (*handler)(void *), void *ih_arg,
    struct cpu_info *ci)
{
	irq_t intrhand;

	intrhand = malloc(sizeof(struct xicu_intrhand), M_TEMP, M_NOWAIT);
	if (intrhand == NULL)
		return NULL;
	intrhand->ih_func = handler;
	intrhand->ih_arg = ih_arg;
	if (xicu_establish(xicu, type, irq, ipl, name, intrstr, intrhand,
	    ci)) {
		free(intrhand, M_TEMP);
		return NULL;
	}
	evcnt_attach_dynamic(&intrhand->ih_ev, EVCNT_TYPE_INTR, NULL,
	    name, "intr");
	return intrhand;
}
