/* $NetBSD: $ */

/*-
  * Copyright (c) 2009 UPMC/LIP6
  * All rights reserved.
  * This software is distributed under the following condiions
  * compliant with the NetBSD foundation policy.
  *
  * Redistribution and use in source and binary forms, with or without
  * modification, are permitted provided that the following conditions
  * are met:
  * 1. Redistributions of source code must retain the above copyright
  *    notice, this list of conditions and the following disclaimer.
  * 2. Redistributions in binary form must reproduce the above copyright
  *    notice, this list of conditions and the following disclaimer in the
  *    documentation and/or other materials provided with the distribution.
  *
  * THIS SOFTWARE IS PROVIDED BY THE NETBSD FOUNDATION, INC. AND CONTRIBUTORS
  * ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED
  * TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
  * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE FOUNDATION OR CONTRIBUTORS
  * BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
  * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
  * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
  * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
  * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
  * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
  * POSSIBILITY OF SUCH DAMAGE.
  */

/* driver for soclib's dma engine */

#include <sys/cdefs.h>
__KERNEL_RCSID(0, "$NetBSD: machdep.c,v 1.223 2008/07/02 17:28:56 ad Exp $");

#include <sys/param.h>
#include <sys/device.h>
#include <sys/proc.h>
#include <sys/systm.h>
#include <sys/conf.h>

#include <uvm/uvm.h>

#include <machine/autoconf.h>
#include <machine/bus.h>
#include <machine/vcidma.h>

static struct vcidma_softc *main_vcidma_softc = NULL;

static int vcidma_match(device_t, cfdata_t, void *);
static void vcidma_attach(device_t, device_t, void *);

struct vcidma_softc {
	device_t sc_dev;
	paddr_t sc_p; /* device mapping informations */
	vaddr_t sc_psize;
	bus_space_tag_t sc_tag;
	bus_space_handle_t sc_handle;
	vaddr_t sc_pagezero;
	paddr_t sc_pzero;
};

/* vcidma devices definition: only 4 32bits registers */
#define DMA_FROM (0 * 4)
#define DMA_TO   (1 * 4)
#define DMA_LEN  (2 * 4)
#define DMA_LEN_ST_SUCCESS 0
#define DMA_LEN_ST_RDERR   1
#define DMA_LEN_ST_WRERR   3
#define DMA_LEN_ST_IDLE	   2
#define DMA_RESET  (3 * 4)

CFATTACH_DECL_NEW(vcidma, sizeof(struct vcidma_softc),
    vcidma_match, vcidma_attach, NULL, NULL);

int
vcidma_match(device_t parent, cfdata_t match, void *aux)
{
	struct tsardevs_attach_args *tsd = aux;
	if (strcmp(tsd->tsd_devtype, "soclib:dma") != 0)
		return 0;
	return 1;
}

void
vcidma_attach(device_t parent, device_t self, void *aux)
{
	struct vcidma_softc *sc = device_private(self);
	struct tsardevs_attach_args *tsd = aux;

	aprint_normal(": SocLib DMA Driver\n");

	sc->sc_dev = self;
	sc->sc_p = tsd->tsd_reg.reg_addr;
	sc->sc_psize = tsd->tsd_reg.reg_size;
	sc->sc_tag = tsd->tsd_tag;

	/* map the device */
	if (bus_space_map(sc->sc_tag, sc->sc_p, sc->sc_psize,
	    0, &sc->sc_handle)) {
		aprint_error_dev(self, ": couldn't map registers\n");
		return;
	}
	/* runs interruptless at this time */
	/* allocate a page that will be used for memset(, 0) */
	sc->sc_pagezero = uvm_km_alloc(kernel_map, PAGE_SIZE, 0,
	    UVM_KMF_WIRED | UVM_KMF_NOWAIT | UVM_KMF_ZERO);
	if (sc->sc_pagezero == 0) {
		aprint_error_dev(self, "can't allocate zero page\n");
		return;
	}
	if (!pmap_extract(pmap_kernel(), sc->sc_pagezero, &sc->sc_pzero)) {
		panic("vcidma: can't pmap_extract");
	}
	if (main_vcidma_softc == NULL)
		main_vcidma_softc = sc;
}

bool
vcidma_copy(paddr_t from, paddr_t to, size_t len)
{
	int err;
	struct vcidma_softc *sc = main_vcidma_softc;
	if (sc == NULL)
		return 0;
	KASSERT(bus_space_read_4(sc->sc_tag, sc->sc_handle, DMA_LEN) ==
	    DMA_LEN_ST_IDLE);
	__asm volatile("sync\n" ::: "memory");
	bus_space_write_4(sc->sc_tag, sc->sc_handle, DMA_FROM, from);
	bus_space_write_4(sc->sc_tag, sc->sc_handle, DMA_TO, to);
	bus_space_write_4(sc->sc_tag, sc->sc_handle, DMA_LEN, len);
	tsar_inval_caches_parange(to, len);
	while ((err = bus_space_read_4(sc->sc_tag, sc->sc_handle, DMA_LEN))
	    > DMA_LEN_ST_WRERR)
		; /* wait for DMA to complete */
	KASSERT(err == DMA_LEN_ST_SUCCESS);
	bus_space_write_4(sc->sc_tag, sc->sc_handle, DMA_RESET, 0);
	return 1;
		
}

bool
vcidma_zero(paddr_t to, size_t len)
{
	struct vcidma_softc *sc = main_vcidma_softc;
	int err;
	if (sc == NULL)
		return 0;
	KASSERT(bus_space_read_4(sc->sc_tag, sc->sc_handle, DMA_LEN) ==
	    DMA_LEN_ST_IDLE);
	bus_space_write_4(sc->sc_tag, sc->sc_handle, DMA_FROM, sc->sc_pzero);
	bus_space_write_4(sc->sc_tag, sc->sc_handle, DMA_TO, to);
	bus_space_write_4(sc->sc_tag, sc->sc_handle, DMA_LEN, len);
	tsar_inval_caches_parange(to, len);
	while ((err = bus_space_read_4(sc->sc_tag, sc->sc_handle, DMA_LEN))
	    > DMA_LEN_ST_WRERR)
		; /* wait for DMA to complete */
	KASSERT(err == DMA_LEN_ST_SUCCESS);
	bus_space_write_4(sc->sc_tag, sc->sc_handle, DMA_RESET, 0);
	return 1;
}
