/* $NetBSD: $ */

/*-
  * Copyright (c) 2009 UPMC/LIP6
  * All rights reserved.
  * This software is distributed under the following condiions
  * compliant with the NetBSD foundation policy.
  *
  * Redistribution and use in source and binary forms, with or without
  * modification, are permitted provided that the following conditions
  * are met:
  * 1. Redistributions of source code must retain the above copyright
  *    notice, this list of conditions and the following disclaimer.
  * 2. Redistributions in binary form must reproduce the above copyright
  *    notice, this list of conditions and the following disclaimer in the
  *    documentation and/or other materials provided with the distribution.
  *
  * THIS SOFTWARE IS PROVIDED BY THE NETBSD FOUNDATION, INC. AND CONTRIBUTORS
  * ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED
  * TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
  * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE FOUNDATION OR CONTRIBUTORS
  * BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
  * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
  * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
  * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
  * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
  * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
  * POSSIBILITY OF SUCH DAMAGE.
  */

#include <sys/cdefs.h>
__KERNEL_RCSID(0, "$NetBSD: cpu.c,v 1.24 2005/12/11 12:18:39 christos Exp $");

#include <sys/atomic.h>
#include <sys/param.h>
#include <sys/device.h>
#include <sys/systm.h>
#include <sys/cpu.h>
#include <uvm/uvm_extern.h>

#include <mips/locore.h>

#include <machine/autoconf.h>
#include <machine/xicu.h>
#include <machine/xicureg.h>
#include <machine/cpuset.h>

static int	xicu_match(device_t, cfdata_t, void *);
static void	xicu_attach(device_t, device_t, void *);

#define XICU_NCPU 4 /* max number of CPU per xicu */
#define XICU_OUT  3 /* max number of output line per CPU */
struct xicu2cpu {
	struct cpu_info *x2c_ci;
	int x2c_mips2xicu[XICU_OUT];
	struct evcnt x2c_ci_ev; /* evt counter for this CPU clock */
};


struct xicu_softc {
	device_t sc_dev;
	paddr_t sc_p; /* device mapping info */
	vaddr_t sc_psize;
	bus_space_tag_t sc_tag;
	bus_space_handle_t sc_handle;
	/* properties */
	int sc_input_lines;
	int sc_ipis;
	int sc_timers;
	/* mapping to CPUs */
	struct xicu2cpu sc_x2c[XICU_NCPU];
	int sc_last_cpuirq; /* for irq round robbin */
	/*
	 * we keep interrupt handler descriptors in an array, one per type.
	 * the array is indexed by the interrupt line number.
	 */
	struct xicu_intrhand *sc_pti_ih[XICU_NIRQ];
	struct xicu_intrhand *sc_wti_ih[XICU_NIRQ];
	struct xicu_intrhand *sc_hwi_ih[XICU_NIRQ];
};

static void	xicu_init_cpu(struct xicu_softc *, int);
static int	xicu_fill_x2c(const struct fdt_node *,
		    const struct boot_fdt_prop *, void *);
#ifdef MULTIPROCESSOR
static int	xicu_ipi(void *);
#endif

static inline
uint32_t xicuread(struct xicu_softc *sc, uint32_t reg)
{
	return bus_space_read_4(sc->sc_tag, sc->sc_handle, reg);
}
static inline
void xicuwrite(struct xicu_softc *sc, uint32_t reg, uint32_t val)
{
	bus_space_write_4(sc->sc_tag, sc->sc_handle, reg, val);
}

CFATTACH_DECL_NEW(xicu, sizeof(struct xicu_softc),
    xicu_match, xicu_attach, NULL, NULL);

static int
xicu_match(device_t parent, cfdata_t match, void *aux)
{
	struct tsardevs_attach_args *tsd = aux;
	if (strcmp(tsd->tsd_devtype, "soclib:xicu:root") != 0)
		return 0;

	return 1;
}

static void
xicu_attach(device_t parent, device_t self, void *aux)
{
	struct xicu_softc *sc = device_private(self);
	struct tsardevs_attach_args *tsd = aux;
	const struct boot_fdt_prop *prop;
	int i;

	aprint_normal(": programmable interrupt controller\n");
	sc->sc_dev = self;
	sc->sc_tag = tsd->tsd_tag;
	sc->sc_p = tsd->tsd_reg.reg_addr;
	sc->sc_psize = tsd->tsd_reg.reg_size;


	/* map the device */
	if (bus_space_map(sc->sc_tag, sc->sc_p, sc->sc_psize,
	    0, &sc->sc_handle)) {
		aprint_error_dev(self, "couldn't map registers\n");
		return;
	}
	aprint_normal_dev(self, "mapped at 0x%x\n", sc->sc_handle);

	prop = fdt_find_prop(tsd->tsd_node->fdt_node_data, "input_lines");
	if (prop == NULL) {
		aprint_error_dev(self, "no input_lines\n");
		return;
	}
	sc->sc_input_lines = be32toh(prop->prop_value[0]);

	prop = fdt_find_prop(tsd->tsd_node->fdt_node_data, "ipis");
	if (prop == NULL) {
		aprint_error_dev(self, "no ipis\n");
		return;
	}
	sc->sc_ipis = be32toh(prop->prop_value[0]);

	prop = fdt_find_prop(tsd->tsd_node->fdt_node_data, "timers");
	if (prop == NULL) {
		aprint_error_dev(self, "no timers\n");
		return;
	}
	sc->sc_timers = be32toh(prop->prop_value[0]);

	aprint_normal_dev(self, "%d inputs, %d IPIs, %d timers\n",
	     sc->sc_input_lines, sc->sc_ipis, sc->sc_timers);

	/* initialize mapping registers */
	for(i = 0; i < sc->sc_input_lines; i++)
		xicuwrite(sc, XICU_MSK_HWI(i), 0);
	for(i = 0; i < sc->sc_timers; i++)
		xicuwrite(sc, XICU_MSK_PTI(i), 0);
	for(i = 0; i < sc->sc_ipis; i++)
		xicuwrite(sc, XICU_MSK_WTI(i), 0);

	/* parse the out@ nodes and fill sc_x2c[] */
	fdt_walk(tsd->tsd_node, "device_type", xicu_fill_x2c, sc);
	/* and establish a timer and IPI for each CPU */
	for (i = 0; i < XICU_NCPU; i++) {
		if (sc->sc_x2c[i].x2c_ci) {
			sc->sc_x2c[i].x2c_ci->ci_xicu = self;
			sc->sc_x2c[i].x2c_ci->ci_xicuidx = i;
			xicu_init_cpu(sc, i);
		}
	}
}

static int
xicu_fill_x2c(const struct fdt_node *node, const struct boot_fdt_prop *prop,
    void *p)
{
	struct xicu_softc *sc = p;
	struct fdt_node *cpu_node;
	struct cpu_info *ci;
	const struct boot_fdt_prop *irq, *outline;
	int i;

	if (strcmp((const char *)&prop->prop_value[0],
	    "soclib:xicu:filter") != 0)
		return 0; /* not right node */
	irq = fdt_find_prop(node->fdt_node_data, "irq");
	if (irq == NULL) {
		aprint_error_dev(sc->sc_dev, "missing irq in node %s\n",
		    node->fdt_node_name);
		return EINVAL;
	}
	outline = fdt_find_prop(node->fdt_node_data, "output_line");
	if (outline == NULL) {
		aprint_error_dev(sc->sc_dev, "missing output_line in node %s\n",
		    node->fdt_node_name);
		return EINVAL;
	}
	cpu_node = fdt_find_node_from_ref(irq->prop_value[0]);
	if (cpu_node == NULL) {
		aprint_error_dev(sc->sc_dev, "no CPU node for %d node %s\n",
		    be32toh(irq->prop_value[0]), node->fdt_node_name);
		return EINVAL;
	}
	ci = device_private(cpu_node->fdt_node_dev);
	/* lookup x2c, grab a new one if noone match */
	for (i = 0; i < XICU_NCPU; i++) {
		if (sc->sc_x2c[i].x2c_ci == ci)
			break;
		if (sc->sc_x2c[i].x2c_ci == NULL) {
			sc->sc_x2c[i].x2c_ci = ci;
			break;
		}
	}
	if (i == XICU_NCPU) {
		aprint_error_dev(sc->sc_dev, "too many CPUs\n");
		return EINVAL;
	}
	if (be32toh(irq->prop_value[1]) >= XICU_OUT) {
		aprint_error_dev(sc->sc_dev, "MIPS irq %d out of range\n",
		    be32toh(irq->prop_value[1]));
		return EINVAL;
	}
	aprint_debug_dev(sc->sc_dev, "line %d to %s line %d (slot %d)\n",
	    be32toh(outline->prop_value[0]), device_xname(ci->ci_dev),
	    be32toh(irq->prop_value[1]), i);
	sc->sc_x2c[i].x2c_mips2xicu[be32toh(irq->prop_value[1])] =
	    be32toh(outline->prop_value[0]);
	return 0;
}

int
xicu_establish(device_t xicu, irq_type_t type, irq_line_t irq, ipl_t ipl,
   const char *name, char *intrstr, struct xicu_intrhand *ih,
   struct cpu_info *ci)
{
	int s;
	struct xicu_softc *sc = device_private(xicu);
	char outstr[50];
	/* get target CPU */
	/* get xicu output line for target CPU and IPL */
	int xirq;

	if (irq >= XICU_NIRQ || irq < 0)
		return EINVAL;
	if (ci == NULL)
		ci = sc->sc_x2c[sc->sc_last_cpuirq].x2c_ci;
	xirq = sc->sc_x2c[ci->ci_xicuidx].x2c_mips2xicu[ipl2irq(ipl)];
	printf("xicu_establish on %s/%p/%p\n", device_xname(xicu), xicu, sc);
	aprint_debug_dev(xicu, "route irq %d type %d for %s to %s "
	    "via output %d (slot %d)\n", irq, type, name,
	    device_xname(ci->ci_dev), xirq, ci->ci_xicuidx);
	s = splhigh();
	switch (type) {
	case IRQ_HWI:
		/* register handler */
		KASSERT(sc->sc_hwi_ih[irq] == NULL);
		sc->sc_hwi_ih[irq] = ih;
		/* route and unmask interrupt */
		xicuwrite(sc, XICU_MSK_HWI_E(xirq), (1U << irq));
		printf("xicu_establish for %s %d %d %d reg 0x%x=0x%x\n",
		    name, type, irq, ipl, XICU_MSK_HWI(xirq),
		    xicuread(sc, XICU_MSK_HWI(xirq)));
		sprintf(outstr, "%s line %d, %s irq %d",
		    device_xname(xicu), irq, device_xname(ci->ci_dev),
		    ipl2irq(ipl));
		break;
	case IRQ_PTI:
		/* register handler */
		KASSERT(sc->sc_pti_ih[irq] == NULL);
		sc->sc_pti_ih[irq] = ih;
		/* route and unmask interrupt */
		xicuwrite(sc, XICU_MSK_PTI_E(xirq), (1U << irq));
		printf("xicu_establish for %s %d %d %d reg 0x%x=0x%x\n",
		    name, type, irq, ipl, XICU_MSK_PTI(xirq),
		    xicuread(sc, XICU_MSK_PTI(xirq)));
		sprintf(outstr, "%s timer %d, %s irq %d",
		    device_xname(xicu), irq, device_xname(ci->ci_dev),
		    ipl2irq(ipl));
		break;
	case IRQ_WTI:
		/* register handler */
		KASSERT(sc->sc_wti_ih[irq] == NULL);
		sc->sc_wti_ih[irq] = ih;
		/* route and unmask interrupt */
		xicuwrite(sc, XICU_MSK_WTI_E(xirq), (1U << irq));
		printf("xicu_establish for %s %d %d %d reg 0x%x=0x%x\n",
		    name, type, irq, ipl, XICU_MSK_WTI(xirq),
		    xicuread(sc, XICU_MSK_WTI(xirq)));
		sprintf(outstr, "%s trigger %d, %s irq %d",
		    device_xname(xicu), irq, device_xname(ci->ci_dev),
		    ipl2irq(ipl));
		break;
	}
	if (intrstr)
		strcpy(intrstr, outstr);
	return 0;
}

void
xicu_intr(int ppl, vaddr_t pc, u_int32_t status)
{
	struct cpu_info * const ci = curcpu();
	struct xicu_softc *sc = device_private(curcpu()->ci_xicu);
	int cpuidx = curcpu()->ci_xicuidx;
	int i, ipl;
	uint32_t pending;
	uint32_t xicupending;
	uint32_t irq;
	int outirq;

#if 0
	aprint_debug_dev(curcpu()->ci_dev,
	    "xicu_intr(%p, 0x%x, 0x%x, 0x%x 0x%x %d)\n", curlwp, ppl, pc, status, (vaddr_t)&ci, ci->ci_idepth);
#endif
	KASSERT(ci->ci_cpl == IPL_HIGH);
	uvmexp.intrs++;
	while (ppl < (ipl = splintr(&pending))) {
		splx(ipl);
		/* find pending spls for this CPU, starting from high */
		for (i = 0; pending != 0; i++, pending = pending >>1) {
			if ((pending & MIPS_INT_MASK_0) == 0)
				continue;
			/* output pin for this MIPS irq */
			outirq = sc->sc_x2c[cpuidx].x2c_mips2xicu[i];
			/* find pending interrupts for this SPL on this CPU */
			xicupending = xicuread(sc, XICU_PRIO(outirq));
#if 0
			aprint_debug("xicupending 0x%x\n", xicupending);
#endif
			while ((xicupending & XICU_PRIO_PENDING) != 0) {
				if (xicupending & XICU_PRIO_PTI) {
					irq = XICU_PRIO_PTII(xicupending);
					/* special-case clock interrupt */
					if (irq == cpuidx) {
						struct clockframe cf;
						cf.pc = pc;
						cf.sr = status;
						cf.intr = (ci->ci_idepth > 1);
						hardclock(&cf);
						(void)xicuread(sc,
						    XICU_PTI_ACK(irq));
						sc->sc_x2c[cpuidx].x2c_ci_ev.ev_count++;
					} else {
						if (sc->sc_pti_ih[irq] == NULL)
							printf("irq: %d\n", irq);
						KASSERT(sc->sc_pti_ih[irq] != NULL);
						sc->sc_pti_ih[irq]->ih_func(
						    sc->sc_pti_ih[irq]->ih_arg);
						sc->sc_pti_ih[irq]->ih_ev.ev_count++;
					}
				}
				if (xicupending & XICU_PRIO_WTI) {
					irq = XICU_PRIO_WTII(xicupending);
					KASSERT(sc->sc_wti_ih[irq] != NULL);
					sc->sc_wti_ih[irq]->ih_func(
					    sc->sc_wti_ih[irq]->ih_arg);
					sc->sc_wti_ih[irq]->ih_ev.ev_count++;
				}
				if (xicupending & XICU_PRIO_HWI) {
					KERNEL_LOCK(1, NULL);
					irq = XICU_PRIO_HWII(xicupending);
					KASSERT(sc->sc_hwi_ih[irq] != NULL);
					sc->sc_hwi_ih[irq]->ih_func(
					    sc->sc_hwi_ih[irq]->ih_arg);
					sc->sc_hwi_ih[irq]->ih_ev.ev_count++;
					KERNEL_UNLOCK_ONE(NULL);
				}
				xicupending = xicuread(sc, XICU_PRIO(outirq));
			}
		}
		(void)splhigh();
	}
	KASSERT(ci->ci_cpl == IPL_HIGH);
}

/*
 * initialize timer and IPI for the target CPU
 */
static void
xicu_init_cpu(struct xicu_softc *sc, int cpui)
{
	/* get CPU and output line for IPL_CLOCK for cpui */
	int xirq = sc->sc_x2c[cpui].x2c_mips2xicu[ipl2irq(IPL_CLOCK)];
	struct cpu_info *ci = sc->sc_x2c[cpui].x2c_ci;
	/* assume about 500Khz clock -> 10 interrupt per second (for now) */
	xicuwrite(sc, XICU_PTI_PER(cpui), 70000);
	/* clear any pending interrupt */
	(void)xicuread(sc, XICU_PTI_ACK(cpui));
	/* map interrupt to IPL_CLOCK */
	xicuwrite(sc, XICU_MSK_PTI_E(xirq), (1U << cpui));
	evcnt_attach_dynamic(&sc->sc_x2c[cpui].x2c_ci_ev, EVCNT_TYPE_INTR,
	    NULL, device_xname(ci->ci_dev), "clock");

#ifdef MULTIPROCESSOR
	/* establish IPI handler */
	ipi_init(ci);
	intr_establish(sc->sc_dev, IRQ_WTI, cpui, IPL_SCHED,
	    device_xname(ci->ci_dev), NULL, xicu_ipi, ci, ci);
#endif

	printf("xicu out %d map 0x%x 0x%x 0x%x\n", 
	    xirq,
	    xicuread(sc, XICU_MSK_HWI(xirq)),
	    xicuread(sc, XICU_MSK_PTI(xirq)),
	    xicuread(sc, XICU_MSK_WTI(xirq)));
}

#ifdef MULTIPROCESSOR
int
xicu_send_ipi(struct cpu_info *ci, int tag)
{
	uint32_t req = 1 << tag;
	struct xicu_softc *sc = device_private(ci->ci_xicu);
	int cpuidx = ci->ci_xicuidx;

	if (!CPUSET_HAS_P(cpus_running, cpu_index(ci)))
		return -1;
#if 0
	printf("%s IPI %d to %s\n", device_xname(curcpu()->ci_dev),
	    tag, device_xname(ci->ci_dev));
#endif
	KASSERT(tag < NIPIS);
	atomic_or_ulong(&ci->ci_request_ipis, req);
	xicuwrite(sc, XICU_WTI_REG(cpuidx), 1);
	return 0;
}

static int
xicu_ipi(void *p)
{
	struct cpu_info *ci = curcpu();
	int cpuidx = ci->ci_xicuidx;
	struct xicu_softc *sc = device_private(ci->ci_xicu);
	uint32_t ipi_mask;

	(void)xicuread(sc, XICU_WTI_REG(cpuidx));
	ipi_mask = atomic_swap_ulong(&ci->ci_request_ipis, 0);
	if (ipi_mask == 0)
		return 0;

#if 0
	printf("%s got IPIs 0x%x\n", device_xname(ci->ci_dev),
	    ipi_mask);
#endif
	ipi_process(ci, ipi_mask);
	return 1;
}
#endif /* MULTIPROCESSOR */
