/* $NetBSD: $ */

/*-
  * Copyright (c) 2009 UPMC/LIP6
  * All rights reserved.
  * This software is distributed under the following condiions
  * compliant with the NetBSD foundation policy.
  *
  * Redistribution and use in source and binary forms, with or without
  * modification, are permitted provided that the following conditions
  * are met:
  * 1. Redistributions of source code must retain the above copyright
  *    notice, this list of conditions and the following disclaimer.
  * 2. Redistributions in binary form must reproduce the above copyright
  *    notice, this list of conditions and the following disclaimer in the
  *    documentation and/or other materials provided with the distribution.
  *
  * THIS SOFTWARE IS PROVIDED BY THE NETBSD FOUNDATION, INC. AND CONTRIBUTORS
  * ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED
  * TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
  * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE FOUNDATION OR CONTRIBUTORS
  * BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
  * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
  * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
  * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
  * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
  * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
  * POSSIBILITY OF SUCH DAMAGE.
  */

#include <sys/cdefs.h>
__KERNEL_RCSID(0, "$NetBSD: mainbus.c,v 1.36 2005/12/11 12:18:39 christos Exp $");

#define _MIPS_BUS_DMA_PRIVATE
#include <sys/param.h>
#include <sys/systm.h>
#include <sys/device.h>
#include <uvm/uvm_extern.h>

#include <machine/autoconf.h>
#include <machine/bootinfo.h>
#include <machine/pte.h>
#include <machine/locore.h>
#include <machine/xicu.h>
#include "locators.h"

static int  mainbus_match(device_t, cfdata_t, void *);
static void mainbus_attach(device_t, device_t, void *);
static int  mainbus_cluster_print(void *, const char *);

CFATTACH_DECL_NEW(mainbus, 0,
    mainbus_match, mainbus_attach, NULL, NULL);

struct cluster_softc {
	device_t sc_dev; /* our device info */
	device_t sc_xicu; /* xicu associated with this cluster */
};

static int  cluster_match(device_t, cfdata_t, void *);
static void cluster_attach(device_t, device_t, void *);

CFATTACH_DECL_NEW(cluster, sizeof(struct cluster_softc),
    cluster_match, cluster_attach, NULL, NULL);

struct mainbus_attach_args {
	struct fdt_node cla_node; /* FDT node of cluster */
	struct fdt_reg cla_reg; /* and its locator data */
};

static int  tsardev_print(void *, const char *);
static int  cpubus_print(void *, const char *);

#define TSARMIPS_BUS_DMAMAP_OPS_INITIALIZER {				\
                .dmamap_create          = _bus_dmamap_create,		\
		.dmamap_destroy         = _bus_dmamap_destroy,		\
		.dmamap_load            = tsarmips_bus_dmamap_load,	\
		.dmamap_load_mbuf       = tsarmips_bus_dmamap_load_mbuf,\
		.dmamap_load_uio        = tsarmips_bus_dmamap_load_uio,	\
		.dmamap_load_raw        = _bus_dmamap_load_raw,		\
		.dmamap_unload          = _bus_dmamap_unload,		\
		.dmamap_sync            = _bus_dmamap_sync,		\
}

#define TSARMIPS_BUS_DMAMEM_OPS_INITIALIZER {				\
		.dmamem_alloc =         _bus_dmamem_alloc,		\
		.dmamem_free =          _bus_dmamem_free,		\
		.dmamem_map =           tsarmips_bus_dmamem_map,	\
		.dmamem_unmap =         tsarmips_bus_dmamem_unmap,	\
		.dmamem_mmap =          _bus_dmamem_mmap,		\
}

#define TSARMIPS_BUS_DMATAG_OPS_INITIALIZER {				\
		.dmatag_subregion =     _bus_dmatag_subregion,		\
		.dmatag_destroy =       _bus_dmatag_destroy,		\
}

static int
mainbus_match(device_t parent, cfdata_t match, void *aux)
{
	return (1);
}

static void
mainbus_attach(device_t parent, device_t self, void *aux)
{
	struct mainbus_attach_args cla;
	struct fdt_node topologyn;
	int err;

	printf("\n");

	/* look for the topology node, clusters are under it */
	err = fdt_find_node(&fdt->fdt_root_node,"topology", &topologyn);
	if (err) {
		aprint_error_dev(self, "can't find topology: %d\n", err);
		panic("mainbus_attach");
	}
	/* now parse subnodes of topologyn looking for clusters */
	err = fdt_find_subnode(&topologyn, &cla.cla_node);
	if (err) {
		aprint_error_dev(self, "no nodes in topology: %d\n", err);
		panic("mainbus_attach2");
	}
	while (1) {
		err = fdt_read_node_reg(&cla.cla_node, &cla.cla_reg);
		if (err) {
			aprint_error_dev(self, "can't read regs for %s: %d\n",
			    cla.cla_node.fdt_node_name, err);
		} else {
			config_found_ia(self, "mainbus", &cla,
			    mainbus_cluster_print);
		}
		err = fdt_next_node(&topologyn, cla.cla_node.fdt_node_data,
		    &cla.cla_node);
		if (err)
			break;
	}
}

static int
mainbus_cluster_print(aux, pnp)
	void *aux;
	const char *pnp;
{
	struct mainbus_attach_args *cla = aux;
	const uint32_t *r = (uint32_t *)(&cla->cla_reg.reg_addr);

	if (pnp)
		aprint_normal("%s at %s", cla->cla_node.fdt_node_name, pnp);

	aprint_normal(" cluster %d, %d", r[1], r[0]);
	return (UNCONF);
}

static int
cluster_match(device_t parent, cfdata_t match, void *aux)
{
	return (1);
}

static void
cluster_attach(device_t parent, device_t self, void *aux)
{
	struct mainbus_attach_args *cla = aux;
	struct tsardevs_attach_args tsda;
	const struct boot_fdt_prop *cl_devices, *tsd_prop;
	struct fdt_node *xicu_node = NULL;
	int i, err;
	//int bootcpu = mips_cp0_ebase_read() & 0x1ff;
	struct cluster_softc *sc = device_private(self);
	static struct mips_bus_dma_tag _dmatag = {
		._dmamap_ops = TSARMIPS_BUS_DMAMAP_OPS_INITIALIZER,
		._dmamem_ops = TSARMIPS_BUS_DMAMEM_OPS_INITIALIZER,
		._dmatag_ops = TSARMIPS_BUS_DMATAG_OPS_INITIALIZER,
	};
	sc->sc_dev = self;

	printf("\n");

	/*
	 * parse the FDT propery "devices" of the cluster node,
	 * which points to nodes attached to this cluster.
	 * This has to be done in the right order, so we have to
	 * traverse the list several times
	 */

	cl_devices = fdt_find_prop(cla->cla_node.fdt_node_data, "devices");
	if (cl_devices == NULL) {
		aprint_error_dev(self, "no devices\n");
		return;
	}
	/* first attach xicu; we need it to wake up CPUs */
	for (i = 0; i < be32toh(cl_devices->prop_value_size) / sizeof(uint32_t);
	    i++) {
		tsda.tsd_node =
		    fdt_find_node_from_ref(cl_devices->prop_value[i]);
		if (tsda.tsd_node == NULL) {
			aprint_error_dev(self, "no node for ref %d\n",
			    be32toh(cl_devices->prop_value[i]));
			continue;
		}
		tsd_prop = fdt_find_prop(tsda.tsd_node->fdt_node_data,
		    "device_type");
		if (tsd_prop == NULL)
			continue;
		tsda.tsd_devtype = (const char *)&tsd_prop->prop_value[0];
		if (strcmp(tsda.tsd_devtype, "soclib:xicu:root") != 0)
			continue;
		err = fdt_read_node_reg(tsda.tsd_node, &tsda.tsd_reg);
		if (err) {
			aprint_error_dev(self, "can't read regs for %s: %d\n",
			    tsda.tsd_node->fdt_node_name, err);
		} else {
			tsda.tsd_irq = -1;
			tsda.tsd_node->fdt_node_dev =
			    config_found_ia(self, "cluster", &tsda,
			    tsardev_print);
			tsda.tsd_xicu = tsda.tsd_node->fdt_node_dev;
			xicu_node = tsda.tsd_node;
		}
	}
	/* then attach CPUs */
	pmap_seccpus_bootstrap_add((vaddr_t)cputramp);
	for (i = 0; i < be32toh(cl_devices->prop_value_size) / sizeof(uint32_t);
	    i++) {
		tsda.tsd_node =
		    fdt_find_node_from_ref(cl_devices->prop_value[i]);
		if (tsda.tsd_node == NULL) {
			aprint_error_dev(self, "no node for ref %d\n",
			    be32toh(cl_devices->prop_value[i]));
			continue;
		}
		tsd_prop = fdt_find_prop(tsda.tsd_node->fdt_node_data,
		    "device_type");
		if (tsd_prop == NULL)
			continue;
		tsda.tsd_devtype = (const char *)&tsd_prop->prop_value[0];
		if (strcmp(tsda.tsd_devtype, "cpu") != 0)
			continue;
		err = fdt_read_node_reg(tsda.tsd_node, &tsda.tsd_reg);
		if (err) {
			aprint_error_dev(self, "can't read regs for %s: %d\n",
			    tsda.tsd_node->fdt_node_name, err);
		} else {
			tsda.tsd_irq = -1;
			tsda.tsd_node->fdt_node_dev =
			    config_found_ia(self, "cpubus", &tsda,
			    cpubus_print);
		}
	}
	pmap_seccpus_bootstrap_remove((vaddr_t)cputramp);

	/* then finish setting up xicus */
	xicu_init(xicu_node);

	/* now we can handle remaining devices */
	for (i = 0; i < be32toh(cl_devices->prop_value_size) / sizeof(uint32_t);
	    i++) {
		tsda.tsd_node =
		    fdt_find_node_from_ref(cl_devices->prop_value[i]);
		if (tsda.tsd_node == NULL) {
			aprint_error_dev(self, "no node for ref %d\n",
			    be32toh(cl_devices->prop_value[i]));
			continue;
		}
		tsd_prop = fdt_find_prop(tsda.tsd_node->fdt_node_data,
		    "device_type");
		if (tsd_prop == NULL)
			continue;
		tsda.tsd_devtype = (const char *)&tsd_prop->prop_value[0];
		if (strcmp(tsda.tsd_devtype, "soclib:xicu:root") == 0)
			continue;
		if (strcmp(tsda.tsd_devtype, "cpu") == 0)
			continue;
		err = fdt_read_node_reg(tsda.tsd_node, &tsda.tsd_reg);
		if (err) {
			aprint_error_dev(self, "can't read regs for %s: %d\n",
			    tsda.tsd_node->fdt_node_name, err);
			continue;
		}
		tsd_prop = fdt_find_prop(tsda.tsd_node->fdt_node_data, "irq");
		if (tsd_prop == NULL)
			tsda.tsd_irq = -1;
		else
			tsda.tsd_irq = be32toh(tsd_prop->prop_value[1]); /* XXX */
		tsda.tsd_tag = 0; /* unused */
		memset(&tsda.tsd_dmatag, 0, sizeof(tsda.tsd_dmatag));
		tsda.tsd_dmatag = &_dmatag;
		tsda.tsd_node->fdt_node_dev =
		    config_found_ia(self, "cluster", &tsda, tsardev_print);
	}
}

static int
tsardev_print(aux, pnp)
	void *aux;
	const char *pnp;
{
	struct tsardevs_attach_args *tsd = aux;

	if (pnp)
		aprint_normal("%s at %s", tsd->tsd_node->fdt_node_name, pnp);

	if (tsd->tsd_reg.reg_size > 0)
		aprint_normal(" addr 0x%" PRIx64 "/%u",
		    (uint64_t)tsd->tsd_reg.reg_addr,
		    (u_int)tsd->tsd_reg.reg_size);
	if (tsd->tsd_irq >= 0)
		aprint_normal(" irq %d", tsd->tsd_irq);
	return (UNCONF);
}

static int
cpubus_print(aux, pnp)
	void *aux;
	const char *pnp;
{
	struct tsardevs_attach_args *tsd = aux;

	if (pnp)
		aprint_normal("%s at %s", tsd->tsd_node->fdt_node_name, pnp);

	if (tsd->tsd_reg.reg_size > 0)
		aprint_normal(" cpuid 0x%x", (u_int)tsd->tsd_reg.reg_addr);
	return (UNCONF);
}
