/* $NetBSD: $ */
/*-
  * Copyright (c) 2009 UPMC/LIP6
  * All rights reserved.
  * This software is distributed under the following condiions
  * compliant with the NetBSD foundation policy.
  *
  * Redistribution and use in source and binary forms, with or without
  * modification, are permitted provided that the following conditions
  * are met:
  * 1. Redistributions of source code must retain the above copyright
  *    notice, this list of conditions and the following disclaimer.
  * 2. Redistributions in binary form must reproduce the above copyright
  *    notice, this list of conditions and the following disclaimer in the
  *    documentation and/or other materials provided with the distribution.
  *
  * THIS SOFTWARE IS PROVIDED BY THE NETBSD FOUNDATION, INC. AND CONTRIBUTORS
  * ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED
  * TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
  * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE FOUNDATION OR CONTRIBUTORS
  * BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
  * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
  * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
  * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
  * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
  * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
  * POSSIBILITY OF SUCH DAMAGE.
  */

/*
 * Copyright (c) 2006 Mathieu Ropert <mro@adviseo.fr>
 *
 * Permission to use, copy, modify, and distribute this software for any
 * purpose with or without fee is hereby granted, provided that the above
 * copyright notice and this permission notice appear in all copies.
 *
 * THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
 * WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
 * ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
 * ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
 * OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 */

/*
 *
 * Copyright (c) 1997 Charles D. Cranor and Washington University.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *      This product includes software developed by Charles D. Cranor and
 *      Washington University.
 * 4. The name of the author may not be used to endorse or promote products
 *    derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

/*
 * Copyright 2001 (c) Wasabi Systems, Inc.
 * All rights reserved.
 *
 * Written by Frank van der Linden for Wasabi Systems, Inc.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *      This product includes software developed for the NetBSD Project by
 *      Wasabi Systems, Inc.
 * 4. The name of Wasabi Systems, Inc. may not be used to endorse
 *    or promote products derived from this software without specific prior
 *    written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY WASABI SYSTEMS, INC. ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED
 * TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL WASABI SYSTEMS, INC
 * BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include <sys/cdefs.h>
__KERNEL_RCSID(0, "$NetBSD: machdep.c,v 1.223 2008/07/02 17:28:56 ad Exp $");

#include <sys/param.h>
#include <sys/systm.h>
#include <sys/proc.h>
#include <sys/malloc.h>
#include <sys/pool.h>
#include <sys/user.h>
#include <sys/kernel.h>
#include <sys/atomic.h>
#include <sys/cpu.h>
#include <sys/intr.h>
#include <sys/xcall.h>
#include <sys/queue.h>

#include <uvm/uvm.h>

#include <machine/pmap.h>
#include <machine/vcidma.h>
#include <machine/pte.h>

#include "vcidma.h"

void printx(int);

#define PMD_PTE_SET	0x0001
#define PMD_PTE_GET	0x0002
#define PMD_PMAP_ALLOC	0x0004
#define PMD_PMAP_SWITCH	0x0008
#define PMD_PMAP_INIT	0x0010
#define PMD_PTE_SET2	0x0020
#define PMD_PTE_GET2	0x0040
#if 0
int pmap_debug = PMD_PMAP_ALLOC | PMD_PMAP_SWITCH |PMD_PMAP_INIT /* | PMD_PTE_SET2 | PMD_PTE_GET2*/;
#else
int pmap_debug = 0;
#endif

#define DPRINTF(x, s)  if (pmap_debug & (x)) printf s
#define UDPRINTF(p, x, s)  if ((p) != pmap_kernel() && (pmap_debug & (x))) printf s

struct  user *proc0paddr; /* for kern_proc.c */

extern char kernel_text[], __data_start[], edata[];
extern pde1_t kernelpd[];

static pte2_t *kpte2_base;
static vaddr_t kpte2_va;

/* virtual addresses used by pmap_zero_page and pmap_copy_page */
static vaddr_t pmap_va1;
static pte2_t *pmap_pte1;
static vaddr_t pmap_va2;
static pte2_t *pmap_pte2;

static bool pmap_initialized = false;

int nkptp; /* number of kernel L2 page table */
#define NPTE2_EXTRA 1024 /* enough to map 4M extra memory */
static vaddr_t pmap_maxkvaddr; /* what we can map with nkptp */

static vaddr_t virtual_avail; /* VA of first free KVA */
static vaddr_t virtual_end;   /* VA of last free KVA */

static kmutex_t pmaps_lock;
static kmutex_t pmaps_copy_lock;

/* prot2pte array: convert a vm_prot_t to PTE2 flags */
static const uint32_t prot2pte[] = {
	[VM_PROT_NONE] = 0, /* XXX can't prevent read */
	[VM_PROT_EXECUTE] = PTE2_X,
	[VM_PROT_READ] = 0,
	[VM_PROT_READ|VM_PROT_EXECUTE] = PTE2_X, /* XXX can't prevent read */
	[VM_PROT_WRITE] = PTE2_W,
	[VM_PROT_WRITE|VM_PROT_EXECUTE] = PTE2_W | PTE2_X,
	[VM_PROT_WRITE|VM_PROT_READ] = PTE2_W, /* XXX can't prevent read */
	[VM_PROT_ALL] = PTE2_W | PTE2_X
};

/* management of vm pages and pv_list */

#define VM_PAGE_TO_PP(pg)	(&(pg)->mdpage.mp_pp)

#define pp_lock(pp)		mutex_spin_enter(&(pp)->pp_lock)
#define pp_unlock(pp)		mutex_spin_exit(&(pp)->pp_lock)
#define pp_locked(pp)		mutex_owned(&(pp)->pp_lock)

#define PV_HASH_SIZE	    32768
#define PV_HASH_LOCK_CNT	32

struct pv_hash_lock {
	kmutex_t lock;
} __aligned(CACHE_LINE_SIZE) pv_hash_locks[PV_HASH_LOCK_CNT]
    __aligned(CACHE_LINE_SIZE);

struct pv_hash_head {
	SLIST_HEAD(, pv_entry) hh_list;
} pv_hash_heads[PV_HASH_SIZE];

static u_int
pvhash_hash(struct vm_page *ptp, vaddr_t va)
{

	return (uintptr_t)ptp / sizeof(*ptp) + (va >> PAGE_SHIFT);
}

static struct pv_hash_head *
pvhash_head(u_int hash)
{

	return &pv_hash_heads[hash % PV_HASH_SIZE];
}

static kmutex_t *
pvhash_lock(u_int hash)
{

	return &pv_hash_locks[hash % PV_HASH_LOCK_CNT].lock;
}

static struct pv_entry *
pvhash_remove(struct pv_hash_head *hh, struct vm_page *ptp, vaddr_t va)
{
	struct pv_entry *pve;
	struct pv_entry *prev;

	prev = NULL;
	SLIST_FOREACH(pve, &hh->hh_list, pve_hash) {
		if (pve->pve_pte.pte_ptp == ptp &&
		    pve->pve_pte.pte_va == va) {
			if (prev != NULL) {
				SLIST_REMOVE_AFTER(prev, pve_hash);
			} else {
				SLIST_REMOVE_HEAD(&hh->hh_list, pve_hash);
			}
			break;
		}
		prev = pve;
	}
	return pve;
}

static struct pv_pte * pv_pte_first(struct pmap_page *);
static struct pv_pte * pv_pte_next(struct pmap_page *, struct pv_pte *);

static struct pool_cache pmap_pv_cache;

static void pmap_do_remove(struct pmap *, vaddr_t, vaddr_t, int);

/* ptp-related functions */
static pte2_t *pmap_map_ptes(struct pmap *, int);
#define PM_NOWAIT 0
#define PM_WAITOK 1
static void   pmap_unmap_ptes(struct pmap *);

static inline struct vm_page * pmap_find_ptp(struct pmap *,
    vaddr_t, paddr_t, pte2_t *);
static struct vm_page * pmap_get_ptp(struct pmap *, vaddr_t, pte2_t *);

struct pmap kernel_pmap_store;

static struct pmap_head pmaps; /* list of non-kernel pmaps */

/* pool that pmap structures are allocated from */
static struct pool_cache pmap_cache;
static int	pmap_ctor(void *, void *, int);
static void 	pmap_dtor(void *, void *);

/* statistics management */
static inline void
pmap_stats_update(struct pmap *pmap, int resid_diff, int wired_diff)
{
        if (pmap == pmap_kernel()) {
		atomic_add_long(&pmap->pm_stats.resident_count, resid_diff);
		atomic_add_long(&pmap->pm_stats.wired_count, wired_diff);
	} else {
		KASSERT(mutex_owned(&pmap->pm_lock));
		pmap->pm_stats.resident_count += resid_diff;
		pmap->pm_stats.wired_count += wired_diff;
	}
}

static inline void
pmap_stats_update_byflags(struct pmap *pmap, uint32_t nf, uint32_t of)
{
        int resid_diff = ((nf & PTE2_V) ? 1 : 0) - ((of & PTE2_V) ? 1 : 0);
        int wired_diff = ((nf & PTE2_w) ? 1 : 0) - ((of & PTE2_w) ? 1 : 0);

#if PTE2_w != 0
        KASSERT((nf & (PTE2_V | PTE2_w)) != PTE2_w);
        KASSERT((of & (PTE2_V | PTE2_w)) != PTE2_w);
#endif

        pmap_stats_update(pmap, resid_diff, wired_diff);
}

static void
pmap_tlb_inval(vaddr_t va)
{
#ifdef VCACHE
	dtlbinval(va);
	itlbinval(va);
#endif
}

static void
pmap_dtlb_inval(vaddr_t va)
{
#ifdef VCACHE
	dtlbinval(va);
#endif
}

static void pmap_free(struct pmap *pmap);

/* helper function for locore.S */
void pmap_bootstrap_map(int phys, int size, int virt, pte1_t *pte);
void
pmap_bootstrap_map(int phys, int size, int virt, pte1_t *pte)
{
	/* map I/O uncached */
	int flags = (phys == 0) ? 0 :
	    (PTE1_V | PTE1_W | PTE1_X | PTE1_G | PTE1_w);
	if (phys && phys <= VM_MAX_KERNEL_ADDRESS)
		flags |= PTE1_C;

	int i = (virt & PTE1_FRAME) >> PTE1_SHIFT; /* slot index */
	phys = (phys & PTE1_FRAME) >> PTE1_SHIFT; /* physical PFN */
	for (; size > 0 ; phys++, i++, size--) {
		pte[i] = phys | flags;
		//printx((vaddr_t)&pte[i]); printx(pte[i]);
	}
}

void
pmap_bootstrap(vaddr_t *vaend, vaddr_t *msgbp, paddr_t *paendp,
    vaddr_t *fdt_vap, paddr_t fdt_pa, int fdt_size)
{
	vsize_t kdata_size; /*kernel data size */
	int npde1, pnpde1; /* number of PTD needed for above */
	pte2_t *pte2;
	vaddr_t va;
	paddr_t paend;
	int i;
	struct pmap *pm = pmap_kernel();

	virtual_avail = *vaend;
	paend = *paendp;

	/* allocate proc0 uarea */
	lwp0.l_addr = proc0paddr = (void *)virtual_avail;
	lwp0.l_md.md_utf = (struct trapframe *)(virtual_avail + USPACE) - 1;
	virtual_avail += USPACE;
	paend += USPACE;

	/* allocate exception vector page */
	cp0_ebase = virtual_avail;
	virtual_avail += PAGE_SIZE;
	paend += PAGE_SIZE;

	/* allocate msgbuf */
	*msgbp = virtual_avail;
	virtual_avail += round_page(MSGBUFSIZE);
	paend += round_page(MSGBUFSIZE);

	/* allocate the PTE2 mapping page table (shadow PDP) */
	pmap_kernel()->pm_pte2map = (pte2_t *)virtual_avail;
	virtual_avail += PAGE_SIZE * N_L1_PTE2SLOTS;
	for (i = 0; i < N_L1_PTE2SLOTS; i++) {
		pmap_kernel()->pm_pte2map_pa[i] = paend;
		paend += PAGE_SIZE;
	}

	/*
	 * remap kernel data using 4K pages to not waste space
	 */
	kpte2_base = pte2 = (void *)virtual_avail;
	kpte2_va = (vaddr_t)__data_start;
	kdata_size = virtual_avail - (vaddr_t)__data_start;
	/* compute number of pt2 pages needed */
	npde1 = 0;
	do {
		pnpde1 = npde1;
		/* compute how many PT and allocate them */
		kdata_size =
		    virtual_avail - (vaddr_t)__data_start +
		    (npde1 * PAGE_SIZE) + NPTE2_EXTRA * PTE2_NBPTE;
		npde1 = (kdata_size + PDE1_NBPTD - 1) / PDE1_NBPTD;
	} while (npde1 != pnpde1);
	nkptp = npde1;
	virtual_avail += npde1 * PAGE_SIZE;
	paend += npde1 * PAGE_SIZE;
	DPRINTF(PMD_PMAP_INIT,
	    ("virtual_avail 0x%x npdt %d paend 0x%lx\n", virtual_avail,
	    npde1, (u_long)paend));

	/* steal one page of VA for pmap_zero_page() */
	pmap_va1 = virtual_avail;
	virtual_avail += PAGE_SIZE;
	pmap_va2 = virtual_avail;
	virtual_avail += PAGE_SIZE;

	/* make sure memory up to virtual_avail is mapped */
	for (va =  (vaddr_t)__data_start; va < virtual_avail; va += PTE1_NBPTE) {
		DPRINTF(PMD_PTE_SET,
		   ("%p kernelpd[%d] = 0x%x\n", &kernelpd[VADDR_TO_PDE1I(va)],
		    VADDR_TO_PDE1I(va),
		    ((va - KERNBASE) >> PTE1_SHIFT) |
		    PTE1_V | PTE1_C | PTE1_W | PTE1_G | PTE1_w));
		kernelpd[VADDR_TO_PDE1I(va)] =
		    (va - KERNBASE) >> PTE1_SHIFT |
		    PTE1_V | PTE1_C | PTE1_W | PTE1_G | PTE1_w;
	}
	/* zero out new memory */
	memset((void *)mips_round_page(*vaend), 0,
	    virtual_avail - mips_round_page(*vaend));
	/* now build real PTE2 entries */
	va =  (vaddr_t)__data_start;
	for (; npde1 > 0; npde1--) {
		vaddr_t vabase = va; /* va mapped by this pt2 page */
		/* fill in pte2 */
		for(i = 0;
		    i < PAGE_SIZE / sizeof(pte2_t);
		    i++, va += PTE2_NBPTE) {
			if (va == pmap_va1) {
				/* do not map this va, but remember pte */
				pmap_pte1 = &pte2[i];
				pte2[i].pte2_flags = pte2[i].pte2_ppn = 0;
			} else if (va == pmap_va2) {
				/* do not map this va, but remember pte */
				pmap_pte2 = &pte2[i];
				pte2[i].pte2_flags = pte2[i].pte2_ppn = 0;
			} else if (va < virtual_avail) {
				pte2[i].pte2_flags =
				    PTE2_V | PTE2_C | PTE2_W | PTE2_G | PTE2_w;
				if (va == cp0_ebase)
					pte2[i].pte2_flags |= PTE2_X;
				pte2[i].pte2_ppn =
				    (va - KERNBASE) >> PAGE_SHIFT;
			} else {
				/* unmapped pte */
				pte2[i].pte2_flags = pte2[i].pte2_ppn = 0;
			}
			if (pte2[i].pte2_flags)
				DPRINTF(PMD_PTE_SET,
				    ("pte2 %p 0x%x 0x%x\n", &pte2[i],
				    pte2[i].pte2_flags, pte2[i].pte2_ppn));
		}
		/* make pde1 point to pte2 */
		DPRINTF(PMD_PTE_SET,
		    ("pde1 %p 0x%x\n", &kernelpd[VADDR_TO_PDE1I(vabase)],
		    (((vaddr_t)pte2 - KERNBASE) >> PAGE_SHIFT) |
		    PDE1_V | PDE1_T));
		kernelpd[VADDR_TO_PDE1I(vabase)] =
		    (((vaddr_t)pte2 - KERNBASE) >> PAGE_SHIFT) |
		    PDE1_V | PDE1_T;
		/* also fill in shadow PD */
		pmap_kernel()->pm_pte2map[VADDR_TO_PDE1I(vabase)].pte2_ppn =
		    (((vaddr_t)pte2 - KERNBASE) >> PAGE_SHIFT);
		pmap_kernel()->pm_pte2map[VADDR_TO_PDE1I(vabase)].pte2_flags =
		    PTE2_V | PTE2_C | PTE2_W | PTE2_G | PTE2_w;
		DPRINTF(PMD_PTE_SET, ("pte2map @0x%p 0x%x 0x%x\n",
		    &pmap_kernel()->pm_pte2map[VADDR_TO_PDE1I(vabase)],
		    pmap_kernel()->pm_pte2map[VADDR_TO_PDE1I(vabase)].pte2_ppn,
		    pmap_kernel()->pm_pte2map[VADDR_TO_PDE1I(vabase)].pte2_flags
		));
		/* flush tlb to start using new PTE2 */
		membar_producer();
		pmap_tlb_inval(vabase);
		/* point to next pte2 */
		pte2 = (void *)((vaddr_t)pte2 + PAGE_SIZE);
	}
	/*
	 * make L1_PTE2_SLOT point to shadow PD, this effectively maps
	 * our page table pages in virtual space
	 */
	for (i = L1_PTE2_KERNSLOT; i < N_L1_PTE2SLOTS; i++) {
		kernelpd[L1_PTE2_SLOT + i] =
		    (pm->pm_pte2map_pa[i] >> PAGE_SHIFT) | PDE1_V | PDE1_T;
		DPRINTF(PMD_PTE_SET, ("map PTP: ptd @0x%p: 0x%x\n",
		    &kernelpd[L1_PTE2_SLOT + i], kernelpd[L1_PTE2_SLOT + i]));
	}
	membar_producer();
	memset(lwp0.l_addr, 0, USPACE);
	lwp0.l_addr->u_pcb.pcb_context.val[_L_SR] = MIPS_SR_INT_IE;
	pm->pm_pd = kernelpd;
	pm->pm_pdpa = (vaddr_t)kernelpd - KERNBASE;
	pm->pm_mapped = pm;
	memset(&pm->pm_list, 0, sizeof(pm->pm_list));  /* pm_list not used */
	pm->pm_stats.wired_count = pm->pm_stats.resident_count =
	    (virtual_avail - (vaddr_t)kernel_text) >> PAGE_SHIFT;

	/* init global lists and locks */
	mutex_init(&pmaps_lock, MUTEX_DEFAULT, IPL_NONE);
	mutex_init(&pmaps_copy_lock, MUTEX_DEFAULT, IPL_NONE);
	LIST_INIT(&pmaps);
	UVM_OBJ_INIT(&pmap_kernel()->pm_obj, NULL, 1);
	/* XXX init pools */
	pool_cache_bootstrap(&pmap_cache, sizeof(struct pmap), 0, 0, 0,
	    "pmappl", NULL, IPL_NONE, pmap_ctor, pmap_dtor, NULL);
	pool_cache_bootstrap(&pmap_pv_cache, sizeof(struct pv_entry), 0, 0,
	    PR_LARGECACHE, "pvpl", &pool_allocator_meta, IPL_NONE, NULL,
	    NULL, NULL);

	cpu_info_store.ci_pmap = pmap_kernel();
	LIST_INIT(&cpu_info_store.ci_freepmaps);

	/*
	 * get pmap_maxkvaddr from nkptp
	 * As we started aligned on a 2Mb boundary
	 * there's no space lost in the first ptp
	 */
	pmap_maxkvaddr = nkptp * PDE1_NBPTD + (vaddr_t)__data_start;
	virtual_end = VM_MAX_KERNEL_ADDRESS;
	/* steal some VA to map the FDT */
	*fdt_vap = virtual_avail;
	for (; fdt_size > 0;
	    virtual_avail += PAGE_SIZE, fdt_pa += PAGE_SIZE,
	    fdt_size -= PAGE_SIZE) {
		pmap_kenter_pa(virtual_avail, fdt_pa, VM_PROT_READ);
	}

	*vaend = virtual_avail;
	*paendp = paend;
}

/* finish pmap initialisation */
void
pmap_init(void)
{
	/* init hash and locks XXXX */
	printf(__func__);
	printf("\n");
	pmap_initialized = true;
}

/* pmap cache functions */
int
pmap_ctor(void *arg, void *v, int flags)
{
	struct pmap *pmap = v;
	struct vm_page *pg;
	vaddr_t pdva, shpdva;
	paddr_t shpdpa;
	int error;
	int i;
	uvm_flag_t kmflags = (flags & PR_WAITOK) ? UVM_KMF_CANFAIL :
	    (UVM_KMF_NOWAIT|UVM_KMF_TRYLOCK);

	DPRINTF(PMD_PMAP_ALLOC, ("pmap_ctor %p\n", pmap));
	UVM_OBJ_INIT(&pmap->pm_obj, NULL, 1);
	pmap->pm_stats.wired_count = 0;
	pmap->pm_stats.resident_count = 1;

	/*
	 * allocate the PDP. We need 2 physically-contigous pages, so
	 * we have to get them via uvm_pglistalloc() and map them ourselve
	 */
again:
	error = uvm_pglistalloc(PT1_SIZE, 0, mips_avail_end, PT1_SIZE,
	    0, &pmap->pm_pdlist, 1, (flags & PR_WAITOK) ? 1 : 0);
	if (error)
		goto err;
	pdva = uvm_km_alloc(kernel_map, PT1_SIZE, 0, 
	    UVM_KMF_VAONLY | kmflags);
	if (pdva == 0) {
		error = ENOMEM;
		goto err1;
	}
	pmap->pm_pd = (void *)pdva;
	pmap->pm_pdpa = VM_PAGE_TO_PHYS(TAILQ_FIRST(&pmap->pm_pdlist));
	DPRINTF(PMD_PMAP_ALLOC,
	    ("pmap %p pd %p pa 0x%llx\n", pmap, pmap->pm_pd, pmap->pm_pdpa));
	TAILQ_FOREACH(pg, &pmap->pm_pdlist, pageq.queue) {
		DPRINTF(PMD_PMAP_ALLOC,
		    (" map 0x%llx at va %x\n", VM_PAGE_TO_PHYS(pg), pdva));
		error = pmap_enter(pmap_kernel(), pdva, VM_PAGE_TO_PHYS(pg),
		    VM_PROT_READ | VM_PROT_WRITE,
		    PMAP_WIRED | VM_PROT_READ | VM_PROT_WRITE);
		pdva += PAGE_SIZE;
		if (error)
			goto err2;
	}

	/* allocate shadow PDP */
	shpdva = uvm_km_alloc(kernel_map,
	    L1_PTE2_KERNSLOT * PAGE_SIZE, 0,
	    kmflags | UVM_KMF_ZERO | UVM_KMF_WIRED);
	if (shpdva == 0) {
		error = ENOMEM;
		goto err2;
	}
	pmap->pm_pte2map = (void *)shpdva;
	DPRINTF(PMD_PMAP_ALLOC,
	    ("pmap %p pm_pte2map %p pa", pmap, pmap->pm_pte2map));
	for (i = 0; i < L1_PTE2_KERNSLOT; i++) {
		if (!pmap_extract(pmap_kernel(),
		    shpdva + PAGE_SIZE * i, &shpdpa))
			panic("pmap_create: new shadow PD not mapped");
		DPRINTF(PMD_PMAP_ALLOC, (" 0x%llx", shpdpa));
		pmap->pm_pte2map_pa[i] = shpdpa;
	}
	DPRINTF(PMD_PMAP_ALLOC, ("\n"));

	/* zero init PDP area */
	memset(pmap->pm_pd, 0,
	    VADDR_TO_PDE1I(VM_MIN_KERNEL_ADDRESS) * sizeof(pte1_t));
	/* shadow PDP already zeroed */
	pmap->pm_mapped = pmap_kernel();
	/* copy kernel's PTEs */
	memcpy(&pmap->pm_pd[VADDR_TO_PDE1I(VM_MIN_KERNEL_ADDRESS)],
	    &pmap_kernel()->pm_pd[VADDR_TO_PDE1I(VM_MIN_KERNEL_ADDRESS)],
	    (L1_PTE2_SLOT - VADDR_TO_PDE1I(VM_MIN_KERNEL_ADDRESS)) *
	    sizeof(pte1_t));
	/* map kernel shadow PD */
	for (i = L1_PTE2_KERNSLOT; i < N_L1_PTE2SLOTS; i++) {
		pmap->pm_pd[i + L1_PTE2_SLOT] =
		    pmap_kernel()->pm_pd[i + L1_PTE2_SLOT];
	}
	/* we're done */
	return(0);
err2:
	pmap_remove(pmap_kernel(), (vaddr_t)pmap->pm_pd, pdva);
	uvm_km_free(kernel_map, (vaddr_t)pmap->pm_pd, PT1_SIZE, UVM_KMF_VAONLY);
err1:
	uvm_pglistfree(&pmap->pm_pdlist);
err:
	if (flags & PR_WAITOK) {
		kpause("pmap_ctor", 0, hz / 10, NULL);
		goto again;
	}
	UVM_OBJ_DESTROY(&pmap->pm_obj);
	return error;
}

void
pmap_dtor(void *arg, void *v)
{
	struct pmap *pmap = v;
	DPRINTF(PMD_PMAP_ALLOC, ("pmap_dtor %p\n", pmap));

	KASSERT(pmap->pm_stats.wired_count == 0);
	KASSERT(pmap->pm_stats.resident_count == 1);
	KASSERT(pmap->pm_mapped == pmap_kernel());

	uvm_km_free(kernel_map, (vaddr_t)pmap->pm_pte2map,
	    L1_PTE2_KERNSLOT * PAGE_SIZE, UVM_KMF_WIRED);
	pmap_remove(pmap_kernel(), (vaddr_t)pmap->pm_pd,
	    (vaddr_t)pmap->pm_pd + PT1_SIZE);
	uvm_km_free(kernel_map, (vaddr_t)pmap->pm_pd, PT1_SIZE, UVM_KMF_VAONLY);
	uvm_pglistfree(&pmap->pm_pdlist);
	UVM_OBJ_DESTROY(&pmap->pm_obj);
}

/* map pm's PTE2 pages in the current pmap */
static pte2_t *
pmap_map_ptes(struct pmap *pm, int wait)
{
	struct pmap *cpm;
	int i;
	struct lwp *l;
	uint64_t ncsw;

	KASSERT(kpreempt_disabled());
	if (pm == pmap_kernel()) {
		/*
		 * pmap_kernel is always mapped, no need to do locking
		 * refcount is also not needed, if the mapped pmap
		 * is dropped, it will be remplaced with pmap_kernel().
		 */
		KASSERT(curcpu()->ci_pmap->pm_mapped != NULL);
		return (pte2_t *)L1_PTE2SLOT_VADDR;
	}

retry:
	l = curlwp;
	ncsw = l->l_ncsw;
	cpm = curcpu()->ci_pmap;
	pmap_reference(cpm);
	/* ordered locking, avoids deadlock */
	if ((uintptr_t)pm < (uintptr_t)cpm) {
		pmap_reference(pm);
		if (wait == PM_NOWAIT) {
			if (__predict_false(!mutex_tryenter(&pm->pm_lock))) {
				goto unref_and_return;
			}
			if (__predict_false(!mutex_tryenter(&cpm->pm_lock))) {
				mutex_exit(&pm->pm_lock);
				goto unref_and_return;
			}
		} else {
			mutex_enter(&pm->pm_lock);
			mutex_enter(&cpm->pm_lock);
		}
	} else if ((uintptr_t)cpm < (uintptr_t)pm) {
		pmap_reference(pm);
		if (wait == PM_NOWAIT) {
			if (__predict_false(!mutex_tryenter(&cpm->pm_lock))) {
				goto unref_and_return;
			}
			if (__predict_false(!mutex_tryenter(&pm->pm_lock))) {
				mutex_exit(&cpm->pm_lock);
				goto unref_and_return;
			}
		} else {
			mutex_enter(&cpm->pm_lock);
			mutex_enter(&pm->pm_lock);
		}
	} else { /* cpm == pm */
		if (wait == PM_NOWAIT) {
			if (__predict_false(!mutex_tryenter(&pm->pm_lock))) {
				goto unref_and_return;
			}
		} else {
			mutex_enter(&pm->pm_lock);
		}
	}
	if (l->l_ncsw != ncsw) {
		mutex_exit(&pm->pm_lock);
		pmap_destroy(pm);
		if (cpm != pm) {
			mutex_exit(&cpm->pm_lock);
			pmap_destroy(cpm);
		}
		goto retry;
	}

	if (cpm->pm_mapped == pm && curcpu()->ci_mappm == pm) {
		return (pte2_t *)L1_PTE2SLOT_VADDR;
	}

	if (cpm->pm_mapped != pm) {
		for (i = 0; i < L1_PTE2_KERNSLOT; i++) {
			cpm->pm_pd[i + L1_PTE2_SLOT] =
			    (pm->pm_pte2map_pa[i] >> PAGE_SHIFT) |
			    PDE1_V | PDE1_T;
			UDPRINTF(pm, PMD_PTE_SET2,
			    ("pmap_map_ptes: %p ptd @%p: 0x%x\n", pm, 
			    &cpm->pm_pd[i + L1_PTE2_SLOT],
			    cpm->pm_pd[i + L1_PTE2_SLOT]));
		}
		membar_producer();
		cpm->pm_mapped = pm;
	}
	for (i = 0 ; i < VADDR_TO_PDE1I(VM_MIN_KERNEL_ADDRESS); i++) {
		dtlbinval(L1_PTE2SLOT_VADDR + PTE2_NBPTE * i);
	}
	curcpu()->ci_mappm = pm;
	return (pte2_t *)L1_PTE2SLOT_VADDR;

unref_and_return:
	pmap_destroy(cpm);
	if (pm != cpm)
		pmap_destroy(pm);
	return NULL;
}

static void
pmap_unmap_ptes(struct pmap *pm)
{
	struct pmap *cpm = curcpu()->ci_pmap;
	KASSERT(kpreempt_disabled());
	if (pm == pmap_kernel())
		return;

	if (pm != cpm) {
		mutex_exit(&pm->pm_lock);
		pmap_destroy(pm);
	}
	mutex_exit(&cpm->pm_lock);
	pmap_destroy(cpm);
}

static inline struct vm_page *
pmap_find_ptp(struct pmap *pmap, vaddr_t va, paddr_t pa, pte2_t *ptebase)
{
	struct vm_page *pg;

	KASSERT(mutex_owned(&pmap->pm_lock));

	pg = uvm_pagelookup(&pmap->pm_obj,
	     (vaddr_t)&ptebase[VADDR_TO_PTE2BASEI(va)]);

	KASSERT(pg == NULL || pg->wire_count >= 1);
	return pg;
}

static struct vm_page *
pmap_get_ptp(struct pmap *pmap, vaddr_t va, pte2_t *ptebase)
{
	struct vm_page *ptp;
	paddr_t pa;

	KASSERT(pmap != pmap_kernel());
	KASSERT(mutex_owned(&pmap->pm_lock));
	KASSERT(kpreempt_disabled());

	DPRINTF(PMD_PTE_SET2,
	    ("pmap_get_ptp(%p, 0x%x, %p)", pmap, va, ptebase));

	ptp = NULL;
	if (pde1_valid_entry(&pmap->pm_pd[VADDR_TO_PDE1I(va)])) {
		pa = PTD1_TO_PTE2PADDR(pmap->pm_pd[VADDR_TO_PDE1I(va)]);
		KASSERT(pa != 0);
		ptp = pmap_find_ptp(pmap, va, pa, ptebase);
		DPRINTF(PMD_PTE_SET2,
		    (" got it pa 0x%llx return %p\n", pa, ptp));
		KASSERT(ptp != NULL);
		return ptp;
	}
	KASSERT(PTD1_TO_PTE2PADDR(pmap->pm_pd[VADDR_TO_PDE1I(va)]) == 0);
	ptp = uvm_pagealloc(&pmap->pm_obj,
	    (vaddr_t)&ptebase[VADDR_TO_PTE2BASEI(va)], NULL,
	    UVM_PGA_USERESERVE|UVM_PGA_ZERO);
	DPRINTF(PMD_PTE_SET2, (" alloc %p", ptp));
	if (ptp == NULL)
		return NULL;
	ptp->flags &= ~PG_BUSY; /* never busy */
	ptp->wire_count = 1;
	pa = VM_PAGE_TO_PHYS(ptp);
	DPRINTF(PMD_PTE_SET2, (" pa 0x%llx", pa));
	pmap->pm_pd[VADDR_TO_PDE1I(va)] =
	    (VM_PAGE_TO_PHYS(ptp) >>  PAGE_SHIFT) | PDE1_V | PDE1_T;
	pmap->pm_pte2map[VADDR_TO_PDE1I(va)].pte2_ppn =
	    (VM_PAGE_TO_PHYS(ptp) >>  PAGE_SHIFT);
	pmap->pm_pte2map[VADDR_TO_PDE1I(va)].pte2_flags =
	    PTE2_V | PTE2_C | PTE2_W | PTE2_w;
	DPRINTF(PMD_PTE_SET2, (" pde @%p 0x%x",
	    &pmap->pm_pd[VADDR_TO_PDE1I(va)], pmap->pm_pd[VADDR_TO_PDE1I(va)]));
	DPRINTF(PMD_PTE_SET2, (" spde @%p 0x%x/0x%x\n",
	    &pmap->pm_pte2map[VADDR_TO_PDE1I(va)],
	    pmap->pm_pte2map[VADDR_TO_PDE1I(va)].pte2_flags,
	    pmap->pm_pte2map[VADDR_TO_PDE1I(va)].pte2_ppn));
	membar_producer();
#if 0
	pmap_tlb_inval_range(va & PDE1_MASK, (va & PDE1_MASK) + PTE1_NBPTE);
#endif
	return ptp;
}

static inline void
pmap_freepage(struct pmap *pmap, struct vm_page *ptp)
{
	struct uvm_object *obj;
	struct pmap_page *pp;

	KASSERT(ptp->wire_count == 1);

	obj = &pmap->pm_obj;
	pmap_stats_update(pmap, -1, 0);
	ptp->wire_count = 0;
	uvm_pagerealloc(ptp, NULL, 0);
	pp = VM_PAGE_TO_PP(ptp);
	LIST_INIT(&pp->pp_head.pvh_list);
	uvm_pagefree(ptp);
#if 0
	VM_PAGE_TO_PP(ptp)->pp_link = curlwp->l_md.md_gc_ptp;
	curlwp->l_md.md_gc_ptp = ptp;
#endif
}

static void
pmap_free_ptp(struct pmap *pmap, struct vm_page *ptp, vaddr_t va)
{
	//vaddr_t invaladdr;
	pde1_t opde;

	KASSERT(pmap != pmap_kernel());
	KASSERT(mutex_owned(&pmap->pm_lock));
	KASSERT(kpreempt_disabled());

	opde = atomic_swap_32(&pmap->pm_pd[VADDR_TO_PDE1I(va)], 0);
	pmap->pm_pte2map[VADDR_TO_PDE1I(va)].pte2_flags = 0;
	pmap->pm_pte2map[VADDR_TO_PDE1I(va)].pte2_ppn = 0;
	membar_producer();
	pmap_freepage(pmap, ptp);
#if 0
	invaladdr = L1_PTE2SLOT_VADDR +
	    VADDR_TO_PTE2SLOTI(va) * sizeof(pte2_t);
	pmap_tlb_inval(invaladdr);
#endif
#if 0
	/* XXX not needed ? */
	pmap_tlb_inval_range(va & PDE1_MASK, (va & PDE1_MASK) + PTE1_NBPTE);
#endif
}

/*
 * ptp_to_pmap: lookup pmap by ptp
 */

static struct pmap *
ptp_to_pmap(struct vm_page *ptp)
{
	struct pmap *pmap;

	if (ptp == NULL) {
		return pmap_kernel();
	}
	pmap = (struct pmap *)ptp->uobject;
	KASSERT(pmap != NULL);
	KASSERT(&pmap->pm_obj == ptp->uobject);
	return pmap;
}


static inline struct pv_pte *
pve_to_pvpte(struct pv_entry *pve)
{

	KASSERT((void *)&pve->pve_pte == (void *)pve);
	return &pve->pve_pte;
}

static inline struct pv_entry *
pvpte_to_pve(struct pv_pte *pvpte)
{
	struct pv_entry *pve = (void *)pvpte;

	KASSERT(pve_to_pvpte(pve) == pvpte);
	return pve;
}

/*
 * pmap_free_pvs: free a list of pv_entrys
 */

static void
pmap_free_pvs(struct pv_entry *pve)
{
	struct pv_entry *next;

	for ( /* null */ ; pve != NULL ; pve = next) {
		next = pve->pve_next;
		pool_cache_put(&pmap_pv_cache, pve);
	}
}


/*
 * main pv_entry manipulation functions:
 *   pmap_enter_pv: enter a mapping onto a pv_head list
 *   pmap_remove_pv: remove a mappiing from a pv_head list
 *
 * NOTE: Both pmap_enter_pv and pmap_remove_pv expect the caller to lock 
 *       the pvh before calling
 */

/*
 * insert_pv: a helper of pmap_enter_pv
 */

static void
insert_pv(struct pmap_page *pp, struct pv_entry *pve)
{
	struct pv_hash_head *hh;
	kmutex_t *lock;
	u_int hash;

	KASSERT(pp_locked(pp));

	hash = pvhash_hash(pve->pve_pte.pte_ptp, pve->pve_pte.pte_va);
	lock = pvhash_lock(hash);
	hh = pvhash_head(hash);
	mutex_spin_enter(lock);
	SLIST_INSERT_HEAD(&hh->hh_list, pve, pve_hash);
	mutex_spin_exit(lock);

	LIST_INSERT_HEAD(&pp->pp_head.pvh_list, pve, pve_list);
}

/*
 * pmap_enter_pv: enter a mapping onto a pv_head lst
 *
 * => caller should have the pp_lock locked
 * => caller should adjust ptp's wire_count before calling
 */

static struct pv_entry *
pmap_enter_pv(struct pmap_page *pp,
	      struct pv_entry *pve,	/* preallocated pve for us to use */
	      struct pv_entry **sparepve,
	      struct vm_page *ptp,
	      vaddr_t va)
{

	KASSERT(ptp == NULL || ptp->wire_count >= 2);
	KASSERT(ptp == NULL || ptp->uobject != NULL);
	KASSERT(ptp == NULL ||
	    L1_PTE2SLOT_VADDR + VADDR_TO_PTE2BASEI(va) * sizeof(pte2_t) ==
	    ptp->offset);
	KASSERT(pp_locked(pp));

	if ((pp->pp_flags & PP_EMBEDDED) == 0) {
		if (LIST_EMPTY(&pp->pp_head.pvh_list)) {
			pp->pp_flags |= PP_EMBEDDED;
			pp->pp_pte.pte_ptp = ptp;
			pp->pp_pte.pte_va = va;

			return pve;
		}
	} else {
		struct pv_entry *pve2;

		pve2 = *sparepve;
		*sparepve = NULL;

		pve2->pve_pte = pp->pp_pte;
		pp->pp_flags &= ~PP_EMBEDDED;
		LIST_INIT(&pp->pp_head.pvh_list);
		insert_pv(pp, pve2);
	}

	pve->pve_pte.pte_ptp = ptp;
	pve->pve_pte.pte_va = va;
	insert_pv(pp, pve);

	return NULL;
}

/*
 * pmap_remove_pv: try to remove a mapping from a pv_list
 *
 * => caller should hold pp_lock [so that attrs can be adjusted]
 * => caller should adjust ptp's wire_count and free PTP if needed
 * => we return the removed pve
 */

static struct pv_entry *
pmap_remove_pv(struct pmap_page *pp, struct vm_page *ptp, vaddr_t va)
{
	struct pv_hash_head *hh;
	struct pv_entry *pve;
	kmutex_t *lock;
	u_int hash;

	KASSERT(ptp == NULL || ptp->uobject != NULL);
	KASSERT(ptp == NULL ||
	    L1_PTE2SLOT_VADDR + VADDR_TO_PTE2BASEI(va) * sizeof(pte2_t) == ptp->offset);
	KASSERT(pp_locked(pp));

	if ((pp->pp_flags & PP_EMBEDDED) != 0) {
		if (pp->pp_pte.pte_ptp != ptp)
			printf("PP_EMBEDDED ptp %p != %p\n", pp->pp_pte.pte_ptp, ptp);
		if (pp->pp_pte.pte_va != va)
			printf("PP_EMBEDDED va 0x%x != 0x%x\n", pp->pp_pte.pte_va, va);
		KASSERT(pp->pp_pte.pte_ptp == ptp);
		KASSERT(pp->pp_pte.pte_va == va);

		pp->pp_flags &= ~PP_EMBEDDED;
		LIST_INIT(&pp->pp_head.pvh_list);

		return NULL;
	}

	hash = pvhash_hash(ptp, va);
	lock = pvhash_lock(hash);
	hh = pvhash_head(hash);
	mutex_spin_enter(lock);
	pve = pvhash_remove(hh, ptp, va);
	mutex_spin_exit(lock);

	LIST_REMOVE(pve, pve_list);

	return pve;
}

/*
 * pmap_sync_pv: clear pte bits and return the old value of the pte.
 *
 * => called with pp_lock held. (thus preemption disabled)
 */

static int
pmap_sync_pv(struct pv_pte *pvpte, pte2_t expect, int clearbits, pte2_t *optep)
{
	struct pmap *pmap;
	struct vm_page *ptp;
	vaddr_t va;
	pte2_t *ptep, *ptebase;
	pte2_t opte;
	pte2_t npte;
	bool need_shootdown;

	ptp = pvpte->pte_ptp;
	va = pvpte->pte_va;
	KASSERT(ptp == NULL || ptp->uobject != NULL);
	KASSERT(ptp == NULL ||
	    L1_PTE2SLOT_VADDR + VADDR_TO_PTE2BASEI(va) * sizeof(pte2_t) ==
	    ptp->offset);
	pmap = ptp_to_pmap(ptp);

	KASSERT((expect.pte2_flags & ~(PTE2_V)) == 0);
	KASSERT((expect.pte2_flags & PTE2_V) != 0);
	KASSERT(clearbits == ~0 ||
	    (clearbits & ~(PTE2_D | PTE2_RL | PTE2_W)) == 0);
	KASSERT(kpreempt_disabled());

	ptebase = pmap_map_ptes(pmap, PM_NOWAIT); /* lock pmaps */
	if (ptebase == NULL)
		return EAGAIN;
	ptep = &ptebase[VADDR_TO_PTE2SLOTI(va)];

	do {
		opte.pte2_val = ptep->pte2_val;
		KASSERT((opte.pte2_flags & (PTE2_D | PTE2_RL))
		    != PTE2_D);
		KASSERT((opte.pte2_flags & PTE2_RL) == 0 ||
		    pte2_valid_entry(&opte));
		KASSERT(opte.pte2_val == 0 || pte2_valid_entry(&opte));
		if ((opte.pte2_flags & PTE2_V)  != (expect.pte2_flags & PTE2_V)
		    || opte.pte2_ppn != expect.pte2_ppn) {
			/*
			 * we lost a race with a V->P operation like
			 * pmap_remove().  wait for the competitor
			 * reflecting pte bits into mp_attrs.
			 *
			 * issue a redundant TLB shootdown so that
			 * we can wait for its completion.
			 */

			pmap_unmap_ptes(pmap);
			return EAGAIN;
		}

		/*
		 * check if there's anything to do on this pte.
		 */

		if ((opte.pte2_flags & clearbits) == 0) {
			need_shootdown = false;
			break;
		}

		/*
		 * we need a shootdown if the pte is cached. (PTE2_RL)
		 *
		 */
		need_shootdown = ((opte.pte2_flags & PTE2_RL) != 0);

		npte.pte2_flags = opte.pte2_flags & ~clearbits;

		/*
		 * if we need a shootdown anyway, clear cached and dirty
		 */

		if (need_shootdown) {
			npte.pte2_flags &= ~(PTE2_RL | PTE2_D);
		}
		KASSERT((npte.pte2_flags & (PTE2_D | PTE2_RL)) != PTE2_D);
		KASSERT((npte.pte2_flags & PTE2_RL) == 0 ||
		    pte2_valid_entry(&npte));
		KASSERT((npte.pte2_val == 0 || pte2_valid_entry(&opte)));
	} while (atomic_cas_32(&ptep->pte2_flags, opte.pte2_flags,
	    npte.pte2_flags) != opte.pte2_flags);
	membar_producer();

	if (need_shootdown) {
		pmap_tlb_inval(va);
	}
	pmap_unmap_ptes(pmap);

	optep->pte2_val = opte.pte2_val;
	return 0;
}

/*
 * pmap_page_remove: remove a managed vm_page from all pmaps that map it
 *
 * => R/M bits are sync'd back to attrs
 */

void
pmap_page_remove(struct vm_page *pg)
{
	struct pmap_page *pp;
	struct pv_pte *pvpte;
	struct pv_entry *killlist = NULL;
	struct vm_page *ptp;
	pte2_t expect;
	lwp_t *l;
	int count;

#ifdef DIAGNOSTIC
	int bank, off;

	bank = vm_physseg_find(atop(VM_PAGE_TO_PHYS(pg)), &off);
	if (bank == -1)
		panic("pmap_page_remove: unmanaged page?");
#endif

	l = curlwp;
	pp = VM_PAGE_TO_PP(pg);
	expect.pte2_ppn = VM_PAGE_TO_PHYS(pg) >> PAGE_SHIFT;
	expect.pte2_flags = PTE2_V;
	count = SPINLOCK_BACKOFF_MIN;
	kpreempt_disable();
startover:
	pp_lock(pp);
	while ((pvpte = pv_pte_first(pp)) != NULL) {
		struct pmap *pmap;
		struct pv_entry *pve;
		pte2_t opte;
		vaddr_t va;
		int error;

		/*
		 * add a reference to the pmap before clearing the pte.
		 * otherwise the pmap can disappear behind us.
		 */

		ptp = pvpte->pte_ptp;
		pmap = ptp_to_pmap(ptp);
		if (ptp != NULL) {
			pmap_reference(pmap);
		}

		error = pmap_sync_pv(pvpte, expect, ~0, &opte);
		if (__predict_false(error == EAGAIN)) {
			int hold_count;
			pp_unlock(pp);
			KERNEL_UNLOCK_ALL(curlwp, &hold_count);
			if (ptp != NULL) {
				pmap_destroy(pmap);
			}
			SPINLOCK_BACKOFF(count);
			KERNEL_LOCK(hold_count, curlwp);
			goto startover;
		}

		pp->pp_attrs |= PTE2PATTR(opte.pte2_flags);
		va = pvpte->pte_va;
		pve = pmap_remove_pv(pp, ptp, va);
		pp_unlock(pp);

		/* update the PTP reference count.  free if last reference. */
		if (ptp != NULL) {
			pte2_t *ptebase;

			KASSERT(pmap != pmap_kernel());

			ptebase = pmap_map_ptes(pmap, PM_WAITOK);/*lock pmaps*/
			pmap_stats_update_byflags(pmap, 0, opte.pte2_flags);
			ptp->wire_count--;
			if (ptp->wire_count <= 1) {
				pmap_free_ptp(pmap, ptp, va);
			}
			pmap_unmap_ptes(pmap);
			pmap_destroy(pmap);
		} else {
			KASSERT(pmap == pmap_kernel());
			pmap_stats_update_byflags(pmap, 0, opte.pte2_flags);
		}

		if (pve != NULL) {
			pve->pve_next = killlist;	/* mark it for death */
			killlist = pve;
		}
		pp_lock(pp);
	}
	pp_unlock(pp);
	kpreempt_enable();

	/* Now free unused pvs. */
	pmap_free_pvs(killlist);
}

/*
 * pv_pte_first, pv_pte_next: PV list iterator.
 */

static struct pv_pte *
pv_pte_first(struct pmap_page *pp)
{

	KASSERT(pp_locked(pp));
	if ((pp->pp_flags & PP_EMBEDDED) != 0) {
		return &pp->pp_pte;
	}
	return pve_to_pvpte(LIST_FIRST(&pp->pp_head.pvh_list));
}

static struct pv_pte *
pv_pte_next(struct pmap_page *pp, struct pv_pte *pvpte)
{

	KASSERT(pvpte != NULL);
	KASSERT(pp_locked(pp));
	if (pvpte == &pp->pp_pte) {
		KASSERT((pp->pp_flags & PP_EMBEDDED) != 0);
		return NULL;
	}
	KASSERT((pp->pp_flags & PP_EMBEDDED) == 0);
	return pve_to_pvpte(LIST_NEXT(pvpte_to_pve(pvpte), pve_list));
}

/*
 * pmap_activate: activate a process' pmap
 *
 * => must be called with kernel preemption disabled
 */

void
pmap_activate(struct lwp *l)
{
	struct pmap *pmap = vm_map_pmap(&l->l_proc->p_vmspace->vm_map);
	struct pmap *oldpmap;
	struct cpu_info *ci = curcpu();
	KASSERT(kpreempt_disabled());
	KASSERT(curlwp == ci->ci_curlwp);
	if (l == curlwp && pmap != ci->ci_pmap) {
		if (pmap == pmap_kernel()) {
			/* no need to load it, it's present in all pmaps */
			return;
		}
		/* grab a reference to the new pmap */
		pmap_reference(pmap);
		DPRINTF(PMD_PMAP_SWITCH,
		    ("pmap_activate(%p) for lwp %d.%d pdpa 0x%llx\n", pmap,
		    l->l_proc->p_pid, l->l_lid, pmap->pm_pdpa >> 4));
		tsar_vcache_write(VC_PTPR, pmap->pm_pdpa >> VC_PTPR_PASHIFT);

		oldpmap = ci->ci_pmap;
		ci->ci_pmap = pmap;
		/* now running with new pmap, drop reference to old pmap */
		if (oldpmap != pmap_kernel())
			pmap_destroy(oldpmap);
	} else if (l != curlwp)
		printf("pmap_activate l %p pmap %p curlwp %p curpmap %p 0x%x\n",
		    l, pmap, curlwp, ci->ci_pmap, tsar_vcache_read(VC_PTPR));
}

void
pmap_deactivate(struct lwp *l)
{
	KASSERT(kpreempt_disabled());
	/* noting to do for now */
}

void
pmap_unwire(struct pmap *pmap, vaddr_t va)
{
	pte2_t *ptebase;
	UDPRINTF(pmap, PMD_PTE_SET2, ("pmap_unwire(%p, 0x%x)\n", pmap, va));
	kpreempt_disable();
	if (pmap_pde_valid(pmap, va)) {
		ptebase = pmap_map_ptes(pmap, PM_WAITOK); /* lock pmap */
		pte2_t *ptep = &ptebase[VADDR_TO_PTE2SLOTI(va)];
		uint32_t oflags, nflags;
		oflags = ptep->pte2_flags;
#ifdef DIAGNOSTIC
		if ((oflags &  PTE2_V) == 0)
			panic("pmap_unwire: invalid (unmapped) va 0x%x", va);
#endif
		if ((oflags & PTE2_w) != 0) {
			nflags = oflags & ~PTE2_w;
			oflags = atomic_swap_32(&ptep->pte2_flags, nflags);
			membar_producer();
			pmap_stats_update_byflags(pmap, nflags, oflags);
		}
#ifdef DIAGNOSTIC
		else {
			printf("pmap_unwire: wiring for pmap %p va 0x%x "
			       "didn't change!\n", pmap, va);
		}
#endif
		pmap_unmap_ptes(pmap);
	}
#ifdef DIAGNOSTIC
        else {
		panic("pmap_unwire: invalid PDE");
	}
#endif
	kpreempt_enable();
}

/*
 * p m a p   a t t r i b u t e  f u n c t i o n s
 * functions that test/change managed page's attributes
 * since a page can be mapped multiple times we must check each PTE that
 * maps it by going down the pv lists.
 */

/*
 * pmap_test_attrs: test a page's attributes
 */

bool
pmap_test_attrs(struct vm_page *pg, unsigned testbits)
{
	struct pmap_page *pp;
	struct pv_pte *pvpte;
	pte2_t expect;
	u_int result;
	int count;

	DPRINTF(PMD_PTE_SET, ("pmap_test_attrs(%p, 0x%x)\n", pg, testbits));
#if DIAGNOSTIC
	{
	int bank, off;

	bank = vm_physseg_find(atop(VM_PAGE_TO_PHYS(pg)), &off);
	if (bank == -1)
		panic("pmap_test_attrs: unmanaged page?");
	}
#endif

	pp = VM_PAGE_TO_PP(pg);
	if ((pp->pp_attrs & PTE2PATTR(testbits)) != 0) {
		return true;
	}
	expect.pte2_ppn = VM_PAGE_TO_PHYS(pg) >> PAGE_SHIFT;
	expect.pte2_flags = PTE2_V;
	count = SPINLOCK_BACKOFF_MIN;
startover:
	pp_lock(pp);
	for (pvpte = pv_pte_first(pp); pvpte; pvpte = pv_pte_next(pp, pvpte)) {
		pte2_t opte;
		int error;

		if ((pp->pp_attrs & PTE2PATTR(testbits)) != 0) {
			break;
		}
		error = pmap_sync_pv(pvpte, expect, 0, &opte);
		if (__predict_false(error == EAGAIN)) {
			int hold_count;
			pp_unlock(pp);
			KERNEL_UNLOCK_ALL(curlwp, &hold_count);
			SPINLOCK_BACKOFF(count);
			KERNEL_LOCK(hold_count, curlwp);
			goto startover;
		}
		KASSERT(error == 0);
		pp->pp_attrs |= PTE2PATTR(opte.pte2_flags);
	}
	result = pp->pp_attrs & PTE2PATTR(testbits);
	pp_unlock(pp);

	/*
	 * note that we will exit the for loop with a non-null pve if
	 * we have found the bits we are testing for.
	 */

	return result != 0;
}
/*
 * pmap_clear_attrs: clear the specified attribute for a page.
 *
 * => we return true if we cleared one of the bits we were asked to
 */

bool
pmap_clear_attrs(struct vm_page *pg, unsigned clearbits)
{
	struct pmap_page *pp;
	struct pv_pte *pvpte;
	u_int result;
	pte2_t expect;
	int count;

	DPRINTF(PMD_PTE_SET, ("pmap_clear_attrs(%p, 0x%x)\n", pg, clearbits));
#ifdef DIAGNOSTIC
	{
	int bank, off;

	bank = vm_physseg_find(atop(VM_PAGE_TO_PHYS(pg)), &off);
	if (bank == -1)
		panic("pmap_change_attrs: unmanaged page?");
	}
#endif

	pp = VM_PAGE_TO_PP(pg);
	expect.pte2_ppn = VM_PAGE_TO_PHYS(pg) >> PAGE_SHIFT;
	expect.pte2_flags = PTE2_V;
	count = SPINLOCK_BACKOFF_MIN;
	kpreempt_disable();
startover:
	pp_lock(pp);
	for (pvpte = pv_pte_first(pp); pvpte; pvpte = pv_pte_next(pp, pvpte)) {
		pte2_t opte;
		int error;

		error = pmap_sync_pv(pvpte, expect, clearbits, &opte);
		if (__predict_false(error == EAGAIN)) {
			int hold_count;
			pp_unlock(pp);
			KERNEL_UNLOCK_ALL(curlwp, &hold_count);
			SPINLOCK_BACKOFF(count);
			KERNEL_LOCK(hold_count, curlwp);
			goto startover;
		}
		KASSERT(error == 0);
		pp->pp_attrs |= PTE2PATTR(opte.pte2_flags);
	}
	result = pp->pp_attrs & PTE2PATTR(clearbits);
	pp->pp_attrs &= ~PTE2PATTR(clearbits);
	pp_unlock(pp);
	kpreempt_enable();

	return result != 0;
}


void
pmap_collect(struct pmap *pmap)
{
	panic(__func__);
}

/* Create a physical map and return it to the caller */
struct pmap *
pmap_create(void)
{
	struct pmap *pmap;
	paddr_t pa, shpa;
	int i;

try_again:
	pmap = pool_cache_get(&pmap_cache, PR_WAITOK);
	DPRINTF(PMD_PMAP_ALLOC, ("pmap_create %p %d %d %ld %ld\n", pmap,
	    pmap->pm_obj.uo_npages, pmap->pm_obj.uo_refs, pmap->pm_stats.wired_count, pmap->pm_stats.resident_count));
	KASSERT(pmap->pm_obj.uo_npages == 0);
	KASSERT(pmap->pm_obj.uo_refs == 1);
	KASSERT(pmap->pm_stats.wired_count == 0);
	KASSERT(pmap->pm_stats.resident_count == 1);

	kpreempt_disable();
	mutex_enter(&pmaps_lock);
	/* make sure we did't race with pmap_growkernel() */
	if (__predict_false(
	    pmap->pm_pd[VADDR_TO_PDE1I(pmap_maxkvaddr) - 1] == 0)) {
		mutex_exit(&pmaps_lock);
		kpreempt_enable();
		pool_cache_destruct_object(&pmap_cache, pmap);
		goto try_again;
	}
	LIST_INSERT_HEAD(&pmaps, pmap, pm_list);
	mutex_exit(&pmaps_lock);
#ifdef DIAGNOSTIC
	if (!pmap_extract(pmap_kernel(), (vaddr_t)pmap->pm_pd + PAGE_SIZE,
	    &pa)) {
		panic("pmap_create: new PD[1] not mapped");
	}
	if (pmap->pm_pdpa + PAGE_SIZE != pa) {
		panic("pmap_create: new PD not contigous 0x%llx 0x%llx",
		    pmap->pm_pdpa + PAGE_SIZE, pa);
	}
	for (i = 0; i < L1_PTE2_KERNSLOT; i++) {
		if (!pmap_extract(pmap_kernel(),
		    (vaddr_t)&pmap->pm_pte2map[i], &shpa)) {
			panic("pmap_create: new shadow PD[%d] not mapped", i);
		}
	}
	KASSERT(pmap->pm_mapped == pmap_kernel());
#endif

	kpreempt_enable();
	DPRINTF(PMD_PMAP_ALLOC, ("pmap_create return %p\n", pmap));
	return(pmap);
}

void
pmap_destroy(struct pmap *pmap)
{
	int s;

	/* drop refcount, return if pmap still in use */
	if (atomic_dec_uint_nv((unsigned *)&pmap->pm_obj.uo_refs) > 0) {
		return;
	}
#ifdef DIAGNOSTIC
	{
	struct cpu_info *ci;
	CPU_INFO_ITERATOR cii;
	for (CPU_INFO_FOREACH(cii, ci))
		if (ci->ci_pmap == pmap)
			panic("destroying pmap being used");
	}
#endif
	/* remove from global list */
	s = splsched(); /* also prevent preemption */
	if (mutex_tryenter(&pmaps_lock)) {
		struct pmap *pm;
		pmap_free(pmap);
		/* also process deffered frees while we hold the lock */
		LIST_FOREACH(pm, &curcpu()->ci_freepmaps, pm_freelist) {
			LIST_REMOVE(pm, pm_freelist);
			pmap_free(pm);
		}
		mutex_exit(&pmaps_lock);
	} else {
		/* defer pmap free removal/free */
		LIST_INSERT_HEAD(&curcpu()->ci_freepmaps, pmap, pm_freelist);
	}
	splx(s);
}

static void
pmap_free(struct pmap *pmap)
{
	struct pmap *pm;
	struct cpu_info *ci;
	CPU_INFO_ITERATOR cii;

	KASSERT(mutex_owned(&pmaps_lock));
	KASSERT(pmap->pm_obj.uo_refs == 0);
	KASSERT(pmap->pm_obj.uo_npages == 0);
	KASSERT(TAILQ_EMPTY(&pmap->pm_obj.memq));
	LIST_REMOVE(pmap, pm_list);
	LIST_FOREACH(pm, &pmaps, pm_list) {
		if (pm->pm_mapped == pmap)
			pm->pm_mapped = pmap_kernel();
	}
	for (CPU_INFO_FOREACH(cii, ci)) {
		if (ci->ci_mappm == pmap)
			ci->ci_mappm = NULL;
	}
	memset(pmap->pm_pd, 0, PD1_SIZE);
	memset(pmap->pm_pte2map, 0, L1_PTE2_KERNSLOT * PAGE_SIZE);
	pmap->pm_mapped = pmap_kernel();
	DPRINTF(PMD_PMAP_ALLOC, ("pmap_destroy %p\n", pmap));
	pmap->pm_stats.wired_count = 0;
	pmap->pm_stats.resident_count = 1;
	pmap->pm_obj.uo_npages = 0;
	pmap->pm_obj.uo_refs = 1;
	pool_cache_put(&pmap_cache, pmap);
}

int
pmap_enter(struct pmap *pmap, vaddr_t va, paddr_t pa, vm_prot_t prot, int flags)
{
	pte2_t npte, opte, *ptebase, *ptep;
	uint32_t opte2_flags, npte2_flags;
	bool wired = (flags & PMAP_WIRED) != 0;
	struct vm_page *ptp, *pg;
	struct pmap_page *new_pp, *old_pp;
	struct pv_entry *old_pve = NULL;
	struct pv_entry *new_pve, *new_pve2;
	int error;

	UDPRINTF(pmap, PMD_PTE_SET2, (__func__));
	if (pmap == pmap_kernel()) {
		UDPRINTF(pmap, PMD_PTE_SET2, (" pmap_kernel"));
	} else {
		UDPRINTF(pmap, PMD_PTE_SET2, (" %p", pmap));
	}
	UDPRINTF(pmap, PMD_PTE_SET2,
	    (" va 0x%x pa 0x%lx prot 0x%x flags 0x%x\n", va, (u_long)pa,
	    prot, flags));
	KASSERT(pmap_initialized);
#ifdef DIAGNOSTIC
	/* sanity check: totally out of range? */
	if (va >= VM_MAX_KERNEL_ADDRESS)
		panic("pmap_enter: too big");
	/* sanity check: kernel PTPs should already have been pre-allocated */
	if (va >= VM_MIN_KERNEL_ADDRESS &&
	    !pde1_valid_entry(&pmap->pm_pd[VADDR_TO_PDE1I(va)]))
		panic("pmap_enter: missing kernel PTP for va %x!", va);
#endif /* DIAGNOSTIC */
	npte.pte2_flags = prot2pte[prot] | PTE2_V | PTE2_C;
	npte.pte2_ppn = (pa >> PAGE_SHIFT);

	if (wired)
		npte.pte2_flags |= PTE2_w;
	if (va <= VM_MAXUSER_ADDRESS)
		npte.pte2_flags |= PTE2_U;

	if (pmap == pmap_kernel()) {
		KASSERT(va >= VM_MIN_KERNEL_ADDRESS);
		npte.pte2_flags |= PTE2_G;
	}
	/* XXX use flags to pre-seed npte ?*/
	pg = PHYS_TO_VM_PAGE(pa);
	if (pg != NULL) {
		/* This is a managed page */
		npte.pte2_flags |= PTE2_pv;
		new_pp = VM_PAGE_TO_PP(pg);
	} else {
		new_pp = NULL;
	}
	/* pre-allocate pves. */
	new_pve = pool_cache_get(&pmap_pv_cache, PR_NOWAIT);
	new_pve2 = pool_cache_get(&pmap_pv_cache, PR_NOWAIT);
	if (new_pve == NULL || new_pve2 == NULL) {
		if (flags & PMAP_CANFAIL) {
			printf("pmap_enter pve allocation failed\n");
			error = ENOMEM;
			goto out2;
		}
		panic("pmap_enter: pve allocation failed");
	}
	kpreempt_disable();
	ptebase = pmap_map_ptes(pmap, PM_WAITOK); /* lock pmaps */
	if (pmap == pmap_kernel()) {
		ptp = NULL;
	} else {
		ptp = pmap_get_ptp(pmap, va, ptebase);
		if (ptp == NULL) {
			pmap_unmap_ptes(pmap);
			if (flags & PMAP_CANFAIL) {
				printf("pmap_enter get ptp failed\n");
				error = ENOMEM;
				goto out;
			}
			panic("pmap_enter: get ptp failed");
		}
	}

	/*
	 * update the pte.
	 */
	ptep = &ptebase[VADDR_TO_PTE2SLOTI(va)];
	/* first invalidate the old PTE if needed */
	do {
		do {
			opte = *ptep;
			opte2_flags = opte.pte2_flags & ~PTE2_V;
		} while (atomic_cas_32(&ptep->pte2_flags,
		    opte.pte2_flags, opte2_flags) != opte.pte2_flags);
		membar_producer();
		if (pte2_valid_entry(&opte) &&
		    (opte.pte2_flags & (PTE2_RL|PTE2_D)) != 0) {
			pmap_tlb_inval(va);
		}
		/* now install new PTE */
		npte2_flags = npte.pte2_flags;
		if ((opte.pte2_flags & npte.pte2_flags & PTE2_V) != 0 &&
		    (opte.pte2_ppn & PTE2_ADDRMASK) ==
		    (npte.pte2_ppn & PTE2_ADDRMASK)) {
			/* if keeping the same PA, inherit referenced/dirty */
			npte2_flags |=
			    opte.pte2_flags & (PTE2_RL | PTE2_D);
		} else {
			ptep->pte2_ppn = npte.pte2_ppn;
			membar_producer();
		}
	} while (atomic_cas_32(&ptep->pte2_flags, opte2_flags, npte2_flags)
	    != opte2_flags);
	membar_producer();

	UDPRINTF(pmap, PMD_PTE_SET2, ("pmap_enter ptep %p now 0x%x/0x%x\n",
	    ptep, ptep->pte2_flags, ptep->pte2_ppn));

	/* update statistics and refcounts */
	pmap_stats_update_byflags(pmap, npte.pte2_flags, opte.pte2_flags);
	if (ptp != NULL && !pte2_valid_entry(&opte))
		ptp->wire_count++;
	/* if PA didn't change, skip pv_entry update */
	if ((opte.pte2_flags & npte.pte2_flags & PTE2_V) != 0 &&
	    (opte.pte2_ppn & PTE2_ADDRMASK) ==
	    (npte.pte2_ppn & PTE2_ADDRMASK)) {
		KASSERT((opte.pte2_flags & PTE2_pv) ==
		    (npte.pte2_flags & PTE2_pv));
		goto same_pa;
	}
	/* remove pv_entry for opte if needed */
	if ((opte.pte2_flags & (PTE2_V | PTE2_pv)) == (PTE2_V | PTE2_pv)) {
		pg = PHYS_TO_VM_PAGE(PTE2_TO_PADDR(opte));
		KASSERT(pg != NULL);
		old_pp = VM_PAGE_TO_PP(pg);
		pp_lock(old_pp);
		old_pve = pmap_remove_pv(old_pp, ptp, va);
		old_pp->pp_attrs |= PTE2PATTR(opte.pte2_flags);
		pp_unlock(old_pp);
	}
	/* insert pv_entry for npte if needed */
	if (new_pp) {
		pp_lock(new_pp);
		new_pve = pmap_enter_pv(new_pp, new_pve, &new_pve2, ptp, va);
		pp_unlock(new_pp);
	}
same_pa:
	error = 0;
out:
	pmap_unmap_ptes(pmap);
	kpreempt_enable();
out2:
	if (old_pve)
		pool_cache_put(&pmap_pv_cache, old_pve);
	if (new_pve)
		pool_cache_put(&pmap_pv_cache, new_pve);
	if (new_pve2)
		pool_cache_put(&pmap_pv_cache, new_pve2);

	return error;
}

/*
 * p m a p   r e m o v e   f u n c t i o n s
 *
 * functions that remove mappings
 */

/*
 * pmap_remove_ptes: remove PTEs from a PTP
 *
 * => must have proper locking on pmap_master_lock
 * => caller must hold pmap's lock
 * => PTP must be mapped into KVA
 * => PTP should be null if pmap == pmap_kernel()
 * => must be called with kernel preemption disabled
 * => returns composite pte if at least one page should be shot down
 */

#define PMAP_REMOVE_ALL		0	/* remove all mappings */
#define PMAP_REMOVE_SKIPWIRED	1	/* skip wired mappings */

static uint32_t
pmap_remove_ptes(struct pmap *pmap, struct vm_page *ptp, pte2_t *ptebase,
		 vaddr_t startva, vaddr_t endva, int flags,
		 struct pv_entry **pv_tofree)
{
	struct pv_entry *pve;
	pte2_t *pte = &ptebase[VADDR_TO_PTE2SLOTI(startva)];
	pte2_t opte;
	uint32_t xpte_flags = 0;

	KASSERT(pmap == pmap_kernel() || mutex_owned(&pmap->pm_lock));
	KASSERT(kpreempt_disabled());

	/*
	 * note that pte points to the PTE that maps startva.   this may
	 * or may not be the first PTE in the PTP.
	 *
	 * we loop through the PTP while there are still PTEs to look at
	 * and the wire_count is greater than 1 (because we use the wire_count
	 * to keep track of the number of real PTEs in the PTP).
	 */

	for (/*null*/; startva < endva && (ptp == NULL || ptp->wire_count > 1)
			     ; pte++, startva += PAGE_SIZE) {
		struct vm_page *pg;
		struct pmap_page *pp;

		if (!pte2_valid_entry(pte))
			continue;			/* VA not mapped */
		if ((flags & PMAP_REMOVE_SKIPWIRED) &&
		    (pte->pte2_flags & PTE2_w)) {
			continue;
		}

		/* atomically save the old PTE and zap! it */
		do {
			opte.pte2_val = pte->pte2_val;
		} while (atomic_cas_32(&pte->pte2_flags, opte.pte2_flags, 0) !=
		    opte.pte2_flags);
		membar_producer();
		if (!pte2_valid_entry(&opte)) {
			continue;
		}
		if (opte.pte2_flags & (PTE2_RL|PTE2_D))
			pmap_tlb_inval(startva);

		pmap_stats_update_byflags(pmap, 0, opte.pte2_flags);
		xpte_flags |= opte.pte2_flags;

		if (ptp) {
			ptp->wire_count--;		/* dropping a PTE */
			/* Make sure that the PDE is flushed */
			if (ptp->wire_count <= 1)
				xpte_flags |= PTE2_L;
		}

		/*
		 * if we are not on a pv_head list we are done.
		 */
#if 0 /* XXX */
		if ((opte & PG_PVLIST) == 0) {
#if defined(DIAGNOSTIC)
			if (PHYS_TO_VM_PAGE(PTE2_TO_PADDR(opte)) != NULL)
				panic("pmap_remove_ptes: managed page without "
				      "PG_PVLIST for 0x%lx", startva);
#endif
			continue;
		}
#endif

		pg = PHYS_TO_VM_PAGE(PTE2_TO_PADDR(opte));
		if (pg == NULL)
			continue;
#if 0
#ifdef DIAGNOSTIC
		if (pg == NULL)
			panic("pmap_remove_ptes: unmanaged page marked "
			      "PG_PVLIST, va = 0x%x, pa = 0x%lx",
			      startva, (u_long)PTE2_TO_PADDR(opte));
#endif
#endif

		/* sync R/M bits */
		pp = VM_PAGE_TO_PP(pg);
		pp_lock(pp);
		pp->pp_attrs |= PTE2PATTR(opte.pte2_flags);
		pve = pmap_remove_pv(pp, ptp, startva);
		pp_unlock(pp);

		if (pve != NULL) {
			pve->pve_next = *pv_tofree;
			*pv_tofree = pve;
		}

		/* end of "for" loop: time for next pte */
	}

	return xpte_flags;
}

/*
 * pmap_remove_pte: remove a single PTE from a PTP
 *
 * => must have proper locking on pmap_master_lock
 * => caller must hold pmap's lock
 * => PTP must be mapped into KVA
 * => PTP should be null if pmap == pmap_kernel()
 * => returns true if we removed a mapping
 * => must be called with kernel preemption disabled
 */

static bool
pmap_remove_pte(struct pmap *pmap, struct vm_page *ptp, pte2_t *pte2base,
		vaddr_t va, int flags, struct pv_entry **pv_tofree)
{
	pte2_t opte;
	struct pv_entry *pve;
	struct vm_page *pg;
	struct pmap_page *pp;
	pte2_t *pte = &pte2base[VADDR_TO_PTE2SLOTI(va)];

	KASSERT(pmap == pmap_kernel() || mutex_owned(&pmap->pm_lock));
	KASSERT(pmap == pmap_kernel() || kpreempt_disabled());

	if (!pte2_valid_entry(pte))
		return(false);		/* VA not mapped */
	if ((flags & PMAP_REMOVE_SKIPWIRED) && (pte->pte2_flags & PTE2_w)) {
		return(false);
	}

	/* atomically save the old PTE and zap! it */
	do {
		opte.pte2_val = pte->pte2_val;
	} while (atomic_cas_32(&pte->pte2_flags, opte.pte2_flags, 0) !=
	    opte.pte2_flags);
	membar_producer();

	if (!pte2_valid_entry(&opte)) {
		return false;
	}

	pmap_stats_update_byflags(pmap, 0, opte.pte2_flags);

	if (opte.pte2_flags & (PTE2_RL|PTE2_D)) {
		pmap_tlb_inval(va);
	}

	if (ptp) {
		ptp->wire_count--;		/* dropping a PTE */
	}

	/*
	 * if we are not on a pv_head list we are done.
	 */

#if 0 /* XXX */
	if ((opte.pte2_flags & PG_pv) == 0) {
#if defined(DIAGNOSTIC)
		if (PHYS_TO_VM_PAGE(PTE2_TO_PADDR(opte)) != NULL)
			panic("pmap_remove_pte: managed page without "
			      "PG_PVLIST for 0x%lx", va);
#endif
		return(true);
	}
#endif

	pg = PHYS_TO_VM_PAGE(PTE2_TO_PADDR(opte));
	if (pg == NULL)
		return(true);
#if 0
#ifdef DIAGNOSTIC
	if (pg == NULL)
		panic("pmap_remove_pte: unmanaged page marked "
		    "PG_PVLIST, va = 0x%x, pa = 0x%lx", va,
		    (u_long)(PTE2_TO_PADDR(opte)));
#endif
#endif

	/* sync R/M bits */
	pp = VM_PAGE_TO_PP(pg);
	pp_lock(pp);
	pp->pp_attrs |= PTE2PATTR(opte.pte2_flags);
	pve = pmap_remove_pv(pp, ptp, va);
	pp_unlock(pp);

	if (pve) { 
		pve->pve_next = *pv_tofree;
		*pv_tofree = pve;
	}

	return(true);
}

/*
 * pmap_remove: top level mapping removal function
 *
 * => caller should not be holding any pmap locks
 */

void
pmap_remove(struct pmap *pmap, vaddr_t sva, vaddr_t eva)
{
	pmap_do_remove(pmap, sva, eva, PMAP_REMOVE_ALL);
}

/*
 * pmap_do_remove: mapping removal guts
 *
 * => caller should not be holding any pmap locks
 */

static void
pmap_do_remove(struct pmap *pmap, vaddr_t sva, vaddr_t eva, int flags)
{
	uint32_t xpte2_flags = 0;
	pte2_t *pte2base;
	pde1_t pde;
	struct pv_entry *pv_tofree = NULL;
	bool result;
	paddr_t ptppa;
	vaddr_t blkendva, va = sva;
	struct vm_page *ptp;

	UDPRINTF(pmap, PMD_PTE_SET2, ("pmap_do_remove(%p, 0x%x, 0x%x %d)\n",
	    pmap, sva, eva, flags));
	kpreempt_disable();
	pte2base = pmap_map_ptes(pmap, PM_WAITOK);	/* locks pmap */

	/*
	 * removing one page?  take shortcut function.
	 */

	if (va + PAGE_SIZE == eva) {
		pde = pmap->pm_pd[VADDR_TO_PDE1I(va)];
		if (pde1_valid_entry(&pde)) {
			/* PA of the PTP */
			ptppa = PTD1_TO_PTE2PADDR(pde);

			/* get PTP if non-kernel mapping */
			if (pmap == pmap_kernel()) {
				/* we never free kernel PTPs */
				ptp = NULL;
			} else {
				ptp = pmap_find_ptp(pmap, va, ptppa, pte2base);
#ifdef DIAGNOSTIC
				if (ptp == NULL)
					panic("pmap_remove: unmanaged "
					      "PTP detected");
#endif
			}

			/* do it! */
			result = pmap_remove_pte(pmap, ptp,
			    pte2base, va, flags, &pv_tofree);

			/*
			 * if mapping removed and the PTP is no longer
			 * being used, free it!
			 */

			if (result && ptp && ptp->wire_count <= 1)
				pmap_free_ptp(pmap, ptp, va);
		}
	} else for (/* null */ ; va < eva ; va = blkendva) {
		/* determine range of block */
		blkendva = (va + PDE1_NBPTD) & PDE1_FRAME;
		if (blkendva > eva)
			blkendva = eva;

		pde = pmap->pm_pd[VADDR_TO_PDE1I(va)];
		if (!pde1_valid_entry(&pde)) {
			/*
			 * skip this range (va -> blkendva) as PDE is invalid
			 */
 			continue;
		}

		/* PA of the PTP */
		ptppa = PTD1_TO_PTE2PADDR(pde);

		/* get PTP if non-kernel mapping */
		if (pmap == pmap_kernel()) {
			/* we never free kernel PTPs */
			ptp = NULL;
		} else {
			ptp = pmap_find_ptp(pmap, va, ptppa, pte2base);
#ifdef DIAGNOSTIC
			if (ptp == NULL)
				panic("pmap_remove: unmanaged PTP "
				      "detected");
#endif
		}
		xpte2_flags |= pmap_remove_ptes(pmap, ptp,
		    pte2base, va, blkendva, flags, &pv_tofree);

		/* if PTP is no longer being used, free it! */
		if (ptp && ptp->wire_count <= 1) {
			pmap_free_ptp(pmap, ptp, va);
		}
	}
	pmap_unmap_ptes(pmap);		/* unlock pmap */
	kpreempt_enable();

	/* Now we free unused PVs */
	if (pv_tofree)
		pmap_free_pvs(pv_tofree);
}

void
pmap_remove_all(struct pmap *pmap)
{
	DPRINTF(PMD_PMAP_INIT, ("pmap_remove_all %p\n", pmap));
}

/*
 * p m a p   p r o t e c t i o n   f u n c t i o n s
 */

/*
 * pmap_page_protect: change the protection of all recorded mappings
 *	of a managed page
 *
 * => NOTE: this is an inline function in pmap.h
 */

/* see pmap.h */

/*
 * pmap_protect: set the protection in of the pages in a pmap
 *
 * => NOTE: this is an inline function in pmap.h
 */

/* see pmap.h */

/*
 * pmap_write_protect: write-protect pages in a pmap
 */

void
pmap_write_protect(struct pmap *pmap, vaddr_t sva, vaddr_t eva, vm_prot_t prot)
{
	pte2_t *epte;
	pte2_t *spte;
	pte2_t *ptebase;
	vaddr_t blockend, va;
	pte2_t opte;
	pde1_t *pde;

	UDPRINTF(pmap, PMD_PTE_SET2,
	   ("pmap_write_protect(%p, 0x%x, 0x%x %d)\n", pmap, sva, eva, prot));
	kpreempt_disable();
	ptebase = pmap_map_ptes(pmap, PM_WAITOK);	/* locks pmap */

	/* should be ok, but just in case ... */
	sva &= PTE2_FRAME;
	eva &= PTE2_FRAME;

	for (va = sva ; va < eva ;) {

		blockend = (va & PDE1_FRAME) + PDE1_NBPTD;
		if (blockend > eva)
			blockend = eva;

		/* empty block? */
		pde = &pmap->pm_pd[VADDR_TO_PDE1I(va)];
		if (!pde1_valid_entry(pde))
			continue;

#ifdef DIAGNOSTIC
		if (va >= VM_MAXUSER_ADDRESS &&
		    va < VM_MAX_ADDRESS)
			panic("pmap_write_protect: PTE space");
#endif

		spte = &ptebase[VADDR_TO_PTE2SLOTI(va)];
		epte = &ptebase[VADDR_TO_PTE2SLOTI(blockend)];

		for (/*null */; spte < epte ; spte++) {
			pte2_t npte;

			do {
				opte.pte2_val = spte->pte2_val;
				if ((~opte.pte2_flags & (PTE2_W | PTE2_V)) != 0) {
					goto next;
				}
				npte.pte2_flags = opte.pte2_flags & ~PTE2_W;
			} while (atomic_cas_32(&spte->pte2_flags,
			  opte.pte2_flags, npte.pte2_flags) != opte.pte2_flags);
			membar_producer();
			if ((opte.pte2_flags & (PTE2_RL | PTE2_D)) != 0) {
				pmap_tlb_inval(va);
			}
next:
			va += PAGE_SIZE;
		}
		KASSERT(va == blockend);
	}

	pmap_unmap_ptes(pmap);	/* unlocks pmap */
	kpreempt_enable();
}

/*
 * end of protection functions
 */

/* Increment the reference count on the specified pmap */
void
pmap_reference(struct pmap *pmap)
{
	atomic_inc_uint((unsigned *)&pmap->pm_obj.uo_refs);
}

void
pmap_update(struct pmap *pmap)
{
	//UDPRINTF(pmap, PMD_PMAP_SWITCH, ("pmap_update(%p)\n", pmap));
}

/* zero a page by physical address */
void
pmap_zero_page(paddr_t pa)
{
	DPRINTF(PMD_PTE_SET, (__func__));
	kpreempt_disable();
	mutex_enter(&pmaps_copy_lock);
#if NVCIDMA > 0
	if (vcidma_zero(pa, PAGE_SIZE)) {
		mutex_exit(&pmaps_copy_lock);
		kpreempt_enable();
		return;
	}
#endif
	KASSERT(pmap_pte1->pte2_flags == 0);
	pmap_pte1->pte2_ppn = (pa >> PTE2_SHIFT);
	atomic_swap_32(&pmap_pte1->pte2_flags,
	    PTE2_V | PTE2_W | PTE2_G | PTE2_w);
	membar_producer();
	pmap_dtlb_inval(pmap_va1);
	memset((void *)pmap_va1, 0, PAGE_SIZE);
	atomic_swap_32(&pmap_pte1->pte2_flags, 0);
	pmap_pte1->pte2_ppn = 0;
	membar_producer();
	pmap_dtlb_inval(pmap_va1);
	DPRINTF(PMD_PTE_SET, (" pa 0x%lx done\n", (u_long)pa));
	mutex_exit(&pmaps_copy_lock);
	kpreempt_enable();
}

/* copy a page to another by physical address */
void
pmap_copy_page(paddr_t src, paddr_t dst)
{
	DPRINTF(PMD_PTE_SET, (__func__));
	kpreempt_disable();
	mutex_enter(&pmaps_copy_lock);
#if NVCIDMA > 0
	if (vcidma_copy(src, dst, PAGE_SIZE)) {
		mutex_exit(&pmaps_copy_lock);
		kpreempt_enable();
		return;
	}
#endif
	KASSERT(pmap_pte1->pte2_flags == 0);
	KASSERT(pmap_pte2->pte2_flags == 0);
	pmap_pte1->pte2_ppn = (src >> PTE2_SHIFT);
	atomic_swap_32(&pmap_pte1->pte2_flags,
	    PTE2_V | PTE2_W | PTE2_G | PTE2_w);
	pmap_pte2->pte2_ppn = (dst >> PTE2_SHIFT);
	atomic_swap_32(&pmap_pte2->pte2_flags,
	    PTE2_V | PTE2_W | PTE2_G | PTE2_w);
	membar_producer();
	pmap_dtlb_inval(pmap_va1);
	pmap_dtlb_inval(pmap_va2);
	memcpy((void *)pmap_va2, (void *)pmap_va1, PAGE_SIZE);
	membar_producer();
	atomic_swap_32(&pmap_pte1->pte2_flags, 0);
	atomic_swap_32(&pmap_pte2->pte2_flags, 0);
	pmap_pte1->pte2_ppn = 0;
	pmap_pte2->pte2_ppn = 0;
	membar_producer();
	pmap_dtlb_inval(pmap_va1);
	pmap_dtlb_inval(pmap_va2);
	DPRINTF(PMD_PTE_SET, (" pa 0x%lx -> 0x%lx done\n",
	    (u_long)src, (u_long)dst));
	mutex_exit(&pmaps_copy_lock);
	kpreempt_enable();
}

/*
 * Define the initial bounds of the kernel virtual address space.
 */
void
pmap_virtual_space(vaddr_t *startp, vaddr_t *endp)
{
	*startp = virtual_avail;
	*endp = virtual_end;
}

bool
pmap_extract(struct pmap *pmap, vaddr_t va, paddr_t *pap)
{
	pte2_t *ptebase;

	kpreempt_disable();
	if (pmap == pmap_kernel()) {
		DPRINTF(PMD_PTE_GET, ("pmap_extract(pmap_kernel, 0x%x)", va));
	} else {
		DPRINTF(PMD_PTE_GET, ("pmap_extract(%p, 0x%x)", pmap, va));
	}
	if (!pde1_valid_entry(&pmap->pm_pd[VADDR_TO_PDE1I(va)])) {
		DPRINTF(PMD_PTE_GET, (" not mapped1\n"));
		if (pap)
			*pap = -1;
		kpreempt_enable();
		return false;
	}
	if ((pmap->pm_pd[VADDR_TO_PDE1I(va)] & PDE1_T) == 0) {
		/* this is a PTE1 */
		if (pap) {
			*pap = PTE1_TO_PADDR(pmap->pm_pd[VADDR_TO_PDE1I(va)]) |
			    (va & ~PTE1_FRAME);
		}
		DPRINTF(PMD_PTE_GET, (" pa1 0x%llx\n",
		    PTE1_TO_PADDR(pmap->pm_pd[VADDR_TO_PDE1I(va)])));
		kpreempt_enable();
		return true;
	}
	ptebase = pmap_map_ptes(pmap, PM_WAITOK);
	if (!pte2_valid_entry(&ptebase[VADDR_TO_PTE2SLOTI(va)])) {
		DPRINTF(PMD_PTE_GET, (" not mapped2\n"));
		if (pap)
			*pap = -1;
		pmap_unmap_ptes(pmap);
		kpreempt_enable();
		return false;
	}
	if (pap) {
		*pap = PTE2_TO_PADDR(ptebase[VADDR_TO_PTE2SLOTI(va)]) |
		    (va & ~PTE2_FRAME);
	}
	DPRINTF(PMD_PTE_GET, (" pa2 0x%llx\n",
	    PTE2_TO_PADDR(ptebase[VADDR_TO_PTE2SLOTI(va)])));
	pmap_unmap_ptes(pmap);
	kpreempt_enable();
	return true;
}

#if 0
vaddr_t
pmap_steal_memory(vsize_t size, vaddr_t *startp, vaddr_t *endp)
{
	return 0;
}

void
pmap_fork(struct pmap *src_pmap, struct pmap *dst_pmap)
{
}
#endif

vaddr_t
pmap_growkernel(vaddr_t maxkvaddr)
{
	vsize_t size;
	int nkptp_alloc, i;
	struct vm_page *pg;
	vaddr_t va;
	pte2_t *ptebase;
	struct pmap *pm;
	int s;

	s = splvm();
	DPRINTF(PMD_PMAP_INIT, (__func__));
	if (maxkvaddr <= pmap_maxkvaddr) {
		goto done;
	}
	mutex_enter(&pmap_kernel()->pm_lock);
	ptebase = pmap_map_ptes(pmap_kernel(), PM_WAITOK);
	size = maxkvaddr - pmap_maxkvaddr;
	size = (size + (PDE1_NBPTD - 1)) & PDE1_MASK; /* roundup */
	nkptp_alloc = size / PDE1_NBPTD;
	DPRINTF(PMD_PMAP_INIT, (" size 0x%x alloc %d\n", size, nkptp_alloc));
	for (va = pmap_maxkvaddr, i = 0;
	     i < nkptp_alloc;
	     i++, va += PDE1_NBPTD) {
		pg = uvm_pagealloc(&pmap_kernel()->pm_obj,
		    (vaddr_t)&ptebase[VADDR_TO_PTE2BASEI(va)],
		    NULL, UVM_PGA_USERESERVE|UVM_PGA_ZERO);
		if (pg == NULL)
			panic("pmap_growkernel: can't get new PTP");

		pg->flags &= ~PG_BUSY;
		pg->wire_count = 1;
		KASSERT(
		    (pmap_kernel()->pm_pd[VADDR_TO_PDE1I(va)] & PDE1_V) == 0);
		pmap_kernel()->pm_pd[VADDR_TO_PDE1I(va)] =
		    (VM_PAGE_TO_PHYS(pg) >>  PAGE_SHIFT) | PDE1_V | PDE1_T;
		KASSERT(
		    (pmap_kernel()->pm_pte2map[VADDR_TO_PDE1I(va)].pte2_flags
		    & PTE2_V) == 0);
		pmap_kernel()->pm_pte2map[VADDR_TO_PDE1I(va)].pte2_ppn =
		    (VM_PAGE_TO_PHYS(pg) >>  PAGE_SHIFT);
		pmap_kernel()->pm_pte2map[VADDR_TO_PDE1I(va)].pte2_flags =
		    PTE2_V | PTE2_C | PTE2_W | PTE2_G | PTE2_w;
		DPRINTF(PMD_PMAP_INIT, ("   pmap_growkernel pde @%p 0x%x\n",
		    &pmap_kernel()->pm_pd[VADDR_TO_PDE1I(va)],
		    pmap_kernel()->pm_pd[VADDR_TO_PDE1I(va)]));
	}
	membar_producer();
	pmap_unmap_ptes(pmap_kernel());
	mutex_exit(&pmap_kernel()->pm_lock);
	/*
	 * Now enter it in all non-kernel pmaps.
	 * The shadow kernel PD is shared so we only have the
	 * physical PD to do.
	 */
	mutex_enter(&pmaps_lock);
	LIST_FOREACH(pm, &pmaps, pm_list) {
		DPRINTF(PMD_PMAP_INIT,
		    ("pmap_growkernel copy pm %p to %p from %p size %d\n",
		    pm, &pm->pm_pd[VADDR_TO_PDE1I(pmap_maxkvaddr)],
		    &pmap_kernel()->pm_pd[VADDR_TO_PDE1I(pmap_maxkvaddr)],
		    nkptp_alloc * sizeof(pde1_t)));
		memcpy(&pm->pm_pd[VADDR_TO_PDE1I(pmap_maxkvaddr)],
		    &pmap_kernel()->pm_pd[VADDR_TO_PDE1I(pmap_maxkvaddr)],
		    nkptp_alloc * sizeof(pde1_t));
	}
	mutex_exit(&pmaps_lock);
	pmap_maxkvaddr = va;
done:
	splx(s);
	DPRINTF(PMD_PMAP_INIT, ("pmap_growkernel 0x%x return 0x%x\n",
	    maxkvaddr, pmap_maxkvaddr));
	return pmap_maxkvaddr;
}

/* remove all mappings beteen va and va + sz from kernel_map */
void
pmap_kremove(vaddr_t va, vsize_t sz)
{
	vaddr_t endva = va + sz;
	pte2_t *ptebase;

	DPRINTF(PMD_PTE_SET, ("pmap_kremove 0x%x 0x%x", va, sz));
	/* pmap_kernel() is always accessible, avoids locking pmaps */
	ptebase = (pte2_t *)L1_PTE2SLOT_VADDR;
	for (; va < endva; va += PTE2_NBPTE) {
		DPRINTF(PMD_PTE_SET, (" pte %p 0x%x/0x%x",
		    &ptebase[VADDR_TO_PTE2SLOTI(va)],
		    ptebase[VADDR_TO_PTE2SLOTI(va)].pte2_flags,
		    ptebase[VADDR_TO_PTE2SLOTI(va)].pte2_ppn));
		atomic_swap_32(&ptebase[VADDR_TO_PTE2SLOTI(va)].pte2_flags, 0);
		membar_producer();
		pmap_tlb_inval(va);
	}
	DPRINTF(PMD_PTE_SET, ("\n"));
}

/*
 * Enter an unmanaged (no R/M tracking, no pv_entry)  mapping in kernel_map. 
 */
void
pmap_kenter_pa(vaddr_t va, paddr_t pa, vm_prot_t prot)
{
	pte2_t *ptebase;

	KASSERT(!(prot & ~VM_PROT_ALL));
	KASSERT(va >= VM_MIN_KERNEL_ADDRESS);

	if (pa == 0x0)
		Debugger();
	/* pmap_kernel() is always accessible, avoids locking pmaps */
	ptebase = (pte2_t *)L1_PTE2SLOT_VADDR;
	DPRINTF(PMD_PTE_SET, ("pmap_kenter_pa(0x%x 0x%lx 0x%x) pte %p",
	    va, (u_long)pa, prot, &ptebase[VADDR_TO_PTE2SLOTI(va)]));
	ptebase[VADDR_TO_PTE2SLOTI(va)].pte2_ppn = (pa >> PTE2_SHIFT);
	membar_producer();
	atomic_swap_32(&ptebase[VADDR_TO_PTE2SLOTI(va)].pte2_flags,
	    prot2pte[prot] | PTE2_V | PTE2_G | PTE2_C | PTE2_w);
	membar_producer();
	pmap_tlb_inval(va);
	DPRINTF(PMD_PTE_SET, (" val 0x%x/0x%x\n",
	    ptebase[VADDR_TO_PTE2SLOTI(va)].pte2_flags,
	    ptebase[VADDR_TO_PTE2SLOTI(va)].pte2_ppn));
}

/* bus_space(9) helper */
void
tsarmips_pmap_kenter_pa(vaddr_t va, paddr_t pa , vm_prot_t prot, int cache)
{
	pte2_t *ptebase;
	uint32_t flags;

	KASSERT(!(prot & ~VM_PROT_ALL));
	KASSERT(va >= VM_MIN_KERNEL_ADDRESS);

	flags = prot2pte[prot] | PTE2_V | PTE2_G | PTE2_w;
	if (cache)
		flags |= PTE2_C;

	if (pa == 0x0)
		Debugger();
	/* pmap_kernel() is always accessible, avoids locking pmap */
	ptebase = (pte2_t *)L1_PTE2SLOT_VADDR;
	ptebase[VADDR_TO_PTE2SLOTI(va)].pte2_ppn = (pa >> PTE2_SHIFT);
	membar_producer();
	atomic_swap_32(&ptebase[VADDR_TO_PTE2SLOTI(va)].pte2_flags, flags);
	membar_producer();
	pmap_tlb_inval(va);
	DPRINTF((PMD_PTE_SET|PMD_PMAP_INIT),
	    ("tsarmips_pmap_kenter_pa(0x%x 0x%lx 0x%x) pte %p val 0x%x/0x%x\n",
	    va, (u_long)pa, prot, &ptebase[VADDR_TO_PTE2SLOTI(va)],
	    ptebase[VADDR_TO_PTE2SLOTI(va)].pte2_flags,
	    ptebase[VADDR_TO_PTE2SLOTI(va)].pte2_ppn));
}

/* manage identity mapping (using big pages) for secondary CPUs bootstrap */
void
pmap_seccpus_bootstrap_add(vaddr_t va)
{
	pte1_t *pte = pmap_kernel()->pm_pd;
	va = (va - KERNBASE);

	printf("mapping 0x%x at 0x%x\n", (int)va, (int)va);
	KASSERT((pte[VADDR_TO_PTE1I(va)] & PTE1_V) == 0);
	pte[VADDR_TO_PTE1I(va)] = PTE1_V | PTE1_C | PTE1_X |
	    (va >> PTE1_SHIFT);
	wbflush();
}

void
pmap_seccpus_bootstrap_remove(vaddr_t va)
{
	pte1_t *pte = pmap_kernel()->pm_pd;
	va = (va - KERNBASE);

	printf("unmapping 0x%x at 0x%x\n", (int)va, (int)va);
	KASSERT((pte[VADDR_TO_PTE1I(va)] & PTE1_V) != 0);
	pte[VADDR_TO_PTE1I(va)] = 0;
	wbflush();
}

void pmap_check_pmap(void);

void
pmap_check_pmap(void)
{
	struct pmap *pmap = vm_map_pmap(&curlwp->l_proc->p_vmspace->vm_map);
	u_int ptpr = tsar_vcache_read(VC_PTPR);

	if ((pmap->pm_pdpa >> VC_PTPR_PASHIFT) != ptpr) {
		printf("curlwp %p pmap %p (%s) ptpr 0x%llx != 0x%x\n",
		    curlwp, pmap, 
		    (pmap == pmap_kernel()) ? "kernel" : "user",
		    pmap->pm_pdpa >> VC_PTPR_PASHIFT, ptpr);
	}
}
