/* $NetBSD: $ */

/*-
  * Copyright (c) 2009 UPMC/LIP6
  * All rights reserved.
  * This software is distributed under the following condiions
  * compliant with the NetBSD foundation policy.
  *
  * Redistribution and use in source and binary forms, with or without
  * modification, are permitted provided that the following conditions
  * are met:
  * 1. Redistributions of source code must retain the above copyright
  *    notice, this list of conditions and the following disclaimer.
  * 2. Redistributions in binary form must reproduce the above copyright
  *    notice, this list of conditions and the following disclaimer in the
  *    documentation and/or other materials provided with the distribution.
  *
  * THIS SOFTWARE IS PROVIDED BY THE NETBSD FOUNDATION, INC. AND CONTRIBUTORS
  * ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED
  * TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
  * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE FOUNDATION OR CONTRIBUTORS
  * BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
  * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
  * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
  * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
  * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
  * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
  * POSSIBILITY OF SUCH DAMAGE.
  */

#include <sys/cdefs.h>
__KERNEL_RCSID(0, "$NetBSD: cpu.c,v 1.24 2005/12/11 12:18:39 christos Exp $");

#include <sys/param.h>
#include <sys/device.h>
#include <sys/systm.h>

#include <mips/locore.h>

#include <machine/autoconf.h>
#include <machine/xicu.h>
#include <machine/xicureg.h>

/*
 * we keep interrupt handler descriptors in an array, one per type.
 * the array is indexed by the interrupt line number.
 */
struct xicu_intrhand *pti_ih[XICU_NIRQ];
struct xicu_intrhand *wti_ih[XICU_NIRQ];
struct xicu_intrhand *hwi_ih[XICU_NIRQ];

/* timer 0 is our clock, it's a special case */
struct evcnt xicu_clock_evcnt =
    EVCNT_INITIALIZER(EVCNT_TYPE_INTR, NULL, "clock", "intr");


static int	xicu_match(device_t, cfdata_t, void *);
static void	xicu_attach(device_t, device_t, void *);

struct xicu_softc {
	device_t sc_dev;
	paddr_t sc_p; /* device mapping info */
	vaddr_t sc_psize;
	bus_space_tag_t sc_tag;
	bus_space_handle_t sc_handle;
};

struct xicu_softc *xicu0; /* XXX per-CPU */

static void xicu_init_timer0(struct xicu_softc *);

static inline
uint32_t xicuread(struct xicu_softc *sc, uint32_t reg)
{
	return bus_space_read_4(sc->sc_tag, sc->sc_handle, reg);
}
static inline
void xicuwrite(struct xicu_softc *sc, uint32_t reg, uint32_t val)
{
	bus_space_write_4(sc->sc_tag, sc->sc_handle, reg, val);
}

CFATTACH_DECL_NEW(xicu, sizeof(struct xicu_softc),
    xicu_match, xicu_attach, NULL, NULL);

static int
xicu_match(device_t parent, cfdata_t match, void *aux)
{
	return 1;
}

static void
xicu_attach(device_t parent, device_t self, void *aux)
{
	struct xicu_softc *sc = device_private(self);
	struct tsardevs_attach_args *ma = aux;
	int i;

	aprint_normal(": programmable interrupt controller\n");
	sc->sc_dev = self;
	sc->sc_tag = ma->tag;
	sc->sc_p = ma->device_base;
	sc->sc_psize = ma->device_size;


	/* map the device */
	if (bus_space_map(sc->sc_tag, sc->sc_p, sc->sc_psize,
	    0, &sc->sc_handle)) {
		aprint_error_dev(self, "couldn't map registers\n");
		return;
	}
	aprint_normal_dev(self, "mapped at 0x%lx\n", sc->sc_handle);
	xicu0 = sc;
	/* initialize mapping registers */
	for(i = 0; i < 6; i++) {
		xicuwrite(sc, XICU_MSK_HWI(i), 0);
		xicuwrite(sc, XICU_MSK_PTI(i), 0);
		xicuwrite(sc, XICU_MSK_WTI(i), 0);
	}
	/* initialize interrupt handler arrays */
	for (i = 0; i < XICU_NIRQ; i++) {
		pti_ih[i] = NULL;
		wti_ih[i] = NULL;
		hwi_ih[i] = NULL;
	}
	/* setup timer 0 */
	xicu_init_timer0(sc);
}

int
xicu_establish(irq_type_t type, int irq, int ipl, const char *name,
    struct xicu_intrhand *ih)
{
	int s;
	int mipsirq = IPL2IRQ(ipl);

	if (irq >= XICU_NIRQ || irq < 0)
		return EINVAL;
	s = splhigh();
	switch (type) {
	case IRQ_HWI:
		/* register handler */
		KASSERT(hwi_ih[irq] == NULL);
		hwi_ih[irq] = ih;
		/* route and unmask interrupt */
		xicuwrite(xicu0, XICU_MSK_HWI_E(mipsirq), (1U << irq));
		printf("xicu_establish for %s %d %d %d reg 0x%x=0x%x\n",
		    name, type, irq, ipl, XICU_MSK_HWI(mipsirq),
		    xicuread(xicu0, XICU_MSK_HWI(mipsirq)));
		break;
	case IRQ_PTI:
		/* register handler */
		KASSERT(pti_ih[irq] == NULL);
		pti_ih[irq] = ih;
		/* route and unmask interrupt */
		xicuwrite(xicu0, XICU_MSK_PTI_E(mipsirq), (1U << irq));
		printf("xicu_establish for %s %d %d %d reg 0x%x=0x%x\n",
		    name, type, irq, ipl, XICU_MSK_PTI(mipsirq),
		    xicuread(xicu0, XICU_MSK_PTI(mipsirq)));
		break;
	case IRQ_WTI:
		/* register handler */
		KASSERT(wti_ih[irq] == NULL);
		wti_ih[irq] = ih;
		/* route and unmask interrupt */
		xicuwrite(xicu0, XICU_MSK_WTI_E(mipsirq), (1U << irq));
		printf("xicu_establish for %s %d %d %d reg 0x%x=0x%x\n",
		    name, type, irq, ipl, XICU_MSK_WTI(mipsirq),
		    xicuread(xicu0, XICU_MSK_WTI(mipsirq)));
		break;
	}
	return 0;
}

void
xicu_intr(u_int32_t status, u_int32_t cause, u_int32_t pc, u_int32_t ipending)
{
	int i;
	uint32_t xicupending;
	uint32_t irq;
	ipending &= MIPS_HARD_INT_MASK;

#if 0
	printf("xicu_intr(0x%x, 0x%x, 0x%x, 0x%x)\n", status,
	    cause, pc, ipending);
#endif
	/* find pending spls, starting from high */
	for (i = 0; ipending != 0; i++, ipending = ipending >>1) {
		if ((ipending & MIPS_INT_MASK_0) == 0)
			continue;
		xicupending = xicuread(xicu0, XICU_PRIO(i));
		while ((xicupending & XICU_PRIO_PENDING) != 0) {
			if (xicupending & XICU_PRIO_PTI) {
				irq = XICU_PRIO_PTII(xicupending);
				/* special-case clock interrupt */
				if (irq == 0) {
					struct clockframe cf;
					cf.pc = pc;
					cf.sr = status;
					hardclock(&cf);
					(void)xicuread(xicu0, XICU_PTI_ACK(0));
					xicu_clock_evcnt.ev_count++;
				} else {
					KASSERT(pti_ih[irq] != NULL);
					pti_ih[irq]->ih_func(pti_ih[irq]->ih_arg);
					pti_ih[irq]->ih_ev.ev_count++;
				}
			}
			if (xicupending & XICU_PRIO_WTI) {
				irq = XICU_PRIO_WTII(xicupending);
				KASSERT(wti_ih[irq] != NULL);
				wti_ih[irq]->ih_func(wti_ih[irq]->ih_arg);
				wti_ih[irq]->ih_ev.ev_count++;
			}
			if (xicupending & XICU_PRIO_HWI) {
				irq = XICU_PRIO_HWII(xicupending);
				KASSERT(hwi_ih[irq] != NULL);
				hwi_ih[irq]->ih_func(hwi_ih[irq]->ih_arg);
				hwi_ih[irq]->ih_ev.ev_count++;
			}
			xicupending = xicuread(xicu0, XICU_PRIO(i));
		}
	}
	_splset(MIPS_SR_INT_IE | (status & ~cause & MIPS_HARD_INT_MASK));
}

static void
xicu_init_timer0(struct xicu_softc *sc)
{
	/* assume about 500Khz clock -> 10 interrupt per second (for now) */
	xicuwrite(sc, XICU_PTI_PER(0), 70000);
	/* clear any pending interrupt */
	(void)xicuread(xicu0, XICU_PTI_ACK(0));
	/* map interrupt to IPL_CLOCK */
	xicuwrite(sc, XICU_MSK_PTI_E(IPL2IRQ(IPL_SCHED)), (1U << 0));
	printf("xicu out %d map 0x%x 0x%x 0x%x\n",  IPL2IRQ(IPL_SCHED),
		    xicuread(sc, XICU_MSK_HWI(IPL2IRQ(IPL_SCHED))),
		    xicuread(sc, XICU_MSK_PTI(IPL2IRQ(IPL_SCHED))),
		    xicuread(sc, XICU_MSK_WTI(IPL2IRQ(IPL_SCHED))));
	evcnt_attach_static(&xicu_clock_evcnt);
}
