/* $NetBSD: $ */

/*-
  * Copyright (c) 2017 UPMC/LIP6
  * All rights reserved.
  * This software is distributed under the following condiions
  * compliant with the NetBSD foundation policy.
  *
  * Redistribution and use in source and binary forms, with or without
  * modification, are permitted provided that the following conditions
  * are met:
  * 1. Redistributions of source code must retain the above copyright
  *    notice, this list of conditions and the following disclaimer.
  * 2. Redistributions in binary form must reproduce the above copyright
  *    notice, this list of conditions and the following disclaimer in the
  *    documentation and/or other materials provided with the distribution.
  *
  * THIS SOFTWARE IS PROVIDED BY THE NETBSD FOUNDATION, INC. AND CONTRIBUTORS
  * ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED
  * TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
  * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE FOUNDATION OR CONTRIBUTORS
  * BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
  * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
  * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
  * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
  * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
  * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
  * POSSIBILITY OF SUCH DAMAGE.
  */

/* driver for 16550-compatible UART with VCI attachement */

#include <sys/cdefs.h>
__KERNEL_RCSID(0, "$NetBSD: machdep.c,v 1.223 2008/07/02 17:28:56 ad Exp $");

#include <sys/param.h>
#include <sys/ioctl.h>
#include <sys/proc.h>
#include <sys/tty.h>
#include <sys/systm.h>
#include <sys/device.h>
#include <sys/conf.h>

#include <machine/autoconf.h>
#include <machine/bus.h>

#include <dev/cons.h>
#include <dev/ic/comvar.h>

static int vcicom_match(device_t, cfdata_t, void *);
static void vcicom_attach(device_t, device_t, void *);

struct vcicom_softc {
	struct com_softc vsc_sc;
	paddr_t vsc_p; /* console mapping informations */
	vaddr_t vsc_psize;
	irq_t vsc_ih;
};

CFATTACH_DECL_NEW(com_vci, sizeof(struct vcicom_softc),
    vcicom_match, vcicom_attach, NULL, NULL);

int
vcicom_match(device_t parent, cfdata_t match, void *aux)
{
	struct tsardevs_attach_args *tsd = aux;
	if (strcmp(tsd->tsd_devtype, "ns16550a") != 0)
		return 0;
	return 1;
}

void
vcicom_attach(device_t parent, device_t self, void *aux)
{
	struct vcicom_softc *vsc = device_private(self);
	struct com_softc * const sc = &vsc->vsc_sc;
	struct tsardevs_attach_args *tsd = aux;
	const struct boot_fdt_prop *prop;
	bus_space_handle_t bsh;
	uint32_t reg_shift;
	char intstr[40];

	sc->sc_dev = self;
	vsc->vsc_p = tsd->tsd_reg[0].reg_addr;
	vsc->vsc_psize = tsd->tsd_reg[0].reg_size;

	if (!fdt_get_clock_freq(tsd->tsd_node, &sc->sc_frequency)) {
		aprint_error(": no frequency\n");
		return;
	}
	sc->sc_type = COM_TYPE_NORMAL;

	prop = fdt_find_prop(tsd->tsd_node->fdt_node_data, "reg-shift");
	if (prop != NULL)
		reg_shift = be32toh(prop->prop_value[0]);
	else
		reg_shift = 0;

	if (reg_shift != 0) {
		aprint_error(": unsupported reg-shift %d\n", reg_shift);
		return;
	}

	if (bus_space_map(tsd->tsd_tag, vsc->vsc_p, vsc->vsc_psize, 0, &bsh)) {
		aprint_error(": couldn't map registers\n");
		return;
	}
	COM_INIT_REGS(sc->sc_regs, tsd->tsd_tag, bsh, vsc->vsc_p);
	com_attach_subr(sc);
	aprint_naive("\n");
	vsc->vsc_ih = intr_establish(tsd->tsd_irq, IPL_TTY,
	    device_xname(self), intstr, comintr, sc, NULL);
	if (vsc->vsc_ih == NULL) {
		aprint_error_dev(self, "can't establish interrupt\n");
	} else {
		aprint_normal_dev(self, "interrupting at %s\n", intstr);
	}
}

void iopic_stat(irq_t);
void vci_comstat(void *);
void vci_comstat(void *p)
{
	struct vcicom_softc *vsc = p;
	iopic_stat(vsc->vsc_ih);
}
