/* $NetBSD: $ */

/*-
  * Copyright (c) 2017 UPMC/LIP6
  * All rights reserved.
  * This software is distributed under the following condiions
  * compliant with the NetBSD foundation policy.
  *
  * Redistribution and use in source and binary forms, with or without
  * modification, are permitted provided that the following conditions
  * are met:
  * 1. Redistributions of source code must retain the above copyright
  *    notice, this list of conditions and the following disclaimer.
  * 2. Redistributions in binary form must reproduce the above copyright
  *    notice, this list of conditions and the following disclaimer in the
  *    documentation and/or other materials provided with the distribution.
  *
  * THIS SOFTWARE IS PROVIDED BY THE NETBSD FOUNDATION, INC. AND CONTRIBUTORS
  * ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED
  * TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
  * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE FOUNDATION OR CONTRIBUTORS
  * BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
  * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
  * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
  * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
  * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
  * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
  * POSSIBILITY OF SUCH DAMAGE.
  */

/* driver for the IOPIC controller */

#include <sys/cdefs.h>
__KERNEL_RCSID(0, "$NetBSD: machdep.c,v 1.223 2008/07/02 17:28:56 ad Exp $");

#include <sys/param.h>
#include <sys/device.h>
#include <sys/cpu.h>
#include <sys/malloc.h>

#include <machine/autoconf.h>
#include <machine/bus.h>
#include <machine/intr.h>
#include <machine/xicu.h>
#include <machine/iopic.h>

#undef VCI_DEBUG
#ifdef VCI_DEBUG
#define DPRINTF(m, x) {if (vcidebug_mask & (m)) printf x;}
#define VCIDEBUG_FUNC	0x1
#define VCIDEBUG_INIT	0x2
#define VCIDEBUG_INTR	0x4
static int vcidebug_mask = VCIDEBUG_FUNC|VCIDEBUG_INIT|VCIDEBUG_INTR;
#else
#define DPRINTF(m, x)
#endif

/* VCI device registers */
#define IOPIC_SPAN		4
#define IOPIC_ADDR_LO(i)	((0x00 + (i) * IOPIC_SPAN) * 4)
#define IOPIC_ADDR_HI(i)	((0x01 + (i) * IOPIC_SPAN) * 4)
#define IOPIC_STATUS(i)		((0x02 + (i) * IOPIC_SPAN) * 4)
#define IOPIC_MASK(i)		((0x03 + (i) * IOPIC_SPAN) * 4)

#define IOPIC_STATUS_INTR	0x01
#define IOPIC_STATUS_ERR	0x02

#define IOPIC_MASK_INTR		0x01

#define IOPIC_NIRQ 32

static int  iopic_match(device_t, cfdata_t, void *);
static void iopic_attach(device_t, device_t, void *);

struct iopic_softc;

struct iopic_ih {
	struct intrhand ioih_ih; /* must be first */
	struct intrhand *ioih_xih; /* handler allocated by xicu */
	struct xicu_wti ioih_wti; /* WTI descriptor */
};


struct iopic_softc {
	device_t sc_dev;
	bus_space_tag_t sc_tag;
	bus_space_handle_t sc_handle;
	int	sc_input_lines;
	paddr_t	sc_p; /* device mapping info */
	paddr_t	sc_psize; /* device mapping info */
	struct iopic_intrhand *sc_irq_ih[IOPIC_NIRQ];
};

static int  iopic_intr(void *);
static inline uint32_t
iopic_read(struct iopic_softc *sc, uint32_t reg)
{
	return bus_space_read_4(sc->sc_tag, sc->sc_handle, reg);
}
static inline void
iopic_write(struct iopic_softc *sc, uint32_t reg, uint32_t val)
{
	bus_space_write_4(sc->sc_tag, sc->sc_handle, reg, val);
}

CFATTACH_DECL_NEW(vciiopic, sizeof(struct iopic_softc),
    iopic_match, iopic_attach, NULL, NULL);

static int
iopic_match(device_t parent, cfdata_t match, void *aux)
{
	struct tsardevs_attach_args *tsd = aux;
	if (strcmp(tsd->tsd_devtype, "vci:iopic") == 0)
		return 1;
	if (strcmp(tsd->tsd_devtype, "vci:xpic") == 0)
		return 1;
	return 0;
}

static void
iopic_attach(device_t parent, device_t self, void *aux)
{
	struct iopic_softc *sc = device_private(self);
	struct tsardevs_attach_args *tsd = aux;
	const struct boot_fdt_prop *prop;

	aprint_normal(": IO programmable interrupt controller\n");

	sc->sc_dev = self;
	sc->sc_tag = tsd->tsd_tag;

	prop = fdt_find_prop(tsd->tsd_node->fdt_node_data, "input_lines");
	if (prop == NULL) {
		aprint_error_dev(self, "no input_lines\n");
		return;
	}
	sc->sc_input_lines = be32toh(prop->prop_value[0]);

	aprint_normal_dev(self, "%d input lines\n", sc->sc_input_lines);
	if (sc->sc_input_lines > IOPIC_NIRQ)
		sc->sc_input_lines = IOPIC_NIRQ;

	/* map the device */
	if (bus_space_map(sc->sc_tag,
	    tsd->tsd_reg[0].reg_addr, tsd->tsd_reg[0].reg_size, 0,
	    &sc->sc_handle)) {
			aprint_error_dev(self, "couldn't map registers\n");
			return;
	}
	return;
}

static int
iopic_intr(void *p)
{
	struct iopic_ih *ioih = p;
	struct intrhand *ih = &ioih->ioih_ih;
	struct iopic_softc *sc = device_private(ih->ih_pic_dev);
	int rv = 0;

	DPRINTF(VCIDEBUG_INTR, ("iopic_intr irq %d\n", ih->ih_irq));
#if 0
	/*
	 * ack interrupt now, avoid loop in xicu handler until hardware
	 * does ack
	 */
	(void)bus_space_read_4(ioih->ioih_wti.wti_tag,
	    ioih->ioih_wti.wti_handle, 0);
#endif

	while (iopic_read(sc, IOPIC_STATUS(ih->ih_irq)) & IOPIC_STATUS_INTR) {
		KERNEL_LOCK(1, NULL);
		KASSERT(ih->ih_func != NULL);
		rv = ih->ih_func(ih->ih_arg);
		ih->ih_ev.ev_count++;
		KERNEL_UNLOCK_ONE(NULL);
	}
	return rv;
}

irq_t
iopic_intr_establish(device_t iopic, irq_line_t irq, ipl_t ipl,
    const char *name, char *intrstr, int (*handler)(void *), void *ih_arg,
    struct cpu_info *ci)
{
	irq_t intrhand;
	struct iopic_ih *iopic_ih;
	struct iopic_softc *sc = device_private(iopic);
	uint32_t reg;

	KASSERT(device_is_a(iopic, "iopic"));
	DPRINTF(VCIDEBUG_FUNC | VCIDEBUG_INTR, ("iopic_intr_establish %s/%d\n", device_xname(iopic), irq));

	iopic_ih = malloc(sizeof(struct iopic_ih), M_TEMP, M_NOWAIT);
	if (iopic_ih == NULL)
		return NULL;

	intrhand = &iopic_ih->ioih_ih;

	intrhand->ih_pic_dev = iopic;
	intrhand->ih_irq = irq;
	intrhand->ih_func = handler;
	intrhand->ih_arg = ih_arg;

	if (!xicu_wti_alloc(ci, &iopic_ih->ioih_wti)) {
		aprint_error_dev(iopic, "no WTI available for irq %d\n", irq);
		goto bad_wti;
	}
	DPRINTF(VCIDEBUG_INIT, ("iopic_intr_establish: wti %s/%d\n", device_xname(iopic_ih->ioih_wti.wti_dev), iopic_ih->ioih_wti.wti_line));
	reg = iopic_read(sc, IOPIC_STATUS(irq));
	DPRINTF(VCIDEBUG_INIT, ("iopic_intr_establish: status 0x%x\n", reg));
	iopic_write(sc, IOPIC_ADDR_LO(irq),
	    (iopic_ih->ioih_wti.wti_paddr & 0xffffffff));
	iopic_write(sc, IOPIC_ADDR_HI(irq),
	    (iopic_ih->ioih_wti.wti_paddr >> 32));
	reg = iopic_read(sc, IOPIC_STATUS(irq));
	DPRINTF(VCIDEBUG_INIT, ("iopic_intr_establish: status2 0x%x\n", reg));

	iopic_ih->ioih_xih = xicu_intr_establish(iopic_ih->ioih_wti.wti_dev,
	    IRQ_WTI, iopic_ih->ioih_wti.wti_line, ipl, name, intrstr,
	    iopic_intr, iopic_ih, ci);
	if (iopic_ih->ioih_xih == NULL) {
		aprint_error_dev(iopic, "can't establish WTI %s:%d for irq %d\n",
		    device_xname(iopic_ih->ioih_wti.wti_dev),
		    iopic_ih->ioih_wti.wti_line, irq);
		goto bad;
	}
	DPRINTF(VCIDEBUG_INIT, ("iopic_intr_establish: ih %p xih %p\n", iopic_ih, iopic_ih->ioih_xih));

	evcnt_attach_dynamic(&intrhand->ih_ev, EVCNT_TYPE_INTR, NULL,
	    name, "iointr");
	iopic_write(sc, IOPIC_MASK(irq), 1);
	return intrhand;

bad:
	iopic_write(sc, IOPIC_MASK(irq), 0);
	iopic_write(sc, IOPIC_ADDR_LO(irq), 0);
	iopic_write(sc, IOPIC_ADDR_HI(irq), 0);
	xicu_wti_free(&iopic_ih->ioih_wti);
bad_wti:
	free(iopic_ih, M_TEMP);
	return NULL;
}

void iopic_stat(irq_t);
void xicu_stat(irq_t);
void iopic_stat(irq_t ih)
{
	struct iopic_ih *ioih = (void *)ih;
	struct iopic_softc *sc = device_private(ih->ih_pic_dev);

	printf("iopic status 0x%x\n", iopic_read(sc, IOPIC_STATUS(ih->ih_irq)));
	xicu_stat(ioih->ioih_xih);
#if 0
	while (iopic_read(sc, IOPIC_STATUS(ih->ih_irq)) & IOPIC_STATUS_INTR) {
		KASSERT(ih->ih_func != NULL);
		ih->ih_func(ih->ih_arg);
	}
#endif
}
