/* $NetBSD: $ */

/*-
  * Copyright (c) 2009 UPMC/LIP6
  * All rights reserved.
  * This software is distributed under the following condiions
  * compliant with the NetBSD foundation policy.
  *
  * Redistribution and use in source and binary forms, with or without
  * modification, are permitted provided that the following conditions
  * are met:
  * 1. Redistributions of source code must retain the above copyright
  *    notice, this list of conditions and the following disclaimer.
  * 2. Redistributions in binary form must reproduce the above copyright
  *    notice, this list of conditions and the following disclaimer in the
  *    documentation and/or other materials provided with the distribution.
  *
  * THIS SOFTWARE IS PROVIDED BY THE NETBSD FOUNDATION, INC. AND CONTRIBUTORS
  * ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED
  * TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
  * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE FOUNDATION OR CONTRIBUTORS
  * BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
  * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
  * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
  * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
  * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
  * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
  * POSSIBILITY OF SUCH DAMAGE.
  */

#include <sys/cdefs.h>
__KERNEL_RCSID(0, "$NetBSD: autoconf.c,v 1.71 2007/12/03 15:34:09 ad Exp $");

#include "opt_cputype.h"
#include "opt_soclib.h"
#include "opt_ddb.h"
#include "ksyms.h"
#include "opt_multiprocessor.h"
#include "simhelper.h"

#define __INTR_PRIVATE

#include <sys/param.h>
#include <sys/systm.h>
#include <sys/atomic.h> 
#include <sys/conf.h>
#include <sys/reboot.h>
#include <sys/device.h>
#include <sys/mbuf.h>
#include <sys/msgbuf.h>
#include <sys/cpu.h>
#include <sys/kcore.h>
#include <sys/exec.h>
#include <sys/proc.h>
#include <sys/lwp.h>
#include <sys/ucontext.h>

#include <sys/ras.h>
#include <sys/sysctl.h>
#include <sys/ksyms.h>
#include <sys/bitops.h>
#include <sys/kernel.h>
#include <sys/mount.h>

#include <machine/autoconf.h>
#include <machine/bootinfo.h>
#include <machine/pte.h>
#include <machine/vcache.h>
#include <machine/locore.h>
#include <machine/intr.h>
#include <machine/tsar_firmware.h>

#include <mips/cache.h>
#include <mips/psl.h>
#include <mips/userret.h>
#include <mips/frame.h>

#if NKSYMS || defined(DDB) || defined(LKM)
#include <machine/db_machdep.h>
#include <ddb/db_extern.h>
#endif

#include <uvm/uvm_extern.h>

void tsar_icache_sync_all(void);
void tsar_icache_sync_range(vaddr_t, vsize_t);
void tsar_icache_sync_range_index(vaddr_t, vsize_t);
void tsar_pdcache_wbinv_all(void);
void tsar_pdcache_wbinv_range(vaddr_t, vsize_t);
void tsar_pdcache_wbinv_range_index(vaddr_t, vsize_t);
void tsar_pdcache_inv_range(vaddr_t, vsize_t);
void tsar_pdcache_wb_range(vaddr_t, vsize_t);

struct mips_cache_ops mips_cache_ops = {
	.mco_icache_sync_all = tsar_icache_sync_all,
	.mco_icache_sync_range = tsar_icache_sync_range,
	.mco_icache_sync_range_index = tsar_icache_sync_range_index,
	.mco_pdcache_wbinv_all = tsar_pdcache_wbinv_all,
	.mco_pdcache_wbinv_range = tsar_pdcache_wbinv_range,
	.mco_pdcache_wbinv_range_index = tsar_pdcache_wbinv_range_index,
	.mco_pdcache_inv_range = tsar_pdcache_inv_range,
	.mco_pdcache_wb_range = tsar_pdcache_wb_range
};
	

void mach_init(void);
void lopanic(const char *);
void early_putc(char);
void printx(int);

int physmem;
struct vm_map *mb_map = NULL;
struct vm_map *phys_map = NULL;

paddr_t mips_avail_end; /* PA of last available physical page */
extern char kernel_text[], end[];

struct cluster_info cluster_info[MAXCLUSTERS];
int nclusters = 0;

/* callback filling the memory array */
struct mem_callback_arg {
	vaddr_t vaend;
	paddr_t paend;
	uint32_t fdt_cookie;
};
static int mem_callback(const struct fdt_node *, const struct boot_fdt_prop *,
			void *);
static void register_mem_cluster(paddr_t, paddr_t, int);
static int cluster_callback(const struct fdt_node *,
                            const struct boot_fdt_prop *, void *);
static void find_clusters(struct mem_callback_arg *);

struct cpu_info cpu_info_store
#ifdef MULTIPROCESSOR
	__section(".data1")
	__aligned(1LU << ilog2((2*sizeof(struct cpu_info)-1)))
#endif
    = {
	.ci_curlwp = &lwp0,
#ifndef NOFPU
	.ci_fpcurlwp = &lwp0,
#endif
	.ci_cpl = IPL_HIGH,
#ifdef MULTIPROCESSOR
	.ci_flags = CPUF_PRIMARY|CPUF_PRESENT|CPUF_RUNNING,
#endif
}; /* boot CPU info */

vaddr_t cp0_ebase;

void
lopanic(const char *c)
{
	while (*c) {
		early_putc(*c);
		c++;
	}
	early_putc('\n');
	while (1);
}

void
printx(int d)
{
	int i, v;
	early_putc(' ');
	early_putc('0');
	early_putc('x');
	for (i = 7; i >= 0; i--) {
		v = (d >> (4 * i) & 0xf);
		if (v >= 10)
			early_putc(v - 10 + 'A');
		else
			early_putc(v + '0');
	}
	early_putc(' ');
}

static int
mem_callback(const struct fdt_node * node, const struct boot_fdt_prop *prop,
    void *v)
{
	struct mem_callback_arg *memp = v;
	struct fdt_reg reg;
	int err;
	paddr_t mstart, mend;

	if (strcmp((const char *)&prop->prop_value[0], "memory") != 0)
		return 0;

	prop = fdt_find_prop(node->fdt_node_data, "linux,phandle");
	if (prop == NULL)
		return 0;

	if (prop->prop_value[0] != memp->fdt_cookie)
		return 0;

	err = fdt_read_node_reg(node, &reg, 1);
	if (err)
		return 0;

	mstart = reg.reg_addr;
	mend = reg.reg_addr + reg.reg_size;

#if 0	
	/* if this is our boot memory, register now */
	if (mstart <= (vaddr_t)kernel_text - KERNBASE &&
	    mend >= memp->paend) {
		/* split in 2 segments */
		register_mem_cluster(mstart, (vaddr_t)kernel_text - KERNBASE, 0);
		physmem += ((vaddr_t)kernel_text - KERNBASE - mstart) / PAGE_SIZE;
		register_mem_cluster(memp->paend, mend, nclusters);
		physmem += (mend - memp->paend) / PAGE_SIZE;
	} else {
		register_mem_cluster(mstart, mend, nclusters);
		physmem += reg.reg_size / PAGE_SIZE;
	}
#endif
	/* preserve preloader/fdt/etc */
	if (mstart < memp->paend)
		mstart = memp->paend;
	register_mem_cluster(mstart, mend, nclusters);
	physmem += reg.reg_size / PAGE_SIZE;
	return 0;
}

static void
register_mem_cluster(paddr_t mstart, paddr_t mend, int freelist)
{
        if (mstart >= mend)
                return;
#if 0
        printf("mem_clusters 0x%" PRIx64 " -> 0x%" PRIx64 "\n",
            mstart, mend);
#else
	printf("mem_clusters 0x%08x%08x -> 0x%08x%08x freelist %d\n",
	   (uint)(mstart >> 32), (uint)(mstart & 0xffffffff),
	   (uint)(mend >> 32), (uint)(mend & 0xffffffff), freelist);
#endif

        uvm_page_physload(atop(mstart),
            atop(mend),
            atop((mstart == 0) ? PAGE_SIZE : mstart),
            atop(mend),
            freelist);

        if (mips_avail_end < mips_trunc_page_paddr(mend - 1))
                mips_avail_end = mips_trunc_page_paddr(mend - 1);
}


static int
cluster_callback(const struct fdt_node *node, const struct boot_fdt_prop *prop,
    void *v)
{
	struct mem_callback_arg *memp = v;
	int err;
	int i;
	struct fdt_reg reg;
	const uint32_t *r = (uint32_t *)&reg.reg_addr;

	err = fdt_read_node_reg(node, &reg, 1);
	if (err)
		return 0;

	cluster_info[nclusters].cl_y = r[0];
	cluster_info[nclusters].cl_x = r[1];
	cluster_info[nclusters].cl_node = *node;
	cluster_info[nclusters].cl_uvm_freelist = nclusters;

	for (i = 0;
	    i < be32toh(prop->prop_value_size) / sizeof(uint32_t);
	    i++) {
		memp->fdt_cookie = prop->prop_value[i];
		fdt_walk(&fdt->fdt_root_node, "device_type", mem_callback, memp);
	}
	nclusters++;
	return 0;
}

static void
find_clusters(struct mem_callback_arg *memp)
{
	struct fdt_node topologyn;
	int err;

	/* look for the topology node, clusters are under it */
	err = fdt_find_node(&fdt->fdt_root_node,"topology", &topologyn);
	if (err)
		panic("can't find topology: %d", err);
	/* now parse subnodes of topologyn looking for clusters */
	nclusters = 0;
	fdt_walk(&topologyn, "devices", cluster_callback, memp);
}

void
mach_init()
{
	vaddr_t msgb;
	vaddr_t fdt_va;
	struct btinfo_magic *bi_magic;
	struct btinfo_symtab *bi_sym;
	struct btinfo_fdt *bi_fdt;
	struct mem_callback_arg mem;
	int err;

	extern const struct splsw std_splsw;
	mips_splsw = std_splsw;

	intr_init();
	mttycn_attach();

	/* look up informations from boot loader */
	bi_magic = lookup_bootinfo(BTINFO_MAGIC, bootinfo);
	if (bi_magic && bi_magic->magic == BOOTINFO_MAGIC) {
		bi_sym = lookup_bootinfo(BTINFO_SYMTAB, bootinfo);
		bi_fdt = lookup_bootinfo(BTINFO_FDT, bootinfo);
	} else {
		panic("no bootinfo");
	}
	if (bi_fdt == NULL)
		panic("no FDT provided");
	printf("fdt @0x%lx, size 0x%lx\n", (u_long)bi_fdt->fdt_physaddr,
	    (u_long)bi_fdt->fdt_size);

	mem.vaend = mips_round_page((vaddr_t)end);

#if NKSYM > 0 || defined(DDB) || defined(LKM)
	if (bi_sym)
		mem.vaend = mips_round_page(bi_sym->esym + KERNBASE);
#endif
	mem.paend = mem.vaend - KERNBASE;
	printf("vaend 0x%lx paend 0x%lx\n", (u_long)mem.vaend, (u_long)mem.paend);
	pmap_bootstrap(&mem.vaend, &msgb, &mem.paend,
	    &fdt_va, bi_fdt->fdt_physaddr, bi_fdt->fdt_size);
	printf("vaend now 0x%lx paend 0x%lx\n", (u_long)mem.vaend, (u_long)mem.paend);
	printf("uvm_setpagesize");
	uvm_setpagesize();
#if NKSYMS || defined(DDB) || defined(LKM)
	if (bi_sym) {
		printf(" load symyab");
		ksyms_init(bi_sym->nsym, (char *)(bi_sym->ssym + KERNBASE),
		    (char *)(bi_sym->esym + KERNBASE));
	}
#endif
	printf(" done\n");
	/*
	 * Copy exception-dispatch code down to exception vector.
	 * Initialize locore-function vector.
	 */
	printf("mips_vector_init");
	mips_vector_init(NULL, true);
	printf(" done\n");

	boothowto = /* RB_SINGLE | */ AB_DEBUG | AB_VERBOSE;

#if 0 /* NKSYMS || defined(DDB) || defined(LKM) XXX */
	/* init symbols if present */
	if (esym)
		ksyms_init((char *)esym - (char *)ssym, ssym, esym);
#endif
	
	err = fdt_parse_header((void *)fdt_va, fdt);
	if (err != 0) {
		printf("fdt_parse_header failed: %d\n", err);
		panic("fdt_parse_header");
	}

	find_clusters(&mem);

	/* initialise kernel message buffer */
	printf("initmsgbuf");
	initmsgbuf((void *)msgb, round_page(MSGBUFSIZE));
	printf(" done\n");
}

void *
lookup_bootinfo(int type, char *bootinfop)
{
	struct btinfo_common *bt;
	char *bi;

	bi = bootinfop;
	do {
		bt = (struct btinfo_common *)bi;
		//printf("bi %p %d %d\n", bi, bt->type, bt->next);
		if (bt->type == type)
			return (void *)bi;
		bi += bt->next;
	} while (bt->next != 0 &&
	    (vaddr_t)bi < (vaddr_t)bootinfop + BOOTINFO_SIZE);
	return NULL;
}

void
consinit(void)
{
	mttycn_attach();
}

void
cpu_initclocks()
{
}

void setstatclockrate(int r)
{
}

void
cpu_startup()
{
	vaddr_t minaddr, maxaddr;
	char pbuf[9];
	/*
	 * Allocate a submap for physio
	 */
	phys_map = uvm_km_suballoc(kernel_map, &minaddr, &maxaddr,
	    VM_PHYS_SIZE, 0, false, NULL);

	/*
	 * allocate mbuf cluster submap.
	 */
	mb_map = uvm_km_suballoc(kernel_map, &minaddr, &maxaddr,
	    nmbclusters * mclbytes, VM_MAP_INTRSAFE, false, NULL);

	/* say hello to the world */
	printf("%s%s", copyright, version);

	format_bytes(pbuf, sizeof(pbuf), (psize_t)physmem << PAGE_SHIFT);
	printf("total memory = %s\n", pbuf);
	format_bytes(pbuf, sizeof(pbuf), ptoa(uvmexp.free));
	printf("avail memory = %s\n", pbuf);
}

void
cpu_reboot(howto, bootstr)
        volatile int howto;     /* XXX volatile to keep gcc happy */
	char *bootstr;
{
	static int waittime = -1;
        /* If system is cold, just halt. */
	if (cold) {
		boothowto |= RB_HALT;
	} else {
		if ((boothowto & RB_NOSYNC) == 0 && waittime < 0) {
			waittime = 0;
			vfs_shutdown();
		}
	}
	/* disable interrupts */
	splhigh();
	/* run any shutdown hooks */
	doshutdownhooks();


	/* and finally halt/reboot */
	printf("%s\n\n", ((howto & RB_HALT) != 0) ? "halted." : "rebooting...");
#if NSIMHELPER > 0
	if (howto & RB_POWERDOWN)
		simhelper_end(0);
	else if (howto & RB_HALT)
		simhelper_end(1);
	else
		simhelper_end(2);
#endif
	while (1);
	/*NOTREACHED*/
}


void
tsar_icache_sync_all(void)
{
#ifdef VCACHE
	tsar_vcache_write(VC_ICACHE_FLUSH, 1);
#endif
}

void
tsar_icache_sync_range(vaddr_t va, vsize_t size)
{
#ifdef VCACHE
	tsar_vcache_write(VC_ICACHE_FLUSH, 1); /* XXX could do better */
#endif
}

void
tsar_icache_sync_range_index(vaddr_t va, vsize_t size)
{
#ifdef VCACHE
	tsar_vcache_write(VC_ICACHE_FLUSH, 1); /* XXX could do better */
#endif
}

void
tsar_pdcache_wbinv_all(void)
{
	wbflush();
#ifdef VCACHE
	tsar_vcache_write(VC_DCACHE_FLUSH, 1); /* XXX could do better */
#endif
}

void
tsar_pdcache_wbinv_range(vaddr_t va, vsize_t size)
{
	wbflush();
#ifdef VCACHE
	tsar_vcache_write(VC_DCACHE_FLUSH, 1); /* XXX could do better */
#endif
}

void
tsar_pdcache_wbinv_range_index(vaddr_t va, vsize_t size)
{
	wbflush();
#ifdef VCACHE
	tsar_vcache_write(VC_DCACHE_FLUSH, 1); /* XXX could do better */
#endif
}

void
tsar_pdcache_inv_range(vaddr_t va, vsize_t size)
{
#ifdef VCACHE
	tsar_vcache_write(VC_DCACHE_FLUSH, 1); /* XXX could do better */
#endif
}

void
tsar_pdcache_wb_range(vaddr_t va, vsize_t size)
{
	wbflush();
}

void
tsar_inval_icache_parange(paddr_t pa, size_t sz)
{
#ifdef VCACHE
	paddr_t pend = pa + sz;
	int lnsz = curcpu()->ci_l1_cls;

	pa = pa & ~((paddr_t)lnsz -1);
	while (pa < pend) {
		tsar_vcache_write(VC_DATA_H, pa >> 32);
		tsar_vcache_write(VC_DATA_L, pa & 0xffffffff);
		tsar_vcache_write(VC_ICACHE_INVAL_PA, 0);
		pa += lnsz;
	}
#endif
}

void
tsar_inval_dcache_parange(paddr_t pa, size_t sz)
{
#ifdef VCACHE
	paddr_t pend = pa + sz;
	int lnsz = curcpu()->ci_l1_cls;

	pa = pa & ~((paddr_t)lnsz -1);
	while (pa < pend) {
		tsar_vcache_write(VC_DATA_H, pa >> 32);
		tsar_vcache_write(VC_DATA_L, pa & 0xffffffff);
		tsar_vcache_write(VC_DCACHE_INVAL_PA, 0);
		pa += lnsz;
	}
#endif
}

void
tsar_inval_caches_parange(paddr_t pa, size_t sz)
{
#ifdef VCACHE
	paddr_t pend = pa + sz;
	int lnsz = curcpu()->ci_l1_cls;

	pa = pa & ~((paddr_t)lnsz -1);
	while (pa < pend) {
		tsar_vcache_write(VC_DATA_H, pa >> 32);
		tsar_vcache_write(VC_DATA_L, pa & 0xffffffff);
		tsar_vcache_write(VC_ICACHE_INVAL_PA, 0);
		tsar_vcache_write(VC_DCACHE_INVAL_PA, 0);
		pa += lnsz;
	}
#endif
}

extern struct locoresw mips32_locoresw;

struct locoresw mips_locoresw;

extern const struct splsw std_splsw;
struct splsw mips_splsw;

extern const mips_locore_jumpvec_t mips32_locore_vec;
mips_locore_jumpvec_t mips_locore_jumpvec;

struct mips_options mips_options = {
        .mips_cpu_id = 0xffffffff,
	.mips_fpu_id = 0xffffffff,
};

/* the following is used externally (sysctl_hw) */
char	machine[] = MACHINE;		/* from <machine/param.h> */
char	machine_arch[] = MACHINE_ARCH;	/* from <machine/param.h> */
char	cpu_model[128];

static const struct pridtab tsar_cputab[] = {
	{ 0, 0x00163200, -1, -1, CPU_ARCH_MIPS32, -1,
	  CPU_MIPS_CAUSE_IV | CPU_MIPS_USE_WAIT, 0 , 0,
	  "TSAR mips32"},
};

/*
 * Initialize the hardware exception vectors, and the jump table used to
 * call locore cache and TLB management functions, based on the kind
 * of CPU the kernel is running on.
 */
void
mips_vector_init(const struct splsw *splsw, bool multicpu_p)
{
	struct mips_options * const opts = &mips_options;
	/* r4000 exception handler address and end */
	extern char mips32_exception[];
	/* MIPS32/MIPS64 interrupt exception handler */
	extern char mips32_intr[], mips32_intr_end[];

	/*
	 * XXX Set-up curlwp/curcpu again.  They may have been clobbered
	 * beween verylocore and here.
	 */
	opts->mips_cpu = &tsar_cputab[0];
	opts->mips_cpu_arch = tsar_cputab[0].cpu_isa;
	opts->mips_num_tlb_entries = -1;

	opts->mips_cpu_flags = opts->mips_cpu->cpu_flags;
	opts->mips_has_llsc = (opts->mips_cpu_flags & CPU_MIPS_NO_LLSC) == 0;
	opts->mips3_pg_shift = PAGE_SHIFT;

	cpu_info_store.ci_cpuid = mips_cp0_ebase_read() & 0x1ff;
	lwp0.l_cpu = &cpu_info_store;
	curlwp = &lwp0;

	mips_splsw = std_splsw; /* XXX already done in mach_init */

	mips_locore_atomicvec = mips_llsc_locore_atomicvec;

	mips_locore_jumpvec = mips32_locore_vec;

	mips_locoresw = mips32_locoresw;
	mips_locoresw.lsw_cpu_idle = mips_wait_idle;
	mips_locoresw.lsw_send_ipi = xicu_send_ipi;

	/*
	 * Now initialize our ISA-dependent function vector.
	 */

	printf(" ebase");
	/* write ebase register */
	mips_cp0_ebase_write(cp0_ebase);
	cp0_ebase = mips_cp0_ebase_read() & (~0xfff);
	printf("=0x%x", (uint32_t)cp0_ebase);

	/*
	 * Copy down exception vector code.
	 */

	printf(" memcpy1");
	if (mips32_intr - mips32_exception > 0x80)
		panic("startup: General exception vector code too large");
	memcpy((void *)(cp0_ebase + (MIPS3_GEN_EXC_VEC & PAGE_MASK)),
	    mips32_exception, mips32_intr - mips32_exception);
	printf(" memcpy2");
	if (mips32_intr_end - mips32_intr > 0x80)
		panic("startup: interrupt exception vector code too large");
#if 1	/* XXX - why doesn't mipsNN_intr() work? */
	memcpy((void *)(cp0_ebase + (MIPS3_INTR_EXC_VEC & PAGE_MASK)),
	    mips32_intr, mips32_intr_end - mips32_intr);
#else
	memcpy((void *)(cp0_ebase + (MIPS3_INTR_EXC_VEC & PAGE_MASK)),
	    mips32_exception, mips32_intr - mips32_exception);
#endif
	printf(" mips_icache_sync_all");
	mips_icache_sync_all();
	printf(" mips_dcache_wbinv_all");
	mips_dcache_wbinv_all();

	/* Clear BEV in SR so we start handling our own exceptions */
	printf(" mips_cp0_status_write");
	mips_cp0_status_write(mips_cp0_status_read() & ~MIPS_SR_BEV);
	printf(" done\n");
}

/*
 * Set registers on exec.
 * Clear all registers except sp, pc, and t9.
 * $sp is set to the stack pointer passed in.  $pc is set to the entry
 * point given by the exec_package passed in, as is $t9 (used for PIC
 * code by the MIPS elf abi).
 */
void
setregs(l, pack, stack)
	struct lwp *l;
	struct exec_package *pack;
	u_long stack;
{
	struct trapframe * const tf = l->l_md.md_utf;
	struct pcb * const pcb = lwp_getpcb(l);

	memset(tf, 0, sizeof(struct trapframe));
        tf->tf_regs[_R_SP] = (int)stack;
        tf->tf_regs[_R_PC] = (int)pack->ep_entry & ~3;
        tf->tf_regs[_R_T9] = (int)pack->ep_entry & ~3; /* abicall requirement */
        tf->tf_regs[_R_SR] = PSL_USERSET;


	/*
	 * Set up arguments for _start():
	 *	_start(stack, obj, cleanup, ps_strings);
	 *
	 * Notes:
	 *	- obj and cleanup are the auxiliary and termination
	 *	  vectors.  They are fixed up by ld.elf_so.
	 *	- ps_strings is a NetBSD extension.
	 */
        tf->tf_regs[_R_A0] = (intptr_t)stack;
        tf->tf_regs[_R_A1] = 0;
        tf->tf_regs[_R_A2] = 0;
        tf->tf_regs[_R_A3] = (intptr_t)l->l_proc->p_psstr;

	fpu_discard();
	memset(&pcb->pcb_fpregs, 0, sizeof(struct fpreg));
	l->l_md.md_flags &= ~MDP_FPUSED;
	l->l_md.md_ss_addr = 0;
}

SYSCTL_SETUP(sysctl_machdep_setup, "sysctl machdep subtree setup")
{

	sysctl_createv(clog, 0, NULL, NULL,
		       CTLFLAG_PERMANENT,
		       CTLTYPE_NODE, "machdep", NULL,
		       NULL, 0, NULL, 0,
		       CTL_MACHDEP, CTL_EOL);

	sysctl_createv(clog, 0, NULL, NULL,
		       CTLFLAG_PERMANENT,
		       CTLTYPE_STRUCT, "console_device", NULL,
		       sysctl_consdev, 0, NULL, sizeof(dev_t),
		       CTL_MACHDEP, CPU_CONSDEV, CTL_EOL);
#ifdef __HAVE_BOOTINFO_H
	sysctl_createv(clog, 0, NULL, NULL,
		       CTLFLAG_PERMANENT,
		       CTLTYPE_STRING, "booted_kernel", NULL,
		       sysctl_machdep_booted_kernel, 0, NULL, 0,
		       CTL_MACHDEP, CPU_BOOTED_KERNEL, CTL_EOL);
#endif
	sysctl_createv(clog, 0, NULL, NULL,
		       CTLFLAG_PERMANENT,
		       CTLTYPE_STRING, "root_device", NULL,
		       sysctl_root_device, 0, NULL, 0,
		       CTL_MACHDEP, CPU_ROOT_DEVICE, CTL_EOL);
	sysctl_createv(clog, 0, NULL, NULL,
		       CTLFLAG_PERMANENT|CTLFLAG_IMMEDIATE,
                       CTLTYPE_INT, "llsc", NULL,
                       NULL, MIPS_HAS_LLSC, NULL, 0,
                       CTL_MACHDEP, CPU_LLSC, CTL_EOL);
}

/*
 * These variables are needed by /sbin/savecore.
 */
u_int32_t dumpmag = 0x8fca0101;	/* magic number */
int	dumpsize = 0;		/* pages */
long	dumplo = 0;		/* blocks */

struct user dumppcb;		/* Actually, struct pcb would do. */

/*
 * This is called by main to set dumplo and dumpsize.
 * Dumps always skip the first CLBYTES of disk space
 * in case there might be a disk label stored there.
 * If there is extra space, put dump at the end to
 * reduce the chance that swapping trashes it.
 */
void
cpu_dumpconf(void)
{
	/* XXX notyer */
}

/*
 * Dump the kernel's image to the swap partition.
 */
#define	BYTES_PER_DUMP	PAGE_SIZE

void
dumpsys(void)
{
	/* XXX notyet */
}


/* 
 * Start a new LWP
 */
void
startlwp(arg)
	void *arg;
{
	ucontext_t *uc = arg;
	int err;

	err = cpu_setmcontext(curlwp, &uc->uc_mcontext, uc->uc_flags);
#if DIAGNOSTIC
	if (err) {
		printf("Error %d from cpu_setmcontext.", err);
	}
#endif
	pool_put(&lwp_uc_pool, uc);

	userret(curlwp);
}

/* look up clock node and read freq */
bool
fdt_get_clock_freq(const struct fdt_node *node, uint32_t *freqp)
{
	const struct boot_fdt_prop *clkref;
	const struct boot_fdt_prop *clkprop;
	struct fdt_node *clk_node;

	clkref = fdt_find_prop(node->fdt_node_data, "clocks");
	if (clkref == NULL) {
		printf("clkref fail\n");
		return false;
	}

	clk_node = fdt_find_node_from_ref(clkref->prop_value[0]);
	if (clk_node == NULL) {
		printf("clknode fail 0x%x\n", clkref->prop_value[0]);
		return false;
	}

	clkprop = fdt_find_prop(clk_node->fdt_node_data, "clock-frequency");
	if (clkprop == NULL) {
		printf("clkprop fail\n");
		aprint_error("%s: no clock-frequency property\n",
		    clk_node->fdt_node_name);
		return false;
	}
	*freqp = be32toh(clkprop->prop_value[0]);
	return true;
}
